import { redirect } from '@remix-run/server-runtime';
import { getSessionTokenHeader } from '../../helpers/get-session-token-header.mjs';
import { renderAppBridge } from './render-app-bridge.mjs';
import { redirectWithAppBridgeHeaders } from './redirect-with-app-bridge-headers.mjs';

function redirectFactory(params, request, shop) {
    const { config, logger } = params;
    return function redirect$1(url, init) {
        const { searchParams } = new URL(request.url);
        const { url: parsedUrl, target } = parseURL({
            params,
            url,
            base: config.appUrl,
            shop,
            init,
        });
        logger.debug('Redirecting', { shop, url: parsedUrl.toString() });
        const isSameOrigin = parsedUrl.origin === config.appUrl;
        if (isSameOrigin || url.startsWith('/')) {
            searchParams.forEach((value, key) => {
                if (!parsedUrl.searchParams.has(key)) {
                    parsedUrl.searchParams.set(key, value);
                }
            });
        }
        if (target === '_self') {
            if (isBounceRequest(request)) {
                throw renderAppBridge(params, request, {
                    url: parsedUrl.toString(),
                    target,
                });
            }
            else {
                return redirect(parsedUrl.toString(), init);
            }
        }
        else if (isDataRequest(request)) {
            throw redirectWithAppBridgeHeaders(parsedUrl.toString());
        }
        else if (isEmbeddedRequest(request)) {
            throw renderAppBridge(params, request, {
                url: parsedUrl.toString(),
                target,
            });
        }
        return redirect(url, init);
    };
}
function isBounceRequest(request) {
    return (Boolean(getSessionTokenHeader(request)) &&
        request.headers.has('X-Shopify-Bounce'));
}
function isDataRequest(request) {
    const isGet = request.method === 'GET';
    const sessionTokenHeader = Boolean(getSessionTokenHeader(request));
    return (sessionTokenHeader &&
        !isBounceRequest(request) &&
        (!isEmbeddedRequest(request) || !isGet));
}
function isEmbeddedRequest(request) {
    const { searchParams } = new URL(request.url);
    return searchParams.get('embedded') === '1';
}
function parseURL({ params, base, init, shop, url }) {
    let target = typeof init !== 'number' && init?.target ? init.target : undefined;
    if (isAdminRemotePath(url)) {
        const { config } = params;
        const adminPath = getAdminRemotePath(url);
        const cleanShopName = shop.replace('.myshopify.com', '');
        if (!target) {
            target = config.isEmbeddedApp ? '_parent' : '_self';
        }
        return {
            url: new URL(`https://admin.shopify.com/store/${cleanShopName}${adminPath}`),
            target,
        };
    }
    else {
        return {
            url: new URL(url, base),
            target: target ?? '_self',
        };
    }
}
const ADMIN_REGEX = /^shopify:\/*admin\//i;
function isAdminRemotePath(url) {
    return ADMIN_REGEX.test(url);
}
function getAdminRemotePath(url) {
    const parsedUrl = removeRestrictedParams(new URL(url)).href;
    return parsedUrl.replace(ADMIN_REGEX, '/');
}
const embeddedFrameParamsToRemove = [
    'hmac',
    'locale',
    'protocol',
    'session',
    'id_token',
    'shop',
    'timestamp',
    'host',
    'embedded',
    // sent when clicking rel="home" nav item
    'appLoadId',
];
function removeRestrictedParams(url) {
    const newUrl = new URL(url);
    embeddedFrameParamsToRemove.forEach((param) => newUrl.searchParams.delete(param));
    return newUrl;
}

export { redirectFactory };
//# sourceMappingURL=redirect.mjs.map
