'use strict';

var serverRuntime = require('@remix-run/server-runtime');
var types = require('../../types.js');

function loginFactory(params) {
    const { api, config, logger } = params;
    return async function login(request) {
        const url = new URL(request.url);
        const shopParam = url.searchParams.get('shop');
        if (request.method === 'GET' && !shopParam) {
            return {};
        }
        const shop = shopParam || (await request.formData()).get('shop');
        if (!shop) {
            logger.debug('Missing shop parameter', { shop });
            return { shop: types.LoginErrorType.MissingShop };
        }
        const shopWithoutProtocol = shop
            .replace(/^https?:\/\//, '')
            .replace(/\/$/, '');
        const shopWithDomain = shop?.indexOf('.') === -1
            ? `${shopWithoutProtocol}.myshopify.com`
            : shopWithoutProtocol;
        const sanitizedShop = api.utils.sanitizeShop(shopWithDomain);
        if (!sanitizedShop) {
            logger.debug('Invalid shop parameter', { shop });
            return { shop: types.LoginErrorType.InvalidShop };
        }
        const authPath = `${config.appUrl}${config.auth.path}?shop=${sanitizedShop}`;
        const adminPath = api.utils.legacyUrlToShopAdminUrl(sanitizedShop);
        const installPath = `https://${adminPath}/oauth/install?client_id=${config.apiKey}`;
        const shouldInstall = config.isEmbeddedApp && config.future.unstable_newEmbeddedAuthStrategy;
        const redirectUrl = shouldInstall ? installPath : authPath;
        logger.info(`Redirecting login request to ${redirectUrl}`, {
            shop: sanitizedShop,
        });
        throw serverRuntime.redirect(redirectUrl);
    };
}

exports.loginFactory = loginFactory;
//# sourceMappingURL=login.js.map
