import { graphqlClientClass } from '../clients/admin/graphql/client.mjs';
import '@shopify/admin-api-client';
import '@shopify/network';
import { BillingError } from '../error.mjs';
import '../types.mjs';
import '../../runtime/crypto/crypto.mjs';
import '../../runtime/crypto/types.mjs';
import 'compare-versions';
import { APP_SUBSCRIPTION_FRAGMENT } from './types.mjs';
import { convertLineItems } from './utils.mjs';

function check(config) {
    return async function check(params) {
        if (!config.future?.unstable_managedPricingSupport && !config.billing) {
            throw new BillingError({
                message: 'Attempted to look for purchases without billing configs',
                errorData: [],
            });
        }
        const { session, isTest = true, plans } = params;
        const returnObject = params
            .returnObject ?? false;
        const GraphqlClient = graphqlClientClass({ config });
        const client = new GraphqlClient({ session });
        const payments = await assessPayments({ client, isTest, plans });
        if (config.future?.unstable_managedPricingSupport || returnObject) {
            return payments;
        }
        else {
            return payments.hasActivePayment;
        }
    };
}
async function assessPayments({ client, isTest, plans, }) {
    const returnValue = {
        hasActivePayment: false,
        oneTimePurchases: [],
        appSubscriptions: [],
    };
    let installation;
    let endCursor = null;
    do {
        const currentInstallations = await client.request(HAS_PAYMENTS_QUERY, { variables: { endCursor } });
        installation = currentInstallations.data?.currentAppInstallation;
        installation.activeSubscriptions.forEach((subscription) => {
            if (subscriptionMeetsCriteria({ subscription, isTest, plans })) {
                returnValue.hasActivePayment = true;
                if (subscription.lineItems) {
                    subscription.lineItems = convertLineItems(subscription.lineItems);
                }
                returnValue.appSubscriptions.push(subscription);
            }
        });
        installation.oneTimePurchases.edges.forEach(({ node: purchase }) => {
            if (purchaseMeetsCriteria({ purchase, isTest, plans })) {
                returnValue.hasActivePayment = true;
                returnValue.oneTimePurchases.push(purchase);
            }
        });
        endCursor = installation.oneTimePurchases.pageInfo.endCursor;
    } while (installation?.oneTimePurchases.pageInfo.hasNextPage);
    return returnValue;
}
function subscriptionMeetsCriteria({ subscription, isTest, plans, }) {
    return ((typeof plans === 'undefined' || plans.includes(subscription.name)) &&
        (isTest || !subscription.test));
}
function purchaseMeetsCriteria({ purchase, isTest, plans, }) {
    return ((typeof plans === 'undefined' || plans.includes(purchase.name)) &&
        (isTest || !purchase.test) &&
        purchase.status === 'ACTIVE');
}
const HAS_PAYMENTS_QUERY = `
  ${APP_SUBSCRIPTION_FRAGMENT}
  query appSubscription($endCursor: String) {
    currentAppInstallation {
      activeSubscriptions {
        ...AppSubscriptionFragment
      }
      oneTimePurchases(first: 250, sortKey: CREATED_AT, after: $endCursor) {
        edges {
          node {
            id
            name
            test
            status
          }
        }
        pageInfo {
          hasNextPage
          endCursor
        }
      }
    }
  }
`;

export { assessPayments, check };
//# sourceMappingURL=check.mjs.map
