'use strict';

const plugin = (_, __, { sourcesWithOperations, interfaceExtensionCode }, _info) => {
  const code = getDocumentRegistryChunk(sourcesWithOperations);
  code.push(interfaceExtensionCode);
  return code.join("") + "\n";
};
const GENERATED_QUERY_INTERFACE_NAME = "GeneratedQueryTypes";
const GENERATED_MUTATION_INTERFACE_NAME = "GeneratedMutationTypes";
const isMutationRE = /(^|}\s|\n\s*)mutation[\s({]/im;
const normalizeOperation = (rawSDL, variablesMap) => {
  let variableNotFound = false;
  while (/#REQUIRED_VAR=/.test(rawSDL) && !variableNotFound) {
    let requiredVariables = rawSDL.matchAll(/#REQUIRED_VAR=(\w+)/g);
    for (const [match, variableName] of requiredVariables) {
      if (variablesMap.has(variableName)) {
        rawSDL = rawSDL.replace(match, variablesMap.get(variableName));
      } else {
        variableNotFound = true;
        console.error(
          new Error(
            `Variable "${variableName}" not found. This might be a bug in @shopify/graphql-codegen, please report it.`
          )
        );
      }
    }
  }
  return rawSDL;
};
const buildTypeLines = (name, operations) => {
  const lines = [`interface ${name} {
`];
  for (const [originalString, typeNames] of operations) {
    lines.push(
      `  ${JSON.stringify(originalString)}: {return: ${typeNames.length === 1 ? typeNames[0] : "never"}, variables: ${typeNames.map((n) => n + "Variables").join(" & ")}},
`
    );
  }
  lines.push(`}
`);
  return lines;
};
function getDocumentRegistryChunk(sourcesWithOperations = []) {
  const queries = /* @__PURE__ */ new Map();
  const mutations = /* @__PURE__ */ new Map();
  const variablesMap = /* @__PURE__ */ new Map();
  for (const { source } of sourcesWithOperations) {
    const variableName = source.rawSDL?.match(/#VAR_NAME=(\w+)/)?.[1];
    if (variableName) {
      source.rawSDL = source.rawSDL.replace(/#VAR_NAME=\w+$/, "");
      variablesMap.set(variableName, source.rawSDL);
    }
  }
  for (const { operations, source } of sourcesWithOperations) {
    const actualOperations = operations.filter(
      (op) => op.definition.kind === "OperationDefinition"
    );
    if (actualOperations.length === 0)
      continue;
    const sdlString = source.rawSDL;
    const target = isMutationRE.test(sdlString) ? mutations : queries;
    target.set(
      normalizeOperation(sdlString, variablesMap),
      actualOperations.map((o) => o.initialName)
    );
  }
  return [
    ...buildTypeLines(GENERATED_QUERY_INTERFACE_NAME, queries),
    "\n",
    ...buildTypeLines(GENERATED_MUTATION_INTERFACE_NAME, mutations)
  ];
}

exports.GENERATED_MUTATION_INTERFACE_NAME = GENERATED_MUTATION_INTERFACE_NAME;
exports.GENERATED_QUERY_INTERFACE_NAME = GENERATED_QUERY_INTERFACE_NAME;
exports.plugin = plugin;
//# sourceMappingURL=out.js.map
//# sourceMappingURL=plugin.cjs.map