/**
 * @remix-run/route-config v2.17.0
 *
 * Copyright (c) Remix Software Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE.md file in the root directory of this source tree.
 *
 * @license MIT
 */
'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var node_path = require('node:path');
var pick = require('lodash/pick');
var dev = require('@remix-run/dev');

function _interopDefaultLegacy (e) { return e && typeof e === 'object' && 'default' in e ? e : { 'default': e }; }

var pick__default = /*#__PURE__*/_interopDefaultLegacy(pick);

/**
 * Provides the absolute path to the app directory, for use within `routes.ts`.
 * This is designed to support resolving file system routes.
 */
function getAppDirectory() {
  return dev.UNSAFE_getRouteConfigAppDirectory();
}
const routeOptionKeys = ["id", "index", "caseSensitive"];

/**
 * Helper function for creating a route config entry, for use within
 * `routes.ts`.
 */

function route(path, file, optionsOrChildren, children) {
  let options = {};
  if (Array.isArray(optionsOrChildren) || !optionsOrChildren) {
    children = optionsOrChildren;
  } else {
    options = optionsOrChildren;
  }
  return {
    file,
    children,
    path: path ?? undefined,
    ...pick__default["default"](options, routeOptionKeys)
  };
}
const indexOptionKeys = ["id"];
/**
 * Helper function for creating a route config entry for an index route, for use
 * within `routes.ts`.
 */
function index(file, options) {
  return {
    file,
    index: true,
    ...pick__default["default"](options, indexOptionKeys)
  };
}
const layoutOptionKeys = ["id"];

/**
 * Helper function for creating a route config entry for a layout route, for use
 * within `routes.ts`.
 */

function layout(file, optionsOrChildren, children) {
  let options = {};
  if (Array.isArray(optionsOrChildren) || !optionsOrChildren) {
    children = optionsOrChildren;
  } else {
    options = optionsOrChildren;
  }
  return {
    file,
    children,
    ...pick__default["default"](options, layoutOptionKeys)
  };
}

/**
 * Helper function for adding a path prefix to a set of routes without needing
 * to introduce a parent route file, for use within `routes.ts`.
 */
function prefix(prefixPath, routes) {
  return routes.map(route => {
    if (route.index || typeof route.path === "string") {
      return {
        ...route,
        path: route.path ? joinRoutePaths(prefixPath, route.path) : prefixPath,
        children: route.children
      };
    } else if (route.children) {
      return {
        ...route,
        children: prefix(prefixPath, route.children)
      };
    }
    return route;
  });
}
/**
 * Creates a set of route config helpers that resolve file paths relative to the
 * given directory, for use within `routes.ts`. This is designed to support
 * splitting route config into multiple files within different directories.
 */
function relative(directory) {
  return {
    /**
     * Helper function for creating a route config entry, for use within
     * `routes.ts`. Note that this helper has been scoped, meaning that file
     * path will be resolved relative to the directory provided to the
     * `relative` call that created this helper.
     */
    route: (path, file, ...rest) => {
      return route(path, node_path.resolve(directory, file), ...rest);
    },
    /**
     * Helper function for creating a route config entry for an index route, for
     * use within `routes.ts`. Note that this helper has been scoped, meaning
     * that file path will be resolved relative to the directory provided to the
     * `relative` call that created this helper.
     */
    index: (file, ...rest) => {
      return index(node_path.resolve(directory, file), ...rest);
    },
    /**
     * Helper function for creating a route config entry for a layout route, for
     * use within `routes.ts`. Note that this helper has been scoped, meaning
     * that file path will be resolved relative to the directory provided to the
     * `relative` call that created this helper.
     */
    layout: (file, ...rest) => {
      return layout(node_path.resolve(directory, file), ...rest);
    },
    // Passthrough of helper functions that don't need relative scoping so that
    // a complete API is still provided.
    prefix
  };
}
function joinRoutePaths(path1, path2) {
  return [path1.replace(/\/+$/, ""),
  // Remove trailing slashes
  path2.replace(/^\/+/, "") // Remove leading slashes
  ].join("/");
}

exports.getAppDirectory = getAppDirectory;
exports.index = index;
exports.layout = layout;
exports.prefix = prefix;
exports.relative = relative;
exports.route = route;
