/**
 * Unified WhatsApp Message Preview Utility
 * Supports both Template messages and Service (interactive/media/text) messages
 */

// Shared markdown parser that protects {{variables}} and supports *bold*, _italic_, ~strikethrough~ and line breaks
export function parseMarkdown(text) {
  if (!text) return text;

  const variables = [];
  let protectedText = String(text).replace(/\{\{[^}]+\}\}/g, (match) => {
    const index = variables.length;
    variables.push(match);
    return `XVARX${index}XVARX`;
  });

  protectedText = protectedText
    .replace(/\*([^*]+)\*/g, '<strong>$1</strong>')
    .replace(/_([^_]+)_/g, '<em>$1</em>')
    .replace(/~([^~]+)~/g, '<del>$1</del>')
    .replace(/```([^`]+)```/g, '<code class="monospace">$1</code>')
    .replace(/`([^`]+)`/g, '<code class="inline">$1</code>')
    .replace(/^> (.+)$/gm, '<blockquote>$1</blockquote>')
    .replace(/^\* (.+)$/gm, '<li class="bullet">$1</li>')
    .replace(/^(\d+)\. (.+)$/gm, '<li class="numbered" data-number="$1">$2</li>')
    .replace(/<br\s*\/?>(?=\s*|$)/gi, '\n')
    .replace(/\\r\\n/g, '\n')
    .replace(/\\n/g, '\n')
    .replace(/<(?!\/?(strong|em|del|code|blockquote|li)\b)[^>]*>/g, '')
    .replace(/&nbsp;/g, ' ')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>');

  variables.forEach((variable, index) => {
    protectedText = protectedText.replace(`XVARX${index}XVARX`, variable);
  });

  return protectedText;
}

// Convert any WhatsApp message (template or service) into preview components
export function convertWhatsAppMessageToPreview(input) {
  if (!input) return null;

  // If plain string and not JSON, treat as simple text body
  if (typeof input === 'string' && !input.trim().startsWith('{')) {
    return { type: 'text', body: parseMarkdown(input) };
  }

  let data = input;
  try {
    if (typeof input === 'string') {
      data = JSON.parse(input);
    }
  } catch (e) {
    // Fallback to text parsing if JSON fails
    return { type: 'text', body: parseMarkdown(String(input)) };
  }

  // Handle template data from API response (data.data[0] structure)
  if (data?.data && Array.isArray(data.data) && data.data.length > 0) {
    data = data.data[0];
  }

  // Detect template structure
  const asTemplate = data?.template?.components || data?.components;
  if (asTemplate) {
    const components = {};
    const list = data.template?.components || data.components || [];
    
    // Check if this is a carousel template
    const carouselComponent = list.find(component => component.type === 'CAROUSEL');
    if (carouselComponent) {
      components.type = 'carousel';
      components.carousel = parseCarouselComponent(carouselComponent, data);
      
      // Also parse body text if present (carousel templates have body text)
      const bodyComponent = list.find(component => component.type === 'BODY');
      if (bodyComponent) {
        components.body = parseMarkdown(replaceVariablesWithExamples(bodyComponent.text, bodyComponent.example));
      }
      
      return components;
    }
    
    // Regular template parsing
    list.forEach((component) => {
      switch (component.type) {
        case 'HEADER':
          components.header = parseTemplateHeaderComponent(component);
          break;
        case 'BODY':
          components.body = parseMarkdown(replaceVariablesWithExamples(component.text, component.example));
          break;
        case 'FOOTER':
          components.footer = parseMarkdown(component.text);
          break;
        case 'BUTTONS':
          components.buttons = parseTemplateButtonsComponent(component);
          break;
        default:
          break;
      }
    });
    return components;
  }

  // Otherwise treat as service message
  const messageType = data?.type;
  if (messageType === 'interactive' && data.interactive) {
    return parseInteractiveServiceMessage(data.interactive);
  }
  if (messageType === 'text') {
    return { type: 'text', body: parseMarkdown(data.text?.body || data.body || '') };
  }
  if (messageType === 'image') {
    return { type: 'image', image: { link: data.image?.link || data.image?.url, caption: parseMarkdown(data.image?.caption || '') } };
  }
  if (messageType === 'video') {
    return { type: 'video', video: { link: data.video?.link || data.video?.url, caption: parseMarkdown(data.video?.caption || '') } };
  }
  if (messageType === 'audio') {
    return { type: 'audio', audio: { link: data.audio?.link || data.audio?.url } };
  }
  if (messageType === 'document') {
    return { type: 'document', document: { link: data.document?.link || data.document?.url, filename: data.document?.filename || 'Document', caption: parseMarkdown(data.document?.caption || '') } };
  }
  if (messageType === 'location') {
    return { type: 'location', location: { latitude: data.location?.latitude, longitude: data.location?.longitude, name: data.location?.name || '', address: data.location?.address || '' } };
  }

  // Default fallback
  return { type: 'text', body: parseMarkdown(JSON.stringify(data)) };
}

// Replace template variables with example values
function replaceVariablesWithExamples(text, example) {
  if (!text || !example) return text;
  
  let result = text;
  
  // Handle header text examples
  if (example.header_text && Array.isArray(example.header_text)) {
    example.header_text.forEach((value, index) => {
      const variablePattern = new RegExp(`\\{\\{${index + 1}\\}\\}`, 'g');
      result = result.replace(variablePattern, value);
    });
  }
  
  // Handle body text examples
  if (example.body_text && Array.isArray(example.body_text) && example.body_text.length > 0) {
    const bodyExamples = example.body_text[0]; // First array contains the examples
    if (Array.isArray(bodyExamples)) {
      bodyExamples.forEach((value, index) => {
        const variablePattern = new RegExp(`\\{\\{${index + 1}\\}\\}`, 'g');
        result = result.replace(variablePattern, value);
      });
    }
  }
  
  // Handle URL button examples (for buttons with variables in URL)
  if (example && Array.isArray(example)) {
    example.forEach((value, index) => {
      const variablePattern = new RegExp(`\\{\\{${index + 1}\\}\\}`, 'g');
      result = result.replace(variablePattern, value);
    });
  }
  
  return result;
}

function parseTemplateHeaderComponent(component) {
  if (!component) return null;
  switch (component.format) {
    case 'TEXT':
      return { type: 'text', content: parseMarkdown(replaceVariablesWithExamples(component.text, component.example)) };
    case 'IMAGE':
      return { type: 'image', content: component.example?.header_handle?.[0] || component.url || null };
    case 'VIDEO':
      return { type: 'video', content: component.example?.header_handle?.[0] || component.url || null };
    case 'DOCUMENT':
      return { type: 'document', content: component.example?.header_handle?.[0] || component.url || null, filename: component.filename || 'Document' };
    default:
      return null;
  }
}

function parseTemplateButtonsComponent(component) {
  if (!component?.buttons || !Array.isArray(component.buttons)) return null;
  return component.buttons.map((button) => ({
    type: button.type || 'QUICK_REPLY',
    text: button.text,
    url: button.url ? replaceVariablesWithExamples(button.url, button.example) : null,
    phone_number: button.phone_number || null,
  }));
}

function parseInteractiveServiceMessage(interactive) {
  const components = { type: 'interactive', interactiveType: interactive.type };
  if (interactive.header) {
    components.header = parseServiceHeader(interactive.header);
  }
  if (interactive.body) {
    components.body = parseMarkdown(interactive.body.text);
  }
  if (interactive.footer) {
    components.footer = parseMarkdown(interactive.footer.text);
  }
  if (interactive.action) {
    if (interactive.action.buttons) {
      components.buttons = parseServiceButtons(interactive.action.buttons);
    } else if (interactive.action.sections) {
      components.list = parseServiceList(interactive.action.sections);
    }
  }
  return components;
}

function parseServiceHeader(header) {
  if (!header) return null;
  switch (header.type) {
    case 'text':
      return { type: 'text', content: parseMarkdown(header.text) };
    case 'image':
      return { type: 'image', content: header.image?.link || header.image?.url || null };
    case 'video':
      return { type: 'video', content: header.video?.link || header.video?.url || null };
    case 'document':
      return { type: 'document', content: header.document?.link || header.document?.url || null, filename: header.document?.filename || 'Document' };
    default:
      return null;
  }
}

function parseServiceButtons(buttons) {
  if (!buttons || !Array.isArray(buttons)) return null;
  return buttons.map((button) => ({
    type: button.type === 'reply' ? 'QUICK_REPLY' : button.type?.toUpperCase() || 'QUICK_REPLY',
    text: button.reply?.title || button.text || '',
    id: button.reply?.id || button.id || '',
    url: button.url || null,
    phone_number: button.phone_number || null,
  }));
}

function parseServiceList(sections) {
  if (!sections || !Array.isArray(sections)) return null;
  return sections.map((section) => ({
    title: section.title || '',
    rows: section.rows?.map((row) => ({ id: row.id || '', title: row.title || '', description: row.description || '' })) || [],
  }));
}

// Parse carousel component for WhatsApp carousel templates
function parseCarouselComponent(component, templateData) {
  if (!component?.cards || !Array.isArray(component.cards)) return null;
  
  return component.cards.map((card, index) => {
    const cardData = {
      id: card.card_index || index,
      components: {}
    };
    
    if (card.components && Array.isArray(card.components)) {
      card.components.forEach((cardComponent) => {
        switch (cardComponent.type) {
          case 'HEADER':
            cardData.components.header = parseTemplateHeaderComponent(cardComponent);
            break;
          case 'BODY':
            cardData.components.body = parseMarkdown(
              replaceVariablesWithExamples(
                cardComponent.text || extractTextFromParameters(cardComponent.parameters),
                cardComponent.example
              )
            );
            break;
          case 'BUTTONS':
            cardData.components.buttons = parseTemplateButtonsComponent(cardComponent);
            break;
          default:
            break;
        }
      });
    }
    
    return cardData;
  });
}

// Extract text from template parameters (for carousel cards)
function extractTextFromParameters(parameters) {
  if (!parameters || !Array.isArray(parameters)) return '';
  return parameters
    .filter(param => param.type === 'text')
    .map(param => param.text || '')
    .join(' ');
}

// Extract variables from any WhatsApp message
export function extractVariablesFromWhatsAppMessage(input) {
  if (!input) return [];

  let textToSearch = '';

  if (typeof input === 'string' && input.trim().startsWith('{')) {
    try {
      const data = JSON.parse(input);
      // Template components
      const comps = data?.template?.components || data?.components;
      if (Array.isArray(comps)) {
        comps.forEach((c) => {
          if (c.text) textToSearch += ' ' + c.text;
        });
      }
      // Interactive/service parts
      if (data?.interactive) {
        const it = data.interactive;
        if (it.header?.text) textToSearch += ' ' + it.header.text;
        if (it.body?.text) textToSearch += ' ' + it.body.text;
        if (it.footer?.text) textToSearch += ' ' + it.footer.text;
      }
      if (data?.text) textToSearch += ' ' + (data.text.body || data.text);
      if (data?.image?.caption) textToSearch += ' ' + data.image.caption;
      if (data?.video?.caption) textToSearch += ' ' + data.video.caption;
      if (data?.document?.caption) textToSearch += ' ' + data.document.caption;
    } catch (e) {
      textToSearch = String(input);
    }
  } else {
    textToSearch = String(input);
  }

  const regex = /\{\{([^}]+)\}\}/g;
  const variables = [];
  let match;
  while ((match = regex.exec(textToSearch)) !== null) {
    const variable = match[1].trim();
    if (!variables.includes(variable)) variables.push(variable);
  }
  return variables;
}

// Convert formatted HTML-like string into renderable parts for React
export function getFormattedTextData(text) {
  if (!text) return [];
  const parts = String(text).split(/(<\/?(?:strong|em|del|code|blockquote|li)(?:\s[^>]*)?>)/g);
  const elements = [];
  let current = [];
  parts.forEach((part, index) => {
    if (part === '<strong>') current.push('bold');
    else if (part === '</strong>') current = current.filter((s) => s !== 'bold');
    else if (part === '<em>') current.push('italic');
    else if (part === '</em>') current = current.filter((s) => s !== 'italic');
    else if (part === '<del>') current.push('strikethrough');
    else if (part === '</del>') current = current.filter((s) => s !== 'strikethrough');
    else if (part.startsWith('<code class="monospace">')) current.push('monospace');
    else if (part === '</code>' && current.includes('monospace')) current = current.filter((s) => s !== 'monospace');
    else if (part.startsWith('<code class="inline">')) current.push('inline-code');
    else if (part === '</code>' && current.includes('inline-code')) current = current.filter((s) => s !== 'inline-code');
    else if (part === '<blockquote>') current.push('quote');
    else if (part === '</blockquote>') current = current.filter((s) => s !== 'quote');
    else if (part.startsWith('<li class="bullet">')) current.push('bullet');
    else if (part === '</li>' && current.includes('bullet')) current = current.filter((s) => s !== 'bullet');
    else if (part.startsWith('<li class="numbered"')) current.push('numbered');
    else if (part === '</li>' && current.includes('numbered')) current = current.filter((s) => s !== 'numbered');
    else if (part && !part.startsWith('<')) {
      const textLines = part.split('\n');
      textLines.forEach((line, lineIndex) => {
        const style = {};
        if (current.includes('bold')) style.fontWeight = 'bold';
        if (current.includes('italic')) style.fontStyle = 'italic';
        if (current.includes('strikethrough')) style.textDecoration = 'line-through';
        if (current.includes('monospace')) style.fontFamily = 'monospace';
        if (current.includes('inline-code')) { style.fontFamily = 'monospace'; style.backgroundColor = '#f1f1f1'; style.padding = '2px 4px'; style.borderRadius = '3px'; }
        if (current.includes('quote')) { style.borderLeft = '3px solid #ccc'; style.paddingLeft = '10px'; style.fontStyle = 'italic'; }
        if (current.includes('bullet')) { style.listStyleType = 'disc'; style.marginLeft = '1px'; }
        if (current.includes('numbered')) { style.listStyleType = 'decimal'; style.marginLeft = '1px'; }
        
        elements.push({ 
          text: line, 
          style, 
          key: `${index}-${lineIndex}`, 
          isLineBreak: lineIndex < textLines.length - 1,
          isBullet: current.includes('bullet'),
          isNumbered: current.includes('numbered'),
          isQuote: current.includes('quote')
        });
      });
    }
  });
  return elements;
}
