/**
 * Template API utilities for fetching WhatsApp templates from GetGabs API
 */

import { API_BASE_URL, API_TOKEN, ENDPOINTS, buildApiUrl } from './apiConfig';

/**
 * Fetch templates from GetGabs API
 * @param {Object} options - API request options
 * @param {string} options.templateStatus - Template status filter (All, Approved, Draft, Pending, Disabled)
 * @param {number} options.page - Page number (default: 1)
 * @param {number} options.perPageData - Items per page (default: 20)
 * @returns {Promise<Object>} API response with template data
 */
export const fetchTemplates = async (options = {}) => {
  const {
    templateStatus = 'All',
    page = 1,
    perPageData = 20
  } = options;

  try {
    const url = new URL(buildApiUrl(`${ENDPOINTS.TEMPLATE.BASE}/${ENDPOINTS.TEMPLATE.GET_TEMPLATES}`));
    url.searchParams.append('TemplateStatus', templateStatus);
    url.searchParams.append('page', page.toString());
    url.searchParams.append('perPageData', perPageData.toString());
    url.searchParams.append('token', API_TOKEN);

    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error fetching templates:', error);
    throw error;
  }
};

/**
 * Transform API template data to component format
 * @param {Object} apiTemplate - Raw template data from API
 * @returns {Object} Transformed template object
 */
export const transformTemplateData = (apiTemplate) => {
  try {
    const templateData = JSON.parse(apiTemplate.template_data || '{}');
    const templateInfo = templateData.data?.[0] || {};
    
  
    let templateStructure = {};
    let components = [];
    let language = 'en_US';
    
    if (apiTemplate.draft_json && apiTemplate.draft_json !== 'null') {
      templateStructure = JSON.parse(apiTemplate.draft_json);
      components = templateStructure.components || [];
      language = templateStructure.language || 'en_US';
    } else if (templateInfo.components) {
      components = templateInfo.components || [];
      language = templateInfo.language || 'en_US';
    }

    const category = apiTemplate.category || 'Custom';
    
    const statusMap = {
      'APPROVED': 'Approved',
      'PENDING': 'Pending', 
      'REJECTED': 'Rejected',
      'DISABLED': 'Disabled',
      'DRAFT': 'Draft'
    };
    const status = statusMap[apiTemplate.status] || apiTemplate.status;

    let qualityRating = 'yellow'; 
    if (status === 'Approved') {
      qualityRating = 'green';
    }else if (status === 'Draft' ) {
        qualityRating = 'grey';
      } else if (status === 'Pending' ) {
        qualityRating = 'yellow';
      }
     else if (status === 'Disabled' || status === 'Rejected' ) { 
      qualityRating = 'red';
    } 

    const updatedOn = new Date(apiTemplate.updated_at || apiTemplate.datetime).toISOString().split('T')[0];

    const templateValue = {
      sender: "",
      campaign_id: "put_campaign_id", 
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: apiTemplate.template_name,
        language: {
          code: language
        },
        components: components
      }
    };

    return {
      id: apiTemplate.id ? apiTemplate.id.toString() : 'unknown',
      label: apiTemplate.template_name || 'Unknown Template',
      category: category,
      status: status,
      qualityRating: qualityRating,
      language: getLanguageName(language),
      updatedOn: updatedOn,
      template: JSON.stringify(templateValue),
      template_id: apiTemplate.template_id,
      reason: apiTemplate.reason,
      templateforapi: apiTemplate.templateforapi,
      draft_json: apiTemplate.draft_json,
      template_data: apiTemplate.template_data
    };
  } catch (error) {
    console.error('Error transforming template data:', error);
    return {
      id: apiTemplate.id?.toString() || 'unknown',
      label: apiTemplate.template_name || 'Unknown Template',
      category: 'Marketing',
      status: 'Draft',
      qualityRating: 'red',
      language: 'English',
      updatedOn: new Date().toISOString().split('T')[0],
      template: null
    };
  }
};

/**
 * Get human-readable language name from language code
 * @param {string} languageCode - Language code (e.g., 'en_US')
 * @returns {string} Human-readable language name
 */
const getLanguageName = (languageCode) => {
  const languageMap = {
    'en_US': 'English',
    'es_ES': 'Spanish',
    'fr_FR': 'French',
    'de_DE': 'German',
    'it_IT': 'Italian',
    'pt_BR': 'Portuguese',
    'hi_IN': 'Hindi',
    'ar_AR': 'Arabic',
    'zh_CN': 'Chinese',
    'ja_JP': 'Japanese',
    'ko_KR': 'Korean',
    'ru_RU': 'Russian'
  };
  
  return languageMap[languageCode] || 'English';
};

/**
 * Handle API errors gracefully
 * @param {Error} error - The error object
 * @returns {Object} Error information object
 */
export const handleApiError = (error) => {
  console.error('Template API Error:', error);
  
  if (error.name === 'TypeError' && error.message.includes('fetch')) {
    return {
      message: 'Network error. Please check your internet connection.',
      type: 'network'
    };
  }
  
  if (error.message.includes('401')) {
    return {
      message: 'Authentication failed. Please check your API token.',
      type: 'auth'
    };
  }
  
  if (error.message.includes('403')) {
    return {
      message: 'Access denied. You do not have permission to access templates.',
      type: 'permission'
    };
  }
  
  if (error.message.includes('404')) {
    return {
      message: 'Templates not found. The API endpoint may have changed.',
      type: 'notfound'
    };
  }
  
  if (error.message.includes('500')) {
    return {
      message: 'Server error. Please try again later.',
      type: 'server'
    };
  }
  
  return {
    message: 'An unexpected error occurred while fetching templates.',
    type: 'unknown'
  };
};

/**
 * Get template statistics from API response
 * @param {Object} apiResponse - Full API response
 * @returns {Object} Template statistics
 */
export const getTemplateStats = (apiResponse) => {
  const { counts, datas } = apiResponse;
  
  return {
    totalTemplates: datas?.total || 0,
    currentPage: datas?.current_page || 1,
    lastPage: datas?.last_page || 1,
    perPage: parseInt(datas?.per_page) || 20,
    from: datas?.from || 0,
    to: datas?.to || 0,
    hasNextPage: !!datas?.next_page_url,
    hasPrevPage: !!datas?.prev_page_url,
    nextPageUrl: datas?.next_page_url,
    prevPageUrl: datas?.prev_page_url
  };
};

/**
 * Extract template content for preview
 * @param {Object} template - Template object
 * @returns {string|null} Template content for preview
 */
export const extractTemplateContent = (template) => {
  try {
    if (template.template && typeof template.template === 'string') {
      return template.template;
    }
    
    if (template.draft_json) {
      const draftJson = JSON.parse(template.draft_json);
      const templateValue = {
        sender: "",
        campaign_id: "put_campaign_id",
        messaging_product: "whatsapp", 
        recipient_type: "individual",
        to: "receiver_number",
        type: "template",
        template: {
          name: template.label || 'template',
          language: {
            code: draftJson.language || 'en_US'
          },
          components: draftJson.components || []
        }
      };
      return JSON.stringify(templateValue);
    }
    
    return null;
  } catch (error) {
    console.error('Error extracting template content:', error);
    return null;
  }
};

/** 
 * getall template list for template sellection in create broadcast page and create catalog template page
 * @returns {Object} Template list
 */
export const getAllTemplateList = async () => {
  const response = await fetch('https://app.getgabs.com/partners/broadcast/campaignmanage/gettemplatelists?template_name=template_name', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${API_TOKEN}`
    }
  });
  if (!response.ok) throw new Error('Failed to fetch template list');
  const data = await response.json();
  console.log( "template list", data);
  return data;
};
export const getAllTemplateListForAbandonedCart = async () => {
  const response = await fetch('https://app.getgabs.com/partners/broadcast/campaignmanage/gettemplatelists?template_name=template_name&template_type=2', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${API_TOKEN}`
    }
  });
  if (!response.ok) throw new Error('Failed to fetch template list');
  const data = await response.json();
  console.log( "template list", data);
  return data;
};
export const getAllTemplateListForOrderConfirmation = async () => {
  const response = await fetch('https://app.getgabs.com/partners/broadcast/campaignmanage/gettemplatelists?template_name=template_name&template_type=1', {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${API_TOKEN}`
    }
  });
  if (!response.ok) throw new Error('Failed to fetch template list');
  const data = await response.json();
  console.log( "template list", data);
  return data;
};
/** 
 * get selected template data by template name for chat page and create broadcast page
 * @returns {Object} Template list
 */
export const getSelectedTemplatebyName = async (template_name) => { 
  const response = await fetch(`https://app.getgabs.com/partners/getmessagetemplateapi?template_name=${template_name}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${API_TOKEN}`
    }
  });
  if (!response.ok) throw new Error('Failed to fetch template list');
  const data = await response.json();
  console.log( "template list", data);
  return data;
};
/** 
 * to save temporary file to saveon getgabs
 * @returns {Object} Template list
 */
export const saveTemporaryFile = async (filepayload) => { 
  const response = await fetch(`https://app.getgabs.com/partners/broadcast/campaignmanage/saveFileTemporarily`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${API_TOKEN}`
    },
    body: JSON.stringify(filepayload)
  });
  if (!response.ok) throw new Error('Failed to save temporary file');
  const data = await response.json();
  console.log( "save temporary file", data);
  return data;
};
export const Checkisteplatenameavailable = async (templatename) => { 
  const response = await fetch(`https://app.getgabs.com/partners/template/chechtemplatename?templatename=${templatename}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${API_TOKEN}`
    }
  });
  if (!response.ok) throw new Error('Failed to check template name');
  const data = await response.json();
  console.log( "check template name", data);
  return data;
};