// Note: This file contains server-only functions
// Client-side code should not import this file directly

import { 
  SUBSCRIPTION_PLANS,
  isSubscriptionActive,
  getSubscriptionPlan,
  formatSubscriptionForDisplay 
} from "./subscriptionConstants";

/**
 * Shopify App Subscription Utility (Server-only)
 * Handles subscription creation, management, and billing for the WhatsApp app
 */

// GraphQL mutations for subscription management
const CREATE_SUBSCRIPTION_MUTATION = `
  mutation appSubscriptionCreate($name: String!, $lineItems: [AppSubscriptionLineItemInput!]!, $test: Boolean, $trialDays: Int, $returnUrl: URL!) {
    appSubscriptionCreate(name: $name, lineItems: $lineItems, test: $test, trialDays: $trialDays, returnUrl: $returnUrl) {
      appSubscription {
        id
        name
        status
        createdAt
        trialDays
        currentPeriodEnd
        lineItems {
          id
          plan {
            pricingDetails {
              ... on AppRecurringPricing {
                price {
                  amount
                  currencyCode
                }
                interval
              }
            }
          }
        }
      }
      confirmationUrl
      userErrors {
        field
        message
      }
    }
  }
`;

const CANCEL_SUBSCRIPTION_MUTATION = `
  mutation appSubscriptionCancel($id: ID!) {
    appSubscriptionCancel(id: $id) {
      appSubscription {
        id
        status
      }
      userErrors {
        field
        message
      }
    }
  }
`;

const GET_SUBSCRIPTION_QUERY = `
  query {
    currentAppInstallation {
      activeSubscriptions {
        id
        name
        status
        createdAt
        currentPeriodEnd
        trialDays
        lineItems {
          id
          plan {
            pricingDetails {
              ... on AppRecurringPricing {
                price {
                  amount
                  currencyCode
                }
                interval
              }
            }
          }
        }
      }
    }
  }
`;

/**
 * Create a new subscription for the given plan
 */
export async function createSubscription(request, planKey, isTest = false) {
  try {
    const { authenticate } = await import("../shopify.server");
    const { admin } = await authenticate.admin(request);
    const plan = SUBSCRIPTION_PLANS[planKey];
    
    if (!plan) {
      throw new Error(`Invalid plan: ${planKey}`);
    }

    const variables = {
      name: plan.name,
      test: isTest,
      trialDays: plan.trialDays,
      returnUrl: `${process.env.SHOPIFY_APP_URL}/app/pricing-plan?subscription=success`,
      lineItems: [
        {
          plan: {
            appRecurringPricingDetails: {
              price: {
                amount: plan.price,
                currencyCode: "USD"
              },
              interval: plan.interval
            }
          }
        }
      ]
    };

    const response = await admin.graphql(CREATE_SUBSCRIPTION_MUTATION, {
      variables
    });

    const result = await response.json();
    
    if (result.data?.appSubscriptionCreate?.userErrors?.length > 0) {
      throw new Error(result.data.appSubscriptionCreate.userErrors[0].message);
    }

    return {
      success: true,
      subscription: result.data?.appSubscriptionCreate?.appSubscription,
      confirmationUrl: result.data?.appSubscriptionCreate?.confirmationUrl
    };

  } catch (error) {
    console.error("Error creating subscription:", error);
    return {
      success: false,
      error: error.message
    };
  }
}

/**
 * Get current active subscription
 */
export async function getCurrentSubscription(request) {
  try {
    const { authenticate } = await import("../shopify.server");
    const { admin } = await authenticate.admin(request);
    
    const response = await admin.graphql(GET_SUBSCRIPTION_QUERY);
    const result = await response.json();
    
    const activeSubscriptions = result.data?.currentAppInstallation?.activeSubscriptions || [];
    
    return {
      success: true,
      subscription: activeSubscriptions[0] || null,
      hasActiveSubscription: activeSubscriptions.length > 0
    };

  } catch (error) {
    console.error("Error fetching subscription:", error);
    return {
      success: false,
      error: error.message,
      subscription: null,
      hasActiveSubscription: false
    };
  }
}

/**
 * Cancel current subscription
 */
export async function cancelSubscription(request, subscriptionId) {
  try {
    const { authenticate } = await import("../shopify.server");
    const { admin } = await authenticate.admin(request);
    
    const variables = {
      id: subscriptionId
    };

    const response = await admin.graphql(CANCEL_SUBSCRIPTION_MUTATION, {
      variables
    });

    const result = await response.json();
    
    if (result.data?.appSubscriptionCancel?.userErrors?.length > 0) {
      throw new Error(result.data.appSubscriptionCancel.userErrors[0].message);
    }

    return {
      success: true,
      subscription: result.data?.appSubscriptionCancel?.appSubscription
    };

  } catch (error) {
    console.error("Error canceling subscription:", error);
    return {
      success: false,
      error: error.message
    };
  }
}

// Functions isSubscriptionActive, getSubscriptionPlan, and formatSubscriptionForDisplay
// are now imported from subscriptionConstants.js to avoid client/server import issues
