/**
 * WhatsApp Message Templates
 * 
 * This file contains predefined WhatsApp message templates that can be reused
 * across the application for various automated messaging scenarios.
 * Now includes API integration for fetching real templates from GetGabs API.
 */

import { fetchTemplates, transformTemplateData, handleApiError } from './templateApi';

export const messageTemplateOptions = [
  
  { id: 'JDJD723KDKD1',
    label: 'Diwali Offer Template', 
    category: 'Marketing',
    value: JSON.stringify({
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: 'pandit_diwali_offer',
        language: {
          code: "en_US"
        },
        components: [
          {
            type: 'HEADER',
            format: 'IMAGE',
            example: {
              header_handle: [
                'https://scontent.whatsapp.net/v/t61.29466-34/516934587_1244266223766536_2941614163967020774_n.jpg?ccb=1-7&_nc_sid=8b1bef&_nc_ohc=tKid-3GgrswQ7kNvwHo84gW&_nc_oc=AdnphcBXJRz-TbiJtAGzDpS9t_UOdtFRT7cwRkT2zwXz8S4AP8R14BT5AzskHhHm0Ks&_nc_zt=3&_nc_ht=scontent.whatsapp.net&edm=AH51TzQEAAAA&_nc_gid=vrOUkqB-cMtr5uhuY6dcGA&oh=01_Q5Aa2gHZYGa3Bvw7r5MEu3IvlVy6SsiPCOLVGcxkQhOVFr8x-w&oe=68EF2357',
              ],
            },
          },
          {
            type: 'BODY',
            text: "*Diwali Offer on Recharge on 99Pandit*<br><br>Namskar {{customer_name}}<br><br>We have launched an Offer on *99Pandit Wallet Recharge*.<br><br>*SAVE GST TAX of 18%* <br><br>Also *Get upto 48% Extra Wallet Points* on all Recharge till 31st Oct. <br><br>_Don't Miss this opportunity._ <br><br>Please send screen shot of payment on this number or to 99pandit.",
          },
          {
            type: 'FOOTER',
            text: 'reply STOP to unsubscribe',
          },
          {
            type: 'BUTTONS',
            buttons: [
              {
                type: 'QUICK_REPLY',
                text: 'Interesting Diwali Offer',
              },
            ],
          },
        ]
      }
    })
  },
  { 
    id: 'JDJD723KDK12',
    label: 'Abandoned Cart Recovery', 
    category: 'Marketing',
    value: JSON.stringify({
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: "abandoned_cart_recovery",
        language: {
          code: "en_US"
        },
        components: [
          {
            type: "BODY",
            text: "Hi {{1}}! 👋 You left some amazing items in your cart. Complete your purchase now and get 10% off with code SAVE10!",
            example: {
              body_text: [
                [
                  "customer name"
                ]
              ]
            }
          },
          {
            type: "BUTTONS",
            buttons: [
              {
                type: "QUICK_REPLY",
                text: "Complete Purchase"
              }
            ]
          }
        ]
      }
    })
  },
  { 
    id: 'JDJD723KDKH23',
    label: 'Order Confirmation', 
    category: 'Marketing',
    value: JSON.stringify({
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: "order_confirmation",
        language: {
          code: "en_US"
        },
        components: [
          {
            type: "HEADER",
            format: "TEXT",
            text: "🎉 Order Confirmed"
          },
          {
            type: "BODY",
            text: "Thank you {{1}}! Your order #{{2}} has been confirmed. We'll send you tracking details soon!",
            example: {
              body_text: [
                [
                  "customer name",
                  "order number"
                ]
              ]
            }
          },
          {
            type: "FOOTER",
            text: "Questions? Reply to this message"
          }
        ]
      }
    })
  },
  { 
    id: 'JDJD723KDKH234',
    label: 'Welcome New Customer', 
    category: 'Utility',
    value: JSON.stringify({
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: "welcome_new_customer",
        language: {
          code: "en_US"
        },
        components: [
          {
            type: "HEADER",
            format: "TEXT",
            text: "🌟 Welcome!"
          },
          {
            type: "BODY",
            text: "Welcome to our store {{1}}! Thanks for joining us. Enjoy 15% off your first order with code WELCOME15!",
            example: {
              body_text: [
                [
                  "customer name"
                ]
              ]
            }
          },
          {
            type: "BUTTONS",
            buttons: [
              {
                type: "QUICK_REPLY",
                text: "Shop Now"
              },
              {
                type: "QUICK_REPLY",
                text: "View Offers"
              }
            ]
          }
        ]
      }
    })
  },
  { 
    id: 'JDJD723KDKH2345',
    label: 'Support Auto Reply', 
    category: 'Utility',
    value: JSON.stringify({
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: "support_auto_reply",
        language: {
          code: "en_US"
        },
        components: [
          {
            type: "BODY",
            text: "Hi {{1}}! Thanks for reaching out. Our support team will get back to you within 24 hours. For urgent matters, call us at +1-800-SUPPORT.",
            example: {
              body_text: [
                [
                  "customer name"
                ]
              ]
            }
          },
          {
            type: "FOOTER",
            text: "We're here to help!"
          }
        ]
      }
    })
  },
  { 
    id: 'JDJD723KDKH23456',
    label: 'Shipping Notification', 
    category: 'Utility',
    value: JSON.stringify({
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: "shipping_notification",
        language: {
          code: "en_US"
        },
        components: [
          {
            type: "HEADER",
            format: "TEXT",
            text: "📦 Package Update"
          },
          {
            type: "BODY",
            text: "Great news {{1}}! Your order #{{2}} is on its way. Track your package: {{3}}",
            example: {
              body_text: [
                [
                  "customer name",
                  "order number",
                  "tracking link"
                ]
              ]
            }
          },
          {
            type: "BUTTONS",
            buttons: [
              {
                type: "URL",
                text: "Track Package",
                url: "{{3}}"
              }
            ]
          }
        ]
      }
    })
  },
  { 
    id: 'JDJD723KDKH234567',
    label: 'Product Launch with Video', 
    category: 'Marketing',
    value: JSON.stringify({
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: "product_launch_video",
        language: {
          code: "en_US"
        },
        components: [
          {
            type: "HEADER",
            format: "VIDEO",
            example: {
              header_handle: [
                'https://sample-videos.com/zip/10/mp4/SampleVideo_1280x720_1mb.mp4'
              ]
            }
          },
          {
            type: "BODY",
            text: "🎉 Introducing our latest product: {{product_name}}! Watch the video above to see it in action. Get 20% off with code LAUNCH20!",
            example: {
              body_text: [
                [
                  "Amazing Widget Pro"
                ]
              ]
            }
          },
          {
            type: "FOOTER",
            text: "Limited time offer - ends soon!"
          },
          {
            type: "BUTTONS",
            buttons: [
              {
                type: "QUICK_REPLY",
                text: "Shop Now"
              },
              {
                type: "QUICK_REPLY",
                text: "Learn More"
              }
            ]
          }
        ]
      }
    })
  },
];

/**
 * Get a specific template by its label
 * @param {string} label - The label of the template to retrieve
 * @returns {Object|null} The template object or null if not found
 */
export const getTemplateByLabel = (label) => {
  return messageTemplateOptions.find(template => template.label === label) || null;
};

/**
 * Get a specific template by its ID
 * @param {string} id - The ID of the template to retrieve
 * @returns {Object|null} The template object or null if not found
 */
export const getTemplateById = (id) => {
  return messageTemplateOptions.find(template => template.id === id) || null;
};

/**
 * Get all template labels
 * @returns {string[]} Array of all template labels
 */
export const getTemplateLabels = () => {
  return messageTemplateOptions.map(template => template.label);
};

/**
 * Get templates filtered by a specific criteria
 * @param {Function} filterFn - Function to filter templates
 * @returns {Object[]} Array of filtered templates
 */
export const getFilteredTemplates = (filterFn) => {
  return messageTemplateOptions.filter(filterFn);
};

/**
 * Fetch templates from API and transform them to component format
 * @param {Object} options - API request options
 * @param {string} options.templateStatus - Template status filter
 * @param {number} options.page - Page number
 * @param {number} options.perPageData - Items per page
 * @returns {Promise<Object>} Object containing templates array and pagination info
 */
export const fetchApiTemplates = async (options = {}) => {
  try {
    const apiResponse = await fetchTemplates(options);
    
    if (!apiResponse.datas || !apiResponse.datas.data) {
      throw new Error('Invalid API response format');
    }

    // Transform each template from API format to component format
    const transformedTemplates = apiResponse.datas.data.map(transformTemplateData);
    
    return {
      templates: transformedTemplates,
      pagination: {
        currentPage: apiResponse.datas.current_page,
        lastPage: apiResponse.datas.last_page,
        total: apiResponse.datas.total,
        perPage: parseInt(apiResponse.datas.per_page),
        from: apiResponse.datas.from,
        to: apiResponse.datas.to,
        hasNextPage: !!apiResponse.datas.next_page_url,
        hasPrevPage: !!apiResponse.datas.prev_page_url,
        nextPageUrl: apiResponse.datas.next_page_url,
        prevPageUrl: apiResponse.datas.prev_page_url
      },
      counts: apiResponse.counts
    };
  } catch (error) {
    const errorInfo = handleApiError(error);
    throw new Error(errorInfo.message);
  }
};

/**
 * Get template options for dropdowns (combines static and API templates)
 * @param {Object[]} apiTemplates - Templates fetched from API
 * @returns {Object[]} Combined template options
 */
export const getCombinedTemplateOptions = (apiTemplates = []) => {
  // Convert API templates to the same format as messageTemplateOptions
  const apiTemplateOptions = apiTemplates.map(template => ({
    id: template.id,
    label: template.label,
    category: template.category,
    value: template.template,
    status: template.status,
    qualityRating: template.qualityRating
  }));

  // For dropdown usage, prioritize API templates over static ones
  // Return API templates first, then static templates that don't conflict
  const combined = [...apiTemplateOptions];
  
  messageTemplateOptions.forEach(staticTemplate => {
    const existingIndex = combined.findIndex(t => t.label === staticTemplate.label);
    if (existingIndex < 0) {
      combined.push(staticTemplate);
    }
  });

  return combined;
};

/**
 * Search templates by name or category
 * @param {Object[]} templates - Array of templates to search
 * @param {string} searchTerm - Search term
 * @returns {Object[]} Filtered templates
 */
export const searchTemplates = (templates, searchTerm) => {
  if (!searchTerm || searchTerm.trim() === '') {
    return templates;
  }

  const term = searchTerm.toLowerCase().trim();
  return templates.filter(template => 
    template.label.toLowerCase().includes(term) ||
    template.category.toLowerCase().includes(term) ||
    (template.status && template.status.toLowerCase().includes(term))
  );
};

/**
 * Filter templates by status
 * @param {Object[]} templates - Array of templates to filter
 * @param {string} status - Status to filter by (All, Approved, Draft, Pending, Disabled)
 * @returns {Object[]} Filtered templates
 */
export const filterTemplatesByStatus = (templates, status) => {
  if (!status || status === 'All') {
    return templates;
  }

  return templates.filter(template => 
    template.status && template.status.toLowerCase() === status.toLowerCase()
  );
};

/**
 * Sort templates by specified criteria
 * @param {Object[]} templates - Array of templates to sort
 * @param {string} sortBy - Sort criteria (name, status, category, updated)
 * @param {string} sortDirection - Sort direction (asc, desc)
 * @returns {Object[]} Sorted templates
 */
export const sortTemplates = (templates, sortBy = 'name', sortDirection = 'asc') => {
  const sorted = [...templates];
  
  sorted.sort((a, b) => {
    let aValue, bValue;
    
    switch (sortBy) {
      case 'name':
      case 'label':
        aValue = a.label?.toLowerCase() || '';
        bValue = b.label?.toLowerCase() || '';
        break;
      case 'status':
        aValue = a.status?.toLowerCase() || '';
        bValue = b.status?.toLowerCase() || '';
        break;
      case 'category':
        aValue = a.category?.toLowerCase() || '';
        bValue = b.category?.toLowerCase() || '';
        break;
      case 'updated':
      case 'updatedOn':
        aValue = new Date(a.updatedOn || 0);
        bValue = new Date(b.updatedOn || 0);
        break;
      case 'quality':
      case 'qualityRating':
        const qualityOrder = { 'green': 3, 'yellow': 2, 'red': 1, 'gray': 0 };
        aValue = qualityOrder[a.qualityRating?.toLowerCase()] || 0;
        bValue = qualityOrder[b.qualityRating?.toLowerCase()] || 0;
        break;
      default:
        return 0;
    }
    
    if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
    if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
    return 0;
  });
  
  return sorted;
};
