/**
 * Template API utilities for fetching WhatsApp templates from GetGabs API
 */

import { API_BASE_URL, API_TOKEN, ENDPOINTS, buildApiUrl } from './apiConfig';

/**
 * Fetch templates from GetGabs API
 * @param {Object} options - API request options
 * @param {string} options.templateStatus - Template status filter (All, Approved, Draft, Pending, Disabled)
 * @param {number} options.page - Page number (default: 1)
 * @param {number} options.perPageData - Items per page (default: 20)
 * @returns {Promise<Object>} API response with template data
 */
export const fetchStaticContactList = async (options = {}) => {
  const {
    page = 1,
    perPageData = 20,
    actionType= 0
  } = options;

  try {
    const url = new URL(buildApiUrl(`${ENDPOINTS.STATIC_CONTACT_LIST.BASE}/${ENDPOINTS.STATIC_CONTACT_LIST.GET_STATIC_CONTACT_LIST}`));
    url.searchParams.append('actionType', actionType);
    url.searchParams.append('page', page.toString());
    url.searchParams.append('perPageData', perPageData.toString());
    url.searchParams.append('token', API_TOKEN);

    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error fetching templates:', error);
    throw error;
  }
};
export const fetchStaticContactListContacts = async (options = {}) => {
  const {
    page = 1,
    perPageData = 20,
    actionType= 1,
    contactListId= 0
  } = options;

  try {
    const url = new URL(buildApiUrl(`${ENDPOINTS.STATIC_CONTACT_LIST.BASE}/${ENDPOINTS.STATIC_CONTACT_LIST_CONTACTS.GET_STATIC_CONTACT_LIST_CONTACTS}`));
    url.searchParams.append('actionType', actionType);
    url.searchParams.append('page', page.toString());
    url.searchParams.append('perPageData', perPageData.toString());
    url.searchParams.append('tagId', contactListId.toString());
    url.searchParams.append('token', API_TOKEN);

    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error fetching templates:', error);
    throw error;
  }
};



/** 
* Add new contact list name to GetGabs, then upload CSV if created successfully
* @param {Object} contactListNamePAYLOAD - Payload for creating new contact list
* @param {File} csvFile - CSV file to upload
* @param {Object} columnMappings - Column mappings for CSV columns
* @returns {Object} Combined response
*/
export const addNewContactListName = async (contactListNamePAYLOAD, csvFile, columnMappings = {}) => {
 try {
   const response = await fetch(`https://app.getgabs.com/partners/broadcast/addnewcontacttag`, {
     method: 'POST',
     headers: {
       'Content-Type': 'application/json',
       'Authorization': `Bearer ${API_TOKEN}`,
      'X-Requested-With': 'XMLHttpRequest',
     },
     body: JSON.stringify(contactListNamePAYLOAD),
   });

   if (!response.ok) throw new Error('Failed to add new contact list name');
   const data = await response.json();
   
 console.log('Contact list created successfully:', data);
 if (data?.status === true || data?.success === true || data?.status === 'success' || typeof data?.tagId !== 'undefined' || typeof data?.tag !== 'undefined' || typeof data?.id !== 'undefined') {
   const tagId = data?.tagId ?? data?.tag ?? data?.id;
   const formData = new FormData();
   formData.append('file', csvFile);
   formData.append('tag', tagId);
   
  
   if (Object.keys(columnMappings).length > 0) {
     const mappingsArray = Object.keys(columnMappings)
       .sort((a, b) => parseInt(a) - parseInt(b)) 
       .map(columnIndex => columnMappings[columnIndex]);
     formData.append('columnMappings', JSON.stringify(mappingsArray));
   }

  console.log('create contactlist Formdata contents:');
  for (let [key, value] of formData.entries()) {
    console.log(`${key}:`, value);
  }

    const uploadResponse = await fetch(`https://app.getgabs.com/partners/broadcast/uploadbulkcontact`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${API_TOKEN}`,
         
        
      },
      body: formData,
    });

    if (!uploadResponse.ok) throw new Error('CSV upload failed');
    const uploadData = await uploadResponse.json();
    console.log(' CSV upload response:', uploadData);

    return {
      contactListResponse: data,
      csvUploadResponse: uploadData,
    };
 } else {
   throw new Error('Contact list creation failed');
 }
} catch (error) {
  console.error(' Error:', error);
  throw error;
 }
};


export const addNewVariable = async (variableNamePAYLOAD) => {
 try {
   const response = await fetch(`https://app.getgabs.com/partners/shopify/CustomerLabel/create`, {
     method: 'POST',
     headers: {
       'Content-Type': 'application/json',
       'Authorization': `Bearer ${API_TOKEN}`,
      'X-Requested-With': 'XMLHttpRequest',
     },
     body: JSON.stringify(variableNamePAYLOAD),
   });

   if (!response.ok) throw new Error('Failed to add new variable');
   const data = await response.json();
   console.log('Variable created successfully:', data);
   return data;
 } catch (error) {
   console.error(' Error:', error);
   throw error;
 }
};


/** 
 * to delete contactlist
 * @returns {Object} Contact list
 */
export const deleteContactList = async (contactListId) => { 
  try {
    const url = `https://app.getgabs.com/partners/broadcast/delete/taglist?ids=${encodeURIComponent(contactListId)}`;
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`,
        'X-Requested-With': 'XMLHttpRequest',
      }
    });
    const maybeJson = await response.json().catch(() => null);

    if (!response.ok) {
      const message = (maybeJson && (maybeJson.message || maybeJson.error)) || `HTTP ${response.status}`;
      throw new Error(message);
    }

    const data = maybeJson ?? {};
    const success = data?.status === true || data?.success === true || data?.status === 'success';
    if (!success) {
      throw new Error(data?.message || 'Delete failed');
    }

    console.log('delete contact list', data);
    return data;
  } catch (error) {
    console.error('Error deleting contact list:', error);
    throw error;
  }
};

/** 
 * add single contacts under contact list
 * @returns {Object} Contact
 */
export const addSingleContact = async (singleContactPayload) => { 
  try {
    const url = `https://app.getgabs.com/partners/broadcast/uploadsinglecontact`;
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`,
        'X-Requested-With': 'XMLHttpRequest',
      },
      body: JSON.stringify(singleContactPayload)
    });
    const maybeJson = await response.json().catch(() => null);

    if (!response.ok) {
      const message = (maybeJson && (maybeJson.message || maybeJson.error)) || `HTTP ${response.status}`;
      throw new Error(message);
    }

    const data = maybeJson ?? {};
    const success = data?.status === true || data?.success === true || data?.status === 'success';
    if (!success) {
      throw new Error(data?.message || 'Delete failed');
    }

    console.log('export contact list', data);
    return data;
  } catch (error) {
    console.error('Error exporting contact list:', error);
    throw error;
  }
};
/** 
 * update single contacts under contact list
 * @returns {Object} Contact
 */
export const updateSingleContact = async (updateSingleContactPayload) => { 
  try {
    const url = `https://app.getgabs.com/partners/broadcast/updatesinglecontact`;
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`,
        'X-Requested-With': 'XMLHttpRequest',
      },
      body: JSON.stringify(updateSingleContactPayload)
    });
    const maybeJson = await response.json().catch(() => null);

    if (!response.ok) {
      const message = (maybeJson && (maybeJson.message || maybeJson.error)) || `HTTP ${response.status}`;
      throw new Error(message);
    }

    const data = maybeJson ?? {};
    const success = data?.status === true || data?.success === true || data?.status === 'success';
    if (!success) {
      throw new Error(data?.message || 'Delete failed');
    }

    console.log('export contact list', data);
    return data;
  } catch (error) {
    console.error('Error exporting contact list:', error);
    throw error;
  }
};



// Rate limiting cache to prevent too many requests
const requestCache = new Map();
const CACHE_DURATION = 5000; // 5 seconds cache

// Helper function to wait for retry
const wait = (ms) => new Promise(resolve => setTimeout(resolve, ms));

export const hasContactlisthaveSelectedtemplateVariables = async (options = {}) => {
  const {
    tagIds = [], 
  } = options;

  // Create cache key from sorted tagIds
  const cacheKey = JSON.stringify(tagIds.sort());
  
  // Check if we have a recent cached request
  const now = Date.now();
  if (requestCache.has(cacheKey)) {
    const lastRequest = requestCache.get(cacheKey);
    if (now - lastRequest < CACHE_DURATION) {
      console.log('Rate limiting: Using cached request for getContactLabels');
      // Return a promise that resolves after the cache duration
      return new Promise((resolve) => {
        setTimeout(() => {
          resolve({ status: false, message: 'Rate limited - please try again' });
        }, CACHE_DURATION - (now - lastRequest));
      });
    }
  }

  // Update cache with current request time
  requestCache.set(cacheKey, now);

  let retryCount = 0;
  const maxRetries = 3;
  const baseDelay = 1000; // Start with 1 second

  while (retryCount <= maxRetries) {
    try {
      const response = await fetch("https://app.getgabs.com/partners/broadcast/campaignmanage/getContactLabels", {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${API_TOKEN}`,
          'X-Requested-With': 'XMLHttpRequest',
        },
        body: JSON.stringify({
          tag_id: JSON.stringify(tagIds.map(id => parseInt(id, 10))),
        }),
      });

      if (response.status === 429) {
        retryCount++;
        if (retryCount > maxRetries) {
          throw new Error('Rate limit exceeded. Please try again later.');
        }
        
        // Exponential backoff: 1s, 2s, 4s
        const delay = baseDelay * Math.pow(2, retryCount - 1);
        console.log(`Rate limited. Retrying in ${delay}ms... (Attempt ${retryCount}/${maxRetries})`);
        await wait(delay);
        continue;
      }

      if (!response.ok) {
        throw new Error(`HTTP error! Status: ${response.status}`);
      }

      const data = await response.json();
      return data;

    } catch (error) {
      if (error.message.includes('Rate limit') || retryCount > maxRetries) {
        console.error('Error fetching contact labels:', error);
        throw error;
      }
      
      retryCount++;
      if (retryCount <= maxRetries) {
        const delay = baseDelay * Math.pow(2, retryCount - 1);
        console.log(`Request failed. Retrying in ${delay}ms... (Attempt ${retryCount}/${maxRetries})`);
        await wait(delay);
      } else {
        console.error('Error fetching contact labels:', error);
        throw error;
      }
    }
  }
};

