
/**
 * Chat API utilities for fetching WhatsApp chat data from GetGabs API
 */

import { API_BASE_URL, API_TOKEN, ENDPOINTS, buildApiUrl } from './apiConfig';

/**
 * Build chat API URL with action
 * @param {string} action - The action to perform (e.g., 'fetch-customer-list')
 * @returns {string} Complete API URL
 */
export function buildChatApiUrl(action) {
  return buildApiUrl(`${ENDPOINTS.CHAT.BASE}/${action}`);
}

/**
 * Fetch customer list from API
 * @param {number} page - Page number for pagination
 * @param {boolean} rollingOver - Whether to fetch rolling over customers (0 for active, 1 for rolling over)
 * @returns {Promise<Object>} Customer list response
 */
export async function fetchCustomerList(page = 1, rollingOver = false) {
  try {
    const rollingOverParam = rollingOver ? 1 : 0;
    const jsonString = encodeURIComponent('["dropdownchatfilterid-all"]');
    
    const url = `${buildChatApiUrl(ENDPOINTS.CHAT.CUSTOMER_LIST)}?page=${page}&token=${API_TOKEN}&rollingover=${rollingOverParam}&jsonString=${jsonString}`;
    
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error fetching customer list:', error);
    throw error;
  }
}

/**
 * Fetch customer list using next page URL
 * @param {string} nextPageUrl - The next page URL from previous API response
 * @returns {Promise<Object>} Customer list response
 */
export async function fetchCustomerListByUrl(nextPageUrl) {
  try {
    if (!nextPageUrl) {
      throw new Error('No next page URL provided');
    }
    
    const response = await fetch(nextPageUrl, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error fetching customer list by URL:', error);
    throw error;
  }
}

/**
 * Fetch chat messages for a specific customer
 * @param {string} profileKey - Customer's profile key
 * @param {number} page - Page number for pagination
 * @param {number} lastChatId - Last chat ID that was fetched (for pagination)
 * @returns {Promise<Object>} Chat messages response
 */
export async function fetchCustomerChat(profileKey, page = 1, lastChatId = 0) {
  try {
    const url = `${buildChatApiUrl(ENDPOINTS.CHAT.CUSTOMER_CHAT)}?page=${page}&profile_key=customerkey-${profileKey}&chatlastidthatfetchedd=${lastChatId}&token=${API_TOKEN}`;
    
    const response = await fetch(url, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error fetching customer chat:', error);
    throw error;
  }
}

/**
 * Transform API customer data to component format
 * @param {Object} apiCustomer - Customer data from API
 * @param {number} index - Index for unique ID generation
 * @returns {Object} Transformed customer object
 */
export function transformCustomerData(apiCustomer, index) {
  const name = apiCustomer.profile_name || `User ${apiCustomer.profile_wa_id}`;
  const lastMessageTime = new Date(apiCustomer.updatedtime.replace(' ', 'T'));
  
  return {
    id: index + 1,
    profileWaId: apiCustomer.profile_wa_id,
    profileWaKey: apiCustomer.profile_wa_key,
    name: name,
    phone: `+${apiCustomer.profile_wa_id}`,
    avatar: `https://ui-avatars.com/api/?name=${encodeURIComponent(name)}&background=random`,
    lastMessageTime: apiCustomer.updatedtime,
    lastMessageTimestamp: lastMessageTime,
    isOnline: isRecentlyActive(lastMessageTime),
    unreadCount: apiCustomer.getpendingmsg_count || 0,
  };
}

/**
 * Check if customer was recently active (within last 5 minutes)
 * @param {Date} lastMessageTime - Last message timestamp
 * @returns {boolean} Whether customer is considered online
 */
function isRecentlyActive(lastMessageTime) {
  const now = new Date();
  const diffMinutes = (now - lastMessageTime) / (1000 * 60);
  return diffMinutes <= 5;
}

/**
 * Transform API message data to component format
 * @param {Object} apiMessage - Message data from API
 * @returns {Object} Transformed message object
 */
export function transformMessageData(apiMessage) {
  const isReceived = apiMessage.sender === 1; 
  const timestamp = new Date(apiMessage.created_at);
  
  let content = '';
  let messageType = 'text';
  let templateData = null;
  let templateName = '';
  let mediaUrl = null;
  let mediaType = null;
  let fileName = null;
  let fileSize = null;
  let previewData = null;
  
  try {
    // Handle template messages
    if (apiMessage.message_type === 'template') {
      messageType = 'template';
      
      // Parse template data if available - this contains the actual template structure
      if (apiMessage.template_data) {
        try {
          const templateInfo = JSON.parse(apiMessage.template_data);
          templateData = templateInfo;
          templateName = templateInfo.data?.[0]?.name || 'Unknown Template';
          
          // Use template_data for preview as it contains the proper template structure
          // Convert template_data format to WhatsApp template format for preview
          const templateForPreview = {
            template: {
              name: templateName,
              language: { code: templateInfo.data?.[0]?.language || 'en_US' },
              components: templateInfo.data?.[0]?.components || []
            }
          };
          content = JSON.stringify(templateForPreview);
        } catch (e) {
          console.warn('Failed to parse template_data:', e);
          // Fallback to message_text
          content = apiMessage.message_text;
        }
      } else {
        // Fallback to message_text if template_data is not available
        content = apiMessage.message_text;
      }
      
      // Always check message_text for image URL in template components (even if template_data exists)
      if (apiMessage.message_text && !mediaUrl) {
        try {
          const messageData = JSON.parse(apiMessage.message_text);
          
          // Check for image URL in template components
          if (messageData.template?.components) {
            for (const component of messageData.template.components) {
              if (component.type === 'HEADER' && component.parameters) {
                for (const param of component.parameters) {
                  if (param.type === 'IMAGE' && param.image?.link) {
                    // Extract image URL from template
                    mediaUrl = param.image.link;
                    mediaType = 'image';
                    messageType = 'template_with_image';
                    break;
                  }
                }
                if (mediaUrl) break;
              }
            }
          }
        } catch (e) {
          // Ignore parsing errors for message_text image extraction
        }
      }
      
      // Try to extract template name from message_text if not found in template_data
      if (!templateName && apiMessage.message_text) {
        try {
          const messageData = JSON.parse(apiMessage.message_text);
          templateName = messageData.template?.name || 'Template Message';
        } catch (e) {
          templateName = 'Template Message';
        }
      }
    }
    // Handle image messages
    else if (apiMessage.message_type === 'image') {
      messageType = 'image';
      mediaType = apiMessage.media_type || 'image';
      
     
      fileName = apiMessage.message_text;
      
      // Construct media URL (assuming GetGabs media URL structure
      if (fileName && fileName.includes('.')) {
        mediaUrl = (apiMessage.media_url)?apiMessage.media_url:`https://app.getgabs.com/customers/mediafile/${fileName}`;
      }
      
      content = apiMessage.caption_text || 'Image';
      fileSize = estimateFileSize(fileName, 'image');
    }
    // Handle video messages
    else if (apiMessage.message_type === 'video') {
      messageType = 'video';
      mediaType = apiMessage.media_type || 'video';
      
      // For video messages, message_text contains the filename
      fileName = apiMessage.message_text;
      
      // Construct media URL
      if (fileName && fileName.includes('.')) {
        mediaUrl = `https://app.getgabs.com/customers/mediafile/${fileName}`;
      }
      
      content = apiMessage.caption_text || 'Video';
      fileSize = estimateFileSize(fileName, 'video');
    }
    // Handle audio messages
    else if (apiMessage.message_type === 'audio') {
      messageType = 'audio';
      mediaType = apiMessage.media_type || 'audio';
      fileName = apiMessage.message_text;
      
      if (fileName && fileName.includes('.')) {
        mediaUrl = `https://app.getgabs.com/customers/mediafile/${fileName}`;
      }
      
      content = 'Voice message';
      fileSize = estimateFileSize(fileName, 'audio');
    }
    // Handle document messages
    else if (apiMessage.message_type === 'document') {
      messageType = 'document';
      mediaType = apiMessage.media_type || 'document';
      fileName = apiMessage.message_text;
      
      // Parse additional_info to get proper filename and file details
      if (apiMessage.additional_info) {
        try {
          const additionalInfo = JSON.parse(apiMessage.additional_info);
          if (additionalInfo.file_name) {
            fileName = additionalInfo.file_name;
          }
          if (additionalInfo.fileSizeKB) {
            fileSize = Math.round(additionalInfo.fileSizeKB * 1024); // Convert KB to bytes
          }
        } catch (e) {
          console.warn('Failed to parse additional_info for document:', e);
        }
      }
      
      if (fileName && fileName.includes('.')) {
        mediaUrl = `https://app.getgabs.com/customers/mediafile/${fileName}`;
      }
      
      content = apiMessage.caption_text || fileName || 'Document';
      
      // Use fileSize from additional_info if available, otherwise estimate
      if (!fileSize) {
        fileSize = estimateFileSize(fileName, 'document');
      }
    }
    // Handle button interactions
    else if (apiMessage.message_type === 'button') {
      try {
        const buttonData = JSON.parse(apiMessage.message_text);
        content = buttonData.button?.text || buttonData.text || 'Button clicked';
        messageType = 'button';
      } catch (e) {
        content = 'Button interaction';
        messageType = 'button';
      }
    }
    // Handle interactive/flow messages 
    else if (apiMessage.message_type === 'interactive' || 
             (apiMessage.message_type === 'template' && apiMessage.message_text.includes('"type":"interactive"'))) {
      messageType = 'interactive';
      content = apiMessage.message_text;
      
      try {
        const interactiveData = JSON.parse(apiMessage.message_text);
        
       
        if (interactiveData.type === 'interactive' && interactiveData.interactive?.type === 'button') {
          messageType = 'interactive_button';
          templateName = 'Interactive Button Message';
          
         
          const bodyText = interactiveData.interactive?.body?.text || '';
          const buttons = interactiveData.interactive?.action?.buttons || [];
          const buttonTitles = buttons.map(btn => btn.reply?.title).filter(Boolean);
          
          content = JSON.stringify({
            body: bodyText,
            buttons: buttonTitles,
            originalData: interactiveData
          });
        }
       
        else if (interactiveData.type === 'interactive' && interactiveData.interactive?.type === 'button_reply') {
          messageType = 'button_reply';
          templateName = 'Button Reply';
          
          
          const buttonTitle = interactiveData.interactive?.button_reply?.title || 'Button clicked';
          const buttonId = interactiveData.interactive?.button_reply?.id || '';
          
          content = buttonTitle;
          
         
          previewData = {
            buttonId: buttonId,
            buttonTitle: buttonTitle,
            originalData: interactiveData
          };
        }
        // Handle flow messages
        else if (interactiveData.interactive?.type === 'flow') {
          templateName = 'Flow Message';
        } 
        // Handle other interactive messages
        else {
          templateName = 'Interactive Message';
        }
      } catch (e) {
        templateName = 'Interactive Message';
      }
    }
    // Handle sticker messages
    else if (apiMessage.message_type === 'sticker') {
      messageType = 'sticker';
      mediaType = apiMessage.media_type || 'image/webp';
      fileName = apiMessage.message_text;
      
      if (fileName && fileName.includes('.')) {
        mediaUrl = `https://app.getgabs.com/customers/mediafile/${fileName}`;
      }
      
      content = 'Sticker';
      fileSize = estimateFileSize(fileName, 'image');
    }
    // Handle reply messages
    else if (apiMessage.message_type === 'reply_msg') {
      messageType = 'reply_msg';
      
      try {
        const replyData = JSON.parse(apiMessage.message_text);
        
        // Handle different reply message formats
        if (replyData.text?.body) {
          // Incoming reply message format (from customer)
          content = replyData.text.body;
          previewData = {
            originalMessageId: replyData.context?.id,
            originalFrom: replyData.context?.from,
            replyText: replyData.text.body,
            timestamp: replyData.timestamp,
            originalData: replyData
          };
        } else if (replyData.msgcontent) {
          
          content = replyData.msgcontent;
          previewData = {
            originalMessageId: replyData.message_id,
            replyText: replyData.msgcontent,
            contentSendType: replyData.content_send_type,
            originalData: replyData
          };
        } else {
          // Fallback for unknown format
          content = replyData.content || replyData.body || 'Reply message';
          previewData = {
            originalData: replyData
          };
        }
        
        templateName = 'Reply Message';
      } catch (e) {
        console.warn('Failed to parse reply message:', e);
        content = apiMessage.message_text || 'Reply message';
        messageType = 'reply_msg';
      }
    }
    // Handle regular text messages
    else {
      messageType = 'text';
      
      // Try to parse as JSON first
      try {
        const parsedMessage = JSON.parse(apiMessage.message_text);
        if (parsedMessage.text) {
          content = parsedMessage.text;
        } else if (parsedMessage.button) {
          content = parsedMessage.button.text || 'Button interaction';
          messageType = 'button';
        } else {
          content = apiMessage.message_text;
        }
      } catch {
        // If not JSON, treat as plain text
        content = apiMessage.message_text;
      }
    }
  } catch (error) {
    console.error('Error parsing message:', error);
    content = apiMessage.message_text || 'Message could not be parsed';
    messageType = 'text';
  }
  
  return {
    id: apiMessage.id,
    type: isReceived ? 'received' : 'sent',
    content: content,
    timestamp: timestamp,
    status: getMessageStatus(apiMessage),
    messageType: messageType,
    templateData: templateData,
    templateName: templateName,
    messageId: apiMessage.message_id,
    originalData: apiMessage,
    // Media-specific properties
    mediaUrl: mediaUrl,
    mediaType: mediaType,
    fileName: fileName,
    fileSize: fileSize,
    fileUrl: mediaUrl, // Alias for backward compatibility
    previewData: previewData
  };
}

/**
 * Estimate file size based on filename and type
 * @param {string} fileName - The filename
 * @param {string} type - The file type (image, video, audio, document)
 * @returns {number} Estimated file size in bytes
 */
function estimateFileSize(fileName, type) {
  if (!fileName) return 0;
  
  // Basic estimation based on file type
  const estimations = {
    image: 500000,    // ~500KB
    video: 5000000,   // ~5MB
    audio: 1000000,   // ~1MB
    document: 100000  // ~100KB
  };
  
  return estimations[type] || 100000;
}

/**
 * Get message delivery status
 * @param {Object} apiMessage - Message data from API
 * @returns {string} Message status
 */
function getMessageStatus(apiMessage) {
  // Handle different status formats from GetGabs API
  const status = apiMessage.delivery_status || apiMessage.status;
  
  if (status === 'read' || status === 'seen') return 'read';
  if (status === 'delivered') return 'delivered';
  if (status === 'sent') return 'sent';
  if (status === 'failed') return 'failed';
  
  // For admin/user messages (sender === 2), default to sent
  if (apiMessage.sender === 2) return 'sent';
  
  // For customer messages (sender === 1), default to delivered
  return 'delivered';
}

/**
 * Format timestamp for display
 * @param {Date} timestamp - Message timestamp
 * @returns {string} Formatted time string
 */
export function formatMessageTime(timestamp) {
  const now = new Date();
  const diff = now - timestamp;
  const minutes = Math.floor(diff / 60000);
  const hours = Math.floor(diff / 3600000);
  const days = Math.floor(diff / 86400000);

  if (minutes < 1) return 'now';
  if (minutes < 60) return `${minutes}m ago`;
  if (hours < 24) return `${hours}h ago`;
  if (days < 7) return `${days}d ago`;
  return timestamp.toLocaleDateString();
}



/**
 * Send text message to customer via GetGabs API
 * @param {string} customerKey - Customer's profile key
 * @param {string} textMessage - Message content
 * @param {string} messageType - Type of message (default: 'text')
 * @returns {Promise<Object>} Send message response
 */
export async function sendMessage(customerKey, textMessage, messageType = 'text') {
  try {
    const url = `${API_BASE_URL}/partners/managechat/chat/send-message`;
    
    const payload = {
      messageType: messageType,
      textMessage: textMessage,
      customerKey: customerKey
    };
    
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`
      },
      body: JSON.stringify(payload)
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error sending message:', error);
    throw error;
  }
}

/**
 * Send reply message to customer via GetGabs API
 * @param {string} customerKey - Customer's profile key
 * @param {string} replyText - Reply message content
 * @param {string} originalMessageId - ID of the message being replied to
 * @returns {Promise<Object>} Send reply message response
 */
export async function sendReplyMessage(customerKey, replyText, originalMessageId) {
  try {
    const url = `${API_BASE_URL}/partners/managechat/chat/sendReplyChat`;
    
    // Construct the messageText JSON as required by the API
    const messageTextData = {
      content_send_type: "6",
      message_id: originalMessageId,
      msgcontent: replyText
    };
    
    const payload = {
      messageType: "reply_msg",
      messageText: JSON.stringify(messageTextData),
      customerKey: customerKey
    };
    
    console.log('Sending reply message:', {
      customerKey,
      replyText,
      originalMessageId,
      payload
    });
    
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`
      },
      body: JSON.stringify(payload)
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    console.log(' Reply message sent successfully:', data);
    return data;
  } catch (error) {
    console.error('Error sending reply message:', error);
    throw error;
  }
}

/**
 * Send media file (image/video) to customer via GetGabs API
 * @param {string} customerKey - Customer's profile key
 * @param {string} base64Data - Base64 encoded file data with data URL prefix
 * @param {string} fileType - Type of file ('image' or 'video')
 * @param {string} mimeType - MIME type of the file (e.g., 'image/png', 'video/mp4')
 * @returns {Promise<Object>} Send message response
 */
export async function sendMediaMessage(customerKey, base64Data, fileType, mimeType) {
  try {
    const url = `${API_BASE_URL}/partners/managechat/chat/send-message`;
    
    const payload = {
      messageType: "imgvidfiles",
      fileMessage: [base64Data],
      imgvidfiletype: [fileType],
      filemimetype: [mimeType],
      customerKey: customerKey
    };       
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`
      },
      body: JSON.stringify(payload)
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error sending media message:', error);
    throw error;
  }
}

/**
 * Convert File object to base64 data URL
 * @param {File} file - File object to convert
 * @returns {Promise<string>} Base64 data URL
 */
export function fileToBase64(file) {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onload = () => resolve(reader.result);
    reader.onerror = error => reject(error);
    reader.readAsDataURL(file);
  });
}

/**
 * Send image file to customer
 * @param {string} customerKey - Customer's profile key
 * @param {File|string} imageFile - Image file object or base64 data URL
 * @returns {Promise<Object>} Send message response
 */
export async function sendImageMessage(customerKey, imageFile) {
  try {
    let base64Data;
    let mimeType;
    
    if (typeof imageFile === 'string') {
     
      base64Data = imageFile;

      const mimeMatch = imageFile.match(/data:([^;]+);base64,/);
      mimeType = mimeMatch ? mimeMatch[1] : 'image/png';
    } else {
      
      base64Data = await fileToBase64(imageFile);
      mimeType = imageFile.type || 'image/png';
    }
    
    return await sendMediaMessage(customerKey, base64Data, 'image', mimeType);
  } catch (error) {
    console.error('Error sending image message:', error);
    throw error;
  }
}

/**
 * Send video file to customer
 * @param {string} customerKey - Customer's profile key
 * @param {File|string} videoFile - Video file object or base64 data URL
 * @returns {Promise<Object>} Send message response
 */
export async function sendVideoMessage(customerKey, videoFile) {
  try {
    let base64Data;
    let mimeType;
    
    if (typeof videoFile === 'string') {
      
      base64Data = videoFile;
     
      const mimeMatch = videoFile.match(/data:([^;]+);base64,/);
      mimeType = mimeMatch ? mimeMatch[1] : 'video/mp4';
    } else {
      
      base64Data = await fileToBase64(videoFile);
      mimeType = videoFile.type || 'video/mp4';
    }
    
    return await sendMediaMessage(customerKey, base64Data, 'video', mimeType);
  } catch (error) {
    console.error('Error sending video message:', error);
    throw error;
  }
}

/**
 * Send document file to customer
 * @param {string} customerKey - Customer's profile key
 * @param {File|string} documentFile - Document file object or base64 data URL
 * @returns {Promise<Object>} Send message response
 */
export async function sendDocumentMessage(customerKey, documentFile) {
  try {
    const url = `${API_BASE_URL}/partners/managechat/chat/send-message`;
    
    let base64Data;
    let fileName;
    
    if (typeof documentFile === 'string') {
      base64Data = documentFile;
      fileName = 'document.pdf'; // Default filename for base64 strings
    } else {
      base64Data = await fileToBase64(documentFile);
      fileName = documentFile.name || 'document.pdf';
    }
    
    const payload = {
      messageType: "document",
      documentMessage: [base64Data],
      filename: [fileName],
      customerKey: customerKey
    };
    
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`
      },
      body: JSON.stringify(payload)
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    console.log('📄 Document sent successfully:', data);
    return data;
  } catch (error) {
    console.error('Error sending document message:', error);
    throw error;
  }
}

/**
 * Get last message preview for customer list
 * @param {Object} customer - Customer object
 * @returns {string} Last message preview text
 */
export function getLastMessagePreview(customer) {
  if (!customer.lastMessage) return 'No messages yet';
  

  const maxLength = 50;
  if (customer.lastMessage.length > maxLength) {
    return customer.lastMessage.substring(0, maxLength) + '...';
  }
  
  return customer.lastMessage;
}

/**
 * Handle API errors gracefully
 * @param {Error} error - Error object
 * @returns {Object} Error response object
 */
export function handleApiError(error) {
  console.error('API Error:', error);
  
  return {
    success: false,
    error: error.message || 'An unexpected error occurred',
    data: null,
  };
}



/**
 * Send template message to customer
 * @param {string} templatepayload - Template payload
 * @returns {Promise<Object>} Send message response
 */
export async function sendTemplateMessage( templatepayload) {
  try {
    
    const url = `${API_BASE_URL}/partners/managechat/chat/send-templated-msg`;
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`
      },
      body: JSON.stringify(templatepayload)
    });
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error sending video message:', error);
    throw error;
  }
}