/**
 * Template API utilities for fetching WhatsApp templates from GetGabs API
 */

import { API_BASE_URL, API_TOKEN, ENDPOINTS, buildApiUrl } from './apiConfig.js';

/**
 * Fetch broadcast campaign list from GetGabs API
 * @param {Object} options - API request options
 * @param {string} options.status - Broadcast status filter (All, In Progress, Waiting, Paused, Finished, Payment Issues)
 * @param {number} options.page - Page number (default: 1)
 * @param {number} options.perPageData - Items per page (default: 20)
 * @returns {Promise<Object>} API response with broadcast campaign list data
 */
export const fetchBroadcastList = async (options = {}) => {
  const {
    statusType = 'All',
    page = 1,
    perPageData = 20
  } = options;

  try {
    const url = new URL(buildApiUrl(`${ENDPOINTS.BROADCAST.BASE}/${ENDPOINTS.BROADCAST.GET_BROADCAST_LIST}`));
    url.searchParams.append('statusType', statusType);
    url.searchParams.append('page', page.toString());
    url.searchParams.append('perPageData', perPageData.toString());
    url.searchParams.append('token', API_TOKEN);

    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error fetching broadcast campaign list:', error);
    throw error;
  }
};


/**
 * Send broadcast campaign to customer
 * @param {string} broadcastpayload - Broadcast payload
 * @returns {Promise<Object>} Send message response
 */
export async function sendBroadcastCampaign( broadcastpayload) {
  try {
    
    const url = `${API_BASE_URL}/partners/broadcast/campaignmanage/campaignSubmit`;
    const response = await fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`
      },
      body: JSON.stringify(broadcastpayload)
    });
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Error sending broadcast campaign:', error);
    throw error;
  }
}
