import { API_TOKEN } from "./apiConfig";

/** 
 * Fetch all variables from GetGabs API
 * @returns {Object} API response with variable list
 */
export const getAllVariables = async () => {
  const response = await fetch(`https://app.getgabs.com/partners/shopify/CustomerLabel/fetchList?token=${API_TOKEN}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${API_TOKEN}`
    }
  });
  if (!response.ok) throw new Error('Failed to fetch variable list');
  const data = await response.json();
  console.log("variable list", data);
  return data;
};

/**
 * Transform API variable data to component format
 * @param {Array} apiVariables - Variables from API response
 * @returns {Array} Transformed variables with label and value
 */
export const transformApiVariables = (apiVariables) => {
  if (!Array.isArray(apiVariables)) return [];
  
  return apiVariables
    .filter(variable => variable.status === 1) // Only active variables
    .map(variable => ({
      label: variable.name,
      value: variable.name
    }));
};

/**
 * Get variables from API and transform them
 * @returns {Array} Transformed variables ready for use in components
 */
export const getApiVariables = async () => {
  try {
    const response = await getAllVariables();
    if (response.status && response.data && response.data.data) {
      return transformApiVariables(response.data.data);
    }
    return [];
  } catch (error) {
    console.error('Error fetching API variables:', error);
    return [];
  }
};

/**
 * Available variables - populated from API data only
 */
export let availableVariables = [];

/**
 * Get variables suitable for WhatsApp template body text
 * Fetches from API only
 * @returns {Array} Variables for template use
 */
export const getTemplateVariables = async () => {
  try {
    const apiVariables = await getApiVariables();
    return apiVariables;
  } catch (error) {
    console.error('Failed to fetch API variables:', error);
    return [];
  }
};

/**
 * Get variables suitable for contact field mapping
 * Fetches from API only
 * @returns {Array} Variables for contact mapping
 */
export const getContactFieldVariables = async () => {
  try {
    const apiVariables = await getApiVariables();
    return apiVariables;
  } catch (error) {
    console.error('Failed to fetch API variables:', error);
    return [];
  }
};

/**
 * Get CSV mapping options including API variables and ignore option
 * @returns {Array} Variables plus ignore option for CSV mapping
 */
export const getCSVMappingOptions = async () => {
  const variables = await getContactFieldVariables();
  return [
    { label: 'Ignore Column', value: 'ignore' },
    ...variables
  ];
};

/**
 * Update available variables with API data
 * @param {Array} apiVariables - Variables from API
 */
export const updateAvailableVariables = (apiVariables) => {
  if (Array.isArray(apiVariables) && apiVariables.length > 0) {
    availableVariables.length = 0; // Clear existing
    availableVariables.push(...apiVariables);
  }
};


/**
 * Get variable by value from current available variables
 * @param {string} value - The variable value to search for
 * @param {Array} variables - Optional variables array to search in
 * @returns {Object|null} - The variable object or null if not found
 */
export const getVariableByValue = (value, variables = availableVariables) => {
  return variables.find(variable => variable.value === value) || null;
};

/**
 * Get variable label by value
 * @param {string} value - The variable value to search for
 * @param {Array} variables - Optional variables array to search in
 * @returns {string} - The variable label or the value itself if not found
 */
export const getVariableLabelByValue = (value, variables = availableVariables) => {
  const variable = getVariableByValue(value, variables);
  return variable ? variable.label : value;
};

/**
 * Format variable for template insertion (adds curly braces)
 * @param {string} value - The variable value
 * @returns {string} - Formatted variable for template insertion
 */
export const formatVariableForTemplate = (value) => {
  return `{{${value}}}`;
};

/**
 * Extract variable value from template format (removes curly braces)
 * @param {string} templateVariable - The template variable with curly braces
 * @returns {string} - Clean variable value
 */
export const extractVariableValue = (templateVariable) => {
  return templateVariable.replace(/{{|}}/g, '');
};

// Export empty array as default when no API data
export default [];
