/**
 * Centralized API Configuration
 * Contains API tokens and base URLs used across the application
 */
export const API_BASE_URL = 'https://app.getgabs.com';



// export const API_TOKEN = 'eyJ0eXAiOiJKV1QiLCJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJodHRwOi8vbG9jYWxob3N0OjgwMDAvcGFydG5lcnMvbG9naW4iLCJpYXQiOjE3NjI0MDU2NjQsImV4cCI6MTc3MTA0NTY2NCwibmJmIjoxNzYyNDA1NjY0LCJqdGkiOiJjWU5uRjZkdWdHWUlSYjRSIiwic3ViIjoiNTM0NyIsInBydiI6IjIzYmQ1Yzg5NDlmNjAwYWRiMzllNzAxYzQwMDg3MmRiN2E1OTc2ZjciLCJMb2dnZWRJblVzZXJJZCI6InN6NlZWcUxVQkp2STZ1ZGQwYnR4T2RXa3FsYzdWbFVmdEt6R0pXUmIiLCJhZG1pbl9jaHVua19ibGFzdCI6NTAwMDAwMDB9.1eqK0LSj3wqmVOc2D_7pQ65SpDG64iJ_szgXKYQzwMQ';

// export const autoReplyToken = 'isLOyZ0hmupKqgiorpYmuONMzR2PiN0Sus1jGKyK';
// export const userInfo = {
//   'role': 'user',
//   'id': '5347', 
//   'user_privilage': '0',
//   'admin_id': '0'
// };












// API Endpoints
export const ENDPOINTS = {
  CHAT: {
    BASE: '/partners/managechat/chat',
    CUSTOMER_LIST: 'fetch-customer-list',
    CUSTOMER_CHAT: 'fetch-customer-chat'
  },
  TEMPLATE: {
    BASE: '/partners/template',
    GET_TEMPLATES: 'getcustomtemplateinfos'
  },
  STATIC_CONTACT_LIST: {
    BASE: '/partners/broadcast',
    GET_STATIC_CONTACT_LIST: 'contactlist'
  },
  STATIC_CONTACT_LIST_CONTACTS: {
    BASE: '/partners/broadcast',
    GET_STATIC_CONTACT_LIST_CONTACTS: 'contactcustomerlist'
  },
  BROADCAST: {
    BASE: '/partners/broadcast',
    GET_BROADCAST_LIST: 'campaignlists'
  },
  AUTOREPLY: {
    BASE: '/partners/automation/autoreply',
    GET_AUTOREPLY_LIST: 'fetchautoreplies'
  },
  PRODUCT: {
    BASE: '/partners/shopify/Products',
    GET_PRODUCT_LIST: 'fetchList'
  },
    CATEGORY: {
    BASE: '/partners/shopify/Category',
    GET_CATEGORY_LIST: 'fetchList'
  },
  ABANDONED_CART: {
    BASE: '/partners/shopify/AutoTrigger',
    GET_ABANDONED_CART_LIST: 'fetchList'
  },
  ORDER_CONFIRMATION: {
    BASE: '/partners/shopify/AutoTrigger',
    GET_ORDER_CONFIRMATION_LIST: 'fetchList'
  },
  ABANDONED_LIST: {
    BASE: '/partners/shopify/Abandoned',
    GET_ABANDONED_LIST: 'fetchList'
  },
};

/**
 * Build complete API URL
 * @param {string} endpoint - The endpoint path
 * @returns {string} Complete API URL
 */
export const buildApiUrl = (endpoint) => {
  return `${API_BASE_URL}${endpoint}`;
};

/**
 * Fetch user data from GetGabs login API
 * @param {string} shopName - Shopify store name (e.g., "getgabstore.myshopify.com")
 * @returns {Promise<Object>} User data response
 */
export const fetchUserData = async (shopName) => {
  try {
    const myHeaders = new Headers();
    myHeaders.append("Content-Type", "application/json");

    const raw = JSON.stringify({
      "ShopName": shopName
    });

    const requestOptions = {
      method: "POST",
      headers: myHeaders,
      body: raw,
      redirect: "follow"
    };

    const response = await fetch("https://app.getgabs.com/partners/shopify/login", requestOptions);
    
    if (!response.ok) {
      throw new Error(`HTTP error! status: ${response.status}`);
    }
    
    const result = await response.text();
    const userData = JSON.parse(result);
    
    // Save to localStorage
    saveUserDataToStorage(userData);
    
    return userData;
  } catch (error) {
    console.error('Error fetching user data:', error);
    throw error;
  }
};

/**
 * Save user data to localStorage
 * @param {Object} userData - User data to save
 */
export const saveUserDataToStorage = (userData) => {
  try {
    localStorage.setItem('getgabs_user_data', JSON.stringify(userData));
    localStorage.setItem('getgabs_user_data_timestamp', Date.now().toString());
  } catch (error) {
    console.error('Error saving user data to localStorage:', error);
  }
};

/**
 * Get user data from localStorage
 * @returns {Object|null} User data or null if not found
 */
export const getUserDataFromStorage = () => {
  try {
    const userData = localStorage.getItem('getgabs_user_data');
    return userData ? JSON.parse(userData) : null;
  } catch (error) {
    console.error('Error getting user data from localStorage:', error);
    return null;
  }
};

/**
 * Check if user data exists and is valid (not expired)
 * @param {number} maxAgeHours - Maximum age in hours (default: 24 hours)
 * @returns {boolean} True if user data is valid
 */
export const isUserDataValid = (maxAgeHours = 24) => {
  try {
    const userData = getUserDataFromStorage();
    const timestamp = localStorage.getItem('getgabs_user_data_timestamp');
    
    if (!userData || !timestamp) {
      return false;
    }
    
    const dataAge = Date.now() - parseInt(timestamp);
    const maxAge = maxAgeHours * 60 * 60 * 1000; 
    
    return dataAge < maxAge;
  } catch (error) {
    console.error('Error checking user data validity:', error);
    return false;
  }
};

/**
 * Clear user data from localStorage
 */
export const clearUserDataFromStorage = () => {
  try {
    localStorage.removeItem('getgabs_user_data');
    localStorage.removeItem('getgabs_user_data_timestamp');
  } catch (error) {
    console.error('Error clearing user data from localStorage:', error);
  }
};

/**
 * Get user data with automatic refresh if expired
 * @param {string} shopName - Shopify store name
 * @param {number} maxAgeHours - Maximum age in hours (default: 24 hours)
 * @returns {Promise<Object>} User data
 */
export const getUserDataWithRefresh = async (shopName, maxAgeHours = 24) => {
  try {
   
    if (isUserDataValid(maxAgeHours)) {
      return getUserDataFromStorage();
    }
    
    
    return await fetchUserData(shopName);
  } catch (error) {
    console.error('Error getting user data with refresh:', error);
    
    return getUserDataFromStorage();
  }
};

/**
 * Initialize user data when app loads
 * Automatically fetches user data if localStorage doesn't have valid data
 * @param {string} shopName - Shopify store name (e.g., "getgabstore.myshopify.com")
 * @param {number} maxAgeHours - Maximum age in hours (default: 24 hours)
 * @returns {Promise<void>}
 */
export const initializeUserData = async (shopName, maxAgeHours = 24) => {
  try {
    console.log('Initializing user data for shop:', shopName);
    
    // Check if we already have valid user data in localStorage
    if (isUserDataValid(maxAgeHours)) {
      const cachedData = getUserDataFromStorage();
      console.log('Using cached user data:', cachedData);
      return cachedData;
    }
    
    // If no valid cached data, fetch fresh data from API
    console.log('No valid cached data found. Fetching fresh user data...');
    const userData = await fetchUserData(shopName);
    console.log('Fresh user data fetched and saved:', userData);
    
    return userData;
  } catch (error) {
    console.error('Error initializing user data:', error);
    
    // Try to use any cached data as fallback, even if expired
    const fallbackData = getUserDataFromStorage();
    if (fallbackData) {
      console.log('Using expired cached data as fallback:', fallbackData);
      return fallbackData;
    }
    
    // If no cached data available, return null
    console.log('No user data available');
    return null;
  }
};

/**
 * Get current user data (from cache or fetch if needed)
 * This is a convenience function for components that need user data
 * @returns {Object|null} Current user data or null if not available
 */
export const getCurrentUserData = () => {
  return getUserDataFromStorage();
};

/**
 * Check if user is authenticated (has valid user data)
 * @returns {boolean} True if user has valid authentication data
 */
export const isUserAuthenticated = () => {
  const userData = getUserDataFromStorage();
  return userData !== null && typeof userData === 'object';
};


/**
 * Map access_value_type to plan names
 * @param {number} accessValueType - The access value type from API
 * @returns {string} Plan name
 */
const mapAccessValueTypeToPlan = (accessValueType) => {
  const planMapping = {
    0: 'Non',
    1: 'Basic Monthly',
    11: 'Basic Annually',
    2: 'Pro Monthly',
    22: 'Pro Annually',
    4: 'Plus Monthly',
    44: 'Plus Annually'
  };
  
  return planMapping[accessValueType] || 'Unknown';
};

 const getAccountDetails = () => {
  try {
    const userData = localStorage.getItem('getgabs_user_data');
    if (userData) {
      const parsedData = JSON.parse(userData);

      // Check if userAllInfo exists
      if (!parsedData.userAllInfo) {
        console.warn('userAllInfo not found in parsed data');
        return '';
      }

      // Check if WhatsApp account details exist and have at least one entry
      const hasWhatsAppDetails = Array.isArray(parsedData.userAllInfo.whasappaccountdetails) && 
                                parsedData.userAllInfo.whasappaccountdetails.length > 0;
      
      const whatsappDetails = hasWhatsAppDetails ? parsedData.userAllInfo.whasappaccountdetails[0] : null;

      const useraccountdetails = {
        'id': parsedData.userAllInfo.id,
        'role': parsedData.userAllInfo.role,
        'isplanpurchased': parsedData.userAllInfo.access_value_type > 0,
        'planName': mapAccessValueTypeToPlan(parsedData.userAllInfo.access_value_type),
        'isAccountConnected': hasWhatsAppDetails, 
        'businesName': whatsappDetails?.verified_name || parsedData.userAllInfo.name,
        'username': parsedData.userAllInfo.username,
        "connectedWhatsappNumber": whatsappDetails?.display_phone_number || '+1xxx xxx xxx',
        'accouuntQualityRating': whatsappDetails?.quality_rating || 'please connect whatsapp number',
       'currentMessageLimit': whatsappDetails?.current_limit || "check limit",
        'user_privilage': parsedData.userAllInfo.user_privilage,
        'admin_id': parsedData.userAllInfo.admin_id,
        'api_key': parsedData.userAllInfo.api_key,
        
     
      };
      return useraccountdetails;
    }
    return '';
  } catch (error) {
    console.error('Error getting user info for socket from localStorage:', error);
    return '';
  }
};
  



 const getAPIToken = () => {
  try {
    const userData = localStorage.getItem('getgabs_user_data');
    if (userData) {
      const parsedData = JSON.parse(userData);
      return parsedData.access_token || '';
    }
    return '';
  } catch (error) {
    console.error('Error getting access_token from localStorage:', error);
    return '';
  }
};


 const getAutoReplyToken = () => {
  try {
    const userData = localStorage.getItem('getgabs_user_data');
    if (userData) {
      const parsedData = JSON.parse(userData);
      return parsedData.userAllInfo.shopify_access_token || '';
    }
    return '';
  } catch (error) {
    console.error('Error getting shopify access_token from localStorage:', error);
    return '';
  }
};

 const getUserInfo = () => {
  try {
    const userData = localStorage.getItem('getgabs_user_data');
    if (userData) {
      const parsedData = JSON.parse(userData);

      const userInfo = {
        'role': parsedData.userAllInfo.role,
        'id': parsedData.userAllInfo.id, 
        'user_privilage': parsedData.userAllInfo.user_privilage,
        'admin_id': parsedData.userAllInfo.admin_id
      };
      return userInfo;
    }
    return '';
  } catch (error) {
    console.error('Error getting user info for socket from localStorage:', error);
    return '';
  }
};


export const API_TOKEN = getAPIToken();
 export const userInfo = getUserInfo();
 export const autoReplyToken = getAutoReplyToken();
 export const accountDetails= getAccountDetails();

 console.log('apitoken from apiConfig',API_TOKEN);
 console.log('userInfo from apiConfig',userInfo);
 console.log('autoReplyToken from apiConfig',autoReplyToken);
 console.log('accountDetails from apiConfig',accountDetails);
 


