import {
  Page,
  Badge,
  LegacyCard,
  Button,
  Modal,
  TextContainer,
  Thumbnail,
  Text,
  InlineStack,
  BlockStack,
  TextField,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  Pagination,
  Icon,
  Tooltip,
  Spinner,
  Banner,
  EmptyState,
} from '@shopify/polaris';
import { RefreshIcon, ViewIcon, PlusIcon, ThemeTemplateIcon, SlideshowIcon, StoreIcon, CartIcon } from '@shopify/polaris-icons';
// IndexFiltersProps and TabProps types will be inferred
import React, { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { messageTemplateOptions, fetchApiTemplates, searchTemplates, filterTemplatesByStatus, sortTemplates } from '../utils/messageTemplates';
import { getSelectedTemplatebyName } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';

function TemplatesPage() {
  const navigate = useNavigate();
  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

  const [selectedTemplate, setSelectedTemplate] = useState(null);
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [templateDetailError, setTemplateDetailError] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  const [templateSelectionModalActive, setTemplateSelectionModalActive] = useState(false);


  const [itemStrings, setItemStrings] = useState([
    'All',
    'Approved',
    'Rejected',
    'Pending',
    'Disabled',
    'Draft'
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['name asc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [categoryFilter, setCategoryFilter] = useState('');
  const [qualityFilter, setQualityFilter] = useState([]);


  const [templates, setTemplates] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [apiPagination, setApiPagination] = useState(null);

  const { mode, setMode } = useSetIndexFiltersMode();


  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 20;


  const fetchTemplatesData = useCallback(async (page = 1, status = 'All') => {
    try {
      setLoading(true);
      setError(null);

      const result = await fetchApiTemplates({
        templateStatus: status,
        page: page,
        perPageData: itemsPerPage
      });

      setTemplates(result.templates);
      setApiPagination(result.pagination);
    } catch (err) {
      console.error('Error fetching templates:', err);
      setError(err.message);

      setTemplates([]);
      setApiPagination(null);
    } finally {
      setLoading(false);
    }
  }, [itemsPerPage]);


  useEffect(() => {
    const statusMap = {
      0: 'All',
      1: 'Approved',
      2: 'Rejected',
      3: 'Pending',
      4: 'Disabled',
      5: 'Draft'
    };

    const selectedStatus = statusMap[selected] || 'All';
    fetchTemplatesData(1, selectedStatus);
    setCurrentPage(1); // Reset to first page when changing status
  }, [selected, fetchTemplatesData]);

  // Refresh templates function
  const handleRefresh = useCallback(() => {
    const statusMap = {
      0: 'All',
      1: 'Approved',
      2: 'Rejected',
      3: 'Pending',
      4: 'Disabled',
      5: 'Draft'
    };

    const selectedStatus = statusMap[selected] || 'All';
    fetchTemplatesData(currentPage, selectedStatus);
  }, [selected, currentPage, fetchTemplatesData]);

  // Tab management
  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => { },
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      {
        type: 'rename',
        onAction: () => { },
        onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => {
            if (idx === index) {
              return value;
            }
            return item.content;
          });
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      {
        type: 'duplicate',
        onPrimaryAction: async (value) => {
          await sleep(1);
          duplicateView(value);
          return true;
        },
      },
      {
        type: 'edit',
      },
      {
        type: 'delete',
        onPrimaryAction: async () => {
          await sleep(1);
          deleteView(index);
          return true;
        },
      },
    ],
  }));

  // Filter and search logic
  const filteredTemplates = useMemo(() => {
    let filtered = [...templates];

    // Apply search filter
    if (queryValue) {
      filtered = searchTemplates(filtered, queryValue);
    }

    // Apply additional filters (status filter is handled by API, but we can apply others)
    filtered = filtered.filter((template) => {
      const matchesStatus = statusFilter.length === 0 || statusFilter.includes(template.status.toLowerCase());
      const matchesCategory = categoryFilter === '' || template.category.toLowerCase().includes(categoryFilter.toLowerCase());
      const matchesQuality = qualityFilter.length === 0 || qualityFilter.includes(template.qualityRating.toLowerCase());

      return matchesStatus && matchesCategory && matchesQuality;
    });

    return filtered;
  }, [templates, queryValue, statusFilter, categoryFilter, qualityFilter]);

  // Sort templates
  const sortedTemplates = useMemo(() => {
    const sorted = [...filteredTemplates];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');

    sorted.sort((a, b) => {
      let aValue, bValue;

      switch (sortKey) {
        case 'label':
          aValue = a.label.toLowerCase();
          bValue = b.label.toLowerCase();
          break;
        case 'status':
          aValue = a.status.toLowerCase();
          bValue = b.status.toLowerCase();
          break;
        case 'category':
          aValue = a.category.toLowerCase();
          bValue = b.category.toLowerCase();
          break;
        case 'quality':
          // Convert quality ratings to numeric values for sorting
          const qualityOrder = { 'green': 3, 'yellow': 2, 'red': 1, 'grey': 0 };
          aValue = qualityOrder[a.qualityRating.toLowerCase()] || 0;
          bValue = qualityOrder[b.qualityRating.toLowerCase()] || 0;
          break;
        case 'language':
          aValue = a.language.toLowerCase();
          bValue = b.language.toLowerCase();
          break;
        case 'updated':
          aValue = new Date(a.updatedOn);
          bValue = new Date(b.updatedOn);
          break;
        default:
          return 0;
      }

      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });

    return sorted;
  }, [filteredTemplates, sortSelected]);

  // For API pagination, we use the sorted templates directly (no client-side pagination)
  const paginatedTemplates = sortedTemplates;

  // API Pagination handlers
  const handlePreviousPage = useCallback(() => {
    if (apiPagination && apiPagination.hasPrevPage) {
      const newPage = Math.max(1, currentPage - 1);
      setCurrentPage(newPage);

      const statusMap = {
        0: 'All',
        1: 'Approved',
        2: 'Rejected',
        3: 'Pending',
        4: 'Disabled',
        5: 'Draft'
      };

      const selectedStatus = statusMap[selected] || 'All';
      fetchTemplatesData(newPage, selectedStatus);
    }
  }, [apiPagination, currentPage, selected, fetchTemplatesData]);

  const handleNextPage = useCallback(() => {
    if (apiPagination && apiPagination.hasNextPage) {
      const newPage = currentPage + 1;
      setCurrentPage(newPage);

      const statusMap = {
        0: 'All',
        1: 'Approved',
        2: 'Rejected',
        3: 'Pending',
        4: 'Disabled',
        5: 'Draft'
      };

      const selectedStatus = statusMap[selected] || 'All';
      fetchTemplatesData(newPage, selectedStatus);
    }
  }, [apiPagination, currentPage, selected, fetchTemplatesData]);

  // API Pagination info
  const totalItems = apiPagination?.total || 0;
  const totalPages = apiPagination?.lastPage || 1;
  const startItem = apiPagination?.from || 0;
  const endItem = apiPagination?.to || 0;
  const hasNext = apiPagination?.hasNextPage || false;
  const hasPrevious = apiPagination?.hasPrevPage || false;

  // Sort options
  const sortOptions = [
    { label: 'Template Name', value: 'name asc', directionLabel: 'A-Z' },
    { label: 'Template Name', value: 'name desc', directionLabel: 'Z-A' },
    { label: 'Category', value: 'category asc', directionLabel: 'A-Z' },
    { label: 'Category', value: 'category desc', directionLabel: 'Z-A' },
    { label: 'Status', value: 'status asc', directionLabel: 'A-Z' },
    { label: 'Status', value: 'status desc', directionLabel: 'Z-A' },
    { label: 'Quality Rating', value: 'quality asc', directionLabel: 'Low to High' },
    { label: 'Quality Rating', value: 'quality desc', directionLabel: 'High to Low' },
    { label: 'Language', value: 'language asc', directionLabel: 'A-Z' },
    { label: 'Language', value: 'language desc', directionLabel: 'Z-A' },
    { label: 'Updated On', value: 'updated asc', directionLabel: 'Oldest First' },
    { label: 'Updated On', value: 'updated desc', directionLabel: 'Newest First' },
  ];

  // Fetch detailed template data
  const fetchTemplateDetails = useCallback(async (templateName) => {
    try {
      setLoadingTemplateDetails(true);
      setTemplateDetailError(null);

      const templateData = await getSelectedTemplatebyName(templateName);
      setSelectedTemplateData(templateData);

      return templateData;
    } catch (error) {
      console.error('Error fetching template details:', error);
      setTemplateDetailError(error.message || 'Failed to fetch template details');
      return null;
    } finally {
      setLoadingTemplateDetails(false);
    }
  }, []);

  // Event handlers
  const handlePreviewClick = useCallback(async (template) => {
    setSelectedTemplate(template);
    setSelectedTemplateData(null); // Reset previous data
    setPreviewModalActive(true);


    await fetchTemplateDetails(template.label);
  }, [fetchTemplateDetails]);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setSelectedTemplate(null);
    setSelectedTemplateData(null);
    setTemplateDetailError(null);
  }, []);

  const handleTemplateSelectionModalClose = useCallback(() => {
    setTemplateSelectionModalActive(false);
  }, []);

  const handleTemplateSelectionModalOpen = useCallback(() => {
    setTemplateSelectionModalActive(true);
  }, []);

  const handleAbandonedCartTemplateSelect = useCallback(() => {
    setTemplateSelectionModalActive(false);
    navigate('/app/create-template/abandoned-cart');
  }, [navigate]);

  const handleOrderConfirmationTemplateSelect = useCallback(() => {
    setTemplateSelectionModalActive(false);
    navigate('/app/create-template/order-confirmation');
  }, [navigate]);

  const handleCustomTemplateSelect = useCallback(() => {
    setTemplateSelectionModalActive(false);
    navigate('/app/create-template/custom');
  }, [navigate]);

  const handleCarouselTemplateSelect = useCallback(() => {
    setTemplateSelectionModalActive(false);
    navigate('/app/create-template/carousel');
  }, [navigate]);

  const handleCatalogueTemplateSelect = useCallback(() => {
    setTemplateSelectionModalActive(false);
    navigate('/app/create-catalog-template');
  }, [navigate]);


  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);

  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);

  const handleCategoryChange = useCallback((value) => {
    setCategoryFilter(value);
  }, []);

  const handleQualityChange = useCallback((value) => {
    setQualityFilter(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleCategoryRemove = useCallback(() => setCategoryFilter(''), []);
  const handleQualityRemove = useCallback(() => setQualityFilter([]), []);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleCategoryRemove();
    handleQualityRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleCategoryRemove, handleQualityRemove, handleQueryValueRemove]);

  const onHandleCancel = () => { };
  const onHandleSave = async () => {
    await sleep(1);
    return true;
  };

  const primaryAction = selected === 0
    ? {
      type: 'save-as',
      onAction: onCreateNewView,
      disabled: false,
      loading: false,
    }
    : {
      type: 'save',
      onAction: onHandleSave,
      disabled: false,
      loading: false,
    };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'Approved': { tone: 'success' },
      'Pending': { tone: 'warning' },
      'Rejected': { tone: 'critical' },
      'Disabled': { tone: 'critical' },
      'Draft': { tone: 'subdued' },
    };

    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };


  const getQualityRating = (rating) => {
    let dotColor, dotLabel, badgeTone;

    switch (rating.toLowerCase()) {
      case 'green':
        dotColor = 'var(--p-color-bg-fill-success)';
        dotLabel = 'Green';
        badgeTone = 'success';
        break;
      case 'yellow':
        dotColor = 'var(--p-color-bg-fill-warning)';
        dotLabel = 'Yellow';
        badgeTone = 'warning';
        break;
      case 'red':
        dotColor = 'var(--p-color-bg-fill-critical)';
        dotLabel = 'Red';
        badgeTone = 'critical';
        break;
      case 'grey':
        dotColor = '#9ca3af';
        dotLabel = 'Grey';
        badgeTone = 'subdued';
        break;
      default:
        dotColor = 'var(--p-color-bg-fill-subdued)';
        dotLabel = 'Unknown';
        badgeTone = 'info';
    }

    return (
      <InlineStack gap="200" align="start">
        <div
          style={{
            width: '12px',
            height: '12px',
            borderRadius: '50%',
            backgroundColor: dotColor,
            marginTop: '2px'
          }}
        />
        <Text as="span" tone="subdued">
          {dotLabel}
        </Text>
      </InlineStack>
    );
  };

  // Filters configuration
  const filters = [
    {
      key: 'status',
      label: 'Template Status',
      filter: (
        <ChoiceList
          title="Template Status"
          titleHidden
          choices={[
            { label: 'Approved', value: 'approved' },
            { label: 'Rejected', value: 'rejected' },
            { label: 'Pending', value: 'pending' },
            { label: 'Disabled', value: 'disabled' },
            { label: 'Draft', value: 'draft' },
          ]}
          selected={statusFilter || []}
          onChange={handleStatusChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
    {
      key: 'category',
      label: 'Category',
      filter: (
        <TextField
          label="Category"
          value={categoryFilter}
          onChange={handleCategoryChange}
          autoComplete="off"
          labelHidden
        />
      ),
      shortcut: true,
    },
    {
      key: 'quality',
      label: 'Quality Rating',
      filter: (
        <ChoiceList
          title="Quality Rating"
          titleHidden
          choices={[
            { label: 'Green', value: 'green' },
            { label: 'Yellow', value: 'yellow' },
            { label: 'Red', value: 'red' },
            { label: 'Grey', value: 'grey' },
          ]}
          selected={qualityFilter || []}
          onChange={handleQualityChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
  ];

  // Applied filters
  const appliedFilters = [];
  if (statusFilter && !isEmpty(statusFilter)) {
    appliedFilters.push({
      key: 'status',
      label: disambiguateLabel('status', statusFilter),
      onRemove: handleStatusRemove,
    });
  }
  if (qualityFilter && !isEmpty(qualityFilter)) {
    appliedFilters.push({
      key: 'quality',
      label: disambiguateLabel('quality', qualityFilter),
      onRemove: handleQualityRemove,
    });
  }
  if (!isEmpty(categoryFilter)) {
    appliedFilters.push({
      key: 'category',
      label: disambiguateLabel('category', categoryFilter),
      onRemove: handleCategoryRemove,
    });
  }

  // Resource state for IndexTable
  const resourceName = {
    singular: 'template',
    plural: 'templates',
  };

  const { selectedResources, allResourcesSelected, handleSelectionChange } =
    useIndexResourceState(paginatedTemplates);

  // Helper functions
  function disambiguateLabel(key, value) {
    switch (key) {
      case 'quality':
        return value.map((val) => {
          return `Quality: ${val.charAt(0).toUpperCase() + val.slice(1)}`;
        }).join(', ');
      case 'category':
        return `Category contains "${value}"`;
      case 'status':
        return value.map((val) => `Status: ${val.charAt(0).toUpperCase() + val.slice(1)}`).join(', ');
      default:
        return value;
    }
  }

  function isEmpty(value) {
    if (Array.isArray(value)) {
      return value.length === 0;
    } else {
      return value === '' || value == null;
    }
  }

  // IndexTable row markup
  const rowMarkup = paginatedTemplates.map(
    (template, index) => (
      <IndexTable.Row
        id={template.id}
        key={template.id}
        selected={selectedResources.includes(template.id)}
        position={index}
      >
        <IndexTable.Cell>
          <Text variant="bodyMd" fontWeight="semibold" as="span">
            {template.label}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          {/* <Badge tone="info">{template.category}</Badge> */}
          <Text as="span">
            {template.category}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>{getStatusBadge(template.status)}</IndexTable.Cell>
        <IndexTable.Cell>
          {getQualityRating(template.qualityRating)}
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span">
            {template.language}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span" tone="subdued">
            {new Date(template.updatedOn).toLocaleDateString()}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end">
            <Tooltip content="Preview template">
              <Button
                size="slim"
                onClick={(event) => {
                  event.stopPropagation();
                  handlePreviewClick(template);
                }}
              >
                <Icon source={ViewIcon} />
              </Button>
            </Tooltip>
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  return (
    <>
      <Page
        title="Templates"
        subtitle="Create and manage templates for automated messages"
        fullWidth
        primaryAction={{
          content: (
            <InlineStack alignment="center">

              <Text as="span">create template</Text>
            </InlineStack>
          ),
          onAction: handleTemplateSelectionModalOpen
        }}
        secondaryActions={[
          {
            content: (
              <InlineStack alignment="center">
                <Icon source={RefreshIcon} />
                <Text as="span">Refresh</Text>
              </InlineStack>
            ),
            onAction: handleRefresh,
            loading: loading
          }
        ]}
      >
        {error && (
          <Banner
            title="Error loading templates"
            tone="critical"
            onDismiss={() => setError(null)}
            action={{
              content: 'Retry',
              onAction: handleRefresh
            }}
          >
            <p>{error}</p>
          </Banner>
        )}

        <LegacyCard>
          <IndexFilters
            sortOptions={sortOptions}
            sortSelected={sortSelected}
            queryValue={queryValue}
            queryPlaceholder="Search templates..."
            onQueryChange={handleFiltersQueryChange}
            onQueryClear={() => setQueryValue('')}
            onSort={setSortSelected}
            primaryAction={primaryAction}
            cancelAction={{
              onAction: onHandleCancel,
              disabled: false,
              loading: false,
            }}
            tabs={tabs}
            selected={selected}
            onSelect={setSelected}
            canCreateNewView
            onCreateNewView={onCreateNewView}
            filters={filters}
            appliedFilters={appliedFilters}
            onClearAll={handleFiltersClearAll}
            mode={mode}
            setMode={setMode}
          />
          {loading ? (
            <div style={{
              display: 'flex',
              justifyContent: 'center',
              alignItems: 'center',
              padding: '60px 20px',
              minHeight: '200px'
            }}>
              <BlockStack gap="300" align="center">
                <Spinner accessibilityLabel="Loading templates" size="large" />
                <Text variant="bodyMd" tone="subdued">Loading templates...</Text>
              </BlockStack>
            </div>
          ) : paginatedTemplates.length === 0 ? (
            <EmptyState
              heading="No templates found"
              image="https://cdn.shopify.com/s/files/1/0262/4071/2726/files/emptystate-files.png"
              action={{
                content: 'Create template',
                onAction: handleTemplateSelectionModalOpen
              }}
              secondaryAction={{
                content: 'Refresh',
                onAction: handleRefresh
              }}
            >
              <p>
                {queryValue || statusFilter.length > 0 || categoryFilter || qualityFilter.length > 0
                  ? 'No templates match your current filters. Try adjusting your search criteria.'
                  : 'Get started by creating your first WhatsApp template.'}
              </p>
            </EmptyState>
          ) : (
            <IndexTable
              resourceName={resourceName}
              itemCount={paginatedTemplates.length}
              selectedItemsCount={
                allResourcesSelected ? 'All' : selectedResources.length
              }
              onSelectionChange={handleSelectionChange}
              headings={[
                { title: 'Template Name' },
                { title: 'Category' },
                { title: 'Status' },
                { title: 'Quality Rating' },
                { title: 'Language' },
                { title: 'Updated On' },
                { title: 'Actions', alignment: 'end' },
              ]}
            >
              {rowMarkup}
            </IndexTable>
          )}

          {totalItems > 20 && (
            <div
              style={{
                maxWidth: '700px',
                margin: 'auto',
                border: '1px solid var(--p-color-border)',
                marginTop: '16px'
              }}
            >
              <Pagination
                onPrevious={hasPrevious ? handlePreviousPage : undefined}
                onNext={hasNext ? handleNextPage : undefined}
                type="table"
                hasNext={hasNext}
                hasPrevious={hasPrevious}
                label={`${startItem}-${endItem} of ${totalItems} templates`}
              />
            </div>
          )}
        </LegacyCard>
      </Page>

      {selectedTemplate && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Preview: ${selectedTemplate.label}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
        >
          <Modal.Section>
            <BlockStack gap="400">
              {/* Template Detail Error Banner */}
              {templateDetailError && (
                <Banner
                  title="Error loading template details"
                  tone="critical"
                  onDismiss={() => setTemplateDetailError(null)}
                  action={{
                    content: 'Retry',
                    onAction: () => fetchTemplateDetails(selectedTemplate.label)
                  }}
                >
                  <p>{templateDetailError}</p>
                </Banner>
              )}

              <BlockStack gap="200">
                <Text variant="headingMd" as="h3">
                  {selectedTemplate.label}
                </Text>
                <InlineStack gap="200">
                  <Badge tone="info">{selectedTemplate.category}</Badge>
                  {getStatusBadge(selectedTemplate.status)}
                  {getQualityRating(selectedTemplate.qualityRating)}
                </InlineStack>
              </BlockStack>

              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Language:</strong> {selectedTemplate.language}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Last Updated:</strong> {new Date(selectedTemplate.updatedOn).toLocaleDateString()}
                </Text>
              </TextContainer>

              <BlockStack gap="200">
                <Text variant="headingMd" as="h4">
                  WhatsApp Preview
                </Text>

                {loadingTemplateDetails ? (
                  <div style={{
                    display: 'flex',
                    justifyContent: 'center',
                    alignItems: 'center',
                    padding: '40px 20px',
                    minHeight: '200px'
                  }}>
                    <BlockStack gap="300" align="center">
                      <Spinner accessibilityLabel="Loading template details" size="large" />
                      <Text variant="bodyMd" tone="subdued">Loading template details...</Text>
                    </BlockStack>
                  </div>
                ) : selectedTemplateData ? (
                  <WhatsAppPreview
                    template={selectedTemplateData}
                    variant="full"
                    showWhatsAppHeader={true}
                  />
                ) : templateDetailError ? (
                  <div style={{
                    padding: '40px 20px',
                    textAlign: 'center',
                    border: '1px dashed var(--p-color-border)',
                    borderRadius: '8px'
                  }}>
                    <Text variant="bodyMd" tone="subdued">
                      Unable to load template preview. Please try again.
                    </Text>
                  </div>
                ) : (
                  <WhatsAppPreview
                    template={selectedTemplate.template}
                    variant="full"
                    showWhatsAppHeader={true}
                  />
                )}
              </BlockStack>
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}

      {/* Template Selection Modal */}
      <Modal
        open={templateSelectionModalActive}
        onClose={handleTemplateSelectionModalClose}
        title="Choose Template Type"
        size="large"
        primaryAction={{
          content: 'Cancel',
          onAction: handleTemplateSelectionModalClose,
        }}
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" as="p">
              Select the type of template you want to create:
            </Text>

            <InlineStack gap="300" align="start" wrap={false}>
              {/* Custom Template Card */}
              <div style={{ flex: '1' }}>
                <LegacyCard>
                  <div style={{ padding: '16px', textAlign: 'center', height: '100%' }}>
                    <BlockStack gap="200">
                      <div style={{
                        width: '48px',
                        height: '48px',
                        backgroundColor: 'rgb(249 160 246)',
                        borderRadius: '8px',
                        margin: '0 auto',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center'
                      }}>
                        <Icon source={CartIcon} tone="base" />
                      </div>
                      <BlockStack gap="100">
                        <Text variant="headingSm" as="h3">
                          Abandoned Cart Template
                        </Text>
                        <Text variant="bodySm" as="p" tone="subdued">
                          Create a abandoned cart message template with custom text, variables, and formatting.
                        </Text>
                      </BlockStack>
                      <Button
                        primary
                        onClick={handleAbandonedCartTemplateSelect}
                        fullWidth
                        size="slim"
                      >
                        Create Template
                      </Button>
                    </BlockStack>
                  </div>
                </LegacyCard>
              </div>
              <div style={{ flex: '1' }}>
                <LegacyCard>
                  <div style={{ padding: '16px', textAlign: 'center', height: '100%' }}>
                    <BlockStack gap="200">
                      <div style={{
                        width: '48px',
                        height: '48px',
                        backgroundColor: 'rgb(184 255 119)',
                        borderRadius: '8px',
                        margin: '0 auto',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center'
                      }}>
                        <Icon source={CartIcon} tone="base" />
                      </div>
                      <BlockStack gap="100">
                        <Text variant="headingSm" as="h3">
                          Order Confirm... Template
                        </Text>
                        <Text variant="bodySm" as="p" tone="subdued">
                          Create a short order confirmation template with custom text, variables, and formatting. </Text>
                      </BlockStack>
                      <Button
                        primary
                        onClick={handleOrderConfirmationTemplateSelect}
                        fullWidth
                        size="slim"
                      >
                        Create Template
                      </Button>
                    </BlockStack>
                  </div>
                </LegacyCard>
              </div>
              <div style={{ flex: '1' }}>
                <LegacyCard>
                  <div style={{ padding: '16px', textAlign: 'center', height: '100%' }}>
                    <BlockStack gap="200">
                      <div style={{
                        width: '48px',
                        height: '48px',
                        backgroundColor: 'var(--p-color-bg-fill-info)',
                        borderRadius: '8px',
                        margin: '0 auto',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center'
                      }}>
                        <Icon source={ThemeTemplateIcon} tone="base" />
                      </div>
                      <BlockStack gap="100">
                        <Text variant="headingSm" as="h3">
                          Custom Template
                        </Text>
                        <Text variant="bodySm" as="p" tone="subdued">
                          Create a personalized message template with custom text, variables, and formatting.
                        </Text>
                      </BlockStack>
                      <Button
                        primary
                        onClick={handleCustomTemplateSelect}
                        fullWidth
                        size="slim"
                      >
                        Create Custom
                      </Button>
                    </BlockStack>
                  </div>
                </LegacyCard>
              </div>



              {/* Carousel Template Card */}
              <div style={{ flex: '1' }}>
                <LegacyCard>
                  <div style={{ padding: '16px', textAlign: 'center', height: '100%' }}>
                    <BlockStack gap="200">
                      <div style={{
                        width: '48px',
                        height: '48px',
                        backgroundColor: 'var(--p-color-bg-fill-warning)',
                        borderRadius: '8px',
                        margin: '0 auto',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center'
                      }}>
                        <Icon source={SlideshowIcon} tone="base" />
                      </div>
                      <BlockStack gap="100">
                        <Text variant="headingSm" as="h3">
                          Carousel Template
                        </Text>
                        <Text variant="bodySm" as="p" tone="subdued">
                          Create an interactive carousel template with multiple cards and call-to-action buttons.
                        </Text>
                      </BlockStack>
                      <Button
                        onClick={handleCarouselTemplateSelect}
                        fullWidth
                        size="slim"
                      >
                        Create Carousel
                      </Button>
                    </BlockStack>
                  </div>
                </LegacyCard>
              </div>

              {/* Catalogue Template Card */}
              <div style={{ flex: '1' }}>
                <LegacyCard>
                  <div style={{ padding: '16px', textAlign: 'center', height: '100%' }}>
                    <BlockStack gap="200">
                      <div style={{
                        width: '48px',
                        height: '48px',
                        backgroundColor: 'rgb(91 213 157)',
                        borderRadius: '8px',
                        margin: '0 auto',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center'
                      }}>
                        <Icon source={StoreIcon} tone="base" />
                      </div>
                      <BlockStack gap="100">
                        <Text variant="headingSm" as="h3">
                          Catalogue Template
                        </Text>
                        <Text variant="bodySm" as="p" tone="subdued">
                          Create a product catalogue template to showcase your products with images and details.
                        </Text>
                      </BlockStack>
                      <Button
                        onClick={handleCatalogueTemplateSelect}
                        fullWidth
                        size="slim"
                      >
                        Create Catalogue
                      </Button>
                    </BlockStack>
                  </div>
                </LegacyCard>
              </div>
            </InlineStack>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </>
  );
}

export default TemplatesPage;