import {
  Page, 
  Badge, 
  LegacyCard, 
  Button, 
  Modal, 
  TextContainer,
  Thumbnail,
  Text,
  InlineStack,
  BlockStack,
  TextField,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  Pagination,
  Icon,
  Tooltip,
  Spinner,
  Banner,
} from '@shopify/polaris';
import { RefreshIcon, ViewIcon } from '@shopify/polaris-icons';
// IndexFiltersProps and TabProps types will be inferred
import React, { useState, useCallback, useMemo, useEffect } from 'react';
import { fetchProductList } from '../utils/productApi';
import { getSelectedTemplatebyName } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';


function ProductsPage() {
  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
  
  // API state
  const [products, setProducts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [apiPagination, setApiPagination] = useState({
    currentPage: 1,
    totalPages: 1,
    total: 0,
    perPage: 20
  });
  
  const [selectedProduct, setSelectedProduct] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  
  // Template preview states
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [templateError, setTemplateError] = useState(null);
  
  
  const [itemStrings, setItemStrings] = useState([
    'All',
    'Active Products',
    'Draft Products'
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['name asc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [collectionFilter, setCollectionFilter] = useState('');
  const [priceRange, setPriceRange] = useState([0, 500]);

  // Transform API product data
  const transformProductData = (apiProducts) => {
    return apiProducts.map((product) => {
      // Parse product_img JSON
      let imageUrl = 'https://cdn.shopify.com/s/files/1/0757/9955/files/only-yesterday-icon.png';
      try {
        const imgData = JSON.parse(product.product_img);
        imageUrl = imgData.image_url || imageUrl;
      } catch (e) {
        // Use default image if parsing fails
      }

      return {
        id: product.id.toString(),
        name: product.product_name || 'Untitled Product',
        status: 'Active', 
        collection: product.category?.category_name || '—',
        template:  product.template_info?.template_name || '—', 
        templateId: product.template_info?.template_id || '—',
        templateStatus: product.template_info?.status || '—',
        triggerName: product.trigger_info?.triggerName || '—',
        triggerId: product.trigger_info?.id || '—',
        image: imageUrl,
        description: `Product ID: ${product.product_id}`,
        
      };
    });
  };

  // Fetch products from API
  const fetchProducts = async (page = 1) => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await fetchProductList({ 
        page, 
        perPageData: apiPagination.perPage 
      });
      
      if (response.status && response.data) {
        const transformedProducts = transformProductData(response.data.data);
        setProducts(transformedProducts);
        
        setApiPagination({
          currentPage: response.data.current_page,
          totalPages: response.data.last_page,
          total: response.data.total,
          perPage: response.data.per_page
        });
      } else {
        throw new Error('Invalid API response');
      }
    } catch (err) {
      console.error('Error fetching products:', err);
      setError(err.message || 'Failed to fetch products');
      setProducts([]);
    } finally {
      setLoading(false);
    }
  };


  useEffect(() => {
    fetchProducts(1);
  }, []);

  const handleRefresh = () => {
    fetchProducts(apiPagination.currentPage);
  };
  
  const {mode, setMode} = useSetIndexFiltersMode();

  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 20; 

  // Products are loaded from the server via loader

  // Tab management
  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => {},
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      {
        type: 'rename',
        onAction: () => {},
        onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => {
            if (idx === index) {
              return value;
            }
            return item.content;
          });
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      {
        type: 'duplicate',
        onPrimaryAction: async (value) => {
          await sleep(1);
          duplicateView(value);
          return true;
        },
      },
      {
        type: 'edit',
      },
      {
        type: 'delete',
        onPrimaryAction: async () => {
          await sleep(1);
          deleteView(index);
          return true;
        },
      },
    ],
  }));

  // Filter and search logic
  const filteredProducts = useMemo(() => {
    let filtered = products;
    
    // Filter by tab selection first
    if (selected === 1) { // Active Products
      filtered = filtered.filter(product => product.status === 'Active');
    } else if (selected === 2) { // Draft Products
      filtered = filtered.filter(product => product.status === 'Draft');
    }
    // selected === 0 is "All" so no additional filtering needed
    
    return filtered.filter((product) => {
      const matchesSearch = product.name.toLowerCase().includes(queryValue.toLowerCase()) ||
                           product.collection.toLowerCase().includes(queryValue.toLowerCase());
      const matchesStatus = statusFilter.length === 0 || statusFilter.includes(product.status.toLowerCase());
      const matchesCollection = collectionFilter === '' || product.collection.toLowerCase().includes(collectionFilter.toLowerCase());
      // Remove price filtering since we removed price field
      const matchesPrice = true;
      
      return matchesSearch && matchesStatus && matchesCollection && matchesPrice;
    });
  }, [products, queryValue, statusFilter, collectionFilter, priceRange, selected]);

  // Sort products
  const sortedProducts = useMemo(() => {
    const sorted = [...filteredProducts];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');
    
    sorted.sort((a, b) => {
      let aValue, bValue;
      
      switch (sortKey) {
        case 'name':
          aValue = a.name.toLowerCase();
          bValue = b.name.toLowerCase();
          break;
        case 'status':
          aValue = a.status.toLowerCase();
          bValue = b.status.toLowerCase();
          break;
        case 'collection':
          aValue = a.collection.toLowerCase();
          bValue = b.collection.toLowerCase();
          break;
        case 'template':
          aValue = a.template.toLowerCase();
          bValue = b.template.toLowerCase();
          break;
        default:
          return 0;
      }
      
      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });
    
    return sorted;
  }, [filteredProducts, sortSelected]);

  // Use all products since API handles pagination
  const paginatedProducts = useMemo(() => {
    return sortedProducts;
  }, [sortedProducts]);

  // Remove local pagination since we use API pagination

  // Sort options
  const sortOptions = [
    {label: 'Product Name', value: 'name asc', directionLabel: 'A-Z'},
    {label: 'Product Name', value: 'name desc', directionLabel: 'Z-A'},
    {label: 'Status', value: 'status asc', directionLabel: 'A-Z'},
    {label: 'Status', value: 'status desc', directionLabel: 'Z-A'},
    {label: 'Collection', value: 'collection asc', directionLabel: 'A-Z'},
    {label: 'Collection', value: 'collection desc', directionLabel: 'Z-A'},
    {label: 'Template Name', value: 'template asc', directionLabel: 'A-Z'},
    {label: 'Template Name', value: 'template desc', directionLabel: 'Z-A'},
  ];

  // Function to fetch template details from API
  const fetchTemplateDetails = useCallback(async (templateName) => {
    if (!templateName || templateName === '—' || templateName === 'Unknown Template') {
      return null;
    }

    try {
      setLoadingTemplateDetails(true);
      setTemplateError(null);
      
      const templateData = await getSelectedTemplatebyName(templateName);
      setSelectedTemplateData(templateData);
      
      return templateData;
    } catch (error) {
      console.error('Error fetching template details:', error);
      setTemplateError(error.message || 'Failed to fetch template details');
      return null;
    } finally {
      setLoadingTemplateDetails(false);
    }
  }, []);

  // Event handlers
  const handlePreviewClick = useCallback(async (product) => {
    setSelectedProduct(product);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setPreviewModalActive(true);
    
    // Fetch template details if template name is available
    if (product.template && product.template !== '—') {
      await fetchTemplateDetails(product.template);
    }
  }, [fetchTemplateDetails]);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setSelectedProduct(null);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setLoadingTemplateDetails(false);
  }, []);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);
  
  const handleCollectionChange = useCallback((value) => {
    setCollectionFilter(value);
  }, []);
  
  const handlePriceRangeChange = useCallback((value) => {
    setPriceRange(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleCollectionRemove = useCallback(() => setCollectionFilter(''), []);
  const handlePriceRangeRemove = useCallback(() => setPriceRange([0, 500]), []);

  // Handle API pagination
  const handleApiPreviousPage = useCallback(async () => {
    if (apiPagination.currentPage > 1) {
      await fetchProducts(apiPagination.currentPage - 1);
    }
  }, [apiPagination.currentPage]);

  const handleApiNextPage = useCallback(async () => {
    if (apiPagination.currentPage < apiPagination.totalPages) {
      await fetchProducts(apiPagination.currentPage + 1);
    }
  }, [apiPagination.currentPage, apiPagination.totalPages]);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleCollectionRemove();
    handlePriceRangeRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleCollectionRemove, handlePriceRangeRemove, handleQueryValueRemove]);

  const onHandleCancel = () => {};
  const onHandleSave = async () => {
    await sleep(1);
    return true;
  };

  const primaryAction = selected === 0
    ? {
        type: 'save-as',
        onAction: onCreateNewView,
        disabled: false,
        loading: false,
      }
    : {
        type: 'save',
        onAction: onHandleSave,
        disabled: false,
        loading: false,
      };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'Active': { tone: 'success' },
      'Draft': { tone: 'attention' },
      'Archived': { tone: 'subdued' }
    };
    
    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };

  // Filters configuration
  const filters = [
    {
      key: 'status',
      label: 'Product Status',
      filter: (
        <ChoiceList
          title="Product Status"
          titleHidden
          choices={[
            {label: 'Active', value: 'active'},
            {label: 'Draft', value: 'draft'},
            {label: 'Archived', value: 'archived'},
          ]}
          selected={statusFilter || []}
          onChange={handleStatusChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
    {
      key: 'collection',
      label: 'Collection',
      filter: (
        <TextField
          label="Collection"
          value={collectionFilter}
          onChange={handleCollectionChange}
          autoComplete="off"
          labelHidden
        />
      ),
      shortcut: true,
    },
    {
      key: 'priceRange',
      label: 'Price Range',
      filter: (
        <RangeSlider
          label="Price range"
          labelHidden
          value={priceRange || [0, 500]}
          prefix="$"
          output
          min={0}
          max={500}
          step={5}
          onChange={handlePriceRangeChange}
        />
      ),
    },
  ];

  // Applied filters
  const appliedFilters = [];
  if (statusFilter && !isEmpty(statusFilter)) {
    appliedFilters.push({
      key: 'status',
      label: disambiguateLabel('status', statusFilter),
      onRemove: handleStatusRemove,
    });
  }
  if (priceRange && (priceRange[0] !== 0 || priceRange[1] !== 500)) {
    appliedFilters.push({
      key: 'priceRange',
      label: disambiguateLabel('priceRange', priceRange),
      onRemove: handlePriceRangeRemove,
    });
  }
  if (!isEmpty(collectionFilter)) {
    appliedFilters.push({
      key: 'collection',
      label: disambiguateLabel('collection', collectionFilter),
      onRemove: handleCollectionRemove,
    });
  }

  // Resource state for IndexTable
  const resourceName = {
    singular: 'product',
    plural: 'products',
  };

  const {selectedResources, allResourcesSelected, handleSelectionChange} =
    useIndexResourceState(paginatedProducts);

  // Helper functions
  function disambiguateLabel(key, value) {
    switch (key) {
      case 'priceRange':
        return `Price is between $${value[0]} and $${value[1]}`;
      case 'collection':
        return `Collection contains "${value}"`;
      case 'status':
        return value.map((val) => `Status: ${val.charAt(0).toUpperCase() + val.slice(1)}`).join(', ');
      default:
        return value;
    }
  }

  function isEmpty(value) {
    if (Array.isArray(value)) {
      return value.length === 0;
    } else {
      return value === '' || value == null;
    }
  }


  const rowMarkup = paginatedProducts.map(
    (product, index) => (
      <IndexTable.Row
        id={product.id}
        key={product.id}
        selected={selectedResources.includes(product.id)}
        position={index}
      >
        <IndexTable.Cell>
          <InlineStack gap="300" align="start">
            <Thumbnail
              source={product.image}
              alt={product.name}
              size="small"
            />
            <Text variant="bodyMd" fontWeight="semibold" as="span">
            {product.name.length > 60 
    ? product.name.slice(0, 60) + "..." 
    : product.name}
            </Text>
          </InlineStack>
        </IndexTable.Cell>
        <IndexTable.Cell>{getStatusBadge(product.status)}</IndexTable.Cell>
        <IndexTable.Cell>{product.collection}</IndexTable.Cell>
        <IndexTable.Cell>
          {product.triggerName}
        </IndexTable.Cell>
        <IndexTable.Cell>
          {product.template}
        </IndexTable.Cell>
        <IndexTable.Cell>
  <InlineStack align="end">
  <Tooltip content="see preview of assigned template">
    <Button
      size="slim"
      onClick={(event) => {
        event.stopPropagation();
        handlePreviewClick(product);
      }}
    >
      <Icon source={ViewIcon} />
    </Button>
    </Tooltip>
  </InlineStack>
</IndexTable.Cell>

      </IndexTable.Row>
    ),
  );

  return (
    <>
      <Page 
        title="All Products List" 
        subtitle="Manage your productwise Automations"
        fullWidth
        primaryAction={{
          content: (
            <InlineStack alignment="center">
              <Icon source={RefreshIcon} />
              <Text as="span">Sync Products</Text>
            </InlineStack>
          ),
          onAction: handleRefresh,
          loading: loading
        }}
      >
        {error && (
          <Banner
            title="Error loading products"
            tone="critical"
            onDismiss={() => setError(null)}
          >
            <p>{error}</p>
          </Banner>
        )}
        
        <LegacyCard>
          <IndexFilters
            sortOptions={sortOptions}
            sortSelected={sortSelected}
            queryValue={queryValue}
            queryPlaceholder="Search products..."
            onQueryChange={handleFiltersQueryChange}
            onQueryClear={() => setQueryValue('')}
            onSort={setSortSelected}
            primaryAction={primaryAction}
            cancelAction={{
              onAction: onHandleCancel,
              disabled: false,
              loading: false,
            }}
            tabs={tabs}
            selected={selected}
            onSelect={setSelected}
            canCreateNewView
            onCreateNewView={onCreateNewView}
            filters={filters}
            appliedFilters={appliedFilters}
            onClearAll={handleFiltersClearAll}
            mode={mode}
            setMode={setMode}
          />
          {loading ? (
            <div style={{ padding: '40px', textAlign: 'center' }}>
              <Spinner size="large" />
              <Text variant="bodyMd" as="p" tone="subdued">
                Loading products...
              </Text>
            </div>
          ) : (
            <IndexTable
              resourceName={resourceName}
              itemCount={paginatedProducts.length}
              selectedItemsCount={
                allResourcesSelected ? 'All' : selectedResources.length
              }
              onSelectionChange={handleSelectionChange}
              headings={[
                {title: 'Product'},
                {title: 'Status'},
                {title: 'Collection'},
                {title: 'Trigger Name'},
                {title: 'Template Name'},
                {title: 'Preview Template', alignment: 'end'},
              ]}
            >
              {rowMarkup}
            </IndexTable>
          )}
          
          {!loading && apiPagination.totalPages > 1 && (
            <div
              style={{
                maxWidth: '700px',
                margin: 'auto',
                border: '1px solid var(--p-color-border)',
                marginTop: '16px'
              }}
            >
              <Pagination
                onPrevious={apiPagination.currentPage > 1 ? handleApiPreviousPage : undefined}
                onNext={apiPagination.currentPage < apiPagination.totalPages ? handleApiNextPage : undefined}
                type="table"
                hasNext={apiPagination.currentPage < apiPagination.totalPages}
                hasPrevious={apiPagination.currentPage > 1}
                label={`Page ${apiPagination.currentPage} of ${apiPagination.totalPages} (${apiPagination.total} total products)`}
              />
            </div>
          )}
        </LegacyCard>
      </Page>

      {selectedProduct && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Preview: ${selectedProduct.name}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
          secondaryActions={[
            {
              content: 'Edit Product',
              onAction: () => {
                console.log('Edit product:', selectedProduct.id);
                handleModalClose();
              }
            }
          ]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <InlineStack align="center" gap="400">
                <Thumbnail
                  source={selectedProduct.image}
                  alt={selectedProduct.name}
                  size="large"
                />
                <BlockStack gap="200">
                  <Text variant="headingMd" as="h3">
                    {selectedProduct.name}
                  </Text>
                  <Badge tone={selectedProduct.template !== '—' ? 'success' : 'subdued'}>
                    Template: {selectedProduct.template}
                  </Badge>
                  {getStatusBadge(selectedProduct.status)}
                </BlockStack>
              </InlineStack>
              
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Description:</strong> {selectedProduct.description}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Collection:</strong> {selectedProduct.collection}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Template Name:</strong> {selectedProduct.template}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Trigger Name:</strong> {selectedProduct.triggerName}
                </Text>
              </TextContainer>

              {/* Template Error Banner */}
              {templateError && (
                <Banner
                  title="Template Loading Error"
                  tone="critical"
                  onDismiss={() => setTemplateError(null)}
                >
                  <Text variant="bodyMd" as="p">
                    {templateError}
                  </Text>
                </Banner>
              )}

              {/* Template Preview Section */}
              {selectedProduct.template && selectedProduct.template !== '—' && (
                <BlockStack gap="200">
                  <Text variant="headingMd" as="h4">
                    WhatsApp Template Preview
                  </Text>
                  
                  {loadingTemplateDetails ? (
                    <div style={{ textAlign: 'center', padding: '40px 20px' }}>
                      <Spinner accessibilityLabel="Loading template details" size="large" />
                      <Text variant="bodyMd" as="p" tone="subdued" alignment="center">
                        Loading template details...
                      </Text>
                    </div>
                  ) : selectedTemplateData ? (
                    <WhatsAppPreview 
                      template={selectedTemplateData} 
                      variant="full" 
                      showWhatsAppHeader={true}
                    />
                  ) : (
                    <Text variant="bodyMd" as="p" tone="subdued">
                      Template preview not available for "{selectedProduct.template}"
                    </Text>
                  )}
                </BlockStack>
              )}
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}
    </>
  );
}

export default ProductsPage;