import { useState, useCallback, useEffect } from "react";
import {
  Page,
  Button,
  Text,
  Banner,
  Badge,
  ButtonGroup,
  Card,
  Icon,
  Modal,
  TextContainer,
  DataTable,
} from "@shopify/polaris";
import { CheckIcon, InfoIcon } from '@shopify/polaris-icons';
import { isPlanPurchased, getPlanName } from '../utils/planProtection';
import { getPricingPlanLink } from '../utils/pricing-planApi';

export default function PricingPlanPage() {
  // Function to get account details from localStorage
  const getAccountDetails = () => {
    try {
      const userData = localStorage.getItem('getgabs_user_data');
      if (userData) {
        const parsedData = JSON.parse(userData);
        
        if (!parsedData.userAllInfo) {
          return null;
        }

        // Map access value type to plan key
        const mapAccessValueTypeToPlanKey = (accessValueType) => {
          const planMapping = {
            0: null, // No plan
            1: 'basic', // Basic Monthly
            11: 'basic', // Basic Annually
            2: 'pro', // Pro Monthly
            22: 'pro', // Pro Annually
            4: 'plus', // Plus Monthly
            44: 'plus' // Plus Annually
          };
          return planMapping[accessValueType] || null;
        };

        return {
          isplanpurchased: parsedData.userAllInfo.access_value_type > 0,
          planKey: mapAccessValueTypeToPlanKey(parsedData.userAllInfo.access_value_type),
          accessValueType: parsedData.userAllInfo.access_value_type
        };
      }
    } catch (error) {
      console.error('Error getting account details:', error);
    }
    return null;
  };

  // Define plans first
  const plans = {
    basic: {
      name: "Basic",
      price: { monthly: 5, yearly: 50 },
      description: "Ideal for Startup",
      features: [
        { name: "Setup / Onboarding Fee", value: "No Cost" },
        { name: "Broadcast", value: "30,000/month" },
        { name: "Team Inbox", included: true },
        { name: "Chatbot Flow", value: "Unlimited" },
        { name: "AI Support Agent", included: false },
        { name: "Chat Widget", value: "3,000 Views/month" },
        { name: "WhatsApp Form", value: "1" },
        { name: "Catalog / Commerce", included: false },
        { name: "Webhooks", included: false },
        { name: "REST APIs", value: "50 calls/minutes" },
        { name: "Support", value: "Email + Chat" },
      ],
      limits: {
        messages: 30000,
        templates: "Unlimited",
        contacts: "Unlimited",
        broadcasts: 30000,
      },
      popular: false,
      color: "success",
    },
    plus: {
      name: "Plus",
      price: { monthly: 11, yearly: 110 },
      description: "Ideal for Growing Business",
      features: [
        { name: "Setup / Onboarding Fee", value: "No Cost" },
        { name: "Broadcast", value: "5,00,000/month" },
        { name: "Team Inbox", included: true },
        { name: "Chatbot Flow", value: "Unlimited" },
        { name: "AI Support Agent", included: true },
        { name: "Chat Widget", value: "30,000 Views/month" },
        { name: "WhatsApp Form", value: "5" },
        { name: "Catalog / Commerce", included: true },
        { name: "Webhooks", included: true },
        { name: "REST APIs", value: "180 calls/minutes" },
        { name: "Support", value: "Priority Support" },
      ],
      limits: {
        messages: 500000,
        templates: "Unlimited",
        contacts: "Unlimited",
        broadcasts: 500000,
      },
      popular: true,
      color: "success",
      badge: "Bestseller",
    },
    pro: {
      name: "Pro",
      price: { monthly: 59, yearly: 590 },
      description: "Ideal for Mid and Large Business",
      features: [
        { name: "Setup / Onboarding Fee", value: "No Cost" },
        { name: "Broadcast", value: "Unlimited" },
        { name: "Team Inbox", included: true },
        { name: "Chatbot Flow", value: "Unlimited" },
        { name: "AI Support Agent", included: true },
        { name: "Chat Widget", value: "10,00,000 Views/month" },
        { name: "WhatsApp Form", value: "50" },
        { name: "Catalog / Commerce", included: true },
        { name: "Webhooks", included: true },
        { name: "REST APIs", value: "360 calls/minutes" },
        { name: "Support", value: "Dedicated Support" },
      ],
      limits: {
        messages: "Unlimited",
        templates: "Unlimited",
        contacts: "Unlimited",
        broadcasts: "Unlimited",
      },
      popular: false,
      color: "info",
    },
  };

  // Get real account details from localStorage
  const accountDetails = getAccountDetails();
  
  // Plan purchase status from real API data
  const [planPurchased, setPlanPurchased] = useState(() => accountDetails?.isplanpurchased || false);
  
  const [currentPlan, setCurrentPlan] = useState(() => {
    // Use real plan data if available, otherwise fallback to utility
    if (accountDetails?.planKey && plans[accountDetails.planKey]) {
      return accountDetails.planKey;
    }
    const planName = getPlanName();
    return planName && plans[planName] ? planName : "plus";
  });
  const [billingCycle, setBillingCycle] = useState("monthly");
  const [showUpgradeModal, setShowUpgradeModal] = useState(false);
  const [selectedPlan, setSelectedPlan] = useState("");
  const [showUsageModal, setShowUsageModal] = useState(false);
  const [loadingPlan, setLoadingPlan] = useState("");

  const [usageData] = useState({
    messages: { used: 1250, limit: 2000 },
    templates: { used: 8, limit: 15 },
    contacts: { used: 450, limit: 1000 },
    broadcasts: { used: 12, limit: 25 },
  });

  // Update plan status when localStorage changes
  useEffect(() => {
    const updatePlanStatus = () => {
      const details = getAccountDetails();
      if (details) {
        setPlanPurchased(details.isplanpurchased);
        if (details.planKey && plans[details.planKey]) {
          setCurrentPlan(details.planKey);
        }
      }
    };

    // Listen for storage changes
    window.addEventListener('storage', updatePlanStatus);
    
    return () => {
      window.removeEventListener('storage', updatePlanStatus);
    };
  }, []);

  // Handlers
  const handlePlanSelect = useCallback(async (planKey) => {
    try {
      setLoadingPlan(planKey);
      
      // Map plan keys to API plan IDs based on billing cycle
      const planIdMapping = {
        basic: billingCycle === 'monthly' ? 1 : 11,
        pro: billingCycle === 'monthly' ? 2 : 22,
        plus: billingCycle === 'monthly' ? 4 : 44
      };
      
      const planId = planIdMapping[planKey];
      if (!planId) {
        console.error('Invalid plan key:', planKey);
        return;
      }
      
      // Call API to get pricing link
      const response = await getPricingPlanLink(planId);
      
      if (response.status && response.redirect_url) {
        // Use the redirect_url from API response
        window.open(response.redirect_url, '_blank', 'noopener,noreferrer');
      } else {
        console.error('Invalid API response:', response);
        // Fallback to hardcoded URLs if API fails
        const fallbackUrls = {
          basic: 'https://getgabs.com/pricing-basic',
          plus: 'https://getgabs.com/pricing-plus',
          pro: 'https://getgabs.com/pricing-pro'
        };
        const fallbackUrl = fallbackUrls[planKey];
        if (fallbackUrl) {
          window.open(fallbackUrl, '_blank', 'noopener,noreferrer');
        }
      }
    } catch (error) {
      console.error('Error fetching pricing link:', error);
      // Fallback to hardcoded URLs if API fails
      const fallbackUrls = {
        basic: 'https://getgabs.com/pricing-basic',
        plus: 'https://getgabs.com/pricing-plus',
        pro: 'https://getgabs.com/pricing-pro'
      };
      const fallbackUrl = fallbackUrls[planKey];
      if (fallbackUrl) {
        window.open(fallbackUrl, '_blank', 'noopener,noreferrer');
      }
    } finally {
      setLoadingPlan("");
    }
  }, [billingCycle]);

  const handleUpgrade = useCallback(() => {
    console.log(`Upgrading to ${selectedPlan} plan...`);
    setCurrentPlan(selectedPlan);
    setShowUpgradeModal(false);
  }, [selectedPlan]);

  const calculateSavings = (plan) => {
    const monthly = plan.price.monthly * 12;
    const yearly = plan.price.yearly;
    return monthly - yearly;
  };

  const getUsagePercentage = (used, limit) => {
    if (limit === "Unlimited") return 0;
    return Math.min((used / limit) * 100, 100);
  };

  const getUsageColor = (percentage) => {
    if (percentage >= 90) return "critical";
    if (percentage >= 75) return "warning";
    return "success";
  };

  const renderPlanCard = (planKey, plan) => {
    const isCurrentPlan = currentPlan === planKey;
    const isPurchasedPlan = planPurchased && isCurrentPlan;
    const price = plan.price[billingCycle];
    const savings = billingCycle === "yearly" ? calculateSavings(plan) : 0;
    const discountPercentage = billingCycle === "yearly" ? "(0% Markup)" : "";

    return (
      <div key={planKey} style={{ position: 'relative' }}>
        {plan.badge && (
          <div style={{
            position: 'absolute',
            top: '-10px',
            left: '50%',
            transform: 'translateX(-50%)',
            backgroundColor: '#F7E96C',
            padding: '8px 24px',
            borderRadius: '20px',
            fontSize: '14px',
            fontWeight: 'bold',
            zIndex: 1,
          }}>
            {plan.badge}
          </div>
        )}
        
        {isPurchasedPlan && (
          <div style={{
            position: 'absolute',
            top: '10px',
            right: '10px',
            backgroundColor: '#008060',
            color: 'white',
            padding: '4px 12px',
            borderRadius: '12px',
            fontSize: '12px',
            fontWeight: 'bold',
            zIndex: 1,
          }}>
            Current Plan
          </div>
        )}
        
        <Card sectioned>
          <div style={{ display: 'flex', flexDirection: 'column', gap: '20px', height: '100%', padding: '20px 0' }}>
            {/* Header */}
            <div style={{ textAlign: 'left' }}>
              <div style={{ display: 'flex', alignItems: 'center', gap: '8px', marginBottom: '4px' }}>
                <Text variant="headingLg" fontWeight="bold">
                  {plan.name}
                </Text>
                <Text variant="bodySm" color="subdued">
                  {discountPercentage}
                </Text>
              </div>
              <Text variant="bodyMd" color="subdued">
                {plan.description}
              </Text>
            </div>

        
            <div style={{ display: 'flex', alignItems: 'baseline', gap: '4px' }}>
              <Text variant="bodyLg" color="subdued">$</Text>
              <Text variant="heading2xl" fontWeight="bold">
                {price.toLocaleString()}
              </Text>
              <Text variant="bodyMd" color="subdued">
                /{billingCycle === "monthly" ? "month" : "year"}
              </Text>
            </div>

           
            <div>
              <Button 
                fullWidth 
                primary={!isPurchasedPlan}
                variant={isPurchasedPlan ? "plain" : "primary"}
                disabled={isPurchasedPlan || loadingPlan === planKey}
                loading={loadingPlan === planKey}
                style={{
                  backgroundColor: isPurchasedPlan ? '#f6f6f7' : (planKey === 'basic' ? '#1a5f3f' : planKey === 'plus' ? '#90EE90' : '#1a5f3f'),
                  color: isPurchasedPlan ? '#6d7175' : (planKey === 'plus' ? '#000' : '#fff'),
                  border: isPurchasedPlan ? '1px solid #c9cccf' : 'none',
                  padding: '12px',
                  fontWeight: 'bold'
                }}
                onClick={() => !isPurchasedPlan && handlePlanSelect(planKey)}
              >
                {isPurchasedPlan ? 'Current Plan' : (loadingPlan === planKey ? 'Getting Link...' : 'Buy Now')}
              </Button>
            </div>

            {/* Features List */}
            <div style={{ display: 'flex', flexDirection: 'column', gap: '12px', flex: 1 }}>
              {plan.features.map((feature, index) => (
                <div key={index} style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', padding: '4px 0' }}>
                  <div style={{ display: 'flex', alignItems: 'center', gap: '8px' }}>
                    <Text variant="bodyMd">{feature.name}</Text>
                   <Icon source={InfoIcon} color="subdued" />
                  </div>
                  
                  <div style={{ display: 'flex', alignItems: 'center' }}>
                    {feature.included === true && (
                      <Icon source={CheckIcon} color="success" />
                    )}
                    {feature.included === false && (
                      <div style={{ 
                        width: '20px', 
                        height: '20px', 
                        borderRadius: '50%', 
                        border: '2px solid #ccc',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center'
                      }}>
                        <span style={{ color: '#ccc', fontSize: '14px' }}>×</span>
                      </div>
                    )}
                    {feature.value && (
                      <Text variant="bodySm" color="subdued" style={{ fontStyle: 'italic' }}>
                        {feature.value}
                      </Text>
                    )}
                  </div>
                </div>
              ))}
            </div>
          </div>
        </Card>
      </div>
    );
  };

  const renderUsageCard = () => {
    const currentPlanData = plans[currentPlan];
    
    if (!currentPlanData) {
      return null; // Return null if plan data is not found
    }
    
    return (
      <Card sectioned title="Current Usage">
        <div style={{ display: 'flex', flexDirection: 'column', gap: '20px' }}>
          <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
            <Text variant="headingMd">
              {currentPlanData.name} Plan
            </Text>
            <Button size="slim" onClick={() => setShowUsageModal(true)}>
              View Details
            </Button>
          </div>

          <div style={{ display: 'flex', flexDirection: 'column', gap: '20px' }}>
            <div>
              <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '8px' }}>
                <Text variant="bodyMd">Messages</Text>
                <Text variant="bodySm" color="subdued">
                  {usageData.messages.used.toLocaleString()} / {currentPlanData.limits.messages.toLocaleString()}
                </Text>
              </div>
              <div style={{
                width: '100%',
                height: '8px',
                backgroundColor: '#f0f0f0',
                borderRadius: '4px',
                overflow: 'hidden'
              }}>
                <div style={{
                  width: `${getUsagePercentage(usageData.messages.used, currentPlanData.limits.messages)}%`,
                  height: '100%',
                  backgroundColor: getUsagePercentage(usageData.messages.used, currentPlanData.limits.messages) >= 90 ? '#d73502' : getUsagePercentage(usageData.messages.used, currentPlanData.limits.messages) >= 75 ? '#bf5000' : '#008060',
                  transition: 'width 0.3s ease'
                }} />
              </div>
            </div>

            <div>
              <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '8px' }}>
                <Text variant="bodyMd">Templates</Text>
                <Text variant="bodySm" color="subdued">
                  {usageData.templates.used} / {currentPlanData.limits.templates}
                </Text>
              </div>
              <div style={{
                width: '100%',
                height: '8px',
                backgroundColor: '#f0f0f0',
                borderRadius: '4px',
                overflow: 'hidden'
              }}>
                <div style={{
                  width: `${getUsagePercentage(usageData.templates.used, currentPlanData.limits.templates)}%`,
                  height: '100%',
                  backgroundColor: getUsagePercentage(usageData.templates.used, currentPlanData.limits.templates) >= 90 ? '#d73502' : getUsagePercentage(usageData.templates.used, currentPlanData.limits.templates) >= 75 ? '#bf5000' : '#008060',
                  transition: 'width 0.3s ease'
                }} />
              </div>
            </div>

            <div>
              <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '8px' }}>
                <Text variant="bodyMd">Contacts</Text>
                <Text variant="bodySm" color="subdued">
                  {usageData.contacts.used.toLocaleString()} / {currentPlanData.limits.contacts.toLocaleString()}
                </Text>
              </div>
              <div style={{
                width: '100%',
                height: '8px',
                backgroundColor: '#f0f0f0',
                borderRadius: '4px',
                overflow: 'hidden'
              }}>
                <div style={{
                  width: `${getUsagePercentage(usageData.contacts.used, currentPlanData.limits.contacts)}%`,
                  height: '100%',
                  backgroundColor: getUsagePercentage(usageData.contacts.used, currentPlanData.limits.contacts) >= 90 ? '#d73502' : getUsagePercentage(usageData.contacts.used, currentPlanData.limits.contacts) >= 75 ? '#bf5000' : '#008060',
                  transition: 'width 0.3s ease'
                }} />
              </div>
            </div>

            <div>
              <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', marginBottom: '8px' }}>
                <Text variant="bodyMd">Broadcasts</Text>
                <Text variant="bodySm" color="subdued">
                  {usageData.broadcasts.used} / {currentPlanData.limits.broadcasts}
                </Text>
              </div>
              <div style={{
                width: '100%',
                height: '8px',
                backgroundColor: '#f0f0f0',
                borderRadius: '4px',
                overflow: 'hidden'
              }}>
                <div style={{
                  width: `${getUsagePercentage(usageData.broadcasts.used, currentPlanData.limits.broadcasts)}%`,
                  height: '100%',
                  backgroundColor: getUsagePercentage(usageData.broadcasts.used, currentPlanData.limits.broadcasts) >= 90 ? '#d73502' : getUsagePercentage(usageData.broadcasts.used, currentPlanData.limits.broadcasts) >= 75 ? '#bf5000' : '#008060',
                  transition: 'width 0.3s ease'
                }} />
              </div>
            </div>
          </div>
        </div>
      </Card>
    );
  };

  const renderFeatureComparison = () => {
    return null; // Remove feature comparison for cleaner layout
  };

  return (
    <Page
      title="Pricing & Plans"
      subtitle="Choose the perfect plan for your business needs"
      fullWidth
    >
      <div style={{ padding: '0 24px', maxWidth: '1200px', margin: '0 auto' }}>
        {/* Current Plan Status Banner */}
        {planPurchased && currentPlan && plans[currentPlan] && (
          <div style={{ marginBottom: '24px' }}>
            <Banner status="success">
              <div style={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
                <div>
                  <Text variant="bodyMd" fontWeight="bold">
                    You are currently on the {plans[currentPlan].name} plan
                  </Text>
                  {accountDetails?.accessValueType && (
                    <Text variant="bodySm" color="subdued">
                      | 
                      {accountDetails.accessValueType % 10 === 1 || accountDetails.accessValueType % 10 === 2 || accountDetails.accessValueType % 10 === 4 ? ' Monthly' : ' Annual'} billing
                    </Text>
                  )}
                </div>
              </div>
            </Banner>
          </div>
        )}
        
        {!planPurchased && (
          <div style={{ marginBottom: '24px' }}>
            <Banner status="warning">
              <Text variant="bodyMd">
                You don't have an active plan. Choose a plan below to get started with all features.
              </Text>
            </Banner>
          </div>
        )}
       
        <div style={{ marginBottom: '24px' }}>
          <Card sectioned>
            <div style={{ display: 'flex', justifyContent: 'center' }}>
              <ButtonGroup segmented>
                <Button
                  pressed={billingCycle === "monthly"}
                  onClick={() => setBillingCycle("monthly")}
                >
                  Monthly
                </Button>
                <Button
                  pressed={billingCycle === "yearly"}
                  onClick={() => setBillingCycle("yearly")}
                >
                  Yearly {" "}
                  <Badge status="success">+ 2 months free</Badge>
                </Button>
              </ButtonGroup>
            </div>
          </Card>
        </div>

    
        <div style={{
          display: 'grid',
          gridTemplateColumns: 'repeat(3, 1fr)',
          gap: '24px',
          marginBottom: '40px',
          '@media (max-width: 768px)': {
            gridTemplateColumns: '1fr',
          }
        }}>
          {renderPlanCard("basic", plans.basic)}
          {renderPlanCard("plus", plans.plus)}
          {renderPlanCard("pro", plans.pro)}
        </div>

        {/* Contact Sales */}
        <div>
          <Card sectioned>
            <div style={{ display: 'flex', flexDirection: 'column', gap: '16px', textAlign: 'center' }}>
              <Text variant="headingMd">Need a custom solution?</Text>
              <Text variant="bodyMd">
                Looking for a custom plan with specific features or higher limits? 
                Our sales team can create a tailored solution for your business needs.
              </Text>
              <div>
                <Button primary url="mailto:sales@yourbusiness.com">
                  Contact Sales
                </Button>
              </div>
            </div>
          </Card>
        </div>
      </div>

      {/* Upgrade Modal */}
      <Modal
        open={showUpgradeModal}
        onClose={() => setShowUpgradeModal(false)}
        title={`Upgrade to ${plans[selectedPlan]?.name} Plan`}
        primaryAction={{
          content: "Confirm Upgrade",
          onAction: handleUpgrade,
        }}
        secondaryActions={[
          {
            content: "Cancel",
            onAction: () => setShowUpgradeModal(false),
          },
        ]}
      >
        <Modal.Section>
          {selectedPlan && plans[selectedPlan] && (
            <TextContainer>
              <p>
                You're about to upgrade to the <strong>{plans[selectedPlan].name}</strong> plan.
              </p>
              <p>
                <strong>Price:</strong> ${plans[selectedPlan].price[billingCycle]} per {billingCycle === "monthly" ? "month" : "year"}
              </p>
              {billingCycle === "yearly" && calculateSavings(plans[selectedPlan]) > 0 && (
                <Banner status="success">
                  You'll save ${calculateSavings(plans[selectedPlan])} per year with yearly billing!
                </Banner>
              )}
              <p>Your new plan will be active immediately, and you'll be charged on your next billing cycle.</p>
            </TextContainer>
          )}
        </Modal.Section>
      </Modal>

      {/* Usage Details Modal */}
      <Modal
        open={showUsageModal}
        onClose={() => setShowUsageModal(false)}
        title="Usage Details"
        large
      >
        <Modal.Section>
          {plans[currentPlan] && (
            <div style={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: '20px' }}>
              <Card sectioned title="Messages">
                <div style={{ display: 'flex', flexDirection: 'column', gap: '16px' }}>
                  <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                    <Text>Used this month</Text>
                    <Text fontWeight="bold">{usageData.messages.used.toLocaleString()}</Text>
                  </div>
                  <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                    <Text>Plan limit</Text>
                    <Text>{plans[currentPlan].limits.messages.toLocaleString()}</Text>
                  </div>
                <div style={{
                  width: '100%',
                  height: '8px',
                  backgroundColor: '#f0f0f0',
                  borderRadius: '4px',
                  overflow: 'hidden'
                }}>
                  <div style={{
                    width: `${getUsagePercentage(usageData.messages.used, plans[currentPlan].limits.messages)}%`,
                    height: '100%',
                    backgroundColor: getUsagePercentage(usageData.messages.used, plans[currentPlan].limits.messages) >= 90 ? '#d73502' : getUsagePercentage(usageData.messages.used, plans[currentPlan].limits.messages) >= 75 ? '#bf5000' : '#008060',
                    transition: 'width 0.3s ease'
                  }} />
                </div>
              </div>
            </Card>
            <Card sectioned title="Templates">
              <div style={{ display: 'flex', flexDirection: 'column', gap: '16px' }}>
                <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                  <Text>Created</Text>
                  <Text fontWeight="bold">{usageData.templates.used}</Text>
                </div>
                <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                  <Text>Plan limit</Text>
                  <Text>{plans[currentPlan].limits.templates}</Text>
                </div>
                <div style={{
                  width: '100%',
                  height: '8px',
                  backgroundColor: '#f0f0f0',
                  borderRadius: '4px',
                  overflow: 'hidden'
                }}>
                  <div style={{
                    width: `${getUsagePercentage(usageData.templates.used, plans[currentPlan].limits.templates)}%`,
                    height: '100%',
                    backgroundColor: getUsagePercentage(usageData.templates.used, plans[currentPlan].limits.templates) >= 90 ? '#d73502' : getUsagePercentage(usageData.templates.used, plans[currentPlan].limits.templates) >= 75 ? '#bf5000' : '#008060',
                    transition: 'width 0.3s ease'
                  }} />
                </div>
              </div>
            </Card>
          </div>
          )}
        </Modal.Section>
      </Modal>
    </Page>
  );
}
