import { useState, useCallback, useRef, useEffect } from "react";
import {
  Page,
  Layout,
  Card,
  FormLayout,
  TextField,
  Select,
  Button,
  Text,
  InlineStack,
  BlockStack,
  Checkbox,
  Tabs,
  Badge,
  Avatar,
  DropZone,
  Thumbnail,
  Banner,
  Icon,
  Divider,
  Spinner,
} from "@shopify/polaris";
import { ImageIcon } from "@shopify/polaris-icons";
import { getWhatsappProfileJson, updateWhatsappProfile } from "../utils/getgabsSettingsApi";

export default function ManageWhatsAppProfilePage() {
  // Loading and error states
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState(null);
  const [successMessage, setSuccessMessage] = useState("");
  
  // Basic business information
  const [businessName, setBusinessName] = useState("Your Business Name");
  const [businessAbout, setBusinessAbout] = useState("Quality products and excellent service");
  const [businessDescription, setBusinessDescription] = useState("We are a leading provider of high-quality products with a focus on customer satisfaction. Our team is dedicated to delivering exceptional service and building lasting relationships with our customers.");
  const [businessEmail, setBusinessEmail] = useState("contact@yourbusiness.com");
  const [businessPhone, setBusinessPhone] = useState("+1234567890");
  const [businessAddress, setBusinessAddress] = useState("123 Business Street, City, State 12345");
  const [businessWebsite, setBusinessWebsite] = useState("https://www.yourbusiness.com");
  const [businessCategory, setBusinessCategory] = useState("RETAIL");
  
  // Logo upload
  const [businessLogo, setBusinessLogo] = useState(null);
  const [logoPreview, setLogoPreview] = useState(null);
  const [logoError, setLogoError] = useState("");
  const fileInputRef = useRef(null);
  
  const [selectedTab, setSelectedTab] = useState(0);

  // Fetch WhatsApp profile data on component mount
  useEffect(() => {
    const fetchProfileData = async () => {
      try {
        setIsLoading(true);
        setError(null);
        const response = await getWhatsappProfileJson();
        
        if (response.status && response.message) {
          const profileData = response.message;
          
         
          setBusinessName(profileData.business_name || "Your Business Name");
          setBusinessAbout(profileData.about || "Quality products and excellent service");
          setBusinessDescription(profileData.description || "We are a leading provider of high-quality products with a focus on customer satisfaction. Our team is dedicated to delivering exceptional service and building lasting relationships with our customers.");
          setBusinessEmail(profileData.email || "contact@yourbusiness.com");
          setBusinessAddress(profileData.address || "123 Business Street, City, State 12345");
          setBusinessWebsite(profileData.website1 || "https://www.yourbusiness.com");
          
         
          setBusinessCategory(profileData.vertical || 'RETAIL');
          
         
          if (profileData.profile_picture_url) {
            setLogoPreview(`https://app.getgabs.com/customers/mediafile/${profileData.profile_picture_url}`);
          }
        }
      } catch (err) {
        console.error('Error fetching WhatsApp profile:', err);
        setError('Failed to load WhatsApp profile data. Please try again.');
      } finally {
        setIsLoading(false);
      }
    };

    fetchProfileData();
  }, []);

  const handleSave = useCallback(async () => {
    try {
      setError(null);
      setSuccessMessage("");

      
      const payload = {
       
        ...(logoPreview && typeof logoPreview === 'string' && logoPreview.startsWith('data:')
          ? { profile_logo: logoPreview }
          : {}),
        vertical: businessCategory,
        about: businessAbout,
        description: businessDescription,
        address: businessAddress,
        email: businessEmail,
        website: businessWebsite,
      };

      const resp = await updateWhatsappProfile(payload);
      if (resp) {
        setSuccessMessage("WhatsApp profile updated successfully.");
      }
    } catch (e) {
      setError(e?.message || "Failed to update WhatsApp profile.");
    }
  }, [logoPreview, businessCategory, businessAbout, businessDescription, businessAddress, businessEmail, businessWebsite]);

  const categoryOptions = [
    { label: "UNDEFINED", value: "UNDEFINED" },
    { label: "OTHER", value: "OTHER" },
    { label: "AUTO", value: "AUTO" },
    { label: "BEAUTY", value: "BEAUTY" },
    { label: "APPAREL", value: "APPAREL" },
    { label: "EDU", value: "EDU" },
    { label: "ENTERTAIN", value: "ENTERTAIN" },
    { label: "EVENT PLAN", value: "EVENT_PLAN" },
    { label: "FINANCE", value: "FINANCE" },
    { label: "GROCERY", value: "GROCERY" },
    { label: "GOVT", value: "GOVT" },
    { label: "HOTEL", value: "HOTEL" },
    { label: "HEALTH", value: "HEALTH" },
    { label: "NONPROFIT", value: "NONPROFIT" },
    { label: "PROF SERVICES", value: "PROF_SERVICES" },
    { label: "RETAIL", value: "RETAIL" },
    { label: "TRAVEL", value: "TRAVEL" },
    { label: "RESTAURANT", value: "RESTAURANT" },
    { label: "ALCOHOL", value: "ALCOHOL" },
    { label: "ONLINE GAMBLING", value: "ONLINE_GAMBLING" },
    { label: "PHYSICAL GAMBLING", value: "PHYSICAL_GAMBLING" },
    { label: "OTC DRUGS", value: "OTC_DRUGS" },
  ];


  const handleLogoUpload = useCallback((files) => {
    const file = files[0];
    if (!file) return;

  
    if (!file.type.includes('image/jpeg') && !file.type.includes('image/png')) {
      setLogoError("Please upload a JPEG or PNG image file.");
      return;
    }

    if (file.size > 5 * 1024 * 1024) {
      setLogoError("File size must not exceed 5MB.");
      return;
    }


    const reader = new FileReader();
    reader.onload = (e) => {
      setLogoPreview(e.target.result);
      setBusinessLogo(file);
      setLogoError("");
    };
    reader.readAsDataURL(file);
  }, []);

  const handleRemoveLogo = useCallback(() => {
    setBusinessLogo(null);
    setLogoPreview(null);
    setLogoError("");
    if (fileInputRef.current) {
      fileInputRef.current.value = "";
    }
  }, []);


  const renderProfilePreview = () => (
    <Card sectioned>
      <BlockStack gap="400">
        <Text variant="headingMd">Business Profile Preview</Text>
        <Divider />
        
        {/* Logo Preview */}
        <BlockStack gap="300">
          <Text variant="bodyMd" fontWeight="medium">Business Logo</Text>
          <div style={{ textAlign: 'center' }}>
            {logoPreview ? (
              <div style={{ 
                width: '120px', 
                height: '120px', 
                borderRadius: '12px',
                overflow: 'hidden',
                margin: '0 auto',
                border: '2px solid #e1e3e5'
              }}>
                <img 
                  src={logoPreview} 
                  alt="Business Logo" 
                  style={{ 
                    width: '100%', 
                    height: '100%', 
                    objectFit: 'cover' 
                  }} 
                />
              </div>
            ) : (
              <div style={{
                width: '120px',
                height: '120px',
                borderRadius: '12px',
                backgroundColor: '#f6f6f7',
                border: '2px dashed #c9cccf',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                margin: '0 auto'
              }}>
                <Icon source={ImageIcon} color="subdued" />
              </div>
            )}
          </div>
        </BlockStack>

        {/* Business Info Preview */}
        <BlockStack gap="300">
        <div style={{ textAlign: 'center' }}>
             
             <Text variant="headingMd">{businessName}</Text>
           </div>
         
          <BlockStack gap="200">
           
            <div>
              <Text variant="bodySm" color="subdued">About </Text>
              <Text variant="bodyMd">{businessAbout}</Text>
            </div>
          </BlockStack>
        </BlockStack>

        
        <BlockStack gap="300">
        
          <BlockStack gap="200">
            <div>
              <Text variant="bodySm" color="subdued">Description</Text>
              <Text variant="bodyMd">{businessDescription}</Text>
            </div>
            <div>
              <Text variant="bodySm" color="subdued">Address</Text>
              <Text variant="bodyMd">{businessAddress}</Text>
            </div>
            <div>
              <Text variant="bodySm" color="subdued">Email</Text>
              <Text variant="bodyMd">{businessEmail}</Text>
            </div>
            <div>
              <Text variant="bodySm" color="subdued">Website</Text>
              <Text variant="bodyMd">{businessWebsite}</Text>
            </div>
          </BlockStack>
        </BlockStack>
      </BlockStack>
    </Card>
  );

  // Render main profile form with whatsapp profile
  const renderProfileForm = () => (
    <BlockStack gap="500">
      {/* Business Logo Section with whatsapp profile */}
      <Card sectioned>
        <BlockStack gap="400">
          <Text variant="headingMd">Business Logo</Text>
          <Text variant="bodySm" color="subdued">
            Use a square image with a recommended size of 500x500 pixels (minimum 192x192 pixels) and a file size not exceeding 5MB.
          </Text>
          
          {logoError && (
            <Banner status="critical">
              <p>{logoError}</p>
            </Banner>
          )}

          <DropZone onDrop={handleLogoUpload} accept="image/jpeg,image/png">
            {logoPreview ? (
              <div style={{ textAlign: 'center' , paddingTop: '10px', paddingBottom: '10px'}}>
              <div style={{ textAlign: 'center', display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
                <Thumbnail
                  source={logoPreview}
                  alt="Business Logo"
                  size="large"
                />
                </div>
                <div style={{ marginTop: '16px' }}>
                  <Button onClick={handleRemoveLogo}>Remove Logo</Button>
                </div>
              </div>
            ) : (
              <DropZone.FileUpload />
            )}
          </DropZone>
          
          <Text variant="bodySm" color="subdued">
            Choose a file • JPEG, PNG
          </Text>
        </BlockStack>
      </Card>

      {/* About Business Section with whatsapp profile */}
      <Card sectioned>
        <BlockStack gap="400">
          <Text variant="headingMd">About Business</Text>
          <Text variant="bodySm" color="subdued">
            Provide your business category and a short bio (up to 20 words) that clearly describe your services, products, or mission.
          </Text>
          
          <FormLayout>
            <TextField
              label="Business Name"
              value={businessName}
              onChange={setBusinessName}
              helpText="This will be displayed as your business name on WhatsApp"
              placeholder="Enter your business name"
            />
            
            <Select
              label="Business Category"
              options={categoryOptions}
              value={businessCategory}
              onChange={setBusinessCategory}
              helpText="Select the category that best describes your business"
            />
            
            <TextField
              label="About"
              value={businessAbout}
              onChange={setBusinessAbout}
              helpText={`Brief description of your business (${businessAbout.split(' ').length}/20 words)`}
              placeholder="Quality products and excellent service"
              maxLength={200}
            />
          </FormLayout>
        </BlockStack>
      </Card>

      {/* More Details Section */}
      <Card sectioned>
        <BlockStack gap="400">
          <Text variant="headingMd">More Details</Text>
          <Text variant="bodySm" color="subdued">
            Add your business description, address, email, and website to your WhatsApp Business profile for better visibility and customer engagement.
          </Text>
          
          <FormLayout>
            <TextField
              label="Description"
              value={businessDescription}
              onChange={setBusinessDescription}
              multiline={4}
              helpText="Detailed description of your business, products, or services"
              placeholder="Tell customers about your business..."
            />
            
            <TextField
              label="Address"
              value={businessAddress}
              onChange={setBusinessAddress}
              helpText="Your business address for customer visits"
              placeholder="123 Business Street, City, State 12345"
            />
            
            <TextField
              label="Email"
              value={businessEmail}
              onChange={setBusinessEmail}
              type="email"
              helpText="Business email for customer inquiries"
              placeholder="contact@yourbusiness.com"
            />
            
            <TextField
              label="Website"
              value={businessWebsite}
              onChange={setBusinessWebsite}
              type="url"
              helpText="Your business website URL"
              placeholder="https://www.yourbusiness.com"
            />
          </FormLayout>
        </BlockStack>
      </Card>

      {/* Save Button at Bottom */}
      <Card sectioned>
        <InlineStack align="end">
          <Button primary onClick={handleSave} size="large">
            Update WhatsApp Profile
          </Button>
        </InlineStack>
      </Card>
    </BlockStack>
  );


  // Show loading spinner while fetching data
  if (isLoading) {
    return (
      <Page title="Manage WhatsApp Profile" fullWidth>
        <Layout>
          <Layout.Section>
            <Card sectioned>
              <div style={{ textAlign: 'center', padding: '40px' }}>
                <Spinner size="large" />
                <div style={{ marginTop: '16px' }}>
                  <Text variant="bodyMd">Loading WhatsApp profile data...</Text>
                </div>
              </div>
            </Card>
          </Layout.Section>
        </Layout>
      </Page>
    );
  }

  return (
    <Page
      title="Manage WhatsApp Profile"
      subtitle="Configure your business profile and WhatsApp settings"
      primaryAction={{
        content: "Update WhatsApp Profile",
        onAction: handleSave,
      }}
      fullWidth
    >
      <Layout>
        {successMessage && (
          <Layout.Section>
            <Banner status="success">
              <p>{successMessage}</p>
            </Banner>
          </Layout.Section>
        )}
        {error && (
          <Layout.Section>
            <Banner status="critical">
              <p>{error}</p>
            </Banner>
          </Layout.Section>
        )}
        
        <Layout.Section variant="twoThirds">
          {renderProfileForm()}
        </Layout.Section>
        
        <Layout.Section variant="oneThird">
          <div style={{ position: 'sticky', top: '20px' }}>
            {renderProfilePreview()}
          </div>
        </Layout.Section>
      </Layout>
    </Page>
  );
}
