import {
  Page,
  Card,
  FormLayout,
  TextField,
  Select,
  Button,
  BlockStack,
  InlineStack,
  Text,
  Grid,
  Divider,
  Badge,
  ChoiceList,
  RadioButton,
  ButtonGroup,
  LegacyCard,
  Tooltip,
  Icon,
  DropZone,
  Thumbnail,
  Modal,
  Banner,
  Autocomplete
} from '@shopify/polaris';
import { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { ArrowLeftIcon, QuestionCircleIcon } from '@shopify/polaris-icons';
import { extractVariablesFromWhatsAppMessage } from '../utils/whatsappMessagePreview';
import { getTemplateVariables } from '../utils/availableVariables';
import { API_TOKEN } from '../utils/apiConfig';
import { Checkisteplatenameavailable } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';
import EmojiPicker from '../components/EmojiPicker';

function CreateCustomTemplatePage() {
  const navigate = useNavigate();
  
  // Form state
  const [templateName, setTemplateName] = useState('');
  const [templateCategory, setTemplateCategory] = useState('marketing');
  const [templateLanguage, setTemplateLanguage] = useState('en_US');
  const [templateType, setTemplateType] = useState('text');
  
  // Template components state
  const [hasHeader, setHasHeader] = useState(false);
  const [headerType, setHeaderType] = useState('text');
  const [headerContent, setHeaderContent] = useState('');
  const [headerMediaUrl, setHeaderMediaUrl] = useState('');
  const [headerDocumentFilename, setHeaderDocumentFilename] = useState('');
  const [headerFile, setHeaderFile] = useState(null);
  const [headerFilePreview, setHeaderFilePreview] = useState('');
  
  const [bodyText, setBodyText] = useState('');
  const [cursorPosition, setCursorPosition] = useState(0);
  
  const [hasFooter, setHasFooter] = useState(false);
  const [footerText, setFooterText] = useState('');
  
  const [hasButtons, setHasButtons] = useState(false);
  const [buttons, setButtons] = useState([{ type: 'QUICK_REPLY', text: '' }]);

  // URL variables state
  const [urlVariables, setUrlVariables] = useState({});

  // Variable mapping state
  const [headerVariableMapping, setHeaderVariableMapping] = useState({});
  const [bodyVariableMapping, setBodyVariableMapping] = useState({});
  const [variableInputValues, setVariableInputValues] = useState({});
  
  // API variables state
  const [availableVariables, setAvailableVariables] = useState([]);
  const [loadingVariables, setLoadingVariables] = useState(true);
  const [showVariablesModal, setShowVariablesModal] = useState(false);
  const [variableSearchQuery, setVariableSearchQuery] = useState('');

  // Template name availability state
  const [templateNameAvailability, setTemplateNameAvailability] = useState(null);
  const [checkingTemplateName, setCheckingTemplateName] = useState(false);
  const [templateNameError, setTemplateNameError] = useState('');

  // API error state
  const [apiError, setApiError] = useState('');
  const [showApiError, setShowApiError] = useState(false);

  // Loading state
  const [isCreating, setIsCreating] = useState(false);

  // Category options
  const categoryOptions = [
    { label: 'Marketing', value: 'marketing' },
    { label: 'Utility', value: 'utility' },
    { label: 'Authentication', value: 'authentication' }
  ];

  // Language options
  const languageOptions = [
    { label: 'English (US)', value: 'en_US' },
    { label: 'English (UK)', value: 'en_GB' },
    { label: 'Spanish', value: 'es' },
    { label: 'French', value: 'fr' },
    { label: 'German', value: 'de' },
    { label: 'Hindi', value: 'hi' }
  ];

  // Header type options
  const headerTypeOptions = [
    { label: 'Text', value: 'text' },
    { label: 'Image', value: 'image' },
    { label: 'Video', value: 'video' },
    { label: 'Document', value: 'document' }
  ];

  // Button type options
  const buttonTypeOptions = [
    { label: 'Quick Reply', value: 'QUICK_REPLY' },
    { label: 'URL', value: 'URL' },
    { label: 'Phone Number', value: 'PHONE_NUMBER' }
  ];

  // Phone country code options (common)
  const phoneCountryOptions = [
    { label: '🇺🇸 +1 (US)', value: '+1' },
    { label: '🇬🇧 +44 (UK)', value: '+44' },
    { label: '🇦🇺 +61 (AU)', value: '+61' },
    { label: '🇮🇳 +91 (IN)', value: '+91' },
    { label: '🇨🇦 +1 (CA)', value: '+1-CA' },
    { label: '🇩🇪 +49 (DE)', value: '+49' },
    { label: '🇫🇷 +33 (FR)', value: '+33' },
    { label: '🇪🇸 +34 (ES)', value: '+34' }
  ];

  // Fetch variables from API on component mount
  useEffect(() => {
    const fetchVariables = async () => {
      try {
        setLoadingVariables(true);
        const variables = await getTemplateVariables();
        setAvailableVariables(variables);
      } catch (error) {
        console.error('Error fetching variables:', error);
        setAvailableVariables([]);
      } finally {
        setLoadingVariables(false);
      }
    };

    fetchVariables();
  }, []);

  // Debounced template name availability check
  useEffect(() => {
    if (!templateName.trim()) {
      setTemplateNameAvailability(null);
      setTemplateNameError('');
      return;
    }

    const timeoutId = setTimeout(async () => {
      try {
        setCheckingTemplateName(true);
        setTemplateNameError('');
        const response = await Checkisteplatenameavailable(templateName.toLowerCase().replace(/\s+/g, '_'));
        
        if (response.status === true) {
          setTemplateNameAvailability(true);
        } else {
          setTemplateNameAvailability(false);
          setTemplateNameError(response.message || 'Template name is not available');
        }
      } catch (error) {
        console.error('Error checking template name:', error);
        setTemplateNameError('Error checking template name availability');
        setTemplateNameAvailability(false);
      } finally {
        setCheckingTemplateName(false);
      }
    }, 500); // 500ms debounce

    return () => clearTimeout(timeoutId);
  }, [templateName]);

  const mappingOptions = useMemo(() => {
    return [
      { label: 'Select mapping...', value: '' },
      ...availableVariables.map(variable => ({
        label: variable.label,
        value: variable.value
      }))
    ];
  }, [availableVariables]);

 
  const handleHeaderVariableMapping = useCallback((variableIndex, mappingValue) => {
    setHeaderVariableMapping(prev => ({
      ...prev,
      [variableIndex]: mappingValue
    }));
  }, []);

  const handleBodyVariableMapping = useCallback((variableIndex, mappingValue) => {
    setBodyVariableMapping(prev => ({
      ...prev,
      [variableIndex]: mappingValue
    }));
  }, []);

  const validateHeaderText = useCallback((text) => {
    if (headerType === 'text') {
      const variables = extractVariablesFromWhatsAppMessage(JSON.stringify({ text }));
      if (variables.length > 1) {
      
        const firstVariable = variables[0];
        const newText = text.replace(/\{\{[^}]+\}\}/g, (match, offset) => {
          return offset === text.indexOf(`{{${firstVariable}}}`) ? match : '';
        });
        return newText;
      }
    }
    return text;
  }, [headerType]);

  // Handle header content change with validation
  const handleHeaderContentChange = useCallback((value) => {
    const validatedValue = validateHeaderText(value);
    setHeaderContent(validatedValue);
    
    // Clear header variable mapping if text changes
    if (validatedValue !== value) {
      setHeaderVariableMapping({});
    }
  }, [validateHeaderText]);

  // Extract variables from header text (limit to 1 variable for text headers)
  const headerVariables = useMemo(() => {
    if (hasHeader && headerType === 'text' && headerContent) {
      const variables = extractVariablesFromWhatsAppMessage(JSON.stringify({ text: headerContent }));
      return variables.slice(0, 1); // Limit to 1 variable for header
    }
    return [];
  }, [hasHeader, headerType, headerContent]);

  // Extract variables from body text (allow multiple variables)
  const bodyVariables = useMemo(() => {
    if (bodyText) {
      return extractVariablesFromWhatsAppMessage(JSON.stringify({ text: bodyText }));
    }
    return [];
  }, [bodyText]);

  const handleVariableSelectionChange = useCallback((variable, value) => {
    // Determine if this is a header or body variable
    const isHeaderVariable = headerVariables.includes(variable);
    const variableIndex = isHeaderVariable ? headerVariables.indexOf(variable) : bodyVariables.indexOf(variable);
    
    if (isHeaderVariable) {
      setHeaderVariableMapping(prev => ({
        ...prev,
        [variableIndex]: value
      }));
    } else {
      setBodyVariableMapping(prev => ({
        ...prev,
        [variableIndex]: value
      }));
    }
    
    // Update input value for display
    if (value) {
      const selectedVar = availableVariables.find(av => av.value === value);
      setVariableInputValues(prev => ({
        ...prev,
        [variable]: selectedVar?.label || ''
      }));
    } else {
      // Clear input when no selection
      setVariableInputValues(prev => ({
        ...prev,
        [variable]: ''
      }));
    }
  }, [headerVariables, bodyVariables, availableVariables]);

  // Generate template JSON
  const generateTemplateJSON = useMemo(() => {
    const template = {
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: templateName.toLowerCase().replace(/\s+/g, '_') || 'new_template',
        language: {
          code: templateLanguage
        },
        components: []
      }
    };

    // Add header component
    if (hasHeader && (headerContent || headerMediaUrl)) {
      const headerComponent = {
        type: 'HEADER',
        format: headerType.toUpperCase()
      };

      if (headerType === 'text') {
        headerComponent.text = headerContent;
        
        // Add parameters for header variables if any exist
        if (headerVariables.length > 0) {
          headerComponent.parameters = headerVariables.map((variable, index) => ({
            type: 'text',
            text: headerVariableMapping[index] || variable
          }));
        }
      } else if (headerType === 'image') {
        headerComponent.format = 'IMAGE';
        headerComponent.example = {
          header_handle: [headerMediaUrl]
        };
      } else if (headerType === 'video') {
        headerComponent.format = 'VIDEO';
        headerComponent.example = {
          header_handle: [headerMediaUrl]
        };
      } else if (headerType === 'document') {
        headerComponent.format = 'DOCUMENT';
        headerComponent.example = {
          header_handle: [headerMediaUrl]
        };
        if (headerDocumentFilename) {
          headerComponent.filename = headerDocumentFilename;
        }
      }

      template.template.components.push(headerComponent);
    }

    // Add body component
    if (bodyText) {
      const bodyComponent = {
        type: 'BODY',
        text: bodyText
      };
      
      // Add parameters for body variables if any exist
      if (bodyVariables.length > 0) {
        bodyComponent.parameters = bodyVariables.map((variable, index) => ({
          type: 'text',
          text: bodyVariableMapping[index] || variable
        }));
      }
      
      template.template.components.push(bodyComponent);
    }

    // Add footer component
    if (hasFooter && footerText) {
      template.template.components.push({
        type: 'FOOTER',
        text: footerText
      });
    }

    // Add buttons component
    if (hasButtons && buttons.some(btn => btn.text)) {
      const validButtons = buttons.filter(btn => btn.text);
      if (validButtons.length > 0) {
        template.template.components.push({
          type: 'BUTTONS',
          buttons: validButtons.map(btn => ({
            type: btn.type,
            text: btn.text,
            ...(btn.type === 'URL' && btn.url && { url: btn.url }),
            ...(btn.type === 'PHONE_NUMBER' && btn.phone_number && { phone_number: btn.phone_number })
          }))
        });
      }
    }

    return JSON.stringify(template);
  }, [templateName, templateLanguage, hasHeader, headerType, headerContent, headerMediaUrl, headerDocumentFilename, bodyText, hasFooter, footerText, hasButtons, buttons, headerVariables, bodyVariables, headerVariableMapping, bodyVariableMapping]);

  // Combined variables for display
  const extractVariables = useMemo(() => {
    return [...headerVariables, ...bodyVariables];
  }, [headerVariables, bodyVariables]);

  // Handle text selection for formatting
  const handleTextSelection = useCallback(() => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    setCursorPosition(textarea.selectionStart);
  }, []);

  // Apply formatting to selected text
  const applyFormatting = useCallback((format) => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const end = textarea.selectionEnd;
    const selectedText = textarea.value.substring(start, end);
    
    let formattedText = '';
    let cursorOffset = 0;
    
    if (selectedText) {
      // If text is selected, wrap it with formatting
      switch (format) {
        case 'bold':
          formattedText = `*${selectedText}*`;
          cursorOffset = formattedText.length;
          break;
        case 'italic':
          formattedText = `_${selectedText}_`;
          cursorOffset = formattedText.length;
          break;
        case 'strikethrough':
          formattedText = `~${selectedText}~`;
          cursorOffset = formattedText.length;
          break;
        case 'monospace':
          formattedText = `\`\`\`${selectedText}\`\`\``;
          cursorOffset = formattedText.length;
          break;
        case 'inline-code':
          formattedText = `\`${selectedText}\``;
          cursorOffset = formattedText.length;
          break;
        case 'quote':
          // Ensure quote starts on new line
          const needsNewLineBeforeQuote = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterQuote = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeQuote ? '\n' : '') + `> ${selectedText}` + (needsNewLineAfterQuote ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          const bulletLines = selectedText.split('\n').map(line => line.trim() ? `* ${line}` : line).join('\n');
          // Ensure list starts on new line
          const needsNewLineBefore = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfter = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBefore ? '\n' : '') + bulletLines + (needsNewLineAfter ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          const numberedLines = selectedText.split('\n').filter(line => line.trim()).map((line, index) => `${index + 1}. ${line}`).join('\n');
          // Ensure list starts on new line
          const needsNewLineBeforeNum = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterNum = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeNum ? '\n' : '') + numberedLines + (needsNewLineAfterNum ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = selectedText;
          cursorOffset = formattedText.length;
      }
    } else {
      // If no text is selected, insert formatting characters with cursor between them
      switch (format) {
        case 'bold':
          formattedText = '**';
          cursorOffset = 1;
          break;
        case 'italic':
          formattedText = '__';
          cursorOffset = 1;
          break;
        case 'strikethrough':
          formattedText = '~~';
          cursorOffset = 1;
          break;
        case 'monospace':
          formattedText = '\`\`\`\n\n\`\`\`';
          cursorOffset = 4;
          break;
        case 'inline-code':
          formattedText = '\`\`';
          cursorOffset = 1;
          break;
        case 'quote':
          // Ensure quote starts on new line
          const needsNewLineBeforeQuoteEmpty = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeQuoteEmpty ? '\n> ' : '> ');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          // Ensure list starts on new line
          const needsNewLineBeforeBullet = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeBullet ? '\n* ' : '* ');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          // Ensure list starts on new line
          const needsNewLineBeforeNumber = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeNumber ? '\n1. ' : '1. ');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = '';
          cursorOffset = 0;
      }
    }
    
    const newValue = textarea.value.substring(0, start) + formattedText + textarea.value.substring(end);
    setBodyText(newValue);
    
    // Set cursor position
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + cursorOffset, start + cursorOffset);
    }, 0);
  }, []);

  // Insert variable at cursor position
  const insertVariable = useCallback((variable) => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const variableText = `{{${variable}}}`;
    const newValue = textarea.value.substring(0, start) + variableText + textarea.value.substring(start);
    
    setBodyText(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + variableText.length, start + variableText.length);
    }, 0);
  }, []);

  // Insert emoji at cursor position
  const insertEmoji = useCallback((emoji) => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const newValue = textarea.value.substring(0, start) + emoji + textarea.value.substring(start);
    
    setBodyText(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + emoji.length, start + emoji.length);
    }, 0);
  }, []);

  // Keyboard shortcuts for formatting
  useEffect(() => {
    const handleKeyDown = (event) => {
      if ((event.ctrlKey || event.metaKey) && event.target.id === 'body-text-field') {
        switch (event.key.toLowerCase()) {
          case 'b':
            event.preventDefault();
            applyFormatting('bold');
            break;
          case 'i':
            event.preventDefault();
            applyFormatting('italic');
            break;
          case 's':
            event.preventDefault();
            applyFormatting('strikethrough');
            break;
          case 'm':
            event.preventDefault();
            applyFormatting('monospace');
            break;
          case 'k':
            event.preventDefault();
            applyFormatting('inline-code');
            break;
          case 'q':
            event.preventDefault();
            applyFormatting('quote');
            break;
          case 'l':
            event.preventDefault();
            applyFormatting('bullet-list');
            break;
          case 'n':
            event.preventDefault();
            applyFormatting('numbered-list');
            break;
        }
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [applyFormatting]);

  // Clear variable mappings when variables change
  useEffect(() => {
    // Clear header variable mapping if header variables change
    setHeaderVariableMapping(prev => {
      const newMapping = {};
      headerVariables.forEach((variable, index) => {
        if (prev[index]) {
          newMapping[index] = prev[index];
        }
      });
      return newMapping;
    });
  }, [headerVariables]);

  useEffect(() => {
    // Clear body variable mapping if body variables change
    setBodyVariableMapping(prev => {
      const newMapping = {};
      bodyVariables.forEach((variable, index) => {
        if (prev[index]) {
          newMapping[index] = prev[index];
        }
      });
      return newMapping;
    });
  }, [bodyVariables]);

  // Initialize variable input values when variables are extracted
  useEffect(() => {
    const allVariables = [...headerVariables, ...bodyVariables];
    if (allVariables.length > 0) {
      const initialInputValues = {};
      allVariables.forEach(variable => {
        if (!variableInputValues[variable]) {
          // Find the current mapping value for this variable
          const isHeaderVariable = headerVariables.includes(variable);
          const variableIndex = isHeaderVariable ? headerVariables.indexOf(variable) : bodyVariables.indexOf(variable);
          const currentMapping = isHeaderVariable ? headerVariableMapping[variableIndex] : bodyVariableMapping[variableIndex];
          
          if (currentMapping) {
            const selectedVar = availableVariables.find(av => av.value === currentMapping);
            initialInputValues[variable] = selectedVar?.label || '';
          }
        }
      });
      if (Object.keys(initialInputValues).length > 0) {
        setVariableInputValues(prev => ({ ...prev, ...initialInputValues }));
      }
    }
  }, [headerVariables, bodyVariables, headerVariableMapping, bodyVariableMapping, availableVariables, variableInputValues]);

 
  const validateUrlButton = useCallback((url) => {
    if (!url) return { isValid: false, error: 'URL is required' };
    
   
    return { isValid: true, error: null };
  }, []);

 
  const checkUrlVariables = useCallback((url) => {
    const hasVariable = /\{\{1\}\}/.test(url);
    return hasVariable;
  }, []);

 
  const handleUrlVariableChange = useCallback((buttonIndex, value) => {
    setUrlVariables(prev => ({
      ...prev,
      [buttonIndex]: value
    }));
  }, []);

  const handleButtonChange = useCallback((index, field, value) => {
    setButtons(prev => prev.map((btn, i) => {
      if (i === index) {
        const updatedBtn = { ...btn, [field]: value };
        
       
        if (field === 'url') {
          updatedBtn.urlError = null;
          if (!checkUrlVariables(value)) {
            setUrlVariables(prev => {
              const newVars = { ...prev };
              delete newVars[index];
              return newVars;
            });
          }
        }
        
        return updatedBtn;
      }
      return btn;
    }));
  }, [checkUrlVariables]);

  const addButton = useCallback(() => {
    if (buttons.length < 8) {
      setButtons(prev => [...prev, { type: 'QUICK_REPLY', text: '' }]);
    }
  }, [buttons.length]);

  const removeButton = useCallback((index) => {
    setButtons(prev => prev.filter((_, i) => i !== index));
  }, []);

 
  const handleHeaderFileUpload = useCallback((files) => {
    if (files.length > 0) {
      const file = files[0];
      setHeaderFile(file);
      
      const reader = new FileReader();
      reader.onload = (e) => {
        setHeaderFilePreview(e.target.result);
        setHeaderMediaUrl(e.target.result); 
      };
      reader.readAsDataURL(file);
      
      if (headerType === 'document') {
        setHeaderDocumentFilename(file.name);
      }
    }
  }, [headerType]);

  const handleRemoveHeaderFile = useCallback(() => {
    setHeaderFile(null);
    setHeaderFilePreview('');
    setHeaderMediaUrl('');
    setHeaderDocumentFilename('');
  }, []);

 
const handleSave = useCallback(async () => {

  // console.log('Saving template:', {
  //   name: templateName,
  //   category: templateCategory,
  //   language: templateLanguage,
  //   template: generateTemplateJSON,
  //   headerFile: headerFile, 
  //   headerVariableMapping,
  //   bodyVariableMapping,
  //   headerVariables,
  //   bodyVariables,
  //   buttons,
  // });

  setIsCreating(true);
  try {
    const payload = {
      template_type: 0,
      draftedtemplateid: 0,
      normaltemplateid: 0,
      _token: "",
      templatename: templateName.toLowerCase().replace(/\s+/g, '_'),
      language: templateLanguage,
      category: templateCategory.toUpperCase(),
      body: bodyText,
      bodyVars: Object.values(bodyVariableMapping),
      phoneCountry: 91, 
      previewfooterText: hasFooter ? footerText : ""
    };

    
    if (hasButtons && Array.isArray(buttons)) {
      const quickReplyButtons = buttons.filter(btn => btn.type === 'QUICK_REPLY' && btn.text);
      if (quickReplyButtons.length > 0) {
        payload.quickReplyButtonText = quickReplyButtons.map(btn => btn.text);
      }

      const urlButtons = buttons.filter(btn => btn.type === 'URL' && btn.text && btn.url);
      if (urlButtons.length > 0) {
        payload.websiteButtonText = urlButtons.map(btn => btn.text);
        payload.previewWebsiteUrl = urlButtons.map(btn => btn.url);
        
        // Add URL variables if any exist
        const urlVarsArray = urlButtons.map((btn, index) => {
          return urlVariables[index] || '';
        }).filter(varValue => varValue.trim() !== '');
        
        if (urlVarsArray.length > 0) {
          payload.UrlVars = urlVarsArray;
        }
      }

      const phoneButtons = buttons.filter(btn => btn.type === 'PHONE_NUMBER' && btn.text && btn.phone_number);
      if (phoneButtons.length > 0) {
        const primaryPhoneButton = phoneButtons[0];
        payload.callbuttonText = primaryPhoneButton.text;
        
        payload.prevviewphoneNumber = String(primaryPhoneButton.phone_number).replace(/^\+/, '');
       
        const selectedCode = (primaryPhoneButton.phone_country_code || '').replace(/^\+/, '');
        let countryCode = selectedCode;
        if (!countryCode && primaryPhoneButton.phone_number) {
          const match = String(primaryPhoneButton.phone_number).match(/^\+?(\d{1,3})/);
          if (match && match[1]) {
            countryCode = match[1];
          }
        }
        if (countryCode) {
          const parsedCode = parseInt(countryCode, 10);
          if (!Number.isNaN(parsedCode)) {
            payload.phoneCountry = parsedCode;
          }
        }
      }
    }

    
    if (hasHeader) {
      payload.headerType = headerType.charAt(0).toUpperCase() + headerType.slice(1); 
      
      if (headerType === 'text') {
        payload.headerText = headerContent;
        if (Object.keys(headerVariableMapping).length > 0) {
          payload.headerVar = Object.values(headerVariableMapping)[0];

        }
      } else {
      
        if (headerFile) {
          const formData = new FormData();
          formData.append('headerFile', headerFile);
          
          Object.entries(payload).forEach(([key, value]) => {
            if (Array.isArray(value)) {
              value.forEach((item) => {
                formData.append(`${key}[]`, item);
              });
            } else {
              formData.append(key, value);
            }
          });
console.log('Payload:', payload);
         
          
          const response = await fetch(`https://app.getgabs.com/partners/template/with-custom-vars/designtemplate?token=${API_TOKEN}`, {
            method: 'POST',
            body: formData,
            headers: {
            'X-Requested-With': 'XMLHttpRequest',

            }
          });
          
          const result = await response.json();
          
   
          if (result.success === true) {
            console.log('Template saved successfully:', result);
            navigate('/app/templates');
            return;
          }
          
         
          if (result.status === false) {
            let errorMessage = 'Failed to save template';
            
            
            if (result[0] && result[0].error) {
              errorMessage = result[0].error.error_user_msg || result[0].error.message || errorMessage;
            }
            
            else if (result.error) {
              errorMessage = result.error.error_user_msg || result.error.message || errorMessage;
            }
           
            else if (result.message) {
              errorMessage = result.message;
            }
            
            setApiError(errorMessage);
            setShowApiError(true);
            return;
          }
          
          if (!response.ok) throw new Error('Failed to save template');
          
        
          if (result.status === true) {
            console.log('Template saved successfully:', result);
            navigate('/app/templates');
            return;
          } else {
            // Handle case where status is not true but no error object
            const errorMessage = result.message || 'Template could not be saved';
            setApiError(errorMessage);
            setShowApiError(true);
            return;
          }
        }
      }
    }
    
    
    const response = await fetch(`https://app.getgabs.com/partners/template/with-custom-vars/designtemplate?token=${API_TOKEN}`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${API_TOKEN}`,
        'X-Requested-With': 'XMLHttpRequest',
      },
      body: JSON.stringify(payload)
    });

    const result = await response.json();
    
    // Handle success response - redirect regardless of alert type
    if (result.success === true) {
      console.log('Template saved successfully:', result);
      navigate('/app/templates');
      return;
    }
    
    // Handle API error response
    if (result.status === false) {
      let errorMessage = 'Failed to save template';
      
      // Check for nested error structure (result[0].error)
      if (result[0] && result[0].error) {
        errorMessage = result[0].error.error_user_msg || result[0].error.message || errorMessage;
      }
      // Check for direct error structure (result.error)
      else if (result.error) {
        errorMessage = result.error.error_user_msg || result.error.message || errorMessage;
      }
      // Check for direct message
      else if (result.message) {
        errorMessage = result.message;
      }
      
      setApiError(errorMessage);
      setShowApiError(true);
      return;
    }
    
    if (!response.ok) throw new Error('Failed to save template');
    
    // Only redirect if status is true (and success is not explicitly true)
    if (result.status === true) {
      console.log('Template saved successfully:', result);
      navigate('/app/templates');
    } else {
      // Handle case where status is not true but no error object
      const errorMessage = result.message || 'Template could not be saved';
      setApiError(errorMessage);
      setShowApiError(true);
    }
    
  } catch (error) {
    console.error('Error saving template:', error);
    setApiError('An unexpected error occurred while saving the template. Please try again.');
    setShowApiError(true);
  } finally {
    setIsCreating(false);
  }
  
}, [
  templateName, 
  templateLanguage, 
  templateCategory, 
  bodyText, 
  hasFooter, 
  footerText, 
  hasHeader, 
  headerType, 
  headerContent, 
  headerFile, 
  headerVariableMapping, 
  bodyVariableMapping, 
  headerVariables,
  bodyVariables,
  urlVariables,
  generateTemplateJSON,
  navigate
]);
  const handleCancel = useCallback(() => {
    navigate('/app/templates');
  }, [navigate]);

  const dismissApiError = useCallback(() => {
    setShowApiError(false);
    setApiError('');
  }, []);

  return (
    <Page
      fullWidth
      backAction={{
        content: 'Templates',
        onAction: () => navigate('/app/templates')
      }}
      title="Create Custom Template"
      subtitle="Create a new custom WhatsApp message template"
      primaryAction={{
        content: 'Save Template',
        onAction: handleSave,
        disabled: !templateName || !bodyText || templateNameAvailability !== true || checkingTemplateName,
        loading: isCreating
      }}
      secondaryActions={[
        {
          content: 'Cancel',
          onAction: handleCancel
        }
      ]}
    >
      {/* API Error Banner */}
      {showApiError && (
        <div style={{ marginBottom: '20px' }}>
          <Banner
            status="critical"
            onDismiss={dismissApiError}
          >
            <p>{apiError}</p>
          </Banner>
        </div>
      )}
      
      <Grid>
        <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 8, lg: 8, xl: 8 }}>
          <div style={{ marginRight: '20px' }}>
            <BlockStack gap="500">
              {/* Basic Information */}
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Basic Information</Text>
                  
                  <FormLayout>
                    <TextField
                      label="Template Name"
                      value={templateName}
                      onChange={(value) => setTemplateName(value.replace(/\s+/g, '_'))}
                      placeholder="Enter template name"
                      helpText={
                        checkingTemplateName 
                          ? "Checking availability..." 
                          : templateNameAvailability === true 
                            ? "✓ Template name is available" 
                            : templateNameAvailability === false 
                              ? templateNameError 
                              : "This will be used as the template identifier"
                      }
                      error={templateNameAvailability === false ? templateNameError : undefined}
                      tone={
                        checkingTemplateName 
                          ? undefined 
                          : templateNameAvailability === true 
                            ? "success" 
                            : templateNameAvailability === false 
                              ? "critical" 
                              : undefined
                      }
                      autoComplete="off"
                    />
                    
                    <InlineStack gap="400">
                      <div style={{ flex: 1 }}>
                        <Select
                          label="Category"
                          options={categoryOptions}
                          value={templateCategory}
                          onChange={setTemplateCategory}
                        />
                      </div>
                      <div style={{ flex: 1 }}>
                        <Select
                          label="Language"
                          options={languageOptions}
                          value={templateLanguage}
                          onChange={setTemplateLanguage}
                        />
                      </div>
                    </InlineStack>
                  </FormLayout>
                </BlockStack>
              </Card>

              {/* Header Section */}
              <Card>
                <BlockStack gap="400">
                  <InlineStack align="space-between">
                    <Text variant="headingMd" as="h2">Header</Text>
                    <Button
                      variant={hasHeader ? 'primary' : 'plain'}
                      onClick={() => setHasHeader(!hasHeader)}
                      size="slim"
                    >
                      {hasHeader ? 'Remove Header' : 'Add Header'}
                    </Button>
                  </InlineStack>
                  
                  {hasHeader && (
                    <FormLayout>
                      <Select
                        label="Header Type"
                        options={headerTypeOptions}
                        value={headerType}
                        onChange={setHeaderType}
                      />
                      
                      {headerType === 'text' && (
                        <TextField
                          label="Header Text"
                          value={headerContent}
                          onChange={handleHeaderContentChange}
                          placeholder="Enter header text (max 1 variable allowed)"
                          maxLength={60}
                          showCharacterCount
                          helpText="Header text can contain maximum 1 variable. Use {{variable_name}} format."
                          autoComplete="off"
                        />
                      )}
                      
                      {(headerType === 'image' || headerType === 'video' || headerType === 'document') && (
                        <div>
                          <Text variant="bodyMd" as="p" fontWeight="medium">
                            Choose {headerType.charAt(0).toUpperCase() + headerType.slice(1)}
                          </Text>
                          <div style={{ marginTop: '8px' }}>
                            {!headerFile ? (
                              <DropZone
                                onDrop={handleHeaderFileUpload}
                                accept={headerType === 'image' ? 'image/*' : headerType === 'video' ? 'video/*' : '*/*'}
                                type="file"
                                allowMultiple={false}
                              >
                                <DropZone.FileUpload actionTitle={`Choose ${headerType}`} actionHint={`or drag and drop your ${headerType} here`} />
                              </DropZone>
                            ) : (
                              <BlockStack gap="200">
                                <InlineStack gap="200" align="space-between">
                                  <InlineStack gap="200">
                                    <Thumbnail
                                      source={headerFilePreview || (headerType === 'document' ? '📄' : '📁')}
                                      alt={headerFile.name}
                                      size="small"
                                    />
                                    <BlockStack gap="050">
                                      <Text variant="bodyMd" as="p">{headerFile.name}</Text>
                                      <Text variant="bodySm" as="p" tone="subdued">
                                        {(headerFile.size / 1024 / 1024).toFixed(2)} MB
                                      </Text>
                                    </BlockStack>
                                  </InlineStack>
                                  <Button
                                    variant="plain"
                                    tone="critical"
                                    onClick={handleRemoveHeaderFile}
                                    size="slim"
                                  >
                                    Remove
                                  </Button>
                                </InlineStack>
                                <Button
                                  variant="plain"
                                  onClick={() => {
                                    handleRemoveHeaderFile();
                                  }}
                                  size="slim"
                                >
                                  Choose Different {headerType.charAt(0).toUpperCase() + headerType.slice(1)}
                                </Button>
                              </BlockStack>
                            )}
                          </div>
                        </div>
                      )}
                      
                      {headerType === 'document' && (
                        <TextField
                          label="Document Filename"
                          value={headerDocumentFilename}
                          onChange={setHeaderDocumentFilename}
                          placeholder="Enter document filename"
                          autoComplete="off"
                        />
                      )}
                      
                      {/* Header Variable Mapping */}
                      {headerType === 'text' && headerVariables.length > 0 && (
                        <div>
                          <Text variant="bodyMd" as="p" fontWeight="semibold" tone="subdued">
                            Header Variable Mapping
                          </Text>
                          <Text variant="bodySm" as="p" tone="subdued" style={{ marginBottom: '12px' }}>
                            Map the variables in your header text to actual data fields:
                          </Text>
                          <BlockStack gap="300">
                            {headerVariables.map((variable, index) => (
                              <FormLayout.Group key={variable}>
                                <TextField
                                  label={`Variable: {{${variable}}}`}
                                  value={`{{${variable}}}`}
                                  disabled
                                  helpText="Template variable"
                                />
                                <Autocomplete
                                  options={availableVariables
                                    .filter(av => {
                                      const searchValue = variableInputValues[variable] || '';
                                      return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                                    })
                                    .map(av => ({
                                      value: av.value,
                                      label: av.label
                                    }))}
                                  selected={headerVariableMapping[index] ? [headerVariableMapping[index]] : []}
                                  onSelect={(selected) => {
                                    const value = selected.length > 0 ? selected[0] : '';
                                    handleVariableSelectionChange(variable, value);
                                    if (value) {
                                      const selectedVar = availableVariables.find(av => av.value === value);
                                      setVariableInputValues(prev => ({
                                        ...prev,
                                        [variable]: selectedVar?.label || ''
                                      }));
                                    } else {
                                      // Clear input when no selection
                                      setVariableInputValues(prev => ({
                                        ...prev,
                                        [variable]: ''
                                      }));
                                    }
                                  }}
                                  textField={
                                    <Autocomplete.TextField
                                      label="Map to"
                                      value={variableInputValues[variable] || ''}
                                      onChange={(value) => {
                                        setVariableInputValues(prev => ({
                                          ...prev,
                                          [variable]: value
                                        }));
                                      }}
                                      placeholder="Search and choose data field"
                                      autoComplete="off"
                                    />
                                  }
                                  loading={loadingVariables}
                                  emptyState={loadingVariables ? "Loading variables..." : "No variables found"}
                                  preferredPosition="below"
                                />
                              </FormLayout.Group>
                            ))}
                          </BlockStack>
                        </div>
                      )}
                    </FormLayout>
                  )}
                </BlockStack>
              </Card>

              {/* Body Section */}
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Body Text *</Text>
                  
                  {/* Variable insertion buttons */}
                  <BlockStack gap="200">
                    <Text variant="bodySm" as="p" tone="subdued">
                      Click to insert variables :
                    </Text>
                    <InlineStack gap="100" wrap>
                      {availableVariables.slice(0, 15).map((variable) => (
                        <Button
                          key={variable.value}
                          size="micro"
                          variant="tertiary"
                          onClick={() => insertVariable(variable.value)}
                        >
                          {variable.label}
                        </Button>
                      ))}
                      {availableVariables.length > 15 && (
                        <Button
                          size="micro"
                          variant="primary"
                          onClick={() => setShowVariablesModal(true)}
                        >
                          More Variables ({availableVariables.length - 15})
                        </Button>
                      )}
                    </InlineStack>
                  </BlockStack>
                  
                  {/* Custom message textarea with attached formatting controls */}
                  <div style={{ position: 'relative' }}>
                    <div style={{ position: 'absolute', bottom: 12, right: 6, zIndex: 99, display: 'flex', gap: '4px', flexWrap: 'wrap' }}>
                      <ButtonGroup variant="segmented">
                        <EmojiPicker onEmojiSelect={insertEmoji} buttonSize="micro" />
                      </ButtonGroup>
                      
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Bold<Text as="span" variant="bodyMd" tone="subdued">⌘B</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bold')} accessibilityLabel="Bold">
                            <Text fontWeight="bold">B</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Italic<Text as="span" variant="bodyMd" tone="subdued">⌘I</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('italic')} accessibilityLabel="Italic">
                            <Text as="em">I</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Strikethrough<Text as="span" variant="bodyMd" tone="subdued">⌘S</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('strikethrough')} accessibilityLabel="Strikethrough">
                            <Text as="s">S</Text>
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Monospace<Text as="span" variant="bodyMd" tone="subdued">⌘M</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('monospace')} accessibilityLabel="Monospace">
                            <Text as="code"> {'</>'}</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Inline Code<Text as="span" variant="bodyMd" tone="subdued">⌘K</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('inline-code')} accessibilityLabel="Inline Code">
                            <Text as="code">{"<>"}</Text>
                          </Button>
                        </Tooltip> 
                        <Tooltip content={<InlineStack gap="200">Quote<Text as="span" variant="bodyMd" tone="subdued">⌘Q</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('quote')} accessibilityLabel="Quote">
                            {'"'}
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Bullet List<Text as="span" variant="bodyMd" tone="subdued">⌘L</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bullet-list')} accessibilityLabel="Bullet List">
                            •
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Numbered List<Text as="span" variant="bodyMd" tone="subdued">⌘N</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('numbered-list')} accessibilityLabel="Numbered List">
                            1.
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                    </div>
                   
                    <TextField
                      id="body-text-field"
                      label="Message Body"
                      value={bodyText}
                      onChange={setBodyText}
                      onSelect={handleTextSelection}
                      multiline={8}
                      maxLength={1024}
                      placeholder="Type your message body here... Formatting options: *bold*, _italic_, ~strikethrough~, ```monospace```, `inline code`, > quote, * bullet list, 1. numbered list. Use keyboard shortcuts or click buttons above."
                      helpText={`${bodyText.length}/1024 characters`}
                      showCharacterCount
                      autoComplete="off"
                    />
                  </div>
                  
                  {/* Body Variable Mapping */}
                  {bodyVariables.length > 0 && (
                    <div>
                      <Text variant="bodyMd" as="p" fontWeight="semibold" tone="subdued">
                        Body Variable Mapping
                      </Text>
                      <Text variant="bodySm" as="p" tone="subdued" style={{ marginBottom: '12px' }}>
                        Map the variables in your body text to actual data fields:
                      </Text>
                      <BlockStack gap="300">
                        {bodyVariables.map((variable, index) => (
                          <FormLayout.Group key={variable}>
                            <TextField
                              label={`Variable: {{${variable}}}`}
                              value={`{{${variable}}}`}
                              disabled
                              helpText="Template variable"
                            />
                            <Autocomplete
                              options={availableVariables
                                .filter(av => {
                                  const searchValue = variableInputValues[variable] || '';
                                  return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                                })
                                .map(av => ({
                                  value: av.value,
                                  label: av.label
                                }))}
                              selected={bodyVariableMapping[index] ? [bodyVariableMapping[index]] : []}
                              onSelect={(selected) => {
                                const value = selected.length > 0 ? selected[0] : '';
                                handleVariableSelectionChange(variable, value);
                                if (value) {
                                  const selectedVar = availableVariables.find(av => av.value === value);
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: selectedVar?.label || ''
                                  }));
                                } else {
                                  // Clear input when no selection
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: ''
                                  }));
                                }
                              }}
                              textField={
                                <Autocomplete.TextField
                                  label="Map to"
                                  value={variableInputValues[variable] || ''}
                                  onChange={(value) => {
                                    setVariableInputValues(prev => ({
                                      ...prev,
                                      [variable]: value
                                    }));
                                  }}
                                  placeholder="Search and choose data field"
                                  autoComplete="off"
                                />
                              }
                              loading={loadingVariables}
                              emptyState={loadingVariables ? "Loading variables..." : "No variables found"}
                              preferredPosition="below"
                            />
                          </FormLayout.Group>
                        ))}
                      </BlockStack>
                    </div>
                  )}
                  
                  {/* {extractVariables.length > 0 && (
                    <div>
                      <Text variant="bodyMd" as="p" fontWeight="semibold">
                        All Variables Found:
                      </Text>
                      <InlineStack gap="200" wrap>
                        {extractVariables.map((variable, index) => (
                          <Badge key={index} tone="info">
                            {`{{${variable}}}`}
                          </Badge>
                        ))}
                      </InlineStack>
                    </div>
                  )} */}
                </BlockStack>
              </Card>

              {/* Footer Section */}
              <Card>
                <BlockStack gap="400">
                  <InlineStack align="space-between">
                    <Text variant="headingMd" as="h2">Footer</Text>
                    <Button
                      variant={hasFooter ? 'primary' : 'plain'}
                      onClick={() => setHasFooter(!hasFooter)}
                      size="slim"
                    >
                      {hasFooter ? 'Remove Footer' : 'Add Footer'}
                    </Button>
                  </InlineStack>
                  
                  {hasFooter && (
                    <TextField
                      label="Footer Text"
                      value={footerText}
                      onChange={setFooterText}
                      placeholder="Enter footer text"
                      maxLength={60}
                      showCharacterCount
                      helpText="Footer text appears at the bottom of the message"
                      autoComplete="off"
                    />
                  )}
                </BlockStack>
              </Card>

              {/* Buttons Section */}
              <Card>
                <BlockStack gap="400">
                  <InlineStack align="space-between">
                    <Text variant="headingMd" as="h2">Buttons</Text>
                    <Button
                      variant={hasButtons ? 'primary' : 'plain'}
                      onClick={() => setHasButtons(!hasButtons)}
                      size="slim"
                    >
                      {hasButtons ? 'Remove Buttons' : 'Add Buttons'}
                    </Button>
                  </InlineStack>
                  
                  {hasButtons && (
                    <BlockStack gap="300">
                      {buttons.map((button, index) => (
                        <Card key={index} background="bg-surface-secondary">
                          <BlockStack gap="300">
                            <InlineStack align="space-between">
                              <Text variant="bodyMd" fontWeight="semibold">
                                Button {index + 1}
                              </Text>
                              {buttons.length > 1 && (
                                <Button
                                  variant="plain"
                                  tone="critical"
                                  onClick={() => removeButton(index)}
                                  size="slim"
                                >
                                  Remove
                                </Button>
                              )}
                            </InlineStack>
                            
                            <FormLayout>
                              <Select
                                label="Button Type"
                                options={buttonTypeOptions}
                                value={button.type}
                                onChange={(value) => handleButtonChange(index, 'type', value)}
                              />
                              
                              <TextField
                                label="Button Text"
                                value={button.text}
                                onChange={(value) => handleButtonChange(index, 'text', value)}
                                placeholder="Enter button text"
                                maxLength={20}
                                showCharacterCount
                                autoComplete="off"
                              />
                              
                              {button.type === 'URL' && (
                                <>
                                  <TextField
                                    label="URL"
                                    value={button.url || ''}
                                    onChange={(value) => {
                                      handleButtonChange(index, 'url', value);
                                     
                                      const validation = validateUrlButton(value);
                                      if (!validation.isValid && value) {
                                        handleButtonChange(index, 'urlError', validation.error);
                                      } else {
                                        handleButtonChange(index, 'urlError', null);
                                      }
                                    }}
                                    placeholder="https://example.com/{{1}}"
                                    error={button.urlError}
                                    helpText="URL can contain a variable in {{1}} format at the end (optional for custom templates)"
                                    autoComplete="off"
                                  />
                                  
                                  {button.url && checkUrlVariables(button.url) && (
                                    <TextField
                                      label="URL Variable Value"
                                      value={urlVariables[index] || ''}
                                      onChange={(value) => handleUrlVariableChange(index, value)}
                                      placeholder="Enter value for {{1}} variable"
                                      helpText="This value will replace the {{1}} variable in the URL when sending"
                                      autoComplete="off"
                                    />
                                  )}
                                </>
                              )}
                              
                              {button.type === 'PHONE_NUMBER' && (
                                <>
                                  <Select
                                    label="Country Code"
                                    options={phoneCountryOptions}
                                    value={button.phone_country_code || '+91'}
                                    onChange={(value) => handleButtonChange(index, 'phone_country_code', value.startsWith('+1-') ? '+1' : value)}
                                  />
                                  <TextField
                                    label="Phone Number"
                                    value={button.phone_number || ''}
                                    onChange={(value) => handleButtonChange(index, 'phone_number', value)}
                                    placeholder="Enter phone number"
                                    autoComplete="off"
                                  />
                                </>
                              )}
                            </FormLayout>
                          </BlockStack>
                        </Card>
                      ))}
                      
                      {buttons.length < 8 && (
                        <Button onClick={addButton} variant="plain">
                          Add Another Button
                        </Button>
                      )}
                      
                      <Text variant="bodySm" tone="subdued">
                        You can add up to 8 buttons per template. URL variables in {'{{1}}'} format are optional for custom templates.
                      </Text>
                    </BlockStack>
                  )}
                </BlockStack>
              </Card>
              
              {/* Save and Cancel Buttons at Bottom */}
              <Card>
                <InlineStack gap="300" align="end">
                  <Button
                    variant="secondary"
                    onClick={handleCancel}
                  >
                    Cancel
                  </Button>
                  <Button
                    variant="primary"
                    onClick={handleSave}
                    disabled={!templateName || !bodyText || templateNameAvailability !== true || checkingTemplateName}
                    loading={isCreating}
                  >
                    Save Template
                  </Button>
                </InlineStack>
              </Card>
            </BlockStack>
          </div>
        </Grid.Cell>

        {/* Preview Column */}
        <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 4, lg: 4, xl: 4 }}>
          <div style={{ position: 'sticky', top: '20px' }}>
            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Template Preview</Text>
                
                {generateTemplateJSON && (bodyText || (hasHeader && (headerContent || headerMediaUrl)) || (hasFooter && footerText) || (hasButtons && buttons.some(btn => btn.text))) ? (
                  <BlockStack gap="300">
                    <Text variant="bodySm" as="p">
                      Template: {(templateName ? templateName.toLowerCase().replace(/\s+/g, '_') : 'untitled_template')}
                    </Text>
                    
                    <WhatsAppPreview 
                      template={generateTemplateJSON}
                      variant="full"
                      maxWidth="280px"
                      showWhatsAppHeader={false}
                    />
                    
                    <Text variant="bodySm" as="p" tone="subdued">
                      Variables like {`{{customer_name}}`} or {`{{1}}`} will be replaced with actual data when sent.
                    </Text>
                  </BlockStack>
                ) : (
                  <BlockStack gap="200">
                    <Text as="p" tone="subdued">
                      Enter template details to see the preview
                    </Text>
                    <LegacyCard sectioned>
                      <Text variant="bodySm" as="p" tone="subdued">
                        Your template preview will appear here once you add template content (header, body text, footer, or buttons).
                      </Text>
                    </LegacyCard>
                  </BlockStack>
                )}
              </BlockStack>
            </Card>
          </div>
        </Grid.Cell>
      </Grid>

      {/* Variables Modal */}
      <Modal
        open={showVariablesModal}
        onClose={() => {
          setShowVariablesModal(false);
          setVariableSearchQuery('');
        }}
        title="All Available Variables"
        primaryAction={{
          content: 'Close',
          onAction: () => {
            setShowVariablesModal(false);
            setVariableSearchQuery('');
          },
        }}
        size="large"
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" tone="subdued">
              Click on any variable to insert it into your template:
            </Text>
            
            {/* Search Field */}
            <TextField
              label="Search Variables"
              value={variableSearchQuery}
              onChange={setVariableSearchQuery}
              placeholder="Type to search variables..."
              clearButton
              onClearButtonClick={() => setVariableSearchQuery('')}
              autoComplete="off"
            />
            
            <InlineStack gap="200" wrap>
              {availableVariables
                .filter(variable => 
                  variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
                  variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
                )
                .map((variable) => (
                  <Button
                    key={variable.value}
                    size="medium"
                    variant="tertiary"
                    onClick={() => {
                      insertVariable(variable.value);
                      setShowVariablesModal(false);
                      setVariableSearchQuery('');
                    }}
                  >
                    {variable.label}
                  </Button>
                ))}
            </InlineStack>
            
            {availableVariables.filter(variable => 
              variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
              variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
            ).length === 0 && variableSearchQuery && (
              <Text variant="bodyMd" tone="subdued" alignment="center">
                No variables found matching "{variableSearchQuery}"
              </Text>
            )}
            
            <Text variant="bodySm" tone="subdued">
              You can also create custom variables using the format: {"{{variable_name}}"}
            </Text>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </Page>
  );
}

export default CreateCustomTemplatePage;
