import {
  Page,
  Card,
  FormLayout,
  TextField,
  Select,
  Button,
  BlockStack,
  InlineStack,
  Text,
  Grid,
  Divider,
  Badge,
  ChoiceList,
  RadioButton,
  ButtonGroup,
  LegacyCard,
  Tooltip,
  Icon,
  DropZone,
  Thumbnail,
  Modal,
  Banner,
  Box,
  Autocomplete
} from '@shopify/polaris';
import { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { ArrowLeftIcon, QuestionCircleIcon, PlusIcon, DeleteIcon } from '@shopify/polaris-icons';
import { extractVariablesFromWhatsAppMessage } from '../utils/whatsappMessagePreview';
import { getTemplateVariables } from '../utils/availableVariables';
import { Checkisteplatenameavailable } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';
import EmojiPicker from '../components/EmojiPicker';
import { API_TOKEN , autoReplyToken} from '../utils/apiConfig';

function CreateCarouselTemplatePage() {
  const navigate = useNavigate();
  
  // Form state
  const [templateName, setTemplateName] = useState('');
  const [templateCategory, setTemplateCategory] = useState('marketing');
  const [templateLanguage, setTemplateLanguage] = useState('en_US');
  
  // Main body text for carousel template
  const [bodyText, setBodyText] = useState('');
  
  // Text selection state for formatting
  const [textSelection, setTextSelection] = useState({ start: 0, end: 0 });
  
  // Carousel-specific state
  const [carouselCards, setCarouselCards] = useState([
    {
      id: 1,
      cardText: '',
      headerType: 'IMAGE', // 'IMAGE' or 'VIDEO'
      imageUrl: '',
      videoUrl: '',
      mediaFile: null,
      mediaPreview: '',
      buttons: [{ type: 'URL', text: '', url: '' }]
    }
  ]);

  // URL variables state for carousel cards
  const [urlVariables, setUrlVariables] = useState({});

  const categoryOptions = [
    { label: 'Marketing', value: 'marketing' },
    { label: 'Utility', value: 'utility' },
    { label: 'Authentication', value: 'authentication' }
  ];
 
  const languageOptions = [
    { label: 'English (US)', value: 'en_US' },
    { label: 'English (UK)', value: 'en_GB' },
    { label: 'Spanish', value: 'es' },
    { label: 'French', value: 'fr' },
    { label: 'German', value: 'de' },
    { label: 'Hindi', value: 'hi' }
  ];

  const buttonTypeOptions = [
    { label: 'URL', value: 'URL' },
    { label: 'Phone Number', value: 'PHONE_NUMBER' },
    { label: 'Quick Reply', value: 'QUICK_REPLY' }
  ];


  
  const [mainBodyVariableMapping, setMainBodyVariableMapping] = useState({});
  const [cardBodyVariableMapping, setCardBodyVariableMapping] = useState({});
  const [variableInputValues, setVariableInputValues] = useState({}); 
  
 
  const [availableVariables, setAvailableVariables] = useState([]);
  const [loadingVariables, setLoadingVariables] = useState(true);
  const [showVariablesModal, setShowVariablesModal] = useState(false);
  const [variableSearchQuery, setVariableSearchQuery] = useState('');

  // Template name availability state
  const [templateNameAvailability, setTemplateNameAvailability] = useState(null);
  const [checkingTemplateName, setCheckingTemplateName] = useState(false);
  const [templateNameError, setTemplateNameError] = useState('');

  
  useEffect(() => {
    const fetchVariables = async () => {
      try {
        setLoadingVariables(true);
        const variables = await getTemplateVariables();
        setAvailableVariables(variables);
      } catch (error) {
        console.error('Error fetching variables:', error);
        setAvailableVariables([]);
      } finally {
        setLoadingVariables(false);
      }
    };

    fetchVariables();
  }, []);

  // Debounced template name availability check
  useEffect(() => {
    if (!templateName.trim()) {
      setTemplateNameAvailability(null);
      setTemplateNameError('');
      return;
    }

    const timeoutId = setTimeout(async () => {
      try {
        setCheckingTemplateName(true);
        setTemplateNameError('');
        const response = await Checkisteplatenameavailable(templateName.toLowerCase().replace(/\s+/g, '_'));
        
        if (response.status === true) {
          setTemplateNameAvailability(true);
        } else {
          setTemplateNameAvailability(false);
          setTemplateNameError(response.message || 'Template name is not available');
        }
      } catch (error) {
        console.error('Error checking template name:', error);
        setTemplateNameError('Error checking template name availability');
        setTemplateNameAvailability(false);
      } finally {
        setCheckingTemplateName(false);
      }
    }, 500); // 500ms debounce

    return () => clearTimeout(timeoutId);
  }, [templateName]);

  const mappingOptions = useMemo(() => {
    return [
      { label: 'Select mapping...', value: '' },
      ...availableVariables.map((variable) => ({
        label: variable.label,
        value: variable.value
      }))
    ];
  }, [availableVariables]);


  const mainBodyVariables = useMemo(() => {
    if (bodyText) {
      return extractVariablesFromWhatsAppMessage(JSON.stringify({ text: bodyText }));
    }
    return [];
  }, [bodyText]);

  const handleVariableSelectionChange = useCallback((variable, value, cardId = null) => {
    if (cardId) {
      // Handle card body variable mapping
      const cardVars = extractVariablesFromWhatsAppMessage(JSON.stringify({ text: carouselCards.find(card => card.id === cardId)?.cardText || '' }));
      const variableIndex = cardVars.indexOf(variable);
      
      setCardBodyVariableMapping(prev => ({
        ...prev,
        [cardId]: {
          ...(prev[cardId] || {}),
          [variableIndex]: value
        }
      }));
    } else {
      // Handle main body variable mapping
      const variableIndex = mainBodyVariables.indexOf(variable);
      setMainBodyVariableMapping(prev => ({
        ...prev,
        [variableIndex]: value
      }));
    }
    
    // Update input value for display
    const variableKey = cardId ? `${cardId}-${variable}` : variable;
    if (value) {
      const selectedVar = availableVariables.find(av => av.value === value);
      setVariableInputValues(prev => ({
        ...prev,
        [variableKey]: selectedVar?.label || ''
      }));
    } else {
      // Clear input when no selection
      setVariableInputValues(prev => ({
        ...prev,
        [variableKey]: ''
      }));
    }
  }, [mainBodyVariables, carouselCards, availableVariables]);

  const generateCarouselTemplateJSON = useMemo(() => {
    const template = {
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: templateName.toLowerCase().replace(/\s+/g, '_') || 'new_carousel_template',
        language: {
          code: templateLanguage
        },
        components: []
      }
    };

   
    if (bodyText) {
      template.template.components.push({
        type: 'BODY',
        text: bodyText,
        parameters: mainBodyVariables.map((variable, index) => ({
          type: 'text',
          text: mainBodyVariableMapping[index] || variable
        }))
      });
    }

   
    if (carouselCards.length > 0 && carouselCards.some(card => card.cardText)) {
      const carouselComponent = {
        type: 'CAROUSEL',
        cards: carouselCards
          .filter(card => card.cardText)
          .map((card, index) => ({
            card_index: index,
            components: [
            
              ...(card.imageUrl || card.videoUrl ? [{
                type: 'HEADER',
                format: card.headerType,
                example: {
                  header_handle: [card.headerType === 'IMAGE' ? card.imageUrl : card.videoUrl]
                },
                parameters: [{
                  type: card.headerType,
                  [card.headerType.toLowerCase()]: {
                    link: card.headerType === 'IMAGE' ? card.imageUrl : card.videoUrl
                  }
                }]
              }] : []),
             
              {
                type: 'BODY',
                text: card.cardText,
                parameters: extractVariablesFromWhatsAppMessage(JSON.stringify({ text: card.cardText || '' })).map((variable, vIndex) => ({
                  type: 'text',
                  text: (cardBodyVariableMapping[card.id] && cardBodyVariableMapping[card.id][vIndex]) || variable
                }))
              },
             
              ...(card.buttons && card.buttons.length > 0 && card.buttons.some(btn => btn.text) ? [{
                type: 'BUTTONS',
                buttons: card.buttons
                  .filter(btn => btn.text)
                  .map(btn => ({
                    type: btn.type,
                    text: btn.text,
                    ...(btn.type === 'URL' && btn.url ? { url: btn.url } : {}),
                    ...(btn.type === 'PHONE_NUMBER' && btn.phone_number ? { phone_number: btn.phone_number } : {})
                  }))
              }] : [])
            ]
          }))
      };
      
      template.template.components.push(carouselComponent);
    }

    return JSON.stringify(template);
  }, [templateName, templateLanguage, bodyText, carouselCards, mainBodyVariables, mainBodyVariableMapping, cardBodyVariableMapping]);


  const extractVariables = useMemo(() => {
    const allText = [
      bodyText,
      ...carouselCards.map(card => card.cardText)
    ].join(' ');
    return extractVariablesFromWhatsAppMessage(allText);
  }, [bodyText, carouselCards]);

  // Handlers for variable mapping
  const handleMainBodyVariableMapping = useCallback((variableIndex, mappingValue) => {
    setMainBodyVariableMapping((prev) => ({
      ...prev,
      [variableIndex]: mappingValue
    }));
  }, []);

  const handleCardBodyVariableMapping = useCallback((cardId, variableIndex, mappingValue) => {
    setCardBodyVariableMapping((prev) => ({
      ...prev,
      [cardId]: {
        ...(prev[cardId] || {}),
        [variableIndex]: mappingValue
      }
    }));
  }, []);


  const handleCardChange = useCallback((cardId, field, value) => {
    setCarouselCards(prev => prev.map(card => 
      card.id === cardId ? { ...card, [field]: value } : card
    ));
  }, []);

  const validateUrlButton = useCallback((url) => {
    if (!url) return { isValid: false, error: 'URL is required' };
    
    // Check if URL contains {{1}} variable
    const variablePattern = /\{\{1\}\}/;
    if (!variablePattern.test(url)) {
      return { isValid: false, error: 'URL must contain a variable in {{1}} format at the end' };
    }
    
    return { isValid: true, error: null };
  }, []);

  // Check if URL has variables and update state
  const checkUrlVariables = useCallback((url) => {
    const hasVariable = /\{\{1\}\}/.test(url);
    return hasVariable;
  }, []);

  // Handle URL variable value change for carousel cards
  const handleUrlVariableChange = useCallback((cardId, buttonIndex, value) => {
    const key = `${cardId}-${buttonIndex}`;
    setUrlVariables(prev => ({
      ...prev,
      [key]: value
    }));
  }, []);

  const handleCardButtonChange = useCallback((cardId, buttonIndex, field, value) => {
    setCarouselCards(prev => prev.map(card => 
      card.id === cardId 
        ? {
            ...card,
            buttons: card.buttons.map((btn, idx) => {
              if (idx === buttonIndex) {
                const updatedBtn = { ...btn, [field]: value };
                
                // Clear URL variable if URL changes and doesn't have variables
                if (field === 'url') {
                  updatedBtn.urlError = null;
                  if (!checkUrlVariables(value)) {
                    const key = `${cardId}-${buttonIndex}`;
                    setUrlVariables(prev => {
                      const newVars = { ...prev };
                      delete newVars[key];
                      return newVars;
                    });
                  }
                }
                
                return updatedBtn;
              }
              return btn;
            })
          }
        : card
    ));
  }, [checkUrlVariables]);

 
  const addCard = useCallback(() => {
    if (carouselCards.length < 10) { 
      const newCard = {
        id: Math.max(...carouselCards.map(c => c.id)) + 1,
        cardText: '',
        headerType: 'IMAGE',
        imageUrl: '',
        videoUrl: '',
        mediaFile: null,
        mediaPreview: '',
        buttons: [{ type: 'URL', text: '', url: '' }]
      };
      setCarouselCards(prev => [...prev, newCard]);
    }
  }, [carouselCards]);


  const removeCard = useCallback((cardId) => {
    if (carouselCards.length > 1) {
      setCarouselCards(prev => prev.filter(card => card.id !== cardId));
    }
  }, [carouselCards.length]);


  const addButtonToCard = useCallback((cardId) => {
    setCarouselCards(prev => prev.map(card => 
      card.id === cardId && card.buttons.length < 1
        ? { ...card, buttons: [...card.buttons, { type: 'URL', text: '', url: '' }] }
        : card
    ));
  }, []);

 
  const removeButtonFromCard = useCallback((cardId, buttonIndex) => {
    setCarouselCards(prev => prev.map(card => 
      card.id === cardId && card.buttons.length > 0
        ? { ...card, buttons: card.buttons.filter((_, idx) => idx !== buttonIndex) }
        : card
    ));
  }, []);


  const handleCardMediaUpload = useCallback((cardId, files) => {
    if (files.length > 0) {
      const file = files[0];
      const isVideo = file.type.startsWith('video/');
      
    
      const reader = new FileReader();
      reader.onload = (e) => {
        setCarouselCards(prev => prev.map(card => 
          card.id === cardId 
            ? {
                ...card,
                mediaFile: file,
                mediaPreview: e.target.result,
                headerType: isVideo ? 'VIDEO' : 'IMAGE',
                imageUrl: isVideo ? '' : e.target.result,
                videoUrl: isVideo ? e.target.result : ''
              }
            : card
        ));
      };
      reader.readAsDataURL(file);
    }
  }, []);


  const handleRemoveCardMedia = useCallback((cardId) => {
    setCarouselCards(prev => prev.map(card => 
      card.id === cardId 
        ? {
            ...card,
            mediaFile: null,
            mediaPreview: '',
            imageUrl: '',
            videoUrl: '',
            headerType: 'IMAGE'
          }
        : card
    ));
  }, []);

  
  const applyFormatting = useCallback((format, cardId, field) => {
    const fieldId = cardId && field ? (field === 'cardText' ? `card-text-${cardId}` : `card-${field}-${cardId}`) : 'body-text-field';
    const textarea = document.querySelector(`#${fieldId}`);
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const end = textarea.selectionEnd;
    const selectedText = textarea.value.substring(start, end);
    
    let formattedText = '';
    let cursorOffset = 0;
    
    if (selectedText) {
    
      switch (format) {
        case 'bold':
          formattedText = `*${selectedText}*`;
          cursorOffset = formattedText.length;
          break;
        case 'italic':
          formattedText = `_${selectedText}_`;
          cursorOffset = formattedText.length;
          break;
        case 'strikethrough':
          formattedText = `~${selectedText}~`;
          cursorOffset = formattedText.length;
          break;
        case 'monospace':
          formattedText = `\`\`\`${selectedText}\`\`\``;
          cursorOffset = formattedText.length;
          break;
        case 'inline-code':
          formattedText = `\`${selectedText}\``;
          cursorOffset = formattedText.length;
          break;
        case 'quote':
          const needsNewLineBeforeQuote = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterQuote = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeQuote ? '\n' : '') + `> ${selectedText}` + (needsNewLineAfterQuote ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          const bulletLines = selectedText.split('\n').map(line => line.trim() ? `* ${line}` : line).join('\n');
          const needsNewLineBefore = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfter = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBefore ? '\n' : '') + bulletLines + (needsNewLineAfter ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          const numberedLines = selectedText.split('\n').filter(line => line.trim()).map((line, index) => `${index + 1}. ${line}`).join('\n');
          const needsNewLineBeforeNum = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterNum = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeNum ? '\n' : '') + numberedLines + (needsNewLineAfterNum ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = selectedText;
          cursorOffset = formattedText.length;
      }
    } else {
     
      switch (format) {
        case 'bold':
          formattedText = '**';
          cursorOffset = 1;
          break;
        case 'italic':
          formattedText = '__';
          cursorOffset = 1;
          break;
        case 'strikethrough':
          formattedText = '~~';
          cursorOffset = 1;
          break;
        case 'monospace':
          formattedText = '\`\`\`\n\n\`\`\`';
          cursorOffset = 4;
          break;
        case 'inline-code':
          formattedText = '\`\`';
          cursorOffset = 1;
          break;
        case 'quote':
          const needsNewLineBeforeQuoteEmpty = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeQuoteEmpty ? '\n> ' : '> ');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          const needsNewLineBeforeBullet = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeBullet ? '\n* ' : '* ');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          const needsNewLineBeforeNumber = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeNumber ? '\n1. ' : '1. ');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = '';
          cursorOffset = 0;
      }
    }
    
    const newValue = textarea.value.substring(0, start) + formattedText + textarea.value.substring(end);
    
    
    if (cardId && field) {
      setCarouselCards(prev => prev.map(card => 
        card.id === cardId 
          ? { ...card, [field]: newValue }
          : card
      ));
    } else if (field === 'body') {
      setBodyText(newValue);
    }
    
    // Set cursor position
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + cursorOffset, start + cursorOffset);
    }, 0);
  }, []);

  // Insert variable at cursor position
  const insertVariable = useCallback((variable, cardId, field) => {
    const fieldId = cardId && field ? (field === 'cardText' ? `card-text-${cardId}` : `card-${field}-${cardId}`) : 'body-text-field';
    const textarea = document.querySelector(`#${fieldId}`);
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const variableText = `{{${variable}}}`;
    const newValue = textarea.value.substring(0, start) + variableText + textarea.value.substring(start);
    
    
    if (cardId && field) {
      setCarouselCards(prev => prev.map(card => 
        card.id === cardId 
          ? { ...card, [field]: newValue }
          : card
      ));
    } else if (field === 'body') {
      setBodyText(newValue);
    }
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + variableText.length, start + variableText.length);
    }, 0);
  }, []);

  // Insert emoji at cursor position
  const insertEmoji = useCallback((emoji, cardId, field) => {
    const fieldId = cardId && field ? (field === 'cardText' ? `card-text-${cardId}` : `card-${field}-${cardId}`) : 'body-text-field';
    const textarea = document.querySelector(`#${fieldId}`);
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const newValue = textarea.value.substring(0, start) + emoji + textarea.value.substring(start);
    

    if (cardId && field) {
      setCarouselCards(prev => prev.map(card => 
        card.id === cardId 
          ? { ...card, [field]: newValue }
          : card
      ));
    } else if (field === 'body') {
      setBodyText(newValue);
    }
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + emoji.length, start + emoji.length);
    }, 0);
  }, []);
  

  const handleTextSelection = useCallback((event) => {
    const target = event.target;
    setTextSelection({
      start: target.selectionStart,
      end: target.selectionEnd
    });
  }, []);

 
  const handleSave = useCallback(async () => {

    console.log('Saving carousel template:', {
      name: templateName,
      category: templateCategory,
      language: templateLanguage,
      bodyText: bodyText,
      cardsCount: carouselCards.length,
      carouselCards: carouselCards.map(card => ({
        id: card.id,
        cardText: card.cardText,
        headerType: card.headerType,
        hasMediaFile: !!card.mediaFile,
        mediaFileName: card.mediaFile?.name,
        buttonsCount: card.buttons?.length || 0
      }))
    });
  
    try {
      const normalizedTemplateName = (templateName || '').toLowerCase().replace(/\s+/g, '_');
      const normalizedCategory = (templateCategory || '').toUpperCase();

      // Extract all body variables from main body and card texts
      const allBodyVars = [];
      
      // Add main body variables
      if (bodyText) {
        const mainBodyVars = extractVariablesFromWhatsAppMessage(JSON.stringify({ text: bodyText }));
        mainBodyVars.forEach((variable, index) => {
          const mappedValue = mainBodyVariableMapping[index] || variable;
          if (!allBodyVars.includes(mappedValue)) {
            allBodyVars.push(mappedValue);
          }
        });
      }

      // Add card body variables
      carouselCards.forEach(card => {
        if (card.cardText) {
          const cardVars = extractVariablesFromWhatsAppMessage(JSON.stringify({ text: card.cardText }));
          cardVars.forEach((variable, index) => {
            const mappedValue = (cardBodyVariableMapping[card.id] && cardBodyVariableMapping[card.id][index]) || variable;
            if (!allBodyVars.includes(mappedValue)) {
              allBodyVars.push(mappedValue);
            }
          });
        }
      });

      // Generate carousel files array
      const carouselFiles = carouselCards.map((card) => {
        if (card && card.mediaFile && card.mediaFile.name) {
          return card.mediaFile.name;
        }
        const link = card?.headerType === 'VIDEO' ? card?.videoUrl : card?.imageUrl;
        if (typeof link === 'string' && link.length > 0) {
          try {
            const withoutQuery = link.split('?')[0];
            const fileName = withoutQuery.split('/').pop();
            return fileName || 'default-image.png';
          } catch {
            return 'default-image.png';
          }
        }
        return 'default-image.png';
      });

      // Determine button type
      const hasUrlButtons = carouselCards.some((c) => (c?.buttons || []).some((b) => b?.type === 'URL' && b?.url));
      const hasQuickReplies = carouselCards.some((c) => (c?.buttons || []).some((b) => b?.type === 'QUICK_REPLY' && b?.text));

     
      const payload = {
        draftedtemplateid: 0,
        normaltemplateid: 0,
        _token: autoReplyToken,
        templatename: normalizedTemplateName || 'new_carousel_template',
        language: templateLanguage,
        category: normalizedCategory || 'MARKETING',
        bodyVars: allBodyVars,
        buttontype: hasUrlButtons ? 'cta_url' : (hasQuickReplies ? 'quick_reply' : 'cta_url'),
        quickReplyButtonText: hasQuickReplies
          ? Array.from(new Set(carouselCards.flatMap((c) => (c?.buttons || [])
            .filter((b) => b?.type === 'QUICK_REPLY' && b?.text)
            .map((b) => b.text))))
          : [],
        savedata: "save_submit",
        mainBody: bodyText || '',
        carouselFiles: carouselFiles,
        carouselCards: carouselCards.map((card, index) => {
          // Get body variables for this specific card
          const cardBodyVars = [];
          if (card.cardText) {
            const cardVars = extractVariablesFromWhatsAppMessage(JSON.stringify({ text: card.cardText }));
            cardVars.forEach((variable, vIndex) => {
              const mappedValue = (cardBodyVariableMapping[card.id] && cardBodyVariableMapping[card.id][vIndex]) || variable;
              cardBodyVars.push(mappedValue);
            });
          }

          // Map buttons to the required format
          const mappedButtons = (card?.buttons || [])
            .filter((btn) => btn?.text)
            .map((btn) => {
              if (btn.type === 'URL') {
                return {
                  type: 'url',
                  text: btn.text,
                  url: btn.url || '',
                  example: btn.url ? [btn.url] : []
                };
              }
              if (btn.type === 'PHONE_NUMBER') {
                return {
                  type: 'phone_number',
                  text: btn.text,
                  phone_number: btn.phone_number || ''
                };
              }
              return {
                type: 'quick_reply',
                text: btn.text
              };
            });

          return {
            body: card?.cardText || '',
            bodyVars: cardBodyVars,
            buttons: {
              type: 'buttons',
              buttons: mappedButtons
            },
            image: `carouselFiles[${index}]`
          };
        })
      };

      // Add URL variables if any exist
      const allUrlVars = [];
      carouselCards.forEach((card) => {
        (card?.buttons || []).forEach((btn, btnIndex) => {
          if (btn.type === 'URL' && btn.url) {
            const key = `${card.id}-${btnIndex}`;
            if (urlVariables[key]) {
              allUrlVars.push(urlVariables[key]);
            }
          }
        });
      });
      
      if (allUrlVars.length > 0) {
        payload.UrlVars = allUrlVars;
      }
  
      console.log('Payload structure:', {
        templatename: payload.templatename,
        language: payload.language,
        category: payload.category,
        bodyVarsCount: payload.bodyVars.length,
        carouselFilesCount: payload.carouselFiles.length,
        carouselCardsCount: payload.carouselCards.length,
        buttontype: payload.buttontype
      });
     
  
     
      const response = await fetch('https://app.getgabs.com/partners/template/CarouselTemplateDesignProcess', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${API_TOKEN}`
        },
        body: JSON.stringify(payload)
      });
  
      if (!response.ok) throw new Error('Failed to save template');
      
      const result = await response.json();
      console.log('Template saved successfully:', result);
      
      navigate('/app/templates');
      
    } catch (error) {
      console.error('Error saving template:', error);
      
    }
    
  }, [
    templateName, 
    templateLanguage, 
    templateCategory, 
    bodyText, 
    carouselCards,
    mainBodyVariableMapping,
    cardBodyVariableMapping,
    urlVariables,
    navigate
  ]);

  const handleCancel = useCallback(() => {
    navigate('/app/templates');
  }, [navigate]);

 
  useEffect(() => {
    const handleKeyDown = (event) => {
      if ((event.ctrlKey || event.metaKey) && 
          (event.target.id === 'body-text-field' || event.target.id.startsWith('card-'))) {
        switch (event.key.toLowerCase()) {
          case 'b':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('bold', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('bold', cardId, 'cardText');
            }
            break;
          case 'i':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('italic', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('italic', cardId, 'cardText');
            }
            break;
          case 's':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('strikethrough', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('strikethrough', cardId, 'cardText');
            }
            break;
          case 'm':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('monospace', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('monospace', cardId, 'cardText');
            }
            break;
          case 'k':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('inline-code', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('inline-code', cardId, 'cardText');
            }
            break;
          case 'q':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('quote', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('quote', cardId, 'cardText');
            }
            break;
          case 'l':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('bullet-list', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('bullet-list', cardId, 'cardText');
            }
            break;
          case 'n':
            event.preventDefault();
            if (event.target.id === 'body-text-field') {
              applyFormatting('numbered-list', null, 'body');
            } else if (event.target.id.startsWith('card-text-')) {
              const cardId = parseInt(event.target.id.split('-')[2]);
              applyFormatting('numbered-list', cardId, 'cardText');
            }
            break;
        }
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [applyFormatting]);

  return (
    <Page
      fullWidth
      backAction={{
        content: 'Templates',
        onAction: () => navigate('/app/templates')
      }}
      title="Create Carousel Template"
      subtitle="Create an interactive carousel template with multiple cards"
      primaryAction={{
        content: 'Save Carousel Template',
        onAction: handleSave,
        disabled: !templateName || !bodyText || carouselCards.length === 0 || !carouselCards.some(card => card.cardText) || templateNameAvailability !== true || checkingTemplateName
      }}
      secondaryActions={[
        {
          content: 'Cancel',
          onAction: handleCancel
        }
      ]}
    >
      <Grid>
        <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 8, lg: 8, xl: 8 }}>
          <div style={{ marginRight: '20px' }}>
            <BlockStack gap="500">
             
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Basic Information</Text>
                  
                  <FormLayout>
                    <TextField
                      label="Template Name"
                      value={templateName}
                      onChange={(value) => setTemplateName(value.replace(/\s+/g, '_'))}
                      placeholder="Enter carousel template name"
                      helpText={
                        checkingTemplateName 
                          ? "Checking availability..." 
                          : templateNameAvailability === true 
                            ? "✓ Template name is available" 
                            : templateNameAvailability === false 
                              ? templateNameError 
                              : "This will be used as the template identifier"
                      }
                      error={templateNameAvailability === false ? templateNameError : undefined}
                      tone={
                        checkingTemplateName 
                          ? undefined 
                          : templateNameAvailability === true 
                            ? "success" 
                            : templateNameAvailability === false 
                              ? "critical" 
                              : undefined
                      }
                      autoComplete="off"
                    />
                    
                    <InlineStack gap="400">
                      <div style={{ flex: 1 }}>
                        <Select
                          label="Category"
                          options={categoryOptions}
                          value={templateCategory}
                          onChange={setTemplateCategory}
                        />
                      </div>
                      <div style={{ flex: 1 }}>
                        <Select
                          label="Language"
                          options={languageOptions}
                          value={templateLanguage}
                          onChange={setTemplateLanguage}
                        />
                      </div>
                    </InlineStack>
                  </FormLayout>
                </BlockStack>
              </Card>

              {/* Main Body Section */}
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Body Text *</Text>
                  
                  {/* Variable insertion buttons */}
                  <BlockStack gap="200">
                    <Text variant="bodySm" as="p" tone="subdued">
                      Click to insert variables :
                    </Text>
                    <InlineStack gap="100" wrap>
                      {availableVariables.slice(0, 15).map((variable) => (
                        <Button
                          key={variable.value}
                          size="micro"
                          variant="tertiary"
                          onClick={() => insertVariable(variable.value, null, 'body')}
                        >
                          {variable.label}
                        </Button>
                      ))}
                      {availableVariables.length > 15 && (
                        <Button
                          size="micro"
                          variant="primary"
                          onClick={() => setShowVariablesModal(true)}
                        >
                          More Variables ({availableVariables.length - 15})
                        </Button>
                      )}
                    </InlineStack>
                  </BlockStack>
                  
                  {/* Custom message textarea with attached formatting controls */}
                  <div style={{ position: 'relative' }}>
                    <div style={{ position: 'absolute', bottom: 12, right: 6, zIndex: 99, display: 'flex', gap: '4px', flexWrap: 'wrap' }}>
                      <ButtonGroup variant="segmented">
                        <EmojiPicker onEmojiSelect={(emoji) => insertEmoji(emoji, null, 'body')} buttonSize="micro" />
                      </ButtonGroup>
                      
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Bold<Text as="span" variant="bodyMd" tone="subdued">⌘B</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bold', null, 'body')} accessibilityLabel="Bold">
                            <Text fontWeight="bold">B</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Italic<Text as="span" variant="bodyMd" tone="subdued">⌘I</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('italic', null, 'body')} accessibilityLabel="Italic">
                            <Text as="em">I</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Strikethrough<Text as="span" variant="bodyMd" tone="subdued">⌘S</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('strikethrough', null, 'body')} accessibilityLabel="Strikethrough">
                            <Text as="s">S</Text>
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Monospace<Text as="span" variant="bodyMd" tone="subdued">⌘M</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('monospace', null, 'body')} accessibilityLabel="Monospace">
                            <Text as="code"> {'</>'}</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Inline Code<Text as="span" variant="bodyMd" tone="subdued">⌘K</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('inline-code', null, 'body')} accessibilityLabel="Inline Code">
                            <Text as="code">{"<>"}</Text>
                          </Button>
                        </Tooltip> 
                        <Tooltip content={<InlineStack gap="200">Quote<Text as="span" variant="bodyMd" tone="subdued">⌘Q</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('quote', null, 'body')} accessibilityLabel="Quote">
                            {'"'}
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Bullet List<Text as="span" variant="bodyMd" tone="subdued">⌘L</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bullet-list', null, 'body')} accessibilityLabel="Bullet List">
                            •
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Numbered List<Text as="span" variant="bodyMd" tone="subdued">⌘N</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('numbered-list', null, 'body')} accessibilityLabel="Numbered List">
                            1.
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                    </div>
                   
                    <TextField
                      id="body-text-field"
                      label="Message Body"
                      value={bodyText}
                      onChange={setBodyText}
                      onSelect={handleTextSelection}
                      multiline={8}
                      maxLength={1024}
                      placeholder="Type your message body here... Formatting options: *bold*, _italic_, ~strikethrough~, ```monospace` ``, ` inline code`, > quote, * bullet list, 1. numbered list. Use keyboard shortcuts or click buttons above."
                      helpText={`${bodyText.length}/1024 characters` }
                      showCharacterCount
                      autoComplete="off"
                    />
                  </div>
                  
                  {/* Main Body Variable Mapping */}
                  {mainBodyVariables.length > 0 && (
                    <div>
                      <Text variant="bodyMd" as="p" fontWeight="semibold" tone="subdued">
                        Body Variable Mapping
                      </Text>
                      <Text variant="bodySm" as="p" tone="subdued" style={{ marginBottom: '12px' }}>
                        Map the variables in your body text to actual data fields:
                      </Text>
                      <BlockStack gap="300">
                        {mainBodyVariables.map((variable, index) => (
                          <FormLayout.Group key={variable}>
                            <TextField
                              label={`Variable: {{${variable}}}`}
                              value={`{{${variable}}}`}
                              disabled
                              helpText="Template variable"
                            />
                            <Autocomplete
                              options={availableVariables
                                .filter(av => {
                                  const searchValue = variableInputValues[variable] || '';
                                  return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                                })
                                .map(av => ({
                                  value: av.value,
                                  label: av.label
                                }))}
                              selected={mainBodyVariableMapping[index] ? [mainBodyVariableMapping[index]] : []}
                              onSelect={(selected) => {
                                const value = selected.length > 0 ? selected[0] : '';
                                handleVariableSelectionChange(variable, value);
                                if (value) {
                                  const selectedVar = availableVariables.find(av => av.value === value);
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: selectedVar?.label || ''
                                  }));
                                } else {
                                  // Clear input when no selection
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: ''
                                  }));
                                }
                              }}
                              textField={
                                <Autocomplete.TextField
                                  label="Map to"
                                  value={variableInputValues[variable] || ''}
                                  onChange={(value) => {
                                    setVariableInputValues(prev => ({
                                      ...prev,
                                      [variable]: value
                                    }));
                                  }}
                                  placeholder="Search and choose data field"
                                  autoComplete="off"
                                />
                              }
                              loading={loadingVariables}
                              emptyState={loadingVariables ? "Loading variables..." : "No variables found"}
                              preferredPosition="below"
                            />
                          </FormLayout.Group>
                        ))}
                      </BlockStack>
                    </div>
                  )}
                </BlockStack>
              </Card>

              {/* Carousel Cards Section */}
              <Card>
                <BlockStack gap="400">
                  <BlockStack gap="100">
                    <Text variant="headingMd" as="h2">Carousel Cards</Text>
                    <Text variant="bodySm" as="p" tone="subdued">
                      Create up to 10 cards for your carousel. Each card can have an image, text content, and buttons.
                    </Text>
                  </BlockStack>
                  
                  <BlockStack gap="400">
                    {carouselCards.map((card, cardIndex) => (
                      <Card key={card.id} background="bg-surface-secondary">
                        <BlockStack gap="400">
                          <InlineStack align="space-between">
                            <Text variant="headingSm" as="h3">
                              Card {cardIndex + 1}
                            </Text>
                            {carouselCards.length > 1 && (
                              <Button
                                variant="plain"
                                tone="critical"
                                onClick={() => removeCard(card.id)}
                                size="slim"
                                icon={DeleteIcon}
                              >
                                Remove Card
                              </Button>
                            )}
                          </InlineStack>
                          
                          {/* Card Header (Image or Video) */}
                          <BlockStack gap="200">
                            <InlineStack align="space-between">
                              <Text variant="bodyMd" as="p" fontWeight="medium">
                                Card Header
                              </Text>
                              <InlineStack gap="200">
                                <Button
                                  size="micro"
                                  variant={card.headerType === 'IMAGE' ? 'primary' : 'tertiary'}
                                  onClick={() => handleCardChange(card.id, 'headerType', 'IMAGE')}
                                >
                                  Image
                                </Button>
                                <Button
                                  size="micro"
                                  variant={card.headerType === 'VIDEO' ? 'primary' : 'tertiary'}
                                  onClick={() => handleCardChange(card.id, 'headerType', 'VIDEO')}
                                >
                                  Video
                                </Button>
                              </InlineStack>
                            </InlineStack>
                            
                            {!card.mediaFile ? (
                              <DropZone
                                onDrop={(files) => handleCardMediaUpload(card.id, files)}
                                accept={card.headerType === 'VIDEO' ? 'video/*' : 'image/*'}
                                type="file"
                                allowMultiple={false}
                              >
                                <DropZone.FileUpload 
                                  actionTitle={`Choose ${card.headerType.toLowerCase()}`} 
                                  actionHint={`or drag and drop your ${card.headerType.toLowerCase()} here`} 
                                />
                              </DropZone>
                            ) : (
                              <BlockStack gap="200">
                                <InlineStack gap="200" align="space-between">
                                  <InlineStack gap="200">
                                    <Thumbnail
                                      source={card.mediaPreview}
                                      alt={card.mediaFile.name}
                                      size="small"
                                    />
                                    <BlockStack gap="050">
                                      <Text variant="bodyMd" as="p">{card.mediaFile.name}</Text>
                                      <Text variant="bodySm" as="p" tone="subdued">
                                        {card.headerType} • {(card.mediaFile.size / 1024 / 1024).toFixed(2)} MB
                                      </Text>
                                    </BlockStack>
                                  </InlineStack>
                                  <Button
                                    variant="plain"
                                    tone="critical"
                                    onClick={() => handleRemoveCardMedia(card.id)}
                                    size="slim"
                                  >
                                    Remove
                                  </Button>
                                </InlineStack>
                              </BlockStack>
                            )}
                          </BlockStack>
                          
                          {/* Card Content */}
                          <BlockStack gap="400">
                            <Text variant="bodyMd" as="p" fontWeight="medium">Card Body *</Text>
                            
                            {/* Variable insertion buttons */}
                            <BlockStack gap="200">
                              <Text variant="bodySm" as="p" tone="subdued">
                                Click to insert variables :
                              </Text>
                              <InlineStack gap="100" wrap>
                                {availableVariables.slice(0, 15).map((variable) => (
                                  <Button
                                    key={variable.value}
                                    size="micro"
                                    variant="tertiary"
                                    onClick={() => insertVariable(variable.value, card.id, 'cardText')}
                                  >
                                    {variable.label}
                                  </Button>
                                ))}
                                {availableVariables.length > 15 && (
                                  <Button
                                    size="micro"
                                    variant="primary"
                                    onClick={() => setShowVariablesModal(true)}
                                  >
                                    More Variables ({availableVariables.length - 15})
                                  </Button>
                                )}
                              </InlineStack>
                            </BlockStack>
                            
                            {/* Custom message textarea with attached formatting controls */}
                            <div style={{ position: 'relative' }}>
                              <div style={{ position: 'absolute', bottom: 12, right: 6, zIndex: 99, display: 'flex', gap: '4px', flexWrap: 'wrap' }}>
                                <ButtonGroup variant="segmented">
                                  <EmojiPicker onEmojiSelect={(emoji) => insertEmoji(emoji, card.id, 'cardText')} buttonSize="micro" />
                                </ButtonGroup>
                                
                                <ButtonGroup variant="segmented">
                                  <Tooltip content={<InlineStack gap="200">Bold<Text as="span" variant="bodyMd" tone="subdued">⌘B</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('bold', card.id, 'cardText')} accessibilityLabel="Bold">
                                      <Text fontWeight="bold">B</Text>
                                    </Button>
                                  </Tooltip>
                                  <Tooltip content={<InlineStack gap="200">Italic<Text as="span" variant="bodyMd" tone="subdued">⌘I</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('italic', card.id, 'cardText')} accessibilityLabel="Italic">
                                      <Text as="em">I</Text>
                                    </Button>
                                  </Tooltip>
                                  <Tooltip content={<InlineStack gap="200">Strikethrough<Text as="span" variant="bodyMd" tone="subdued">⌘S</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('strikethrough', card.id, 'cardText')} accessibilityLabel="Strikethrough">
                                      <Text as="s">S</Text>
                                    </Button>
                                  </Tooltip>
                                </ButtonGroup>
                                <ButtonGroup variant="segmented">
                                  <Tooltip content={<InlineStack gap="200">Monospace<Text as="span" variant="bodyMd" tone="subdued">⌘M</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('monospace', card.id, 'cardText')} accessibilityLabel="Monospace">
                                      <Text as="code"> {'</>'}</Text>
                                    </Button>
                                  </Tooltip>
                                  <Tooltip content={<InlineStack gap="200">Inline Code<Text as="span" variant="bodyMd" tone="subdued">⌘K</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('inline-code', card.id, 'cardText')} accessibilityLabel="Inline Code">
                                      <Text as="code">{"<>"}</Text>
                                    </Button>
                                  </Tooltip> 
                                  <Tooltip content={<InlineStack gap="200">Quote<Text as="span" variant="bodyMd" tone="subdued">⌘Q</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('quote', card.id, 'cardText')} accessibilityLabel="Quote">
                                      {'"'}
                                    </Button>
                                  </Tooltip>
                                </ButtonGroup>
                                <ButtonGroup variant="segmented">
                                  <Tooltip content={<InlineStack gap="200">Bullet List<Text as="span" variant="bodyMd" tone="subdued">⌘L</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('bullet-list', card.id, 'cardText')} accessibilityLabel="Bullet List">
                                      •
                                    </Button>
                                  </Tooltip>
                                  <Tooltip content={<InlineStack gap="200">Numbered List<Text as="span" variant="bodyMd" tone="subdued">⌘N</Text></InlineStack>}>
                                    <Button size="micro" onClick={() => applyFormatting('numbered-list', card.id, 'cardText')} accessibilityLabel="Numbered List">
                                      1.
                                    </Button>
                                  </Tooltip>
                                </ButtonGroup>
                              </div>
                             
                              <TextField
                                id={`card-text-${card.id}`}
                                label="Card Body Message"
                                value={card.cardText}
                                onChange={(value) => handleCardChange(card.id, 'cardText', value)}
                                onSelect={handleTextSelection}
                                multiline={6}
                                maxLength={512}
                                placeholder="Type your card body message here... Formatting options: *bold*, _italic_, ~strikethrough~, ```monospace` ``, ` inline code`, > quote, * bullet list, 1. numbered list. Use keyboard shortcuts or click buttons above."
                                helpText={`${card.cardText.length}/512 characters` }
                                showCharacterCount
                                autoComplete="off"
                              />
                            </div>

                            {/* Card Body Variable Mapping */}
                            {extractVariablesFromWhatsAppMessage(JSON.stringify({ text: card.cardText || '' })).length > 0 && (
                              <div>
                                <Text variant="bodyMd" as="p" fontWeight="semibold" tone="subdued">
                                  Card Body Variable Mapping
                                </Text>
                                <Text variant="bodySm" as="p" tone="subdued" style={{ marginBottom: '12px' }}>
                                  Map the variables in this card's body to actual data fields:
                                </Text>
                                <BlockStack gap="300">
                                  {extractVariablesFromWhatsAppMessage(JSON.stringify({ text: card.cardText || '' })).map((variable, vIndex) => (
                                    <FormLayout.Group key={vIndex}>
                                      <TextField
                                        label={`Variable: {{${variable}}}`}
                                        value={`{{${variable}}}`}
                                        disabled
                                        helpText="Template variable"
                                      />
                                      <Autocomplete
                                        options={availableVariables
                                          .filter(av => {
                                            const searchValue = variableInputValues[`${card.id}-${variable}`] || '';
                                            return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                                          })
                                          .map(av => ({
                                            value: av.value,
                                            label: av.label
                                          }))}
                                        selected={(cardBodyVariableMapping[card.id] && cardBodyVariableMapping[card.id][vIndex]) ? [cardBodyVariableMapping[card.id][vIndex]] : []}
                                        onSelect={(selected) => {
                                          const value = selected.length > 0 ? selected[0] : '';
                                          handleVariableSelectionChange(variable, value, card.id);
                                          if (value) {
                                            const selectedVar = availableVariables.find(av => av.value === value);
                                            setVariableInputValues(prev => ({
                                              ...prev,
                                              [`${card.id}-${variable}`]: selectedVar?.label || ''
                                            }));
                                          } else {
                                            // Clear input when no selection
                                            setVariableInputValues(prev => ({
                                              ...prev,
                                              [`${card.id}-${variable}`]: ''
                                            }));
                                          }
                                        }}
                                        textField={
                                          <Autocomplete.TextField
                                            label="Map to"
                                            value={variableInputValues[`${card.id}-${variable}`] || ''}
                                            onChange={(value) => {
                                              setVariableInputValues(prev => ({
                                                ...prev,
                                                [`${card.id}-${variable}`]: value
                                              }));
                                            }}
                                            placeholder="Search and choose data field"
                                            autoComplete="off"
                                          />
                                        }
                                        loading={loadingVariables}
                                        emptyState={loadingVariables ? "Loading variables..." : "No variables found"}
                                        preferredPosition="below"
                                      />
                                    </FormLayout.Group>
                                  ))}
                                </BlockStack>
                              </div>
                            )}
                          </BlockStack>
                          
                          {/* Card Buttons */}
                          <BlockStack gap="300">
                            <Text variant="bodyMd" fontWeight="medium">
                              Card Buttons
                            </Text>
                            
                            {card.buttons.map((button, buttonIndex) => (
                              <Card key={buttonIndex} background="bg-surface-tertiary">
                                <BlockStack gap="300">
                                  <InlineStack align="space-between">
                                    <Text variant="bodySm" fontWeight="semibold">
                                      Button {buttonIndex + 1}
                                    </Text>
                                    {card.buttons.length > 1 && (
                                      <Button
                                        variant="plain"
                                        tone="critical"
                                        onClick={() => removeButtonFromCard(card.id, buttonIndex)}
                                        size="slim"
                                      >
                                        Remove
                                      </Button>
                                    )}
                                  </InlineStack>
                                  
                                  <FormLayout>
                                    <Select
                                      label="Button Type"
                                      options={buttonTypeOptions}
                                      value={button.type}
                                      onChange={(value) => handleCardButtonChange(card.id, buttonIndex, 'type', value)}
                                    />
                                    
                                    <TextField
                                      label="Button Text"
                                      value={button.text}
                                      onChange={(value) => handleCardButtonChange(card.id, buttonIndex, 'text', value)}
                                      placeholder="Enter button text"
                                      maxLength={20}
                                      showCharacterCount
                                      autoComplete="off"
                                    />
                                    
                                    {button.type === 'URL' && (
                                      <>
                                        <TextField
                                          label="URL"
                                          value={button.url || ''}
                                          onChange={(value) => {
                                            handleCardButtonChange(card.id, buttonIndex, 'url', value);
                                           
                                            const validation = validateUrlButton(value);
                                            if (!validation.isValid && value) {
                                              handleCardButtonChange(card.id, buttonIndex, 'urlError', validation.error);
                                            } else {
                                              handleCardButtonChange(card.id, buttonIndex, 'urlError', null);
                                            }
                                          }}
                                          placeholder="https://example.com/{{1}}"
                                          error={button.urlError}
                                          helpText="URL must contain a variable in {{1}} format at the end"
                                          autoComplete="off"
                                        />
                                        
                                        {button.url && checkUrlVariables(button.url) && (
                                          <TextField
                                            label="URL Variable Value"
                                            value={urlVariables[`${card.id}-${buttonIndex}`] || ''}
                                            onChange={(value) => handleUrlVariableChange(card.id, buttonIndex, value)}
                                            placeholder="Enter value for {{1}} variable"
                                            helpText="This value will replace the {{1}} variable in the URL when sending"
                                            autoComplete="off"
                                          />
                                        )}
                                      </>
                                    )}
                                    
                                    {button.type === 'PHONE_NUMBER' && (
                                      <TextField
                                        label="Phone Number"
                                        value={button.phone_number || ''}
                                        onChange={(value) => handleCardButtonChange(card.id, buttonIndex, 'phone_number', value)}
                                        placeholder="+1234567890"
                                        autoComplete="off"
                                      />
                                    )}
                                  </FormLayout>
                                </BlockStack>
                              </Card>
                            ))}
                            
                            <BlockStack gap="200">
                              <Text variant="bodySm" tone="subdued">
                                You can add up to 1 button per card
                              </Text>
                              
                              {card.buttons.length < 1 && (
                                <Button
                                  onClick={() => addButtonToCard(card.id)}
                                  variant="secondary"
                                  size="medium"
                                  icon={PlusIcon}
                                >
                                  Add Button
                                </Button>
                              )}
                            </BlockStack>
                          </BlockStack>
                        </BlockStack>
                      </Card>
                    ))}
                  </BlockStack>
                  
                  <BlockStack gap="300">
                    <Text variant="bodySm" tone="subdued">
                      You can add up to 10 cards in a carousel template
                    </Text>
                    
                    <Button
                      onClick={addCard}
                      disabled={carouselCards.length >= 10}
                      size="large"
                      icon={PlusIcon}
                      variant="secondary"
                    >
                      Add Card
                    </Button>
                  </BlockStack>
                </BlockStack>
              </Card>

              
              
              
              {/* Save and Cancel Buttons at Bottom */}
              <Card>
                <InlineStack gap="300" align="end">
                  <Button
                    variant="secondary"
                    onClick={handleCancel}
                  >
                    Cancel
                  </Button>
                  <Button
                    variant="primary"
                    onClick={handleSave}
                    disabled={!templateName || !bodyText || carouselCards.length === 0 || !carouselCards.some(card => card.cardText) || templateNameAvailability !== true || checkingTemplateName}
                  >
                    Save Carousel Template
                  </Button>
                </InlineStack>
              </Card>
            </BlockStack>
          </div>
        </Grid.Cell>

        {/* Preview Column */}
        <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 4, lg: 4, xl: 4 }}>
          <div style={{ position: 'sticky', top: '20px' }}>
            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Carousel Preview</Text>
                
                {generateCarouselTemplateJSON && (bodyText || carouselCards.some(card => card.cardText || card.mediaFile)) ? (
                  <BlockStack gap="300">
                    <Text variant="bodySm" as="p">
                      Template: {(templateName ? templateName.toLowerCase().replace(/\s+/g, '_') : 'untitled_carousel_template')}
                    </Text>
                    
                    <WhatsAppPreview 
                      template={generateCarouselTemplateJSON}
                      variant="full"
                      maxWidth="280px"
                      showWhatsAppHeader={false}
                    />
                    
                    <Text variant="bodySm" as="p" tone="subdued">
                      Variables like {`{{product_name}}`} will be replaced with actual data when sent.
                    </Text>
                    
                    <Text variant="bodySm" as="p" tone="info">
                      Body: {bodyText ? '✓' : '✗'} | Cards: {carouselCards.filter(card => card.cardText).length} of {carouselCards.length}
                    </Text>
                  </BlockStack>
                ) : (
                  <BlockStack gap="200">
                    <Text as="p" tone="subdued">
                      Enter template details to see the carousel preview
                    </Text>
                    <LegacyCard sectioned>
                      <Text variant="bodySm" as="p" tone="subdued">
                        Your carousel preview will appear here once you add template content:
                        <br />• Body text, header, or footer
                        <br />• At least one card with text, subtitle, or media
                      </Text>
                    </LegacyCard>
                  </BlockStack>
                )}
              </BlockStack>
            </Card>
          </div>
        </Grid.Cell>
      </Grid>

      {/* Variables Modal */}
      <Modal
        open={showVariablesModal}
        onClose={() => {
          setShowVariablesModal(false);
          setVariableSearchQuery('');
        }}
        title="All Available Variables"
        primaryAction={{
          content: 'Close',
          onAction: () => {
            setShowVariablesModal(false);
            setVariableSearchQuery('');
          },
        }}
        size="large"
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" tone="subdued">
              Click on any variable to insert it into your template:
            </Text>
            
            {/* Search Field */}
            <TextField
              label="Search Variables"
              value={variableSearchQuery}
              onChange={setVariableSearchQuery}
              placeholder="Type to search variables..."
              clearButton
              onClearButtonClick={() => setVariableSearchQuery('')}
              autoComplete="off"
            />
            
            <InlineStack gap="200" wrap>
              {availableVariables
                .filter(variable => 
                  variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
                  variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
                )
                .map((variable) => (
                  <Button
                    key={variable.value}
                    size="medium"
                    variant="tertiary"
                    onClick={() => {
                      insertVariable(variable.value, null, 'body');
                      setShowVariablesModal(false);
                      setVariableSearchQuery('');
                    }}
                  >
                    {variable.label}
                  </Button>
                ))}
            </InlineStack>
            
            {availableVariables.filter(variable => 
              variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
              variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
            ).length === 0 && variableSearchQuery && (
              <Text variant="bodyMd" tone="subdued" alignment="center">
                No variables found matching "{variableSearchQuery}"
              </Text>
            )}
            
            <Text variant="bodySm" tone="subdued">
              You can also create custom variables using the format: {"{{variable_name}}"}
            </Text>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </Page>
  );
}

export default CreateCarouselTemplatePage;
