import { useState, useCallback, useRef, useEffect } from "react";
import {
  Page,
  Layout,
  LegacyCard,
  FormLayout,
  TextField,
  Select,
  Button,
  Text,
  InlineStack,
  BlockStack,
  Checkbox,
  Tabs,
  Modal,
  TextContainer,
  ButtonGroup,
  ChoiceList,
  Banner,
  Divider,
  Box,
  Card,
} from "@shopify/polaris";
import { getChatWidgetDataJson, updateWhatappchatWidget } from "../utils/getgabsSettingsApi";
import { API_BASE_URL } from "../utils/apiConfig";

export default function CreateChatWidgetPage() {
  // Basic Configuration
  const [countryCode, setCountryCode] = useState("+1");
  const [whatsappNumber, setWhatsappNumber] = useState("");
  const [businessName, setBusinessName] = useState("");
  const [currentLogo, setCurrentLogo] = useState(null);
  const [newLogo, setNewLogo] = useState(null);
  const [greetingsText, setGreetingsText] = useState("Hi! How can we help you today?");
  const [buttonText, setButtonText] = useState("Start Chat");
  const [incomingMessageText, setIncomingMessageText] = useState("Thanks for contacting us! We'll get back to you soon.");
  const [brandSubtitle, setBrandSubtitle] = useState("online");

  // Appearance Configuration
  const [alignment, setAlignment] = useState(["right"]);
  const [themeColor, setThemeColor] = useState("#00ab88");
  const [showOn, setShowOn] = useState(["web_mobile"]);
  const [onloadBehavior, setOnloadBehavior] = useState(["closed"]);

  // Loading and error states
  const [isSaving, setIsSaving] = useState(false);
  const [saveError, setSaveError] = useState(null);
  const [saveSuccess, setSaveSuccess] = useState(false);

  // Script key for embedding
  const [ifkey, setIfkey] = useState(null);

  // Modal state for embed to theme
  const [embedToThemeModalActive, setEmbedToThemeModalActive] = useState(false);

  const [selectedTab, setSelectedTab] = useState(0);
  const fileInputRef = useRef(null);



  const buildImageUrl = useCallback((pathOrUrl) => {
    if (!pathOrUrl) return null;
    if (typeof pathOrUrl === "string" && (/^https?:\/\//i).test(pathOrUrl)) return pathOrUrl;

    return `https://app.getgabs.com/img/ifusers/${pathOrUrl}`;
  }, []);


  useEffect(() => {
    let cancelled = false;
    (async () => {
      try {
        const response = await getChatWidgetDataJson();
        if (!response || !response.status) return;
        const data = response.data || {};

        if (cancelled) return;
        const fetchedCountry = data.countryCode ? `+${String(data.countryCode)}` : "+1";
        const fetchedNumber = data.NationalNumber || (data.number ? String(data.number) : "");
        setCountryCode(fetchedCountry);
        setWhatsappNumber(fetchedNumber);

        setBusinessName(data.profile_heading || "");
        setBrandSubtitle(data.brand_subtitle || "online");
        setCurrentLogo(buildImageUrl(data.profile_pic));

        setGreetingsText(data.greeting_msg || greetingsText);
        setButtonText(data.button_text || buttonText);
        setIncomingMessageText(data.chat_query_msg || incomingMessageText);

        setThemeColor(data.header_bg_color || data.button_color || "#00ab88");

        const alignVal = Number(data.aligned) === 1 ? "left" : "right";
        setAlignment([alignVal]);

        const showOnVal = Number(data.open_in_mobile) === 1 ? "web_mobile" : "web_only";
        setShowOn([showOnVal]);

        const onloadVal = Number(data.popup) === 1 ? "opened" : "closed";
        setOnloadBehavior([onloadVal]);

        // Extract ifkey for script embedding
        if (data.ifkey) {
          setIfkey(data.ifkey);
        }
      } catch (e) {

        // console.error(e);
      }
    })();
    return () => {
      cancelled = true;
    };

  }, []);


  const countryCodes = [
    { label: "United States (+1)", value: "+1" },
    { label: "United Kingdom (+44)", value: "+44" },
    { label: "India (+91)", value: "+91" },
    { label: "Canada (+1)", value: "+1" },
    { label: "Australia (+61)", value: "+61" },
    { label: "Germany (+49)", value: "+49" },
    { label: "France (+33)", value: "+33" },
    { label: "Japan (+81)", value: "+81" },
    { label: "China (+86)", value: "+86" },
    { label: "Brazil (+55)", value: "+55" },
    { label: "Mexico (+52)", value: "+52" },
    { label: "Spain (+34)", value: "+34" },
    { label: "Italy (+39)", value: "+39" },
    { label: "Netherlands (+31)", value: "+31" },
    { label: "South Africa (+27)", value: "+27" },
  ];

  const handleSave = useCallback(async () => {
    setIsSaving(true);
    setSaveError(null);
    setSaveSuccess(false);

    try {

      const payload = {
        Aignment: alignment[0] === "left" ? 1 : 0,
        BusinessGreetingsText: greetingsText,
        BusinessNameInput: businessName,
        FormButtonText: buttonText,
        Incomingmessagetext: incomingMessageText,
        newLogoSrc: newLogo ? newLogo.preview : '',
        open_in_mobile: showOn[0] === "web_mobile" ? "1" : "0",
        themeColor: themeColor,
        widgetNumber: `${countryCode}${whatsappNumber}`.replace(/\+/g, ""),
      };

      console.log("Saving widget configuration with payload:", payload);


      const response = await updateWhatappchatWidget(payload);

      console.log("Widget configuration saved successfully:", response);
      setSaveSuccess(true);

      // Update current logo with new logo after successful save
      if (newLogo) {
        setCurrentLogo(newLogo.preview);
        setNewLogo(null); // Clear the new logo state
        if (fileInputRef.current) {
          fileInputRef.current.value = ""; // Clear the file input
        }
      }

      setTimeout(() => {
        setSaveSuccess(false);
      }, 3000);

    } catch (error) {
      console.error("Error saving widget configuration:", error);
      setSaveError(error.message || "Failed to save widget configuration");
    } finally {
      setIsSaving(false);
    }
  }, [countryCode, whatsappNumber, businessName, currentLogo, newLogo, greetingsText, buttonText, incomingMessageText, alignment, themeColor, showOn, onloadBehavior]);

  const handleFileUpload = useCallback((event) => {
    const file = event.target.files[0];
    if (file && (file.type === "image/jpeg" || file.type === "image/png")) {
      const reader = new FileReader();
      reader.onload = (e) => {
        setNewLogo({
          file: file,
          preview: e.target.result,
          name: file.name,
        });
      };
      reader.readAsDataURL(file);
    }
  }, []);

  const handleLogoUploadClick = useCallback(() => {
    fileInputRef.current?.click();
  }, []);

  const removeLogo = useCallback(() => {
    setNewLogo(null);
    if (fileInputRef.current) {
      fileInputRef.current.value = "";
    }
  }, []);

  const copyScriptToClipboard = useCallback(async () => {
    if (!ifkey) return;

    const scriptTag = `<script src="https://app.getgabs.com/getredtowp/iframe/${ifkey}" async></script>`;

    try {
      await navigator.clipboard.writeText(scriptTag);
      // You could add a toast notification here if needed
      console.log('Script copied to clipboard');
    } catch (err) {
      console.error('Failed to copy script: ', err);
    }
  }, [ifkey]);

  const copyIfkeyToClipboard = useCallback(async () => {
    if (!ifkey) return;

    try {
      await navigator.clipboard.writeText(ifkey);
      console.log('Ifkey copied to clipboard');
    } catch (err) {
      console.error('Failed to copy ifkey: ', err);
    }
  }, [ifkey]);

  const openEmbedToThemeModal = useCallback(() => {
    setEmbedToThemeModalActive(true);
  }, []);



  const renderPreview = () => (
    <Card>
      <BlockStack gap="400">
        <Text variant="headingMd" as="h3">Widget Preview</Text>


        <div style={{
          width: "100%",
          height: "500px",
          backgroundColor: "#f6f6f7",
          borderRadius: "8px",
          position: "relative",
          overflow: "hidden"
        }}>
          {/* Website Content */}
          <div style={{ padding: "20px" }}>
            <div style={{ fontSize: "18px", fontWeight: "600", marginBottom: "8px" }}>Your Website</div>
            <div style={{ fontSize: "14px", color: "#666" }}>This is how your chat widget will appear on your website.</div>
          </div>

          {/* WhatsApp Chat Widget Preview */}
          <div style={{
            position: "absolute",
            bottom: "20px",
            [alignment[0] === "left" ? "left" : "right"]: "20px",
            width: "320px",
            height: "400px",
            backgroundColor: "#ffffff",
            borderRadius: "12px",
            boxShadow: "0 4px 20px rgba(0,0,0,0.15)",
            overflow: "hidden",
            display: "flex",
            flexDirection: "column"
          }}>
            {/* Header */}
            <div style={{
              backgroundColor: themeColor,
              color: "white",
              padding: "16px",
              display: "flex",
              alignItems: "center",
              gap: "12px"
            }}>
              {(newLogo || currentLogo) ? (
                <img
                  src={newLogo?.preview || currentLogo}
                  alt="Business Logo"
                  style={{
                    width: "40px",
                    height: "40px",
                    borderRadius: "50%",
                    objectFit: "cover",
                    backgroundColor: "white"
                  }}
                />
              ) : (
                <div style={{
                  width: "40px",
                  height: "40px",
                  borderRadius: "50%",
                  backgroundColor: "rgba(255,255,255,0.2)",
                  display: "flex",
                  alignItems: "center",
                  justifyContent: "center",
                  fontSize: "18px",
                  fontWeight: "bold"
                }}>
                  {businessName ? businessName.charAt(0).toUpperCase() : "B"}
                </div>
              )}
              <div style={{ flex: 1 }}>
                <div style={{ fontWeight: "600", fontSize: "16px" }}>
                  {businessName || "Your Business Name"}
                </div>
                <div style={{ fontSize: "12px", opacity: 0.9 }}>
                  {brandSubtitle || "online"}
                </div>
              </div>
              <div style={{ fontSize: "18px", cursor: "pointer" }}>×</div>
            </div>

            {/* Chat Messages Area */}
            <div style={{
              flex: 1,
              backgroundColor: "#e5ddd5",
              backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23d4c5b9' fill-opacity='0.1'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`,
              padding: "16px",
              display: "flex",
              flexDirection: "column",
              gap: "12px",
              position: "relative"
            }}>
              {/* Date indicator */}
              <div style={{
                textAlign: "center",
                marginBottom: "16px"
              }}>
                <span style={{
                  backgroundColor: "rgba(255,255,255,0.8)",
                  padding: "4px 12px",
                  borderRadius: "12px",
                  fontSize: "12px",
                  color: "#667781"
                }}>
                  Today
                </span>
              </div>

              {/* Greeting Message */}
              <div style={{
                alignSelf: "flex-start",
                maxWidth: "80%"
              }}>
                <div style={{
                  backgroundColor: "white",
                  padding: "8px 12px",
                  borderRadius: "8px",
                  boxShadow: "0 1px 2px rgba(0,0,0,0.1)",
                  fontSize: "14px",
                  color: "#333"
                }}>
                  {greetingsText || "Hello Sir, How can I help you?"}
                </div>
              </div>
            </div>

            {/* Bottom Input/Button Area */}
            <div style={{
              backgroundColor: "white",
              padding: "16px",
              borderTop: "1px solid #e0e0e0"
            }}>
              <button style={{
                width: "100%",
                backgroundColor: themeColor,
                color: "white",
                border: "none",
                padding: "12px 20px",
                borderRadius: "25px",
                fontSize: "14px",
                fontWeight: "500",
                cursor: "pointer",
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                gap: "8px"
              }}>
                <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                  <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.893 3.085" />
                </svg>
                {buttonText || "Start Chat"}
              </button>
            </div>
          </div>
        </div>
      </BlockStack>
    </Card>
  );

  const renderScriptCard = () => {
    if (!ifkey) return null;

    const scriptTag = `<script src="https://app.getgabs.com/getredtowp/iframe/${ifkey}" async></script>`;

    return (
      <Card>
        <BlockStack gap="400">
          <Text variant="headingMd" as="h3">Copy Script</Text>
         

          <div style={{
            backgroundColor: "#f6f6f7",
            border: "1px solid #d1d3d4",
            borderRadius: "8px",
            padding: "16px",
            fontFamily: "monospace",
            fontSize: "14px",
            wordBreak: "break-all",
            position: "relative"
          }}>
            <code>{scriptTag}</code>
            <Button
              size="slim"
              onClick={copyScriptToClipboard}
              style={{
                position: "absolute",
                top: "8px",
                right: "8px"
              }}
            >
              Copy
            </Button>
          </div>

          <Text variant="bodySm" as="p" color="subdued">
            Place this script tag before the closing &lt;/body&gt; tag on all pages where you want the chat widget to appear.
          </Text>
        </BlockStack>
      </Card>
    );
  };

  const isFormValid = countryCode && whatsappNumber && businessName && greetingsText && buttonText && incomingMessageText;

  const renderEmbedToThemeModal = () => (
    <Modal
      open={embedToThemeModalActive}
      onClose={() => setEmbedToThemeModalActive(false)}
      title="Embed Chat Widget to Your Theme"
      primaryAction={{
        content: 'Close',
        onAction: () => setEmbedToThemeModalActive(false),
      }}
      size="large"
    >
      <Modal.Section>
        <BlockStack gap="500">
          <Text variant="headingMd" as="h3">Choose Your Embedding Method</Text>

          {/* Method 1: Copy ifkey */}
          <Card>
            <BlockStack gap="400">
              <Text variant="headingMd" as="h4">Method 1: Using Shopify Theme Customizer</Text>
              <Text variant="bodyMd" as="p" color="subdued">
                Follow these steps to add the chat widget through Shopify's theme customizer:
              </Text>

              <BlockStack gap="300">
                <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                  <div style={{
                    width: '24px',
                    height: '24px',
                    backgroundColor: '#00ab88',
                    color: 'white',
                    borderRadius: '50%',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontSize: '12px',
                    fontWeight: 'bold',
                    flexShrink: 0
                  }}>
                    1
                  </div>
                  <Text variant="bodyMd" as="p">Go to Customize (Online Store → Customize)</Text>
                </div>

                <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                  <div style={{
                    width: '24px',
                    height: '24px',
                    backgroundColor: '#00ab88',
                    color: 'white',
                    borderRadius: '50%',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontSize: '12px',
                    fontWeight: 'bold',
                    flexShrink: 0
                  }}>
                    2
                  </div>
                  <Text variant="bodyMd" as="p">Click on Add Section</Text>
                </div>

                <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                  <div style={{
                    width: '24px',
                    height: '24px',
                    backgroundColor: '#00ab88',
                    color: 'white',
                    borderRadius: '50%',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontSize: '12px',
                    fontWeight: 'bold',
                    flexShrink: 0
                  }}>
                    3
                  </div>
                  <Text variant="bodyMd" as="p">Open Apps tab</Text>
                </div>

                <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                  <div style={{
                    width: '24px',
                    height: '24px',
                    backgroundColor: '#00ab88',
                    color: 'white',
                    borderRadius: '50%',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontSize: '12px',
                    fontWeight: 'bold',
                    flexShrink: 0
                  }}>
                    4
                  </div>
                  <Text variant="bodyMd" as="p">Add GetGabs Chat widget</Text>
                </div>

                <div style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
                  <div style={{
                    width: '24px',
                    height: '24px',
                    backgroundColor: '#00ab88',
                    color: 'white',
                    borderRadius: '50%',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                    fontSize: '12px',
                    fontWeight: 'bold',
                    flexShrink: 0
                  }}>
                    5
                  </div>
                  <Text variant="bodyMd" as="p">Paste the Script Key and save</Text>
                </div>
              </BlockStack>

              {ifkey && (
                <div style={{
                  backgroundColor: "#f6f6f7",
                  border: "1px solid #d1d3d4",
                  borderRadius: "8px",
                  padding: "16px",
                  position: "relative"
                }}>
                  <Text variant="bodyMd" as="p" fontWeight="medium" color="subdued">
                    Your Script Key:
                  </Text>
                  <div style={{
                    fontFamily: "monospace",
                    fontSize: "14px",
                    wordBreak: "break-all",
                    marginTop: "8px",
                    padding: "8px",
                    backgroundColor: "white",
                    borderRadius: "4px",
                    border: "1px solid #e1e3e5"
                  }}>
                    {ifkey}
                  </div>
                  <Button
                    size="slim"
                    onClick={copyIfkeyToClipboard}
                    style={{
                      position: "absolute",
                      top: "8px",
                      right: "8px"
                    }}
                  >
                    Copy Key
                  </Button>
                </div>
              )}
            </BlockStack>
          </Card>

          {/* Method 2: Script tag */}
          <Card>
            <BlockStack gap="400">
              <Text variant="headingMd" as="h4">Method 2: Direct Script Embedding</Text>
              <Text variant="bodyMd" as="p" color="subdued">
                Copy and paste this script tag directly into your theme's footer.liquid file to display the chat widget.
              </Text>

              {renderScriptCard()}
            </BlockStack>
          </Card>
        </BlockStack>
      </Modal.Section>
    </Modal>
  );

  return (
    <Page
      fullWidth
      title="WhatsApp Chat Widget Configuration"
      subtitle="Create and customize your WhatsApp chat widget for your website"
      primaryAction={{
        content: isSaving ? "Saving..." : "Save Widget Configuration",
        onAction: handleSave,
        disabled: !isFormValid || isSaving,
        loading: isSaving,
      }}

    >
      <Layout>
        {!isFormValid && (
          <Layout.Section>
            <Banner status="warning">
              <p>Please fill in all required fields marked with * to save your widget configuration.</p>
            </Banner>
          </Layout.Section>
        )}

        {saveSuccess && (
          <Layout.Section>
            <Banner status="success">
              <p>Widget configuration saved successfully!</p>
            </Banner>
          </Layout.Section>
        )}

        {saveError && (
          <Layout.Section>
            <Banner status="critical">
              <p>Error: {saveError}</p>
            </Banner>
          </Layout.Section>
        )}

        <Layout.Section>
          <div style={{ display: 'grid', gridTemplateColumns: '2fr 1fr', gap: '24px', alignItems: 'start' }}>
            {/* Form Column */}
            <div>
              <BlockStack gap="400">
                {/* Contact Information */}
                <Card>
                  <BlockStack gap="400">
                    <Text variant="headingMd" as="h3">Contact Information</Text>
                    <FormLayout>
                      <FormLayout.Group>
                        <Select
                          label="Country Code"
                          options={countryCodes}
                          value={countryCode}
                          onChange={setCountryCode}
                          requiredIndicator
                        />
                        <TextField
                          label="WhatsApp Number"
                          value={whatsappNumber}
                          onChange={setWhatsappNumber}
                          placeholder="1234567890"
                          helpText="Enter your WhatsApp number without country code"
                          requiredIndicator
                        />
                      </FormLayout.Group>
                      <TextField
                        label="Business Name"
                        value={businessName}
                        onChange={setBusinessName}
                        placeholder="Your Business Name"
                        helpText="This will be displayed in the chat widget"
                        requiredIndicator
                      />
                    </FormLayout>
                  </BlockStack>
                </Card>

                {/* Business Logo */}
                <Card>
                  <BlockStack gap="400">
                    <Text variant="headingMd" as="h3">Business Logo</Text>
                    <FormLayout>
                      {currentLogo && !newLogo && (
                        <div>
                          <Text variant="bodyMd" as="p">Current Business Logo *</Text>
                          <Box paddingBlockStart="200">
                            <img
                              src={currentLogo}
                              alt="Current Logo"
                              style={{ maxWidth: "100px", maxHeight: "100px", objectFit: "contain" }}
                            />
                          </Box>
                        </div>
                      )}

                      <div>
                        <Text variant="bodyMd" as="p">
                          {currentLogo ? "Choose New Business Logo" : "Choose Business Logo"} *
                        </Text>
                        <Box paddingBlockStart="200">
                          <input
                            ref={fileInputRef}
                            type="file"
                            accept="image/jpeg,image/png"
                            onChange={handleFileUpload}
                            style={{ display: "none" }}
                          />
                          <InlineStack gap="200" align="start">
                            <Button onClick={handleLogoUploadClick}>
                              Choose File
                            </Button>
                            {newLogo && (
                              <Button variant="plain" destructive onClick={removeLogo}>
                                Remove
                              </Button>
                            )}
                          </InlineStack>
                          {newLogo && (
                            <Box paddingBlockStart="200">
                              <InlineStack gap="200" align="start">
                                <img
                                  src={newLogo.preview}
                                  alt="New Logo Preview"
                                  style={{ maxWidth: "100px", maxHeight: "100px", objectFit: "contain" }}
                                />
                                <Text variant="bodyMd" as="p">{newLogo.name}</Text>
                              </InlineStack>
                            </Box>
                          )}
                          <Box paddingBlockStart="100">
                            <Text variant="bodySm" as="p" color="subdued">
                              Supported formats: JPG, PNG. ** If You Need To Change The Logo Then Only Choose New Logo
                            </Text>
                          </Box>
                        </Box>
                      </div>
                    </FormLayout>
                  </BlockStack>
                </Card>

                {/* Message Configuration */}
                <Card>
                  <BlockStack gap="400">
                    <Text variant="headingMd" as="h3">Message Configuration</Text>
                    <FormLayout>
                      <TextField
                        label="Greetings Text"
                        value={greetingsText}
                        onChange={setGreetingsText}
                        multiline={3}
                        placeholder="Hi! How can we help you today?"
                        helpText="This message will be pre-filled when users click the widget"
                        requiredIndicator
                      />
                      <TextField
                        label="Button Text"
                        value={buttonText}
                        onChange={setButtonText}
                        placeholder="Start Chat"
                        helpText="Text displayed on the chat widget button"
                        requiredIndicator
                      />
                      <TextField
                        label="Incoming Message Text"
                        value={incomingMessageText}
                        onChange={setIncomingMessageText}
                        multiline={3}
                        placeholder="Thanks for contacting us! We'll get back to you soon."
                        helpText="Auto-reply message when customers send their first message"
                        requiredIndicator
                      />
                    </FormLayout>
                  </BlockStack>
                </Card>

                {/* Appearance Settings */}
                <Card>
                  <BlockStack gap="400">
                    <Text variant="headingMd" as="h3">Appearance & Behavior</Text>
                    <FormLayout>
                      <ChoiceList
                        title="Alignment"
                        choices={[
                          { label: "Left", value: "left" },
                          { label: "Right", value: "right" },
                        ]}
                        selected={alignment}
                        onChange={setAlignment}
                      />

                      <div>
                        <Text variant="bodyMd" as="p" fontWeight="medium">Theme Color</Text>
                        <Box paddingBlockStart="200">
                          <InlineStack gap="300" blockAlign="center" style={{ flexWrap: 'wrap' }}>
                            {/* Color preview box */}
                            {/* <div
      style={{
        width: "32px",
        height: "32px",
        backgroundColor: themeColor,
        borderRadius: "6px",
        border: "1px solid #ccc",
        flexShrink: 0,
      }}
    /> */}

                            {/* HEX input field */}
                            <div style={{ minWidth: "200px" }}>
                              <TextField
                                label="HEX"
                                value={themeColor}
                                onChange={(value) => {
                                  let next = value.trim();
                                  if (next && !next.startsWith("#")) next = `#${next}`;
                                  setThemeColor(next);
                                }}
                                autoComplete="off"
                                helpText="Enter color like #00ab88 or #0a8"
                              />
                            </div>

                            {/* Color picker input */}
                            <div style={{ display: "flex", alignItems: "center" }}>
                              <input
                                type="color"
                                value={
                                  /^#([0-9a-f]{6}|[0-9a-f]{3})$/i.test(themeColor)
                                    ? themeColor
                                    : "#00ab88"
                                }
                                onChange={(e) => setThemeColor(e.target.value)}
                                style={{
                                  width: "40px",
                                  height: "40px",
                                  padding: 0,
                                  border: "1px solid #ccc",
                                  borderRadius: "6px",
                                  background: "none",
                                  cursor: "pointer",
                                  display: "block",
                                }}
                                aria-label="Pick color"
                              />
                            </div>
                          </InlineStack>
                        </Box>


                      </div>

                      <ChoiceList
                        title="Show On"
                        choices={[
                          { label: "Web + Mobile", value: "web_mobile" },
                          { label: "Web Only", value: "web_only" },
                        ]}
                        selected={showOn}
                        onChange={setShowOn}
                      />

                      <div>
                        <Text variant="bodyMd" as="p" fontWeight="medium">Setup Onload Behaviour</Text>
                        <Box paddingBlockStart="100">
                          <Text variant="bodySm" as="p" color="subdued">
                            When your website loads, should the chat bubble be opened or closed?
                          </Text>
                        </Box>
                        <Box paddingBlockStart="200">
                          <ChoiceList
                            title="Initial State"
                            choices={[
                              { label: "Opened Chat Bubble", value: "opened" },
                              { label: "Closed Chat Bubble", value: "closed" },
                            ]}
                            selected={onloadBehavior}
                            onChange={setOnloadBehavior}
                          />
                        </Box>
                      </div>
                    </FormLayout>
                  </BlockStack>
                </Card>

                {/* Bottom Action Buttons */}
                <Card>
                  <BlockStack gap="400">
                    <InlineStack gap="300" align="end">

                      <Button
                        variant="primary"
                        onClick={handleSave}
                        disabled={!isFormValid || isSaving}
                        loading={isSaving}
                      >
                        {isSaving ? "Saving..." : "Save Widget Configuration"}
                      </Button>
                    </InlineStack>
                  </BlockStack>
                </Card>
              </BlockStack>
            </div>

            <div style={{ position: 'sticky', top: '20px' }} >
              <BlockStack gap="400" >
                {renderPreview()}
              </BlockStack>
              {ifkey && (
                <BlockStack gap="400">
                  <Button onClick={openEmbedToThemeModal} variant="primary">
                    Embed to Website
                  </Button>
                </BlockStack>
              )}
            </div>

          </div>
        </Layout.Section>
      </Layout>
      {renderEmbedToThemeModal()}
    </Page>
  );
}
