import {
  Page,
  Card,
  FormLayout,
  TextField,
  Select,
  Button,
  BlockStack,
  InlineStack,
  Text,
  Grid,
  Divider,
  Badge,
  ChoiceList,
  RadioButton,
  ButtonGroup,
  LegacyCard,
  Tooltip,
  Icon,
  Modal,
  Autocomplete
} from '@shopify/polaris';
import { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { ArrowLeftIcon } from '@shopify/polaris-icons';
import { extractVariablesFromWhatsAppMessage } from '../utils/whatsappMessagePreview';
import { getTemplateVariables } from '../utils/availableVariables';
import { Checkisteplatenameavailable } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';
import EmojiPicker from '../components/EmojiPicker';
import { API_TOKEN } from '../utils/apiConfig';

function CreateCatalogTemplatePage() {
  const navigate = useNavigate();
  
 
  const [templateName, setTemplateName] = useState('');
  const [templateCategory, setTemplateCategory] = useState('marketing');
  const [templateLanguage, setTemplateLanguage] = useState('en_US');
  
  
  const [bodyText, setBodyText] = useState('');
  const [cursorPosition, setCursorPosition] = useState(0);
  const [bodyVariableMapping, setBodyVariableMapping] = useState({});
  const [variableInputValues, setVariableInputValues] = useState({});
  
  const [hasFooter, setHasFooter] = useState(false);
  const [footerText, setFooterText] = useState('');
  
  const [catalogButtonText, setCatalogButtonText] = useState('View Catalog');

  // API variables state
  const [availableVariables, setAvailableVariables] = useState([]);
  const [loadingVariables, setLoadingVariables] = useState(true);
  const [showVariablesModal, setShowVariablesModal] = useState(false);
  const [variableSearchQuery, setVariableSearchQuery] = useState('');

  // Template name availability state
  const [templateNameAvailability, setTemplateNameAvailability] = useState(null);
  const [checkingTemplateName, setCheckingTemplateName] = useState(false);
  const [templateNameError, setTemplateNameError] = useState('');
  
  const categoryOptions = [
    { label: 'Marketing', value: 'marketing' },
    { label: 'Utility', value: 'utility' },
    { label: 'Authentication', value: 'authentication' }
  ];

  
  const languageOptions = [
    { label: 'English (US)', value: 'en_US' },
    { label: 'English (UK)', value: 'en_GB' },
    { label: 'Spanish', value: 'es' },
    { label: 'French', value: 'fr' },
    { label: 'German', value: 'de' },
    { label: 'Hindi', value: 'hi' }
  ];

  // Fetch variables from API on component mount
  useEffect(() => {
    const fetchVariables = async () => {
      try {
        setLoadingVariables(true);
        const variables = await getTemplateVariables();
        setAvailableVariables(variables);
      } catch (error) {
        console.error('Error fetching variables:', error);
        setAvailableVariables([]);
      } finally {
        setLoadingVariables(false);
      }
    };

    fetchVariables();
  }, []);

  // Debounced template name availability check
  useEffect(() => {
    if (!templateName.trim()) {
      setTemplateNameAvailability(null);
      setTemplateNameError('');
      return;
    }

    const timeoutId = setTimeout(async () => {
      try {
        setCheckingTemplateName(true);
        setTemplateNameError('');
        const response = await Checkisteplatenameavailable(templateName.toLowerCase().replace(/\s+/g, '_'));
        
        if (response.status === true) {
          setTemplateNameAvailability(true);
        } else {
          setTemplateNameAvailability(false);
          setTemplateNameError(response.message || 'Template name is not available');
        }
      } catch (error) {
        console.error('Error checking template name:', error);
        setTemplateNameError('Error checking template name availability');
        setTemplateNameAvailability(false);
      } finally {
        setCheckingTemplateName(false);
      }
    }, 500); // 500ms debounce

    return () => clearTimeout(timeoutId);
  }, [templateName]);

  const mappingOptions = useMemo(() => {
    return [
      { label: 'Select mapping...', value: '' },
      ...availableVariables.map(variable => ({
        label: variable.label,
        value: variable.value
      }))
    ];
  }, [availableVariables]);

  const handleBodyVariableMapping = useCallback((variableIndex, mappingValue) => {
    setBodyVariableMapping(prev => ({
      ...prev,
      [variableIndex]: mappingValue
    }));
  }, []);

  const bodyVariables = useMemo(() => {
    if (bodyText) {
      return extractVariablesFromWhatsAppMessage(JSON.stringify({ text: bodyText }));
    }
    return [];
  }, [bodyText]);

  const handleVariableSelectionChange = useCallback((variable, value) => {
    const variableIndex = bodyVariables.indexOf(variable);
    setBodyVariableMapping(prev => ({
      ...prev,
      [variableIndex]: value
    }));
    
    // Update input value for display
    if (value) {
      const selectedVar = availableVariables.find(av => av.value === value);
      setVariableInputValues(prev => ({
        ...prev,
        [variable]: selectedVar?.label || ''
      }));
    } else {
      // Clear input when no selection
      setVariableInputValues(prev => ({
        ...prev,
        [variable]: ''
      }));
    }
  }, [bodyVariables, availableVariables]);

  
  const generateTemplateJSON = useMemo(() => {
    const template = {
      sender: "",
      campaign_id: "put_campaign_id",
      messaging_product: "whatsapp",
      recipient_type: "individual",
      to: "receiver_number",
      type: "template",
      template: {
        name: templateName.toLowerCase().replace(/\s+/g, '_') || 'new_catalog_template',
        language: {
          code: templateLanguage
        },
        components: []
      }
    };

    
    if (bodyText) {
      const bodyComponent = {
        type: 'BODY',
        text: bodyText
      };

      if (bodyVariables.length > 0) {
        bodyComponent.parameters = bodyVariables.map((variable, index) => ({
          type: 'text',
          text: bodyVariableMapping[index] || variable
        }));
      }

      template.template.components.push(bodyComponent);
    }

    
    if (hasFooter && footerText) {
      template.template.components.push({
        type: 'FOOTER',
        text: footerText
      });
    }

    
    if (catalogButtonText) {
      template.template.components.push({
        type: 'BUTTONS',
        buttons: [
          {
            type: 'CATALOG',
            text: catalogButtonText
          }
        ]
      });
    }

    return JSON.stringify(template);
  }, [templateName, templateLanguage, bodyText, hasFooter, footerText, catalogButtonText]);

  
  const extractVariables = useMemo(() => {
    return bodyVariables;
  }, [bodyVariables]);

  useEffect(() => {
    setBodyVariableMapping(prev => {
      const newMapping = {};
      bodyVariables.forEach((variable, index) => {
        if (prev[index]) {
          newMapping[index] = prev[index];
        }
      });
      return newMapping;
    });
  }, [bodyVariables]);

  
  const handleTextSelection = useCallback(() => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    setCursorPosition(textarea.selectionStart);
  }, []);

  
  const applyFormatting = useCallback((format) => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const end = textarea.selectionEnd;
    const selectedText = textarea.value.substring(start, end);
    
    let formattedText = '';
    let cursorOffset = 0;
    
    if (selectedText) {
      
      switch (format) {
        case 'bold':
          formattedText = `*${selectedText}*`;
          cursorOffset = formattedText.length;
          break;
        case 'italic':
          formattedText = `_${selectedText}_`;
          cursorOffset = formattedText.length;
          break;
        case 'strikethrough':
          formattedText = `~${selectedText}~`;
          cursorOffset = formattedText.length;
          break;
        case 'monospace':
          formattedText = `\`\`\`${selectedText}\`\`\``;
          cursorOffset = formattedText.length;
          break;
        case 'inline-code':
          formattedText = `\`${selectedText}\``;
          cursorOffset = formattedText.length;
          break;
        case 'quote':
          
          const needsNewLineBeforeQuote = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterQuote = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeQuote ? '\n' : '') + `> ${selectedText}` + (needsNewLineAfterQuote ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          const bulletLines = selectedText.split('\n').map(line => line.trim() ? `* ${line}` : line).join('\n');
          // Ensure list starts on new line
          const needsNewLineBefore = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfter = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBefore ? '\n' : '') + bulletLines + (needsNewLineAfter ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          const numberedLines = selectedText.split('\n').filter(line => line.trim()).map((line, index) => `${index + 1}. ${line}`).join('\n');
          // Ensure list starts on new line
          const needsNewLineBeforeNum = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterNum = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeNum ? '\n' : '') + numberedLines + (needsNewLineAfterNum ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = selectedText;
          cursorOffset = formattedText.length;
      }
    } else {
      
      switch (format) {
        case 'bold':
          formattedText = '**';
          cursorOffset = 1;
          break;
        case 'italic':
          formattedText = '__';
          cursorOffset = 1;
          break;
        case 'strikethrough':
          formattedText = '~~';
          cursorOffset = 1;
          break;
        case 'monospace':
          formattedText = '\`\`\`\n\n\`\`\`';
          cursorOffset = 4;
          break;
        case 'inline-code':
          formattedText = '\`\`';
          cursorOffset = 1;
          break;
        case 'quote':
        
          const needsNewLineBeforeQuoteEmpty = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeQuoteEmpty ? '\n> ' : '> ');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          
          const needsNewLineBeforeBullet = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeBullet ? '\n* ' : '* ');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          
          const needsNewLineBeforeNumber = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeNumber ? '\n1. ' : '1. ');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = '';
          cursorOffset = 0;
      }
    }
    
    const newValue = textarea.value.substring(0, start) + formattedText + textarea.value.substring(end);
    setBodyText(newValue);
    
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + cursorOffset, start + cursorOffset);
    }, 0);
  }, []);

 
  const insertVariable = useCallback((variable) => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const variableText = `{{${variable}}}`;
    const newValue = textarea.value.substring(0, start) + variableText + textarea.value.substring(start);
    
    setBodyText(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + variableText.length, start + variableText.length);
    }, 0);
  }, []);

  
  const insertEmoji = useCallback((emoji) => {
    const textarea = document.querySelector('#body-text-field');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const newValue = textarea.value.substring(0, start) + emoji + textarea.value.substring(start);
    
    setBodyText(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + emoji.length, start + emoji.length);
    }, 0);
  }, []);

 
  useEffect(() => {
    const handleKeyDown = (event) => {
      if ((event.ctrlKey || event.metaKey) && event.target.id === 'body-text-field') {
        switch (event.key.toLowerCase()) {
          case 'b':
            event.preventDefault();
            applyFormatting('bold');
            break;
          case 'i':
            event.preventDefault();
            applyFormatting('italic');
            break;
          case 's':
            event.preventDefault();
            applyFormatting('strikethrough');
            break;
          case 'm':
            event.preventDefault();
            applyFormatting('monospace');
            break;
          case 'k':
            event.preventDefault();
            applyFormatting('inline-code');
            break;
          case 'q':
            event.preventDefault();
            applyFormatting('quote');
            break;
          case 'l':
            event.preventDefault();
            applyFormatting('bullet-list');
            break;
          case 'n':
            event.preventDefault();
            applyFormatting('numbered-list');
            break;
        }
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [applyFormatting]);

 
  const handleSave = useCallback(async () => {
    
    console.log('Saving catalog template:', {
      name: templateName,
      category: templateCategory,
      language: templateLanguage,
      template: generateTemplateJSON,
      hasFooter: hasFooter,
      footerText: footerText,
      catalogButtonText: catalogButtonText
    });

    try {
      const payload = {
        draftedtemplateid: 0,
        normaltemplateid: 0,
        _token: "",
        templatename: templateName.toLowerCase().replace(/\s+/g, '_'),
        language: templateLanguage,
        category: templateCategory.toUpperCase(),
        body: bodyText,
        bodyVars: Object.values(bodyVariableMapping),
        previewfooterText: hasFooter ? footerText : "",
        CatalogButtonText: catalogButtonText
      };

      const response = await fetch(`https://app.getgabs.com/partners/template/with-custom-vars/designtemplate?token=${API_TOKEN}`, {
        method: 'POST',
        body: JSON.stringify(payload),
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${API_TOKEN}`,
          'X-Requested-With': 'XMLHttpRequest',
        }
      });
      
      if (!response.ok) throw new Error('Failed to save template');
      
      const result = await response.json();
      console.log('Template saved successfully:', result);
      
      navigate('/app/templates');
      
    } catch (error) {
      console.error('Error saving template:', error);
    }
    
   
  }, [templateName, templateCategory, templateLanguage, generateTemplateJSON, hasFooter, footerText, catalogButtonText, navigate]);

  const handleCancel = useCallback(() => {
    navigate('/app/templates');
  }, [navigate]);

  return (
    <Page
      fullWidth
      backAction={{
        content: 'Templates',
        onAction: () => navigate('/app/templates')
      }}
      title="Create Catalog Template"
      subtitle="Create a new catalog WhatsApp message template to showcase your products"
      primaryAction={{
        content: 'Save Template',
        onAction: handleSave,
        disabled: !templateName || !bodyText || !catalogButtonText || templateNameAvailability !== true || checkingTemplateName
      }}
      secondaryActions={[
        {
          content: 'Cancel',
          onAction: handleCancel
        }
      ]}
    >
      <Grid>
        <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 8, lg: 8, xl: 8 }}>
          <div style={{ marginRight: '20px' }}>
            <BlockStack gap="500">
             
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Basic Information</Text>
                  
                  <FormLayout>
                    <TextField
                      label="Template Name"
                      value={templateName}
                      onChange={(value) => setTemplateName(value.replace(/\s+/g, '_'))}
                      placeholder="Enter catalog template name"
                      helpText={
                        checkingTemplateName 
                          ? "Checking availability..." 
                          : templateNameAvailability === true 
                            ? "✓ Template name is available" 
                            : templateNameAvailability === false 
                              ? templateNameError 
                              : "This will be used as the template identifier"
                      }
                      error={templateNameAvailability === false ? templateNameError : undefined}
                      tone={
                        checkingTemplateName 
                          ? undefined 
                          : templateNameAvailability === true 
                            ? "success" 
                            : templateNameAvailability === false 
                              ? "critical" 
                              : undefined
                      }
                      autoComplete="off"
                    />
                    
                    <InlineStack gap="400">
                      <div style={{ flex: 1 }}>
                        <Select
                          label="Category"
                          options={categoryOptions}
                          value={templateCategory}
                          onChange={setTemplateCategory}
                        />
                      </div>
                      <div style={{ flex: 1 }}>
                        <Select
                          label="Language"
                          options={languageOptions}
                          value={templateLanguage}
                          onChange={setTemplateLanguage}
                        />
                      </div>
                    </InlineStack>
                  </FormLayout>
                </BlockStack>
              </Card>

              {/* Body Section */}
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Body Text *</Text>
                  
                  {/* Variable insertion buttons */}
                  <BlockStack gap="200">
                    <Text variant="bodySm" as="p" tone="subdued">
                      Click to insert variables or create your own using {"{{ }}"}:
                    </Text>
                    <InlineStack gap="100" wrap>
                      {availableVariables.slice(0, 15).map((variable) => (
                        <Button
                          key={variable.value}
                          size="micro"
                          variant="tertiary"
                          onClick={() => insertVariable(variable.value)}
                        >
                          {variable.label}
                        </Button>
                      ))}
                      {availableVariables.length > 15 && (
                        <Button
                          size="micro"
                          variant="primary"
                          onClick={() => setShowVariablesModal(true)}
                        >
                          More Variables ({availableVariables.length - 15})
                        </Button>
                      )}
                    </InlineStack>
                  </BlockStack>
                  
                  {/* Custom message textarea with attached formatting controls */}
                  <div style={{ position: 'relative' }}>
                    <div style={{ position: 'absolute', bottom: 12, right: 6, zIndex: 99, display: 'flex', gap: '4px', flexWrap: 'wrap' }}>
                      <ButtonGroup variant="segmented">
                        <EmojiPicker onEmojiSelect={insertEmoji} buttonSize="micro" />
                      </ButtonGroup>
                      
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Bold<Text as="span" variant="bodyMd" tone="subdued">⌘B</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bold')} accessibilityLabel="Bold">
                            <Text fontWeight="bold">B</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Italic<Text as="span" variant="bodyMd" tone="subdued">⌘I</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('italic')} accessibilityLabel="Italic">
                            <Text as="em">I</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Strikethrough<Text as="span" variant="bodyMd" tone="subdued">⌘S</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('strikethrough')} accessibilityLabel="Strikethrough">
                            <Text as="s">S</Text>
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Monospace<Text as="span" variant="bodyMd" tone="subdued">⌘M</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('monospace')} accessibilityLabel="Monospace">
                            <Text as="code"> {'</>'}</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Inline Code<Text as="span" variant="bodyMd" tone="subdued">⌘K</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('inline-code')} accessibilityLabel="Inline Code">
                            <Text as="code">{"<>"}</Text>
                          </Button>
                        </Tooltip> 
                        <Tooltip content={<InlineStack gap="200">Quote<Text as="span" variant="bodyMd" tone="subdued">⌘Q</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('quote')} accessibilityLabel="Quote">
                            {'"'}
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Bullet List<Text as="span" variant="bodyMd" tone="subdued">⌘L</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bullet-list')} accessibilityLabel="Bullet List">
                            •
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Numbered List<Text as="span" variant="bodyMd" tone="subdued">⌘N</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('numbered-list')} accessibilityLabel="Numbered List">
                            1.
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                    </div>
                   
                    <TextField
                      id="body-text-field"
                      label="Message Body"
                      value={bodyText}
                      onChange={setBodyText}
                      onSelect={handleTextSelection}
                      multiline={8}
                      maxLength={1024}
                      placeholder="Type your catalog message body here... This message will introduce your product catalog. Formatting options: *bold*, _italic_, ~strikethrough~, ```monospace```, `inline code`, > quote, * bullet list, 1. numbered list. Use keyboard shortcuts or click buttons above."
                      helpText={`${bodyText.length}/1024 characters`}
                      showCharacterCount
                      autoComplete="off"
                    />
                  </div>
                  
                  {extractVariables.length > 0 && (
                    <div>
                      <Text variant="bodyMd" as="p" fontWeight="semibold" tone="subdued">
                        Body Variable Mapping
                      </Text>
                      <Text variant="bodySm" as="p" tone="subdued" style={{ marginBottom: '12px' }}>
                        Map the variables in your body text to actual data fields:
                      </Text>
                      <BlockStack gap="300">
                        {extractVariables.map((variable, index) => (
                          <FormLayout.Group key={index}>
                            <TextField
                              label={`Variable: {{${variable}}}`}
                              value={`{{${variable}}}`}
                              disabled
                              helpText="Template variable"
                            />
                            <Autocomplete
                              options={availableVariables
                                .filter(av => {
                                  const searchValue = variableInputValues[variable] || '';
                                  return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                                })
                                .map(av => ({
                                  value: av.value,
                                  label: av.label
                                }))}
                              selected={bodyVariableMapping[index] ? [bodyVariableMapping[index]] : []}
                              onSelect={(selected) => {
                                const value = selected.length > 0 ? selected[0] : '';
                                handleVariableSelectionChange(variable, value);
                                if (value) {
                                  const selectedVar = availableVariables.find(av => av.value === value);
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: selectedVar?.label || ''
                                  }));
                                } else {
                                  // Clear input when no selection
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: ''
                                  }));
                                }
                              }}
                              textField={
                                <Autocomplete.TextField
                                  label="Map to"
                                  value={variableInputValues[variable] || ''}
                                  onChange={(value) => {
                                    setVariableInputValues(prev => ({
                                      ...prev,
                                      [variable]: value
                                    }));
                                  }}
                                  placeholder="Search and choose data field"
                                  autoComplete="off"
                                />
                              }
                              loading={loadingVariables}
                              emptyState={loadingVariables ? "Loading variables..." : "No variables found"}
                              preferredPosition="below"
                            />
                          </FormLayout.Group>
                        ))}
                      </BlockStack>
                    </div>
                  )}
                </BlockStack>
              </Card>

              {/* Footer Section */}
              <Card>
                <BlockStack gap="400">
                  <InlineStack align="space-between">
                    <Text variant="headingMd" as="h2">Footer Text (Optional)</Text>
                    <Button
                      variant={hasFooter ? 'primary' : 'plain'}
                      onClick={() => setHasFooter(!hasFooter)}
                      size="slim"
                    >
                      {hasFooter ? 'Remove Footer' : 'Add Footer'}
                    </Button>
                  </InlineStack>
                  
                  {hasFooter && (
                    <TextField
                      label="Footer Text"
                      value={footerText}
                      onChange={setFooterText}
                      placeholder="Enter footer text (e.g., Terms & Conditions apply)"
                      maxLength={60}
                      showCharacterCount
                      helpText="Footer text appears at the bottom of the message"
                      autoComplete="off"
                    />
                  )}
                </BlockStack>
              </Card>

              {/* Catalog Button Section */}
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Catalog Button Text *</Text>
                  
                  <TextField
                    label="Button Text"
                    value={catalogButtonText}
                    onChange={setCatalogButtonText}
                    placeholder="Enter catalog button text"
                    maxLength={20}
                    showCharacterCount
                    helpText="This button will open your product catalog for customers"
                    autoComplete="off"
                  />
                  
                  <Text variant="bodySm" as="p" tone="subdued">
                    The catalog button will automatically link to your WhatsApp Business catalog, allowing customers to browse and select products directly from the chat.
                  </Text>
                </BlockStack>
              </Card>

              <Card>
                <InlineStack gap="300" align="end">
                  <Button
                    variant="secondary"
                    onClick={handleCancel}
                  >
                    Cancel
                  </Button>
                  <Button
                    variant="primary"
                    onClick={handleSave}
                    disabled={!templateName || !bodyText || !catalogButtonText || templateNameAvailability !== true || checkingTemplateName}
                  >
                    Save Catalog Template
                  </Button>
                </InlineStack>
              </Card>
            </BlockStack>
          </div>
        </Grid.Cell>

        {/* Preview Section */}
        <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 4, lg: 4, xl: 4 }}>
          <div style={{ position: 'sticky', top: '20px' }}>
            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Template Preview</Text>
                
                {generateTemplateJSON && (bodyText || hasFooter || catalogButtonText) ? (
                  <BlockStack gap="300">
                    <Text variant="bodySm" as="p">
                      Template: {(templateName ? templateName.toLowerCase().replace(/\s+/g, '_') : 'untitled_catalog_template')}
                    </Text>
                    
                    <WhatsAppPreview 
                      template={generateTemplateJSON} 
                      variant="full" 
                      maxWidth="280px"
                      showWhatsAppHeader={false}
                    />
                    
                    <Text variant="bodySm" as="p" tone="subdued">
                      This is how your catalog template will appear to customers. The catalog button will open your product catalog when tapped.
                    </Text>
                  </BlockStack>
                ) : (
                  <BlockStack gap="200">
                    <Text as="p" tone="subdued">
                      Enter template details to see the preview
                    </Text>
                    <div style={{ 
                      padding: '16px', 
                      border: '1px solid #e1e3e5', 
                      borderRadius: '8px', 
                      backgroundColor: '#fafbfb' 
                    }}>
                      <Text variant="bodySm" as="p" tone="subdued">
                        Your catalog template preview will appear here once you add template content (body text, footer, or catalog button).
                      </Text>
                    </div>
                  </BlockStack>
                )}
              </BlockStack>
            </Card>
          </div>
        </Grid.Cell>
      </Grid>

      {/* Variables Modal */}
      <Modal
        open={showVariablesModal}
        onClose={() => {
          setShowVariablesModal(false);
          setVariableSearchQuery('');
        }}
        title="All Available Variables"
        primaryAction={{
          content: 'Close',
          onAction: () => {
            setShowVariablesModal(false);
            setVariableSearchQuery('');
          },
        }}
        size="large"
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" tone="subdued">
              Click on any variable to insert it into your template:
            </Text>
            
            {/* Search Field */}
            <TextField
              label="Search Variables"
              value={variableSearchQuery}
              onChange={setVariableSearchQuery}
              placeholder="Type to search variables..."
              clearButton
              onClearButtonClick={() => setVariableSearchQuery('')}
              autoComplete="off"
            />
            
            <InlineStack gap="200" wrap>
              {availableVariables
                .filter(variable => 
                  variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
                  variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
                )
                .map((variable) => (
                  <Button
                    key={variable.value}
                    size="medium"
                    variant="tertiary"
                    onClick={() => {
                      insertVariable(variable.value);
                      setShowVariablesModal(false);
                      setVariableSearchQuery('');
                    }}
                  >
                    {variable.label}
                  </Button>
                ))}
            </InlineStack>
            
            {availableVariables.filter(variable => 
              variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
              variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
            ).length === 0 && variableSearchQuery && (
              <Text variant="bodyMd" tone="subdued" alignment="center">
                No variables found matching "{variableSearchQuery}"
              </Text>
            )}
            
            <Text variant="bodySm" tone="subdued">
              You can also create custom variables using the format: {"{{variable_name}}"}
            </Text>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </Page>
  );
}

export default CreateCatalogTemplatePage;
