import {
  Page,
  Layout,
  LegacyCard,
  Button,
  TextField,
  Select,
  BlockStack,
  InlineStack,
  Text,
  Badge,
  ChoiceList,
  DatePicker,
  Popover,
  Icon,
  FormLayout,
  Divider,
  Banner,
  RadioButton,
  Checkbox,
  Tooltip,
  ButtonGroup,
  Tag,
  Thumbnail,
  Autocomplete,
  Modal,
} from '@shopify/polaris';
import { CalendarIcon, ClockIcon, ArrowLeftIcon } from '@shopify/polaris-icons';
import React, { useState, useCallback, useMemo, useEffect, useRef } from 'react';
import { useNavigate } from '@remix-run/react';
import { getTemplateVariables } from '../utils/availableVariables';
import { fetchStaticContactList, hasContactlisthaveSelectedtemplateVariables } from '../utils/contactsApi';
import { getAllTemplateList, getSelectedTemplatebyName, saveTemporaryFile } from '../utils/templateApi';
import { sendBroadcastCampaign } from '../utils/broadcastApi';
import WhatsAppPreview from '../components/WhatsAppPreview';


function CreateBroadcastPage() {
  const navigate = useNavigate();
  
 
  const [broadcastName, setBroadcastName] = useState('');
  const [selectedTemplateId, setSelectedTemplateId] = useState('');
  const [templateInputValue, setTemplateInputValue] = useState('');
  const [variableMapping, setVariableMapping] = useState({});
  const [mediaMapping, setMediaMapping] = useState({});
  const [targetAudience, setTargetAudience] = useState('all');
  const [customAudienceFilters, setCustomAudienceFilters] = useState({
    collections: [],
    customerTags: [],
    orderHistory: 'any',
    location: ''
  });
  const [selectedLiveSegments, setSelectedLiveSegments] = useState([]);
  const [selectedStaticLists, setSelectedStaticLists] = useState([]);
  const [liveSegmentInputValue, setLiveSegmentInputValue] = useState('');
  const [staticListInputValue, setStaticListInputValue] = useState('');
  const [liveSegmentDropdownActive, setLiveSegmentDropdownActive] = useState(false);
  const [staticListDropdownActive, setStaticListDropdownActive] = useState(false);
  const [scheduleType, setScheduleType] = useState('immediate');
  const [scheduledDate, setScheduledDate] = useState(new Date());
  const [scheduledTime, setScheduledTime] = useState('09:00');
  const [datePickerActive, setDatePickerActive] = useState(false);
  const [estimatedReach, setEstimatedReach] = useState(0);
  

  const [staticContactLists, setStaticContactLists] = useState([]);
  const [staticListsLoading, setStaticListsLoading] = useState(false);
  const [staticListsLoadingMore, setStaticListsLoadingMore] = useState(false);
  const [staticListsNextPageUrl, setStaticListsNextPageUrl] = useState(null);
  const [staticListsError, setStaticListsError] = useState(null);
  const [debouncedSearchTerm, setDebouncedSearchTerm] = useState('');
  
  const [apiTemplates, setApiTemplates] = useState([]);
  const [loadingTemplates, setLoadingTemplates] = useState(false);
  const [templateError, setTemplateError] = useState(null);
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [uploadingMedia, setUploadingMedia] = useState({});
  const [sendingBroadcast, setSendingBroadcast] = useState(false);
  const [broadcastError, setBroadcastError] = useState(null);
  
  // Contact list fields state
  const [contactListFields, setContactListFields] = useState([]);
  const [loadingContactListFields, setLoadingContactListFields] = useState(false);
  const [contactListFieldsError, setContactListFieldsError] = useState(null);
  
  
  const [staticValueEnabled, setStaticValueEnabled] = useState({});
  const [staticValues, setStaticValues] = useState({});
  const [urlButtonVariableMapping, setUrlButtonVariableMapping] = useState({});
  const [urlButtonStaticValueEnabled, setUrlButtonStaticValueEnabled] = useState({});
  const [urlButtonStaticValues, setUrlButtonStaticValues] = useState({});
  
  // API variables state
  const [availableVariables, setAvailableVariables] = useState([
    { label: 'name', value: '{{name}}' },
    { label: 'number', value: '{{number}}' }
  ]);
  const [loadingVariables, setLoadingVariables] = useState(true);
  const [showVariablesModal, setShowVariablesModal] = useState(false);
  const [variableSearchQuery, setVariableSearchQuery] = useState('');
  const [variableInputValues, setVariableInputValues] = useState({});
  const [urlButtonVariableInputValues, setUrlButtonVariableInputValues] = useState({});
  
  const fetchApiTemplates = useCallback(async () => {
    try {
      setLoadingTemplates(true);
      setTemplateError(null);
      const response = await getAllTemplateList();

      if (response.status && response.data && response.data.customtemplates) {
        const transformedTemplates = response.data.customtemplates.map(template => ({
          id: template.template_id ? template.template_id.toString() : 'unknown',
          name: template.template_name || 'Unknown Template',
          templateId: template.template_id,
          userId: template.user_id,
          label: template.template_name || 'Unknown Template',
          value: template.template_id ? template.template_id.toString() : 'unknown'
        }));

        setApiTemplates(transformedTemplates);
      } else {
        console.error('Invalid API response format:', response);
        setApiTemplates([]);
      }
    } catch (error) {
      console.error('Error fetching templates:', error);
      setTemplateError(`Failed to load templates: ${error.message}`);
      setApiTemplates([]);
    } finally {
      setLoadingTemplates(false);
    }
  }, []);

  useEffect(() => {
    fetchApiTemplates();
  }, [fetchApiTemplates]);


  const allTemplateOptions = useMemo(() => {
    return apiTemplates;
  }, [apiTemplates]);

  
  useEffect(() => {
    if (selectedTemplateData) {
      
      const initialVariableMapping = {};
      
      if (selectedTemplateData.components) {
        selectedTemplateData.components.forEach((component, index) => {
          if (component.type === 'BODY' && component.text) {
            const variables = component.text.match(/\{\{([^}]+)\}\}/g);
            if (variables) {
              variables.forEach((variable, varIndex) => {
                const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                const key = `body_var_${varName}`;
                initialVariableMapping[key] = {
                  placeholder: `{{${varName}}}`,
                  value: '',
                  required: true,
                  componentType: 'BODY'
                };
              });
            }
          } else if (component.type === 'FOOTER' && component.text) {
            const variables = component.text.match(/\{\{([^}]+)\}\}/g);
            if (variables) {
              variables.forEach((variable, varIndex) => {
                const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                const key = `footer_var_${varName}`;
                initialVariableMapping[key] = {
                  placeholder: `{{${varName}}}`,
                  value: '',
                  required: true,
                  componentType: 'FOOTER'
                };
              });
            }
          } else if (component.type === 'HEADER' && component.text) {
            const variables = component.text.match(/\{\{([^}]+)\}\}/g);
            if (variables) {
              variables.forEach((variable, varIndex) => {
                const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                const key = `header_var_${varName}`;
                initialVariableMapping[key] = {
                  placeholder: `{{${varName}}}`,
                  value: '',
                  required: true,
                  componentType: 'HEADER'
                };
              });
            }
          }
        });
      }
      
      setVariableMapping(initialVariableMapping);
      
      // Initialize URL button variable mapping
      const initialUrlButtonVariableMapping = {};
      
      if (selectedTemplateData.components) {
        selectedTemplateData.components.forEach((component, componentIndex) => {
          if (component.type === 'BUTTONS' && component.buttons) {
            component.buttons.forEach((button, buttonIndex) => {
              if (button.type === 'URL' && button.url) {
                const variables = button.url.match(/\{\{([^}]+)\}\}/g);
                if (variables) {
                  variables.forEach((variable, varIndex) => {
                    const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                    const key = `url_button_${componentIndex}_${buttonIndex}_var_${varName}`;
                    initialUrlButtonVariableMapping[key] = {
                      placeholder: `{{${varName}}}`,
                      value: '',
                      required: true,
                      componentType: 'URL_BUTTON',
                      buttonText: button.text,
                      buttonUrl: button.url,
                      componentIndex,
                      buttonIndex
                    };
                  });
                }
              }
            });
          }
        });
      }
      
      setUrlButtonVariableMapping(initialUrlButtonVariableMapping);
      
      // Initialize media mapping directly from template data
      const initialMediaMapping = {};
      
      if (selectedTemplateData.components) {
        selectedTemplateData.components.forEach((component, index) => {
          if (component.type === 'HEADER' && component.format === 'IMAGE') {
            initialMediaMapping[`header_${index}`] = {
              type: 'image',
              url: component.example?.header_handle?.[0] || '',
              required: true,
              fileUrl: null,
              file: null
            };
          } else if (component.type === 'HEADER' && component.format === 'VIDEO') {
            initialMediaMapping[`header_${index}`] = {
              type: 'video',
              url: component.example?.header_handle?.[0] || '',
              required: true,
              fileUrl: null,
              file: null
            };
          } else if (component.type === 'HEADER' && component.format === 'DOCUMENT') {
            initialMediaMapping[`header_${index}`] = {
              type: 'document',
              url: component.example?.header_handle?.[0] || '',
              required: true,
              fileUrl: null,
              file: null
            };
          }
        });
      }
      
      setMediaMapping(initialMediaMapping);
    }
  }, [selectedTemplateData]);

  // Filter templates based on search term
  const filteredTemplateOptions = useMemo(() => {
    if (!templateInputValue.trim()) return allTemplateOptions;
    
    const searchLower = templateInputValue.toLowerCase();
    return allTemplateOptions.filter(template => 
      template.label.toLowerCase().includes(searchLower) ||
      template.name?.toLowerCase().includes(searchLower)
    );
  }, [allTemplateOptions, templateInputValue]);

  
  const templateVariables = useMemo(() => {
    if (!selectedTemplateData) return {};
    
    const variableMapping = {};
    
    if (selectedTemplateData.components) {
      selectedTemplateData.components.forEach((component, index) => {
        if (component.type === 'BODY' && component.text) {
            
          const variables = component.text.match(/\{\{([^}]+)\}\}/g);
          if (variables) {
            variables.forEach((variable, varIndex) => {
              const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
              variableMapping[`body_var_${varName}`] = {
                placeholder: `{{${varName}}}`,
                value: '',
                required: true,
                componentType: 'BODY'
              };
            });
          }
        }  else if (component.type === 'HEADER' && component.text) {
          
          const variables = component.text.match(/\{\{([^}]+)\}\}/g);
          if (variables) {
            variables.forEach((variable, varIndex) => {
              const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
              variableMapping[`header_var_${varName}`] = {
                placeholder: `{{${varName}}}`,
                value: '',
                required: true,
                componentType: 'HEADER'
              };
            });
          }
        }
      });
    }
    
    return variableMapping;
  }, [selectedTemplateData]);

  // Extract URL button variables from template
  const urlButtonVariables = useMemo(() => {
    if (!selectedTemplateData) return {};
    
    const buttonVariableMapping = {};
    
    if (selectedTemplateData.components) {
      selectedTemplateData.components.forEach((component, componentIndex) => {
        if (component.type === 'BUTTONS' && component.buttons) {
          component.buttons.forEach((button, buttonIndex) => {
            if (button.type === 'URL' && button.url) {
              const variables = button.url.match(/\{\{([^}]+)\}\}/g);
              if (variables) {
                variables.forEach((variable, varIndex) => {
                  const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                  const key = `url_button_${componentIndex}_${buttonIndex}_var_${varName}`;
                  buttonVariableMapping[key] = {
                    placeholder: `{{${varName}}}`,
                    value: '',
                    required: true,
                    componentType: 'URL_BUTTON',
                    buttonText: button.text,
                    buttonUrl: button.url,
                    componentIndex,
                    buttonIndex
                  };
                });
              }
            }
          });
        }
      });
    }
    
    return buttonVariableMapping;
  }, [selectedTemplateData]);



  const templateMedia = useMemo(() => {
    if (!selectedTemplateData) return {};
    
    const mediaMapping = {};
    
    if (selectedTemplateData.components) {
      selectedTemplateData.components.forEach((component, index) => {
        if (component.type === 'HEADER' && component.format === 'IMAGE') {
          mediaMapping[`header_${index}`] = {
            type: 'image',
            url: component.example?.header_handle?.[0] || '',
            required: true,
            fileUrl: null,
            file: null
          };
        } else if (component.type === 'HEADER' && component.format === 'VIDEO') {
          mediaMapping[`header_${index}`] = {
            type: 'video',
            url: component.example?.header_handle?.[0] || '',
            required: true,
            fileUrl: null,
            file: null
          };
        } else if (component.type === 'HEADER' && component.format === 'DOCUMENT') {
          mediaMapping[`header_${index}`] = {
            type: 'document',
            url: component.example?.header_handle?.[0] || '',
            required: true,
            fileUrl: null,
            file: null
          };
        }
      });
    }
    
    return mediaMapping;
  }, [selectedTemplateData]);

  // Fetch variables from API on component mount
  useEffect(() => {
    const fetchVariables = async () => {
      try {
        setLoadingVariables(true);
        const variables = await getTemplateVariables();
        // Add predefined variables to the beginning of the variables array
        const predefinedVariables = [
          { label: 'name', value: '{{name}}' },
          { label: 'number', value: '{{number}}' }
        ];
        setAvailableVariables([...predefinedVariables, ...variables]);
      } catch (error) {
        console.error('Error fetching variables:', error);
        setAvailableVariables([]);
      } finally {
        setLoadingVariables(false);
      }
    };

    fetchVariables();
  }, []);

  // Sync input values with existing variable mappings
  useEffect(() => {
    if (availableVariables.length > 0) {
      // Sync regular variable input values
      const newVariableInputValues = {};
      Object.entries(variableMapping).forEach(([variableKey, variableData]) => {
        if (variableData.value) {
          const selectedVar = availableVariables.find(av => av.value === variableData.value);
          if (selectedVar) {
            newVariableInputValues[variableKey] = selectedVar.label;
          }
        }
      });
      setVariableInputValues(newVariableInputValues);

      // Sync URL button variable input values
      const newUrlButtonInputValues = {};
      Object.entries(urlButtonVariableMapping).forEach(([variableKey, variableData]) => {
        if (variableData.value) {
          const selectedVar = availableVariables.find(av => av.value === variableData.value);
          if (selectedVar) {
            newUrlButtonInputValues[variableKey] = selectedVar.label;
          }
        }
      });
      setUrlButtonVariableInputValues(newUrlButtonInputValues);
    }
  }, [availableVariables, variableMapping, urlButtonVariableMapping]);
  
  const convertVariableMappingForPreview = () => {
    const previewMapping = {};
    
    Object.entries(variableMapping).forEach(([variableKey, variableData]) => {
      const varName = variableKey.replace(/^(body_var_|header_var_|footer_var_)/, '');
      
     
      if (staticValueEnabled[variableKey] && staticValues[variableKey]) {
        previewMapping[varName] = staticValues[variableKey];
      } else if (variableData && variableData.value) {
        const availableVar = availableVariables.find(av => av.value === variableData.value);
        const displayValue = availableVar ? availableVar.label : variableData.value;
        
        previewMapping[varName] = displayValue;
      }
    });
    
    return previewMapping;
  };
  
  const convertMediaMappingForPreview = () => {
    const previewMapping = {};
    
    Object.entries(mediaMapping).forEach(([mediaKey, mediaData]) => {
      if (mediaData && (mediaData.file || mediaData.url)) {
        previewMapping[mediaKey] = {
          link: mediaData.fileUrl || mediaData.url,
          file: mediaData.file,
          uploaded: mediaData.uploaded || false,
          fileName: mediaData.fileName
        };
      }
    });
    
    return previewMapping;
  };
  
  const getTemplateForPreview = (templateData) => {
    if (!templateData) return null;
    
    try {
      const templateValue = {
        sender: "",
        campaign_id: "put_campaign_id",
        messaging_product: "whatsapp",
        recipient_type: "individual",
        to: "receiver_number",
        type: "template",
        template: {
          name: templateData.name || 'template',
          language: {
            code: templateData.language || 'en_US'
          },
          components: templateData.components || []
        }
      };
      
      return JSON.stringify(templateValue);
    } catch (error) {
      console.error('Error preparing template for preview:', error);
      return null;
    }
  };

  
  const audienceOptions = [
    // { label: 'All customers', value: 'all' },
    { label: 'Static List (your uploaded Contacts)', value: 'static_list' },
    { label: 'Live Segment', value: 'live_segment' }
  ];
  
  const liveSegmentOptions = [
    { label: 'High Value Customers', value: 'high_value_customers', contacts: 2450 },
    { label: 'Abandoned Cart Users', value: 'abandoned_cart_users', contacts: 1230 },
    { label: 'New Subscribers', value: 'new_subscribers', contacts: 890 },
    { label: 'Frequent Buyers', value: 'frequent_buyers', contacts: 3200 },
    { label: 'VIP Members', value: 'vip_members', contacts: 450 },
    { label: 'Recent Purchasers', value: 'recent_purchasers', contacts: 1800 }
  ];

  const staticListOptions = useMemo(() => {
    const uniqueLists = new Map();
    staticContactLists.forEach(list => {
      if (list.id && !uniqueLists.has(list.id)) {
        uniqueLists.set(list.id, {
          label: list.name || `List ${list.id}`,
          value: list.id?.toString(),
          contacts: list.totalcontatspertag_count || 0,
          id: list.id
        });
      }
    });
    return Array.from(uniqueLists.values());
  }, [staticContactLists]);

  
  useEffect(() => {
    let reach = 0;
    switch (targetAudience) {
      case 'all':
        reach = 0;
        break;
      case 'live_segment':
        
        reach = selectedLiveSegments.reduce((total, segmentValue) => {
          const segment = liveSegmentOptions.find(option => option.value === segmentValue);
          return total + (segment ? segment.contacts : 0);
        }, 0);
        break;
      case 'static_list':
        
        reach = selectedStaticLists.reduce((total, listValue) => {
          const list = staticListOptions.find(option => option.value === listValue);
          return total + (list ? list.contacts : 0);
        }, 0);
        break;
      default:
        reach = 0;
    }
    setEstimatedReach(reach);
  }, [targetAudience, selectedLiveSegments, selectedStaticLists, liveSegmentOptions, staticListOptions]);

 
  const fetchStaticLists = useCallback(async (resetData = true) => {
    if (resetData) {
      setStaticListsLoading(true);
      setStaticListsError(null);
      setStaticContactLists([]);
      setStaticListsNextPageUrl(null);
    } else {
      setStaticListsLoadingMore(true);
    }
    
    try {
      const response = await fetchStaticContactList({
        page: 1,
        perPageData: 50, 
        actionType: 0
      });
      
      let newLists = [];
      let nextPageUrl = null;
      
      if (response?.WhatsAppBusinessTag?.data && Array.isArray(response.WhatsAppBusinessTag.data)) {
        newLists = response.WhatsAppBusinessTag.data;
        nextPageUrl = response.WhatsAppBusinessTag.next_page_url || null;
      } else if (response?.data && Array.isArray(response.data)) {
        newLists = response.data;
        nextPageUrl = response.next_page_url || null;
      } else if (response?.contactLists && Array.isArray(response.contactLists)) {
        newLists = response.contactLists;
        nextPageUrl = response.next_page_url || null;
      } else {
        console.warn('Unexpected API response format:', response);
        newLists = [];
      }
      
      if (resetData) {
        setStaticContactLists(newLists);
      } else {
        setStaticContactLists(prev => [...prev, ...newLists]);
      }
      
      setStaticListsNextPageUrl(nextPageUrl);
      
     
      
    } catch (error) {
      console.error('Error fetching static contact lists:', error);
      setStaticListsError('Failed to load contact lists. Please try again.');
      if (resetData) {
        setStaticContactLists([]);
      }
    } finally {
      if (resetData) {
        setStaticListsLoading(false);
      } else {
        setStaticListsLoadingMore(false);
      }
    }
  }, []);

  const loadMoreStaticLists = useCallback(async (nextPageUrl) => {
    if (!nextPageUrl) return;
    
    setStaticListsLoadingMore(true);
    
    try {
      const response = await fetch(nextPageUrl, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
      
      let newLists = [];
      let nextUrl = null;
      
      if (data?.WhatsAppBusinessTag?.data && Array.isArray(data.WhatsAppBusinessTag.data)) {
        newLists = data.WhatsAppBusinessTag.data;
        nextUrl = data.WhatsAppBusinessTag.next_page_url || null;
      } else if (data?.data && Array.isArray(data.data)) {
        newLists = data.data;
        nextUrl = data.next_page_url || null;
      } else if (data?.contactLists && Array.isArray(data.contactLists)) {
        newLists = data.contactLists;
        nextUrl = data.next_page_url || null;
      }
      
  
      setStaticContactLists(prev => {
        const existingIds = new Set(prev.map(list => list.id));
        const uniqueNewLists = newLists.filter(list => !existingIds.has(list.id));
        return [...prev, ...uniqueNewLists];
      });
      setStaticListsNextPageUrl(nextUrl);
      
    } catch (error) {
      console.error('Error loading more static contact lists:', error);
      
    } finally {
      setStaticListsLoadingMore(false);
    }
  }, []);

  
  const loadAllStaticLists = useCallback(async () => {
    setStaticListsLoading(true);
    setStaticListsError(null);
    setStaticContactLists([]);
    setStaticListsNextPageUrl(null);
    
    try {
      let allLists = [];
      let currentPage = 1;
      let hasMorePages = true;
      
      while (hasMorePages) {
        const response = await fetchStaticContactList({
          page: currentPage,
          perPageData: 50,
          actionType: 0
        });
        
        let newLists = [];
        let nextPageUrl = null;
        
        if (response?.WhatsAppBusinessTag?.data && Array.isArray(response.WhatsAppBusinessTag.data)) {
          newLists = response.WhatsAppBusinessTag.data;
          nextPageUrl = response.WhatsAppBusinessTag.next_page_url || null;
        } else if (response?.data && Array.isArray(response.data)) {
          newLists = response.data;
          nextPageUrl = response.next_page_url || null;
        } else if (response?.contactLists && Array.isArray(response.contactLists)) {
          newLists = response.contactLists;
          nextPageUrl = response.next_page_url || null;
        }
        
        // Add new lists t  o the collection, avoiding duplicates
        const existingIds = new Set(allLists.map(list => list.id));
        const uniqueNewLists = newLists.filter(list => list.id && !existingIds.has(list.id));
        allLists = [...allLists, ...uniqueNewLists];
          
          
        if (nextPageUrl && newLists.length > 0) {
          currentPage++;
        
          await new Promise(resolve => setTimeout(resolve, 100));
        } else {
          hasMorePages = false;
        }
        
        
        if (currentPage > 100) {
          console.warn('Reached maximum page limit (100) while loading static contact lists');
          hasMorePages = false;
        }
      }
      
      setStaticContactLists(allLists);
      console.log(`Loaded ${allLists.length} static contact lists from ${currentPage} pages`);
      
    } catch (error) {
      console.error('Error loading all static contact lists:', error);
      setStaticListsError('Failed to load all contact lists. Please try again.');
      setStaticContactLists([]);
    } finally {
      setStaticListsLoading(false);
    }
  }, []);

  // Load all static lists when static list radio button is selected
  useEffect(() => {
    if (targetAudience === 'static_list' && staticContactLists.length === 0 && !staticListsLoading) {
      loadAllStaticLists();
    }
  }, [targetAudience, staticContactLists.length, staticListsLoading, loadAllStaticLists]);

  // Debounce search input to improve performance
  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedSearchTerm(staticListInputValue);
    }, 300);

    return () => clearTimeout(timer);
  }, [staticListInputValue]);

  
  const handleTemplateChange = useCallback(async (value) => {
    setSelectedTemplateId(value);
    
    const selectedTemplate = allTemplateOptions.find(t => t.id === value || t.value === value);
    if (selectedTemplate) {
      setTemplateInputValue(selectedTemplate.label);
      
      if (selectedTemplate.templateId) {
        try {
          setLoadingTemplateDetails(true);
          setTemplateError(null);
          
          const response = await getSelectedTemplatebyName(selectedTemplate.name);
          
          if (response.data && response.data.length > 0) {
            const templateData = response.data[0];
            setSelectedTemplateData(templateData);
            
            
            
          } else {
            console.error('No template data found');
            setTemplateError('Template not found or invalid');
            setSelectedTemplateData(null);
          }
        } catch (error) {
          console.error('Error fetching template details:', error);
          setTemplateError(`Failed to load template: ${error.message}`);
          setSelectedTemplateData(null);
        } finally {
          setLoadingTemplateDetails(false);
        }
      }
    } else {
      setTemplateInputValue('');
      setSelectedTemplateData(null);
    }
    
    // Reset mappings when template changes
    setVariableMapping({});
    setMediaMapping({});
    setStaticValueEnabled({});
    setStaticValues({});
    setUrlButtonVariableMapping({});
    setUrlButtonStaticValueEnabled({});
    setUrlButtonStaticValues({});
    setVariableInputValues({});
    setUrlButtonVariableInputValues({});
  }, [allTemplateOptions]);

  const handleVariableMappingChange = useCallback((variableKey, value) => {
    setVariableMapping(prev => ({
      ...prev,
      [variableKey]: {
        ...prev[variableKey],
        value: value
      }
    }));
  }, []);

  const handleStaticValueToggle = useCallback((variableKey) => {
    setStaticValueEnabled(prev => {
      const newEnabled = { ...prev, [variableKey]: !prev[variableKey] };
      // Clear static value if disabling
      if (!newEnabled[variableKey]) {
        setStaticValues(prevStatic => {
          const newStatic = { ...prevStatic };
          delete newStatic[variableKey];
          return newStatic;
        });
        
        // Clear input value
        setVariableInputValues(prev => {
          const newInputs = { ...prev };
          delete newInputs[variableKey];
          return newInputs;
        });
        
        // Clear variable mapping
        setVariableMapping(prevMapping => ({
          ...prevMapping,
          [variableKey]: {
            ...prevMapping[variableKey],
            value: ''
          }
        }));
      }
      
      return newEnabled;
    });
  }, []);

  const handleStaticValueChange = useCallback((variableKey, value) => {
    setStaticValues(prev => ({
      ...prev,
      [variableKey]: value
    }));
  }, []);

  const handleUrlButtonVariableMappingChange = useCallback((variableKey, value) => {
    setUrlButtonVariableMapping(prev => ({
      ...prev,
      [variableKey]: {
        ...prev[variableKey],
        value: value
      }
    }));
  }, []);

  const handleUrlButtonStaticValueToggle = useCallback((variableKey) => {
    setUrlButtonStaticValueEnabled(prev => {
      const newEnabled = { ...prev, [variableKey]: !prev[variableKey] };
      
      // Clear static value if disabling
      if (!newEnabled[variableKey]) {
        setUrlButtonStaticValues(prevStatic => {
          const newStatic = { ...prevStatic };
          delete newStatic[variableKey];
          return newStatic;
        });
        
        // Clear variable mapping
        setUrlButtonVariableMapping(prevMapping => ({
          ...prevMapping,
          [variableKey]: {
            ...prevMapping[variableKey],
            value: ''
          }
        }));
        
        // Clear input value
        setUrlButtonVariableInputValues(prev => {
          const newInputs = { ...prev };
          delete newInputs[variableKey];
          return newInputs;
        });
      }
      
      return newEnabled;
    });
  }, []);

  const handleUrlButtonStaticValueChange = useCallback((variableKey, value) => {
    setUrlButtonStaticValues(prev => ({
      ...prev,
      [variableKey]: value
    }));
  }, []);

  const handleMediaMappingChange = useCallback(async (mediaKey, file) => {
    if (file) {
      setUploadingMedia(prev => ({ ...prev, [mediaKey]: true }));

      try {
        const reader = new FileReader();
        reader.onload = async () => {
          try {
            const base64String = reader.result.split(',')[1];

            let fileType = 'image';
            if (file.type.startsWith('video/')) {
              fileType = 'video';
            } else if (file.type.startsWith('audio/')) {
              fileType = 'audio';
            } else if (file.type.includes('pdf') || file.type.includes('document') || file.type.includes('text')) {
              fileType = 'document';
            }

            const filePayload = {
              fileName: file.name,
              fileSrc: `data:${file.type};base64,${base64String}`,
              fileType: fileType
            };

            const uploadResponse = await saveTemporaryFile(filePayload);

            if (uploadResponse.status && uploadResponse.filename) {
              const getGabsUrl = `https://app.getgabs.com/customers/mediafile/newfiles/uploads/${uploadResponse.filename}`;

              setMediaMapping(prev => ({
                ...prev,
                [mediaKey]: {
                  ...prev[mediaKey],
                  file: file,
                  fileUrl: getGabsUrl,
                  fileName: uploadResponse.filename,
                  uploaded: true,
                  uploadError: null
                }
              }));
            } else {
              throw new Error(uploadResponse.message || 'Failed to upload file');
            }
          } catch (error) {
            console.error('Error uploading file:', error);
            setMediaMapping(prev => ({
              ...prev,
              [mediaKey]: {
                ...prev[mediaKey],
                uploadError: error.message
              }
            }));
            setTemplateError(`Failed to upload file: ${error.message}`);
          } finally {
            setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
          }
        };

        reader.onerror = () => {
          setTemplateError('Failed to read file');
          setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
        };

        reader.readAsDataURL(file);
      } catch (error) {
        console.error('Error uploading file:', error);
        setTemplateError(`Failed to upload file: ${error.message}`);
        setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
      }
    } else {
      setMediaMapping(prev => ({
        ...prev,
        [mediaKey]: {
          ...prev[mediaKey],
          file: null,
          fileUrl: null,
          fileName: null,
          uploaded: false,
          uploadError: null
        }
      }));
    }
  }, []);

  const handleTargetAudienceChange = useCallback((value) => {
    setTargetAudience(value);
    if (value !== 'live_segment') {
      setSelectedLiveSegments([]);
      setLiveSegmentInputValue('');
      setLiveSegmentDropdownActive(false);
    }
    if (value !== 'static_list') {
      setSelectedStaticLists([]);
      setStaticListInputValue('');
      setStaticListDropdownActive(false);
    }
   
    if (value === 'static_list' && staticContactLists.length === 0 && !staticListsLoading) {
      loadAllStaticLists();
    }
  }, [staticContactLists.length, staticListsLoading, loadAllStaticLists]);

  const handleCustomAudienceChange = useCallback((field, value) => {
    setCustomAudienceFilters(prev => ({
      ...prev,
      [field]: value
    }));
  }, []);

  const handleDateChange = useCallback((date) => {
    setScheduledDate(date.start);
    setDatePickerActive(false);
  }, []);

  
  const handleLiveSegmentSelect = useCallback((value) => {
    console.log('Live segment selected:', value);
    if (value && !selectedLiveSegments.includes(value)) {
      setSelectedLiveSegments([...selectedLiveSegments, value]);
    }
    setLiveSegmentInputValue('');
    setLiveSegmentDropdownActive(false);
  }, [selectedLiveSegments]);

  const handleLiveSegmentRemove = useCallback((value) => {
    setSelectedLiveSegments(selectedLiveSegments.filter(segment => segment !== value));
  }, [selectedLiveSegments]);

  
  const handleStaticListSelect = useCallback((value) => {
    console.log('Static list selected:', value);
    if (value && !selectedStaticLists.includes(value)) {
      setSelectedStaticLists([...selectedStaticLists, value]);
    }
    setStaticListInputValue('');
    setStaticListDropdownActive(false);
  }, [selectedStaticLists]);

  const handleStaticListRemove = useCallback((value) => {
    setSelectedStaticLists(selectedStaticLists.filter(list => list !== value));
  }, [selectedStaticLists]);

  // Function to fetch contact list fields and auto-map variables
  const fetchContactListFieldsAndAutoMap = useCallback(async (tagIds) => {
    if (!tagIds || tagIds.length === 0 || !selectedTemplateData) {
      return { status: false, message: 'Missing required parameters' };
    }
    
    try {
      setLoadingContactListFields(true);
      setContactListFieldsError(null);
      
      // Get the response from hasContactlisthaveSelectedtemplateVariables
      const response = await hasContactlisthaveSelectedtemplateVariables({ tagIds });
      
      if (response.status && response.data && response.data.length > 0) {
        setContactListFields(response.data);
        
        // Extract template variables from the template
        const templateVars = [];
        if (selectedTemplateData.components) {
          selectedTemplateData.components.forEach((component) => {
            if (component.text) {
              const variables = component.text.match(/\{\{([^}]+)\}\}/g);
              if (variables) {
                variables.forEach((variable) => {
                  const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                  if (!templateVars.includes(varName)) {
                    templateVars.push(varName);
                  }
                });
              }
            }
          });
        }
        
        // Create a map of available variables from newjsonfields
        const availableContactFields = new Map();
        
        response.data.forEach(contactList => {
          if (contactList.newjsonfields) {
            try {
              const fields = JSON.parse(contactList.newjsonfields);
              Object.entries(fields).forEach(([key, value]) => {
                // Use the field value as key and the key as display name
                if (value && !availableContactFields.has(value)) {
                  availableContactFields.set(value, {
                    value: value.toLowerCase().replace(/\s+/g, '_'),
                    label: value,
                    fieldKey: key
                  });
                }
              });
            } catch (error) {
              console.error('Error parsing newjsonfields:', error);
            }
          }
        });
        
        // Convert map to array for the select component
        const availableVariablesList = Array.from(availableContactFields.values());
        
        // Auto-map variables based on name matching
        const newVariableMapping = { ...variableMapping };
        const newVariableInputValues = { ...variableInputValues };
        
        templateVars.forEach(varName => {
          const variableKey = `body_var_${varName}`;
          
          // Try to find a matching field in the contact list
          const matchingField = Array.from(availableContactFields.entries()).find(([fieldName]) => {
            const fieldNameLower = fieldName.toLowerCase();
            const varNameLower = varName.toLowerCase();
            return fieldNameLower.includes(varNameLower) || varNameLower.includes(fieldNameLower);
          });
          
          if (matchingField) {
            const [fieldName, fieldData] = matchingField;
            newVariableMapping[variableKey] = {
              ...newVariableMapping[variableKey],
              value: fieldData.value,
              fieldKey: fieldData.fieldKey
            };
            newVariableInputValues[variableKey] = fieldName;
          }
        });
        
        setVariableMapping(newVariableMapping);
        setVariableInputValues(newVariableInputValues);
        
        // Update available variables for the select component, keeping predefined variables at the top
        const predefinedVariables = [
          { label: 'name', value: '{{name}}' },
          { label: 'number', value: '{{number}}' }
        ];
        setAvailableVariables([...predefinedVariables, ...availableVariablesList]);
      } else {
        setContactListFields([]);
        setAvailableVariables([]);
      }
    } catch (error) {
      console.error('Error fetching contact list fields:', error);
      
      // Provide specific error messages for rate limiting
      if (error.message.includes('Rate limit')) {
        setContactListFieldsError('Rate limit exceeded. Please wait a moment and try again.');
      } else {
        setContactListFieldsError('Failed to load contact list fields');
      }
      setContactListFields([]);
    } finally {
      setLoadingContactListFields(false);
    }
    
    return { status: true };
  }, [selectedTemplateData, availableVariables, variableMapping, variableInputValues, setVariableMapping, setVariableInputValues, setAvailableVariables, setContactListFields, setContactListFieldsError]);

  // Track the last processed template ID to prevent duplicate processing
  const lastProcessedTemplateId = useRef(null);
  const isProcessing = useRef(false);

  // Auto-fetch contact list fields when static lists are selected and template is available
  useEffect(() => {
    // Skip if we don't have all required data or already processing
    if (selectedStaticLists.length === 0 || !selectedTemplateData || targetAudience !== 'static_list' || isProcessing.current) {
      return;
    }

    // Create a unique key for the current template and selected lists
    const currentTemplateKey = `${selectedTemplateData.id || ''}-${selectedStaticLists.sort().join('-')}`;
    
    // Skip if we've already processed this combination
    if (lastProcessedTemplateId.current === currentTemplateKey) {
      return;
    }

    console.log('Analyzing contact list fields for auto-mapping...');
    isProcessing.current = true;
    lastProcessedTemplateId.current = currentTemplateKey;
    
    // Use a small timeout to ensure the state updates are batched
    const timer = setTimeout(() => {
      fetchContactListFieldsAndAutoMap(selectedStaticLists)
        .finally(() => {
          isProcessing.current = false;
        });
    }, 100);

    // Cleanup function
    return () => {
      clearTimeout(timer);
      isProcessing.current = false;
    };
  }, [selectedStaticLists, selectedTemplateData, targetAudience, fetchContactListFieldsAndAutoMap]);

  const handleSaveBroadcast = useCallback(async () => {
    try {
      setSendingBroadcast(true);
      setBroadcastError(null);

      
      const campaignSchedule = scheduleType === 'immediate' 
        ? { scheduleType: "now" }
        : { 
            scheduleType: "schedule",
            scheduleDateTime: `${scheduledDate.getFullYear()}-${String(scheduledDate.getMonth() + 1).padStart(2, '0')}-${String(scheduledDate.getDate()).padStart(2, '0')} ${scheduledTime}:00`
          };

      
      const templateComponents = [];
      
      if (selectedTemplateData?.components) {
        selectedTemplateData.components.forEach((component) => {
          if (component.type === 'BODY') {
            const bodyParameters = [];
            
            if (component.text) {
              const variables = component.text.match(/\{\{([^}]+)\}\}/g);
              if (variables) {
                variables.forEach((variable) => {
                  const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                  const variableKey = `body_var_${varName}`;
                  
                  let parameterValue = varName; 
                  
                  if (staticValueEnabled[variableKey] && staticValues[variableKey]) {
                    parameterValue = staticValues[variableKey];
                  } else if (variableMapping[variableKey]?.value) {
                    const selectedVariableKey = variableMapping[variableKey].value;
                    const availableVar = availableVariables.find(av => av.value === selectedVariableKey);
                    parameterValue = availableVar ? availableVar.label : selectedVariableKey;
                  }
                  
                  bodyParameters.push({
                    type: "text",
                    parameter_name: varName,
                    text: `{${parameterValue}}`
                  });
                });
              }
            }
            
            if (bodyParameters.length > 0) {
              templateComponents.push({
                type: "BODY",
                parameters: bodyParameters
              });
            }
          } else if (component.type === 'HEADER' && component.text) {
            // Handle HEADER text variables
            const headerParameters = [];
            
            const variables = component.text.match(/\{\{([^}]+)\}\}/g);
            if (variables) {
              variables.forEach((variable) => {
                const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                const variableKey = `header_var_${varName}`;
                
                let parameterValue = varName; 
                
                if (staticValueEnabled[variableKey] && staticValues[variableKey]) {
                  parameterValue = staticValues[variableKey];
                } else if (variableMapping[variableKey]?.value) {
                  
                  const selectedVariableKey = variableMapping[variableKey].value;
                  const availableVar = availableVariables.find(av => av.value === selectedVariableKey);
                  parameterValue = availableVar ? availableVar.label : selectedVariableKey;
                }
                
                headerParameters.push({
                  type: "text",
                  parameter_name: varName,
                  text: `{${parameterValue}}`
                });
              });
            }
            
            if (headerParameters.length > 0) {
              templateComponents.push({
                type: "HEADER",
                parameters: headerParameters
              });
            }
          } else if (component.type === 'HEADER' && (component.format === 'IMAGE' || component.format === 'VIDEO' || component.format === 'DOCUMENT')) {
            
            const headerKey = `header_${selectedTemplateData.components.indexOf(component)}`;
            const mediaData = mediaMapping[headerKey];
            
            if (mediaData?.fileUrl || mediaData?.url) {
              templateComponents.push({
                type: "HEADER",
                parameters: [{
                  type: component.format.toLowerCase(),
                  [component.format.toLowerCase()]: {
                    link: mediaData.fileUrl || mediaData.url
                  }
                }]
              });
            }
          } else if (component.type === 'BUTTONS' && component.buttons) {
            const componentIndex = selectedTemplateData.components.indexOf(component);
            
            component.buttons.forEach((button, buttonIndex) => {
              if (button.type === 'CATALOG') {
                templateComponents.push({
                  type: "button",
                  sub_type: "CATALOG",
                  index: buttonIndex,
                  parameters: [{
                    type: "action",
                    action: {
                      thumbnail_product_retailer_id: "iu2y8d292a" // Default catalog ID
                    }
                  }]
                });
              } else if (button.type === 'URL' && button.url) {
                // Handle URL buttons with variables
                const variables = button.url.match(/\{\{([^}]+)\}\}/g);
                if (variables) {
                  const urlParameters = [];
                  
                  variables.forEach((variable) => {
                    const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                    const variableKey = `url_button_${componentIndex}_${buttonIndex}_var_${varName}`;
                    
                    let parameterValue = varName; 
                    
                    if (urlButtonStaticValueEnabled[variableKey] && urlButtonStaticValues[variableKey]) {
                      parameterValue = urlButtonStaticValues[variableKey];
                    } else if (urlButtonVariableMapping[variableKey]?.value) {
                      // Find the actual display value for the selected variable
                      const selectedVariableKey = urlButtonVariableMapping[variableKey].value;
                      const availableVar = availableVariables.find(av => av.value === selectedVariableKey);
                      parameterValue = availableVar ? availableVar.label : selectedVariableKey;
                    }
                    
                    urlParameters.push({
                      type: "text",
                      text: urlButtonStaticValueEnabled[variableKey] && urlButtonStaticValues[variableKey] ? parameterValue : `{${parameterValue}}`
                    });
                  });
                  
                  if (urlParameters.length > 0) {
                    templateComponents.push({
                      type: "button",
                      sub_type: "URL",
                      index: buttonIndex,
                      parameters: urlParameters
                    });
                  }
                }
              }
            });
          } else if (component.type === 'FOOTER' && component.text) {
            // Handle FOOTER text variables
            const footerParameters = [];
            
            const variables = component.text.match(/\{\{([^}]+)\}\}/g);
            if (variables) {
              variables.forEach((variable) => {
                const varName = variable.match(/\{\{([^}]+)\}\}/)[1];
                const variableKey = `footer_var_${varName}`;
                
                let parameterValue = varName; 
                
                if (staticValueEnabled[variableKey] && staticValues[variableKey]) {
                  parameterValue = staticValues[variableKey];
                } else if (variableMapping[variableKey]?.value) {
                  // Find the actual display value for the selected variable
                  const selectedVariableKey = variableMapping[variableKey].value;
                  const availableVar = availableVariables.find(av => av.value === selectedVariableKey);
                  parameterValue = availableVar ? availableVar.label : selectedVariableKey;
                }
                
                footerParameters.push({
                  type: "text",
                  parameter_name: varName,
                  text: parameterValue
                });
              });
            }
            
            if (footerParameters.length > 0) {
              templateComponents.push({
                type: "FOOTER",
                parameters: footerParameters
              });
            }
          }
        });
      }

      
      const templateJson = {
        to: "receiver_number",
        type: "template",
        template: {
          name: selectedTemplateData?.name || 'template',
          language: {
            code: selectedTemplateData?.language || 'en_US'
          },
          components: templateComponents
        },
        recipient_type: "individual",
        messaging_product: "whatsapp"
      };

      
      const currentChoosedTemplateJson = {
        choosedTemplate: {
          name: selectedTemplateData?.name || 'template',
          parameter_format: "POSITIONAL",
          components: selectedTemplateData?.components || [],
          language: selectedTemplateData?.language || 'en_US',
          status: selectedTemplateData?.status || 'APPROVED',
          category: selectedTemplateData?.category || 'MARKETING',
          id: selectedTemplateData?.id || selectedTemplateData?.template_id || ''
        }
      };

      
      const contactTagIds = [];
      if (targetAudience === 'static_list' && selectedStaticLists.length > 0) {
        contactTagIds.push(...selectedStaticLists);
      }
      
      

      
      const broadcastPayload = {
        CampaignSchedule: campaignSchedule,
        campaign_name: broadcastName ,
        templateJson: templateJson,
        currentChoosedTemplateJson: currentChoosedTemplateJson,
        contactTagIds: contactTagIds
      };

      console.log('Sending broadcast payload:', broadcastPayload);

    
      const response = await sendBroadcastCampaign(broadcastPayload);
      
      console.log('Broadcast sent successfully:', response);
      
      
      navigate('/app/broadcast');
      
    } catch (error) {
      console.error('Error sending broadcast:', error);
      setBroadcastError(`Failed to send broadcast: ${error.message}`);
    } finally {
      setSendingBroadcast(false);
    }
  }, [
    broadcastName,
    selectedTemplateId,
    selectedTemplateData,
    variableMapping,
    mediaMapping,
    staticValueEnabled,
    staticValues,
    urlButtonVariableMapping,
    urlButtonStaticValueEnabled,
    urlButtonStaticValues,
    targetAudience,
    selectedLiveSegments,
    selectedStaticLists,
    scheduleType,
    scheduledDate,
    scheduledTime,
    availableVariables,
    navigate
  ]);

  
  const filteredLiveSegmentOptions = useMemo(() => {
    return liveSegmentOptions.filter(option => 
      !selectedLiveSegments.includes(option.value) &&
      option.label.toLowerCase().includes(liveSegmentInputValue.toLowerCase())
    );
  }, [liveSegmentOptions, selectedLiveSegments, liveSegmentInputValue]);

  const filteredStaticListOptions = useMemo(() => {
    const searchTerm = debouncedSearchTerm.trim().toLowerCase();
    
    if (!searchTerm) {
      
      return staticListOptions.filter(option => 
        option.value && !selectedStaticLists.includes(option.value)
      );
    }
    
    
    return staticListOptions.filter(option => 
      option.value && 
      !selectedStaticLists.includes(option.value) &&
      (option.label.toLowerCase().includes(searchTerm) ||
       option.id?.toString().includes(searchTerm))
    );
  }, [staticListOptions, selectedStaticLists, debouncedSearchTerm]);

  const isFormValid = broadcastName.trim() && selectedTemplateId && selectedTemplateData &&
    Object.keys(variableMapping).every(variableKey => {
      
      return (staticValueEnabled[variableKey] && staticValues[variableKey]?.trim()) || 
             (!staticValueEnabled[variableKey] && variableMapping[variableKey]?.value);
    }) &&
    Object.keys(urlButtonVariableMapping).every(variableKey => {
      
      return (urlButtonStaticValueEnabled[variableKey] && urlButtonStaticValues[variableKey]?.trim()) || 
             (!urlButtonStaticValueEnabled[variableKey] && urlButtonVariableMapping[variableKey]?.value);
    }) &&
    Object.keys(mediaMapping).every(mediaKey => {
      const media = mediaMapping[mediaKey];
      
      return (media?.uploaded && media?.fileUrl) || media?.url;
    }) &&
    !Object.values(uploadingMedia).some(isUploading => isUploading) && 
    (targetAudience !== 'live_segment' || selectedLiveSegments.length > 0) &&
    (targetAudience !== 'static_list' || selectedStaticLists.length > 0) &&
    !sendingBroadcast; 

  return (
    <Page
      title="Create Broadcast Campaign"
      subtitle="Set up a new WhatsApp broadcast campaign"
      fullWidth
      backAction={{
        content: 'Back to Broadcasts',
        onAction: () => navigate('/app/broadcast')
      }}
      primaryAction={{
        content: scheduleType === 'immediate' ? 'Send Now' : 'Schedule Broadcast',
        onAction: handleSaveBroadcast,
        disabled: !isFormValid || sendingBroadcast,
        loading: sendingBroadcast,
        tone: scheduleType === 'immediate' ? 'success' : 'primary'
      }}
      secondaryActions={[
        {
          content: 'Cancel',
          onAction: () => {
           
            navigate('/app/broadcast');
          }
        }
      ]}
    >
      <Layout>
        {/* Left Column - Form */}
        <Layout.Section variant="twoThirds">
          <BlockStack gap="500">
            {/* Broadcast Details */}
            <LegacyCard title="Broadcast Details" sectioned>
              <BlockStack gap="400">
                <TextField
                  label="Broadcast Name"
                  value={broadcastName}
                  onChange={setBroadcastName}
                  placeholder="Enter broadcast campaign name"
                  helpText="Choose a descriptive name for your campaign"
                />
                
                {/* Template Error Banner */}
                {templateError && (
                  <Banner status="critical" onDismiss={() => setTemplateError(null)}>
                    <Text variant="bodyMd" as="p">
                      {templateError}
                    </Text>
                  </Banner>
                )}

                {/* Broadcast Error Banner */}
                {broadcastError && (
                  <Banner status="critical" onDismiss={() => setBroadcastError(null)}>
                    <Text variant="bodyMd" as="p">
                      {broadcastError}
                    </Text>
                  </Banner>
                )}
                
                <Autocomplete
                  label="Search and select template"
                  options={filteredTemplateOptions.map(template => ({
                    value: template.id || template.value,
                    label: template.label
                  }))}
                  selected={selectedTemplateId ? [selectedTemplateId] : []}
                  onSelect={(selected) => {
                    const selectedValue = selected[0];
                    if (selectedValue) {
                      handleTemplateChange(selectedValue);
                    }
                  }}
                  textField={
                    <Autocomplete.TextField
                      onChange={(value) => setTemplateInputValue(value)}
                      label="Search and select template"
                      value={templateInputValue}
                      placeholder="Type to search templates..."
                      autoComplete="off"
                    />
                  }
                  loading={loadingTemplates || loadingTemplateDetails}
                  emptyState={loadingTemplates ? "Loading templates..." : "No API templates found"}
                  preferredPosition="below"
                  willLoadMoreResults={false}
                />

                {selectedTemplateData && (
                  <Banner status="info">
                    <Text variant="bodyMd" as="p">
                      Template selected: <strong>{allTemplateOptions.find(t => t.id === selectedTemplateId || t.value === selectedTemplateId)?.label}</strong>
                      {loadingTemplateDetails && <span> (Loading details...)</span>}
                    </Text>
                  </Banner>
                )}
              </BlockStack>
            </LegacyCard>

            {/* Target Audience */}
            <LegacyCard title="Target Audience" sectioned>
              <BlockStack gap="400">
                <ChoiceList
                  title="Select your target audience"
                  choices={audienceOptions}
                  selected={[targetAudience]}
                  onChange={(value) => handleTargetAudienceChange(value[0])}
                />

                {targetAudience === 'live_segment' && (
                  <BlockStack gap="400">
                    <Divider />
                    <Text variant="headingMd" as="h3">Select Live Segments</Text>
                    
                 
                    {selectedLiveSegments.length > 0 && (
                      <BlockStack gap="200">
                        <Text variant="bodyMd" as="p">Selected segments:</Text>
                        <InlineStack gap="200" wrap>
                          {selectedLiveSegments.map((segmentValue) => {
                            const segment = liveSegmentOptions.find(option => option.value === segmentValue);
                            return (
                              <Tag
                                key={segmentValue}
                                onRemove={() => handleLiveSegmentRemove(segmentValue)}
                              >
                                {segment?.label} ({segment?.contacts.toLocaleString()} contacts)
                              </Tag>
                            );
                          })}
                        </InlineStack>
                      </BlockStack>
                    )}
                    
                    <text>
                       Live segments currently not available.
                    </text>
                    
                  </BlockStack>
                )}

                {targetAudience === 'static_list' && (
                  <BlockStack gap="400">
                    <Divider />
                    <InlineStack align="space-between">
                      <Text variant="headingMd" as="h3">Select Static Lists</Text>
                      <Button
                        variant="plain"
                        onClick={loadAllStaticLists}
                        loading={staticListsLoading}
                        size="slim"
                      >
                        Refresh 
                      </Button>
                    </InlineStack>
                    
                    {staticListsError && (
                      <Banner status="critical">
                        <Text variant="bodyMd" as="p">
                          {staticListsError}
                        </Text>
                      </Banner>
                    )}
                    
                    {staticListsLoading && (
                      <Banner status="info">
                        <Text variant="bodyMd" as="p">
                          Loading all contact lists from all pages...
                        </Text>
                      </Banner>
                    )}
                    
                    {/* Contact List Fields Error Banner */}
                    {contactListFieldsError && (
                      <Banner status="warning" onDismiss={() => setContactListFieldsError(null)}>
                        <Text variant="bodyMd" as="p">
                          {contactListFieldsError}
                        </Text>
                      </Banner>
                    )}
                    
                    {/* Contact List Fields Loading Banner */}
                    {loadingContactListFields && (
                      <Banner status="info">
                        <Text variant="bodyMd" as="p">
                          Analyzing contact list fields for auto-mapping...
                        </Text>
                      </Banner>
                    )}
                    
                    {selectedStaticLists.length > 0 && (
                      <BlockStack gap="200">
                        <Text variant="bodyMd" as="p">Selected lists:</Text>
                        <InlineStack gap="200" wrap>
                          {selectedStaticLists.map((listValue) => {
                            const list = staticListOptions.find(option => option.value === listValue);
                            return (
                              <Tag
                                key={listValue}
                                onRemove={() => handleStaticListRemove(listValue)}
                              >
                                {list?.label} ({list?.contacts.toLocaleString()} contacts)
                              </Tag>
                            );
                          })}
                        </InlineStack>
                      </BlockStack>
                    )}
                    
                    <BlockStack gap="300">
                      <InlineStack gap="200">
                        <div style={{ flex: 1 }}>
                          <TextField
                            label="Search static lists"
                            value={staticListInputValue}
                            onChange={(value) => {
                              setStaticListInputValue(value);
                              // Only show dropdown if there's content or user is typing
                              if (value.length > 0 || staticListOptions.length > 0) {
                                setStaticListDropdownActive(true);
                              }
                            }}
                            onFocus={() => {
                              setStaticListDropdownActive(true);
                            }}
                            onBlur={() => {
                              setTimeout(() => setStaticListDropdownActive(false), 200);
                            }}
                            placeholder={staticListsLoading ? "Loading all contact lists..." : "Click here or type to search lists..."}
                            disabled={staticListsLoading}
                            autoComplete="off"
                          />
                        </div>
                        {staticListInputValue && (
                          <Button
                            variant="plain"
                            onClick={() => {
                              setStaticListInputValue('');
                              setStaticListDropdownActive(false);
                            }}
                            size="slim"
                          >
                            Clear
                          </Button>
                        )}
                      </InlineStack>
                      
                      {staticListDropdownActive && !staticListsLoading && (staticListOptions.length > 0 || staticListInputValue) && (
                        <div style={{ 
                          border: '1px solid var(--p-color-border)', 
                          borderRadius: '6px',
                          maxHeight: '200px',
                          overflowY: 'auto',
                          backgroundColor: 'var(--p-color-bg-surface)'
                        }}>
                          {filteredStaticListOptions.length > 0 && !debouncedSearchTerm && (
                            <div style={{ 
                              padding: '8px 16px', 
                              borderBottom: '1px solid var(--p-color-border-subdued)',
                              backgroundColor: 'var(--p-color-bg-surface-secondary)'
                            }}>
                              <Text variant="bodySm" as="p" tone="subdued">
                                {staticListOptions.length} contact list{staticListOptions.length !== 1 ? 's' : ''} available
                              </Text>
                            </div>
                          )}
                          {filteredStaticListOptions.length > 0 && debouncedSearchTerm && (
                            <div style={{ 
                              padding: '8px 16px', 
                              borderBottom: '1px solid var(--p-color-border-subdued)',
                              backgroundColor: 'var(--p-color-bg-surface-secondary)'
                            }}>
                              <Text variant="bodySm" as="p" tone="subdued">
                                {filteredStaticListOptions.length} result{filteredStaticListOptions.length !== 1 ? 's' : ''} for "{debouncedSearchTerm}"
                              </Text>
                            </div>
                          )}
                          {filteredStaticListOptions.length > 0 ? (
                            filteredStaticListOptions.map((option) => (
                              <div
                                key={option.value}
                                onClick={() => handleStaticListSelect(option.value)}
                                style={{
                                  padding: '12px 16px',
                                  cursor: 'pointer',
                                  borderBottom: '1px solid var(--p-color-border-subdued)',
                                  ':hover': { backgroundColor: 'var(--p-color-bg-surface-hover)' }
                                }}
                                onMouseEnter={(e) => e.target.style.backgroundColor = 'var(--p-color-bg-surface-hover)'}
                                onMouseLeave={(e) => e.target.style.backgroundColor = 'transparent'}
                              >
                                <BlockStack gap="100">
                                  <Text variant="bodyMd" as="p">
                                    {option.label}
                                  </Text>
                                  <Text variant="bodySm" as="p" tone="subdued">
                                    {option.contacts.toLocaleString()} contacts
                                  </Text>
                                </BlockStack>
                              </div>
                            ))
                          ) : (
                            <div style={{ padding: '16px', textAlign: 'center' }}>
                              <Text variant="bodyMd" as="p" tone="subdued">
                                {staticContactLists.length === 0 ? 'No contact lists available' : `No lists found matching "${debouncedSearchTerm || staticListInputValue}"`}
                              </Text>
                            </div>
                          )}
                        </div>
                      )}
                      
                      {staticListDropdownActive && staticListsLoading && (
                        <div style={{ 
                          padding: '16px', 
                          textAlign: 'center',
                          border: '1px solid var(--p-color-border)', 
                          borderRadius: '6px'
                        }}>
                          <Text variant="bodyMd" as="p" tone="subdued">
                            Loading contact lists...
                          </Text>
                        </div>
                      )}
                      
                      {staticContactLists.length === 0 && !staticListsLoading && (
                        <Banner status="warning">
                          <Text variant="bodyMd" as="p">
                            No static contact lists found. Create contact lists first to use this option.
                          </Text>
                        </Banner>
                      )}
                    </BlockStack>
                    
                    <Text variant="bodyMd" as="p" tone="subdued">
                      Static lists are fixed collections of contacts that were manually created or imported.
                    </Text>
                  </BlockStack>
                )}

                <InlineStack align="space-between">
                  <Text variant="bodyMd" as="p">
                    <strong>Estimated Reach:</strong>
                  </Text>
                  <Badge tone="info">
                    {estimatedReach.toLocaleString()} customers
                  </Badge>
                </InlineStack>
              </BlockStack>
            </LegacyCard>

            {/* Variable Mapping */}
            {Object.keys(templateVariables).length > 0 && (
              <LegacyCard title="Map Template Variables" sectioned>
                <BlockStack gap="400">
                  <Text variant="bodyMd" as="p" tone="subdued">
                    Map the variables in your template to customer data fields:
                  </Text>
                  
                  {/* Auto-mapping status banner */}
                  {targetAudience === 'static_list' && selectedStaticLists.length > 0 && (
                    <Banner status="info" onDismiss={() => {}}>
                      <Text variant="bodyMd" as="p">
                        {loadingContactListFields 
                          ? "Analyzing contact list fields for auto-mapping..." 
                          : contactListFields.length > 0
                            ? `Auto-mapped variables from ${contactListFields.length} contact list(s). Review mappings below.`
                            : "No field data found for selected contact lists. Please map variables manually."
                        }
                      </Text>
                    </Banner>
                  )}
                  
                  {Object.entries(templateVariables).map(([variableKey, variableData]) => {
                    // Check if this variable was auto-mapped
                    const isAutoMapped = variableMapping[variableKey]?.value && 
                                        contactListFields.some(list => {
                                          if (!list.newjsonfields) return false;
                                          try {
                                            const fields = JSON.parse(list.newjsonfields);
                                            const fieldValues = Object.values(fields);
                                            const varName = variableKey.replace(/^(body_var_|header_var_|footer_var_)/, '');
                                            return fieldValues.some(field => 
                                              field.toLowerCase().includes(varName.toLowerCase()) || 
                                              varName.toLowerCase().includes(field.toLowerCase())
                                            );
                                          } catch {
                                            return false;
                                          }
                                        });
                    
                    return (
                      <BlockStack key={variableKey} gap="200">
                        <InlineStack gap="300" align="space-between" blockAlign="start">
                          <div style={{ flex: 1 }}>
                            <InlineStack gap="200" blockAlign="center">
                              <Text variant="bodyMd" as="p" fontWeight="medium">
                                Map {variableData.placeholder} ({variableData.componentType})
                              </Text>
                              {isAutoMapped && (
                                <Badge tone="success">Auto-mapped</Badge>
                              )}
                            </InlineStack>
                          </div>
                          <Button
                            variant={staticValueEnabled[variableKey] ? "plain" : "plain"} 
                            size="slim"
                            onClick={() => handleStaticValueToggle(variableKey)}
                          >
                            {staticValueEnabled[variableKey] ? "Use Field Mapping" : "Add Static Value"}
                          </Button>
                        </InlineStack>
                        
                        {staticValueEnabled[variableKey] ? (
                          <TextField
                            label={`Static value for ${variableData.placeholder}`}
                            value={staticValues[variableKey] || ''}
                            onChange={(value) => handleStaticValueChange(variableKey, value)}
                            placeholder={`Enter static value for ${variableData.placeholder}`}
                            helpText="This value will be used for all recipients"
                          />
                        ) : (
                          <Autocomplete
                            options={availableVariables
                              .filter(av => {
                                const searchValue = variableInputValues[variableKey] || '';
                                return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                              })
                              .map(av => ({
                                value: av.value,
                                label: av.label
                              }))}
                            selected={variableMapping[variableKey]?.value ? [variableMapping[variableKey].value] : []}
                            onSelect={(selected) => {
                              const value = selected.length > 0 ? selected[0] : '';
                              handleVariableMappingChange(variableKey, value);
                              if (value) {
                                const selectedVar = availableVariables.find(av => av.value === value);
                                setVariableInputValues(prev => ({
                                  ...prev,
                                  [variableKey]: selectedVar?.label || ''
                                }));
                              } else {
                                // Clear input when no selection
                                setVariableInputValues(prev => ({
                                  ...prev,
                                  [variableKey]: ''
                                }));
                              }
                            }}
                            textField={
                              <Autocomplete.TextField
                                onChange={(value) => {
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variableKey]: value
                                  }));
                                }}
                                label={`Select data field for ${variableData.placeholder}`}
                                value={variableInputValues[variableKey] || ''}
                                placeholder={isAutoMapped 
                                  ? `Auto-mapped field: ${variableInputValues[variableKey]} (click to change)`
                                  : `Search and select data field for ${variableData.placeholder}...`
                                }
                                autoComplete="off"
                                helpText={isAutoMapped 
                                  ? "This field was automatically matched from your contact list. You can change it if needed."
                                  : "Select the data field that contains the values for this variable."
                                }
                              />
                            }
                            loading={loadingVariables}
                            emptyState={loadingVariables ? "Loading variables..." : "No variables found"}
                            preferredPosition="below"
                          />
                        )}
                      </BlockStack>
                    );
                  })}
                </BlockStack>
              </LegacyCard>
            )}

            {/* URL Button Variable Mapping */}
            {Object.keys(urlButtonVariables).length > 0 && (
              <LegacyCard title="Map URL Button Variables" sectioned>
                <BlockStack gap="400">
                  <Text variant="bodyMd" as="p" tone="subdued">
                    Map the variables in your URL button links to customer data fields:
                  </Text>
                  
                  {Object.entries(urlButtonVariables).map(([variableKey, variableData]) => (
                    <BlockStack key={variableKey} gap="200">
                      <InlineStack gap="300" align="space-between" blockAlign="start">
                        <div style={{ flex: 1 }}>
                          <Text variant="bodyMd" as="p" fontWeight="medium">
                            Map {variableData.placeholder} in "{variableData.buttonText}" button
                          </Text>
                          <Text variant="bodySm" as="p" tone="subdued">
                            URL: {variableData.buttonUrl}
                          </Text>
                        </div>
                        <Button
                          variant={urlButtonStaticValueEnabled[variableKey] ? "plain" : "plain"} 
                          size="slim"
                          onClick={() => handleUrlButtonStaticValueToggle(variableKey)}
                        >
                          {urlButtonStaticValueEnabled[variableKey] ? "Use Field Mapping" : "Add Static Value"}
                        </Button>
                      </InlineStack>
                      
                      {urlButtonStaticValueEnabled[variableKey] ? (
                        <TextField
                          label={`Static value for ${variableData.placeholder}`}
                          value={urlButtonStaticValues[variableKey] || ''}
                          onChange={(value) => handleUrlButtonStaticValueChange(variableKey, value)}
                          placeholder={`Enter static value for ${variableData.placeholder}`}
                          helpText="This value will be used for all recipients"
                        />
                      ) : (
                        <Autocomplete
                          options={availableVariables
                            .filter(av => {
                              const searchValue = urlButtonVariableInputValues[variableKey] || '';
                              return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                            })
                            .map(av => ({
                              value: av.value,
                              label: av.label
                            }))}
                          selected={urlButtonVariableMapping[variableKey]?.value ? [urlButtonVariableMapping[variableKey].value] : []}
                          onSelect={(selected) => {
                            const value = selected.length > 0 ? selected[0] : '';
                            handleUrlButtonVariableMappingChange(variableKey, value);
                            if (value) {
                              const selectedVar = availableVariables.find(av => av.value === value);
                              setUrlButtonVariableInputValues(prev => ({
                                ...prev,
                                [variableKey]: selectedVar?.label || ''
                              }));
                            } else {
                              // Clear input when no selection
                              setUrlButtonVariableInputValues(prev => ({
                                ...prev,
                                [variableKey]: ''
                              }));
                            }
                          }}
                          textField={
                            <Autocomplete.TextField
                              onChange={(value) => {
                                setUrlButtonVariableInputValues(prev => ({
                                  ...prev,
                                  [variableKey]: value
                                }));
                              }}
                              label={`Select data field for ${variableData.placeholder}`}
                              value={urlButtonVariableInputValues[variableKey] || ''}
                              placeholder={`Search and select data field for ${variableData.placeholder}...`}
                              autoComplete="off"
                            />
                          }
                          loading={loadingVariables}
                          emptyState={loadingVariables ? "Loading variables..." : "No variables found"}
                          preferredPosition="below"
                        />
                      )}
                    </BlockStack>
                  ))}
                </BlockStack>
              </LegacyCard>
            )}

            {/* Media Mapping */}
            {Object.keys(templateMedia).length > 0 && (
              <LegacyCard title="Map Template Media Files" sectioned>
                <BlockStack gap="400">
                  <Text variant="bodyMd" as="p" tone="subdued">
                    Upload new media files to replace the template's default media:
                  </Text>

                  {Object.entries(templateMedia).map(([mediaKey, mediaData]) => {
                    const uploadedFile = mediaMapping[mediaKey]?.file;
                    const fileUrl = mediaMapping[mediaKey]?.fileUrl || mediaData.url;
                    const isUploading = uploadingMedia[mediaKey];
                    const uploadError = mediaMapping[mediaKey]?.uploadError;
                    const isUploaded = mediaMapping[mediaKey]?.uploaded;
                    
                    return (
                      <BlockStack key={mediaKey} gap="200">
                        {/* File row */}
                        <InlineStack gap="200" align="space-between" blockAlign="center">
                          <InlineStack gap="200" blockAlign="center">
                            <Thumbnail
                              source={
                                uploadedFile
                                  ? fileUrl
                                  : mediaData.url ||
                                    (mediaData.type === "document" ? "📄" : "📁")
                              }
                              alt={uploadedFile?.name || `Header ${mediaData.type}`}
                              size="small"
                            />
                            <BlockStack gap="050">
                              <InlineStack gap="200" blockAlign="center">
                                <Text variant="bodyMd" as="p">
                                  {uploadedFile ? uploadedFile.name : `Header ${mediaData.type}`}
                                </Text>
                                {isUploading && (
                                  <Badge tone="info">Uploading...</Badge>
                                )}
                                {isUploaded && !isUploading && (
                                  <Badge tone="success">Uploaded</Badge>
                                )}
                              </InlineStack>
                              <Text variant="bodySm" as="p" tone="subdued">
                                {uploadedFile
                                  ? `${(uploadedFile.size / 1024 / 1024).toFixed(2)} MB`
                                  : `Original ${mediaData.type}`}
                              </Text>
                              {uploadError && (
                                <Text variant="bodySm" as="p" tone="critical">
                                  Error: {uploadError}
                                </Text>
                              )}
                            </BlockStack>
                          </InlineStack>

                          {/* Remove / Replace actions */}
                          {uploadedFile ? (
                            <InlineStack gap="100">
                              <Button
                                variant="plain"
                                tone="critical"
                                size="slim"
                                onClick={() => handleMediaMappingChange(mediaKey, null)}
                                disabled={isUploading}
                              >
                                Remove
                              </Button>
                              <Button
                                variant="plain"
                                size="slim"
                                onClick={() => document.getElementById(`upload-${mediaKey}`).click()}
                                disabled={isUploading}
                              >
                                Replace
                              </Button>
                            </InlineStack>
                          ) : (
                            <Button
                              variant="plain"
                              size="slim"
                              onClick={() => document.getElementById(`upload-${mediaKey}`).click()}
                              disabled={isUploading}
                              loading={isUploading}
                            >
                              Upload {mediaData.type}
                            </Button>
                          )}
                        </InlineStack>

                        {/* Hidden file input */}
                        <input
                          id={`upload-${mediaKey}`}
                          type="file"
                          accept={
                            mediaData.type === "image"
                              ? "image/*"
                              : mediaData.type === "video"
                              ? "video/*"
                              : "application/pdf,.doc,.docx,.txt"
                          }
                          style={{ display: "none" }}
                          onChange={(e) => {
                            if (e.target.files?.length > 0) {
                              handleMediaMappingChange(mediaKey, e.target.files[0]);
                            }
                          }}
                        />
                      </BlockStack>
                    );
                  })}
                </BlockStack>
              </LegacyCard>
            )}

            
            <LegacyCard title="Schedule Broadcast" sectioned>
              <BlockStack gap="400">
                <InlineStack gap="400">
                  <RadioButton
                    label="Send immediately"
                    checked={scheduleType === 'immediate'}
                    id="immediate"
                    name="scheduleType"
                    onChange={() => setScheduleType('immediate')}
                  />
                  <RadioButton
                    label="Schedule for later"
                    checked={scheduleType === 'scheduled'}
                    id="scheduled"
                    name="scheduleType"
                    onChange={() => setScheduleType('scheduled')}
                  />
                </InlineStack>

                {scheduleType === 'scheduled' && (
                  <BlockStack gap="400">
                    <Divider />
                    
                    <InlineStack gap="400">
                      <div style={{ flex: 1 }}>
                        <Popover
                          active={datePickerActive}
                          activator={
                            <Button
                              onClick={() => setDatePickerActive(!datePickerActive)}
                              disclosure
                            >
                              <InlineStack gap="200">
                                <Icon source={CalendarIcon} />
                                <Text as="span">
                                  {scheduledDate.toLocaleDateString()}
                                </Text>
                              </InlineStack>
                            </Button>
                          }
                          onClose={() => setDatePickerActive(false)}
                        >
                          <DatePicker
                            month={scheduledDate.getMonth()}
                            year={scheduledDate.getFullYear()}
                            selected={scheduledDate}
                            onMonthChange={(month, year) => {
                              setScheduledDate(new Date(year, month, scheduledDate.getDate()));
                            }}
                            onChange={handleDateChange}
                            disableDatesBefore={new Date()}
                          />
                        </Popover>
                      </div>

                      <div style={{ flex: 1 }}>
                        <Select
                          label="Time"
                          labelHidden
                          options={[
                            { label: '09:00 AM', value: '09:00' },
                            { label: '10:00 AM', value: '10:00' },
                            { label: '11:00 AM', value: '11:00' },
                            { label: '12:00 PM', value: '12:00' },
                            { label: '01:00 PM', value: '13:00' },
                            { label: '02:00 PM', value: '14:00' },
                            { label: '03:00 PM', value: '15:00' },
                            { label: '04:00 PM', value: '16:00' },
                            { label: '05:00 PM', value: '17:00' },
                            { label: '06:00 PM', value: '18:00' }
                          ]}
                          value={scheduledTime}
                          onChange={setScheduledTime}
                        />
                      </div>
                    </InlineStack>

                    <Text variant="bodyMd" as="p" tone="subdued">
                      <Icon source={ClockIcon} /> 
                      Broadcast will be sent on {scheduledDate.toLocaleDateString()} at {scheduledTime}
                    </Text>
                  </BlockStack>
                )}
              </BlockStack>
            </LegacyCard>

            {/* Bottom Action Buttons */}
            <LegacyCard sectioned>
              <InlineStack align="end" gap="300">
                <Button
                  onClick={() => {
                    console.log('Saving as draft...');
                    navigate('/app/broadcast');
                  }}
                  size="large"
                  disabled={sendingBroadcast}
                >
                  Save as Draft
                </Button>
                <Button
                  variant="primary"
                  onClick={handleSaveBroadcast}
                  disabled={!isFormValid || sendingBroadcast}
                  loading={sendingBroadcast}
                  tone={scheduleType === 'immediate' ? 'success' : 'primary'}
                  size="large"
                >
                  {scheduleType === 'immediate' ? 'Send Now' : 'Schedule Broadcast'}
                </Button>
              </InlineStack>
            </LegacyCard>
          </BlockStack>
        </Layout.Section>

        {/* Right Column - Preview */}
        <Layout.Section variant="oneThird">
          <LegacyCard title="Template Preview" sectioned>
            <BlockStack gap="400">
              {selectedTemplateData ? (
                <BlockStack gap="400">
                  <InlineStack align="space-between">
                    <Text variant="headingMd" as="h3">
                      WhatsApp Preview
                    </Text>
                    <Badge tone="info">
                      {allTemplateOptions.find(t => t.id === selectedTemplateId || t.value === selectedTemplateId)?.label}
                    </Badge>
                  </InlineStack>
                  
                  <div style={{ 
                    border: '1px solid var(--p-color-border)', 
                    borderRadius: '8px',
                    padding: '16px',
                    backgroundColor: 'var(--p-color-bg-surface-secondary)'
                  }}>
                    <WhatsAppPreview 
                      template={getTemplateForPreview(selectedTemplateData)}
                      variant="full"
                      showWhatsAppHeader={true}
                      variableMapping={convertVariableMappingForPreview()}
                      mediaMapping={convertMediaMappingForPreview()}
                    />
                  </div>
                </BlockStack>
              ) : (
                <div style={{ 
                  textAlign: 'center', 
                  padding: '60px 20px',
                  border: '2px dashed var(--p-color-border)',
                  borderRadius: '8px'
                }}>
                  <Text variant="headingMd" as="h3" tone="subdued">
                    Select a Template
                  </Text>
                  <Text variant="bodyMd" as="p" tone="subdued">
                    Choose a WhatsApp template from the form to see the preview here
                  </Text>
                </div>
              )}
            </BlockStack>
          </LegacyCard>
        </Layout.Section>
      </Layout>

      {/* Variables Modal */}
      <Modal
        open={showVariablesModal}
        onClose={() => {
          setShowVariablesModal(false);
          setVariableSearchQuery('');
        }}
        title="All Available Variables"
        primaryAction={{
          content: 'Close',
          onAction: () => {
            setShowVariablesModal(false);
            setVariableSearchQuery('');
          },
        }}
        size="large"
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" tone="subdued">
              Available variables for your broadcast templates:
            </Text>
            
            {/* Search Field */}
            <TextField
              label="Search Variables"
              value={variableSearchQuery}
              onChange={setVariableSearchQuery}
              placeholder="Type to search variables..."
              clearButton
              onClearButtonClick={() => setVariableSearchQuery('')}
              autoComplete="off"
            />
            
            <InlineStack gap="200" wrap>
              {availableVariables
                .filter(variable => 
                  variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
                  variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
                )
                .map((variable) => (
                  <Button
                    key={variable.value}
                    size="medium"
                    variant="tertiary"
                    onClick={() => {
                      // Copy variable to clipboard for easy use
                      navigator.clipboard.writeText(`{{${variable.value}}}`);
                      setShowVariablesModal(false);
                      setVariableSearchQuery('');
                    }}
                  >
                    {variable.label}
                  </Button>
                ))}
            </InlineStack>
            
            {availableVariables.filter(variable => 
              variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
              variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
            ).length === 0 && variableSearchQuery && (
              <Text variant="bodyMd" tone="subdued" alignment="center">
                No variables found matching "{variableSearchQuery}"
              </Text>
            )}
            
            <Text variant="bodySm" tone="subdued">
              Click on any variable to copy it to your clipboard. Variables use the format: {"{{variable_name}}"}
            </Text>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </Page>
  );
}

export default CreateBroadcastPage;
