import {
  Page,
  Card,
  FormLayout,
  TextField,
  Select,
  Button,
  ButtonGroup,
  Text,
  InlineStack,
  BlockStack,
  Banner,
  Checkbox,
  ChoiceList,
  Grid,
  Tooltip,
  LegacyCard,
  Combobox,
  Listbox,
  Tag,
  Modal
} from '@shopify/polaris';
import { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { extractVariablesFromWhatsAppMessage } from '../utils/whatsappMessagePreview';
import { messageTemplateOptions } from '../utils/messageTemplates';
import { getTemplateVariables } from '../utils/availableVariables';
import EmojiPicker from '../components/EmojiPicker';
import WhatsAppPreview from '../components/WhatsAppPreview';

function CreateAutoTriggerPage() {
  const navigate = useNavigate();
  
  // Form state
  const [triggerName, setTriggerName] = useState('');
  const [triggerType, setTriggerType] = useState('abandoned_cart');
  const [selectedTemplate, setSelectedTemplate] = useState('');
  const [messageTemplate, setMessageTemplate] = useState('');
  const [delayTime, setDelayTime] = useState('30');
  const [delayUnit, setDelayUnit] = useState('minutes');
  const [isActive, setIsActive] = useState(true);
  const [targetAudience, setTargetAudience] = useState(['abandoned_cart_customers']);
  const [keywords, setKeywords] = useState('');
  const [variableSelections, setVariableSelections] = useState({});
  const [messageType, setMessageType] = useState(''); // 'template' or 'service'
  const [customServiceMessage, setCustomServiceMessage] = useState('');
  const [selectedText, setSelectedText] = useState('');
  const [cursorPosition, setCursorPosition] = useState(0);
  const [filterByCollection, setFilterByCollection] = useState(false);
  const [selectedCollections, setSelectedCollections] = useState([]);
  const [collectionInputValue, setCollectionInputValue] = useState('');

  // API variables state
  const [availableVariables, setAvailableVariables] = useState([]);
  const [loadingVariables, setLoadingVariables] = useState(true);
  const [showVariablesModal, setShowVariablesModal] = useState(false);
  const [variableSearchQuery, setVariableSearchQuery] = useState('');

  // Handlers
  const handleTriggerNameChange = useCallback((value) => setTriggerName(value), []);
  const handleTriggerTypeChange = useCallback((value) => {
    setTriggerType(value);
    
    // Reset all form values when trigger type changes
    setSelectedTemplate('');
    setMessageTemplate('');
    setCustomServiceMessage('');
    setMessageType('');
    setVariableSelections({});
    setFilterByCollection(false);
    setSelectedCollections([]);
    setCollectionInputValue('');
    
    // Reset target audience when trigger type changes
    switch (value) {
      case 'abandoned_cart':
        setTargetAudience(['abandoned_cart_customers']);
        break;
      case 'order_confirmation':
        setTargetAudience(['order_confirmation_customers']);
        break;
      case 'keywords_auto_reply':
        setTargetAudience(['all_customers']);
        // Reset collection filtering for keywords auto reply
        setFilterByCollection(false);
        setSelectedCollections([]);
        break;
      default:
        setTargetAudience(['all_customers']);
        // Reset collection filtering for other trigger types
        setFilterByCollection(false);
        setSelectedCollections([]);
        break;
    }
  }, []);
  const handleSelectedTemplateChange = useCallback((value) => {
    setSelectedTemplate(value);
    if (value && value !== 'custom') {
      setMessageTemplate(value);
    } else if (value === 'custom') {
      setMessageTemplate('');
    }
  }, []);
  const handleMessageTemplateChange = useCallback((value) => setMessageTemplate(value), []);
  const handleDelayTimeChange = useCallback((value) => setDelayTime(value), []);
  const handleDelayUnitChange = useCallback((value) => setDelayUnit(value), []);
  const handleIsActiveChange = useCallback((value) => setIsActive(value), []);
  const handleTargetAudienceChange = useCallback((value) => setTargetAudience(value), []);
  const handleKeywordsChange = useCallback((value) => setKeywords(value), []);
  const handleVariableSelectionChange = useCallback((variable, value) => {
    setVariableSelections(prev => ({
      ...prev,
      [variable]: value
    }));
  }, []);
  const handleMessageTypeChange = useCallback((value) => {
    setMessageType(value);
    // Reset selections when message type changes
    setSelectedTemplate('');
    setCustomServiceMessage('');
    setMessageTemplate('');
  }, []);
  const handleCustomServiceMessageChange = useCallback((value) => {
    setCustomServiceMessage(value);
    setMessageTemplate(value);
  }, []);

  const handleFilterByCollectionChange = useCallback((value) => {
    setFilterByCollection(value);
    if (!value) {
      setSelectedCollections([]);
      setCollectionInputValue('');
    }
  }, []);

  const handleSelectedCollectionsChange = useCallback((value) => {
    setSelectedCollections(value);
  }, []);

  const handleCollectionInputChange = useCallback((value) => {
    setCollectionInputValue(value);
  }, []);

  const handleCollectionSelect = useCallback((selectedValue) => {
    if (selectedValue && !selectedCollections.includes(selectedValue) && selectedCollections.length < 5) {
      setSelectedCollections([...selectedCollections, selectedValue]);
    }
    setCollectionInputValue('');
  }, [selectedCollections]);

  const handleCollectionRemove = useCallback((collectionToRemove) => {
    setSelectedCollections(selectedCollections.filter(collection => collection !== collectionToRemove));
  }, [selectedCollections]);

  const handleTextSelection = useCallback((event) => {
    const textarea = event.target;
    setSelectedText(textarea.value.substring(textarea.selectionStart, textarea.selectionEnd));
    setCursorPosition(textarea.selectionStart);
  }, []);

  const applyFormatting = useCallback((format) => {
    const textarea = document.querySelector('#custom-service-message');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const end = textarea.selectionEnd;
    const selectedText = textarea.value.substring(start, end);
    
    let formattedText = '';
    let cursorOffset = 0;
    
    if (selectedText) {
      // If text is selected, wrap it with formatting
      switch (format) {
        case 'bold':
          formattedText = `*${selectedText}*`;
          cursorOffset = formattedText.length;
          break;
        case 'italic':
          formattedText = `_${selectedText}_`;
          cursorOffset = formattedText.length;
          break;
        case 'strikethrough':
          formattedText = `~${selectedText}~`;
          cursorOffset = formattedText.length;
          break;
        case 'monospace':
          formattedText = `\`\`\`${selectedText}\`\`\``;
          cursorOffset = formattedText.length;
          break;
        case 'inline-code':
          formattedText = `\`${selectedText}\``;
          cursorOffset = formattedText.length;
          break;
        case 'quote':
          // Ensure quote starts on new line
          const needsNewLineBeforeQuote = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterQuote = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeQuote ? '\n' : '') + `> ${selectedText}` + (needsNewLineAfterQuote ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          const bulletLines = selectedText.split('\n').map(line => line.trim() ? `* ${line}` : line).join('\n');
          // Ensure list starts on new line
          const needsNewLineBefore = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfter = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBefore ? '\n' : '') + bulletLines + (needsNewLineAfter ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          const numberedLines = selectedText.split('\n').filter(line => line.trim()).map((line, index) => `${index + 1}. ${line}`).join('\n');
          // Ensure list starts on new line
          const needsNewLineBeforeNum = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterNum = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeNum ? '\n' : '') + numberedLines + (needsNewLineAfterNum ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = selectedText;
          cursorOffset = formattedText.length;
      }
    } else {
      // If no text is selected, insert formatting characters with cursor between them
      switch (format) {
        case 'bold':
          formattedText = '**';
          cursorOffset = 1;
          break;
        case 'italic':
          formattedText = '__';
          cursorOffset = 1;
          break;
        case 'strikethrough':
          formattedText = '~~';
          cursorOffset = 1;
          break;
        case 'monospace':
          formattedText = '\`\`\`\n\n\`\`\`';
          cursorOffset = 4;
          break;
        case 'inline-code':
          formattedText = '\`\`';
          cursorOffset = 1;
          break;
        case 'quote':
          // Ensure quote starts on new line
          const needsNewLineBeforeQuoteEmpty = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeQuoteEmpty ? '\n> ' : '> ');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          // Ensure list starts on new line
          const needsNewLineBeforeBullet = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeBullet ? '\n* ' : '* ');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          // Ensure list starts on new line
          const needsNewLineBeforeNumber = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeNumber ? '\n1. ' : '1. ');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = '';
          cursorOffset = 0;
      }
    }
    
    const newValue = textarea.value.substring(0, start) + formattedText + textarea.value.substring(end);
    setCustomServiceMessage(newValue);
    setMessageTemplate(newValue);
    
    // Set cursor position
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + cursorOffset, start + cursorOffset);
    }, 0);
  }, []);

  // Keyboard shortcuts
  useEffect(() => {
    const handleKeyDown = (event) => {
      if ((event.ctrlKey || event.metaKey) && event.target.id === 'custom-service-message') {
        switch (event.key.toLowerCase()) {
          case 'b':
            event.preventDefault();
            applyFormatting('bold');
            break;
          case 'i':
            event.preventDefault();
            applyFormatting('italic');
            break;
          case 's':
            event.preventDefault();
            applyFormatting('strikethrough');
            break;
          case 'm':
            event.preventDefault();
            applyFormatting('monospace');
            break;
          case 'k':
            event.preventDefault();
            applyFormatting('inline-code');
            break;
          case 'q':
            event.preventDefault();
            applyFormatting('quote');
            break;
          case 'l':
            event.preventDefault();
            applyFormatting('bullet-list');
            break;
          case 'n':
            event.preventDefault();
            applyFormatting('numbered-list');
            break;
        }
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [applyFormatting]);

  const insertVariable = useCallback((variable) => {
    const textarea = document.querySelector('#custom-service-message');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const variableText = `{{${variable}}}`;
    const newValue = textarea.value.substring(0, start) + variableText + textarea.value.substring(start);
    
    setCustomServiceMessage(newValue);
    setMessageTemplate(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + variableText.length, start + variableText.length);
    }, 0);
  }, []);

  const insertEmoji = useCallback((emoji) => {
    const textarea = document.querySelector('#custom-service-message');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const newValue = textarea.value.substring(0, start) + emoji + textarea.value.substring(start);
    
    setCustomServiceMessage(newValue);
    setMessageTemplate(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + emoji.length, start + emoji.length);
    }, 0);
  }, []);

  const handleSave = useCallback(() => {

    console.log('Saving trigger:', {
      triggerName,
      triggerType,
      messageTemplate,
      delayTime,
      delayUnit,
      isActive,
      targetAudience,
      keywords,
      filterByCollection,
      selectedCollections
    });
    navigate('/app/auto-triggers');
  }, [triggerName, triggerType, messageTemplate, delayTime, delayUnit, isActive, targetAudience, keywords, filterByCollection, selectedCollections, navigate]);

  const handleCancel = useCallback(() => {
    navigate('/app/auto-triggers');
  }, [navigate]);

  // types of triggers
  const triggerTypeOptions = [
    { label: 'Abandoned Cart', value: 'abandoned_cart' },
    { label: 'Order Confirmation', value: 'order_confirmation' },
    { label: 'Keywords Auto Reply', value: 'keywords_auto_reply' },
   
  ];

  const delayUnitOptions = [
    { label: 'Minutes', value: 'minutes' },
    { label: 'Hours', value: 'hours' },
    { label: 'Days', value: 'days' }
  ];

  // Dynamic target audience options based on trigger type
  const getTargetAudienceOptions = () => {
    switch (triggerType) {
      case 'abandoned_cart':
        return [
          { label: 'Abandoned Cart Customers', value: 'abandoned_cart_customers' }
        ];
      case 'order_confirmation':
        return [
          { label: 'Order Confirmation Customers', value: 'order_confirmation_customers' }
        ];
      case 'keywords_auto_reply':
        return [
          { label: 'All Customers', value: 'all_customers' }
        ];
      default:
        return [
          { label: 'All Customers', value: 'all_customers' }
        ];
    }
  };

  const targetAudienceOptions = getTargetAudienceOptions();

  // Sample collections - in a real app, this would come from Shopify API
  const collectionOptions = [
    { label: 'Electronics', value: 'electronics' },
    { label: 'Clothing', value: 'clothing' },
    { label: 'Home & Garden', value: 'home-garden' },
    { label: 'Sports & Outdoors', value: 'sports-outdoors' },
    { label: 'Books', value: 'books' },
    { label: 'Beauty & Personal Care', value: 'beauty-personal-care' },
    { label: 'Toys & Games', value: 'toys-games' },
    { label: 'Automotive', value: 'automotive' }
  ];

  // Filter collections based on search input
  const filteredCollectionOptions = useMemo(() => {
    if (selectedCollections.length >= 5) return [];
    if (!collectionInputValue) return collectionOptions.filter(option => !selectedCollections.includes(option.value));
    return collectionOptions.filter(option =>
      option.label.toLowerCase().includes(collectionInputValue.toLowerCase()) &&
      !selectedCollections.includes(option.value)
    );
  }, [collectionInputValue, selectedCollections]);



  // Extract variables 
  const extractVariables = useMemo(() => {
    return extractVariablesFromWhatsAppMessage(messageTemplate);
  }, [messageTemplate, triggerType, messageType]);

  // Fetch variables from API on component mount
  useEffect(() => {
    const fetchVariables = async () => {
      try {
        setLoadingVariables(true);
        const variables = await getTemplateVariables();
        setAvailableVariables(variables);
      } catch (error) {
        console.error('Error fetching variables:', error);
        setAvailableVariables([]);
      } finally {
        setLoadingVariables(false);
      }
    };

    fetchVariables();
  }, []);

  const messageTypeOptions = [
   
    { label: 'Select Template', value: 'template' },
    { label: 'Create Service Message', value: 'service' }
  ];

  return (
    <Page
      title="Create Auto Trigger"
      subtitle="Set up a new automated trigger for customer engagement"
      backAction={{
        content: 'Auto Triggers',
        onAction: () => navigate('/app/auto-triggers')
      }}
      primaryAction={{
        content: 'Save Trigger',
        onAction: handleSave,
        disabled: !triggerName || !messageTemplate || (triggerType === 'keywords_auto_reply' && !messageType)
      }}
      secondaryActions={[
        {
          content: 'Cancel',
          onAction: handleCancel
        }
      ]}
      fullWidth
    >
      <Grid>
        <Grid.Cell columnSpan={{xs: 6, sm: 6, md: 8, lg: 8, xl: 8}}>
          <BlockStack gap="500">
        <Banner status="info">
          <Text as="p">
            Create automated triggers to engage with your customers at the right moment. 
            Configure when and how messages are sent based on customer behavior.
          </Text>
        </Banner>

        <Card>
          <BlockStack gap="400">
            <Text variant="headingMd" as="h2">Basic Information</Text>
            
            <FormLayout>
              <TextField
                label="Trigger Name"
                value={triggerName}
                onChange={handleTriggerNameChange}
                placeholder="e.g., Abandoned Cart Recovery"
                helpText="Give your trigger a descriptive name"
                requiredIndicator
              />

              <Select
                label="Trigger Type"
                options={triggerTypeOptions}
                value={triggerType}
                onChange={handleTriggerTypeChange}
              />

              {/* <Checkbox
                label="Active"
                checked={isActive}
                onChange={handleIsActiveChange}
                helpText="Enable this trigger to start sending messages"
              /> */}
            </FormLayout>
          </BlockStack>
        </Card>

        {triggerType === 'keywords_auto_reply' && (
          <Card>
            <BlockStack gap="400">
              <Text variant="headingMd" as="h2">Keywords Configuration</Text>
              
              <FormLayout>
                <TextField
                  label="Keywords"
                  value={keywords}
                  onChange={handleKeywordsChange}
                  placeholder="e.g., help, support, refund"
                  helpText="Enter keywords separated by commas"
                  multiline={2}
                />
                
                <Select
                  label="Message Type"
                  options={messageTypeOptions}
                  value={messageType}
                  onChange={handleMessageTypeChange}
                  placeholder="Choose message type"
                  helpText="Select whether to use a template or service message"
                />
              </FormLayout>
            </BlockStack>
          </Card>
        )}

        <Card>
          <BlockStack gap="400">
            <Text variant="headingMd" as="h2">Message Settings</Text>
            
            <FormLayout>
              {/* Show template selection for non-keywords triggers or when template type is selected */}
              {(triggerType !== 'keywords_auto_reply' || messageType === 'template') && (
                <Select
                  label="Select Template"
                  options={messageTemplateOptions}
                  value={selectedTemplate}
                  onChange={handleSelectedTemplateChange}
                  placeholder="Choose a message template"
                  helpText="Select a predefined template or choose 'Custom Template' to create your own"
                />
              )}
              
              {/* Show custom service message editor for keywords auto reply when service type is selected */}
              {triggerType === 'keywords_auto_reply' && messageType === 'service' && (
                <BlockStack gap="300">
                  <Text variant="bodyMd" as="h3">Custom Service Message</Text>
                  
                
                  
                  {/* Variable insertion buttons */}
                  <BlockStack gap="200">
                    <Text variant="bodySm" as="p" tone="subdued">
                      click to insert variables:
                    </Text>
                    <InlineStack gap="100" wrap>
                      {availableVariables.slice(0, 15).map((variable) => (
                        <Button
                          key={variable.value}
                          size="micro"
                          variant="tertiary"
                          onClick={() => insertVariable(variable.value)}
                        >
                          {variable.label}
                        </Button>
                      ))}
                      {availableVariables.length > 15 && (
                        <Button
                          size="micro"
                          variant="primary"
                          onClick={() => setShowVariablesModal(true)}
                        >
                          More Variables ({availableVariables.length - 15})
                        </Button>
                      )}
                    </InlineStack>
                  </BlockStack>
                  
                  {/* Custom message textarea with attached formatting controls */}
                  <div style={{ position: 'relative' }}>
                    <div style={{ position: 'absolute', bottom: 12, right: 6, zIndex: 99, display: 'flex', gap: '4px', flexWrap: 'wrap' }}>
                    <ButtonGroup variant="segmented">
                        <EmojiPicker onEmojiSelect={insertEmoji} buttonSize="micro" />
                      </ButtonGroup>
                      
                      
                      <ButtonGroup variant="segmented">
                       
                        <Tooltip content={<InlineStack gap="200">Bold<Text as="span" variant="bodyMd" tone="subdued">⌘B</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bold')} accessibilityLabel="Bold">
                            <Text fontWeight="bold">B</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Italic<Text as="span" variant="bodyMd" tone="subdued">⌘I</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('italic')} accessibilityLabel="Italic">
                            <Text as="em">I</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Strikethrough<Text as="span" variant="bodyMd" tone="subdued">⌘S</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('strikethrough')} accessibilityLabel="Strikethrough">
                            <Text as="s">S</Text>
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Monospace<Text as="span" variant="bodyMd" tone="subdued">⌘M</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('monospace')} accessibilityLabel="Monospace">
                            <Text as="code"> {'</>'}</Text>
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Inline Code<Text as="span" variant="bodyMd" tone="subdued">⌘K</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('inline-code')} accessibilityLabel="Inline Code">
                              <Text as="code">{"<>"}</Text>
                          </Button>
                        </Tooltip> 
                        <Tooltip content={<InlineStack gap="200">Quote<Text as="span" variant="bodyMd" tone="subdued">⌘Q</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('quote')} accessibilityLabel="Quote">
                            {'"'}
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                      <ButtonGroup variant="segmented">
                        <Tooltip content={<InlineStack gap="200">Bullet List<Text as="span" variant="bodyMd" tone="subdued">⌘L</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('bullet-list')} accessibilityLabel="Bullet List">
                            •
                          </Button>
                        </Tooltip>
                        <Tooltip content={<InlineStack gap="200">Numbered List<Text as="span" variant="bodyMd" tone="subdued">⌘N</Text></InlineStack>}>
                          <Button size="micro" onClick={() => applyFormatting('numbered-list')} accessibilityLabel="Numbered List">
                            1.
                          </Button>
                        </Tooltip>
                      </ButtonGroup>
                     
                    </div>
                   
                      <TextField
                        id="custom-service-message"
                        label="Message Content"
                        value={customServiceMessage}
                        onChange={handleCustomServiceMessageChange}
                        onSelect={handleTextSelection}
                        multiline={8}
                        maxLength={4096}
                        placeholder="Type your custom service message here...    Formatting options:*bold*, _italic_, ~strikethrough~, ```monospace```, `inline code`, > quote,* bullet list, 1. numbered list. Use keyboard shortcuts or click buttons above."
                        helpText={`${customServiceMessage.length}/4096 characters`}
                      />
                     
                  </div>
                </BlockStack>
              )}

              

              {extractVariables.length > 0 && (
                <BlockStack gap="300">
                  <Text variant="bodyMd" as="h3">Variable Mapping</Text>
                  <Text variant="bodySm" as="p" tone="subdued">
                    Map the template variables to actual data fields:
                  </Text>
                  {extractVariables.map((variable, index) => (
                    <FormLayout.Group key={variable}>
                      <TextField
                        label={`Variable: {{${variable}}}`}
                        value={`{{${variable}}}`}
                        disabled
                        helpText="Template variable"
                      />
                      <Select
                        label="Map to"
                        options={[
                          { label: 'Select mapping...', value: '' },
                          ...availableVariables
                        ]}
                        value={variableSelections[variable] || ''}
                        onChange={(value) => handleVariableSelectionChange(variable, value)}
                        placeholder="Choose data field"
                      />
                    </FormLayout.Group>
                  ))}
                </BlockStack>
              )}

              {triggerType !== 'keywords_auto_reply' && (
                <FormLayout.Group>
                  <TextField
                    label="Delay Time"
                    type="number"
                    value={delayTime}
                    onChange={handleDelayTimeChange}
                    min="1"
                  />
                  
                  <Select
                    label="Delay Unit"
                    options={delayUnitOptions}
                    value={delayUnit}
                    onChange={handleDelayUnitChange}
                    
                  />
                </FormLayout.Group>
              )}
            </FormLayout>
          </BlockStack>
        </Card>

        <Card>
          <BlockStack gap="400">
            <Text variant="headingMd" as="h2">Target Audience</Text>
            
            <BlockStack gap="300">
              <Text as="p" tone="subdued">
                Who should receive this trigger?
              </Text>
              <Checkbox
                label={targetAudienceOptions[0]?.label}
                checked={true}
                disabled={true}
                helpText="This audience is automatically selected based on your trigger type"
              />
              
              {/* Collection filtering - only show for abandoned cart and order confirmation */}
              {(triggerType === 'abandoned_cart' || triggerType === 'order_confirmation') && (
                <BlockStack gap="300">
                  <Checkbox
                    label="Do you want to filter target customers based on your product category/collection?"
                    checked={filterByCollection}
                    onChange={handleFilterByCollectionChange}
                    helpText="Filter customers based on the products they have in their cart or ordered"
                  />
                  
                  {filterByCollection && (
                    <FormLayout>
                      <BlockStack gap="200">
                        <InlineStack align="space-between">
                          <Text variant="bodyMd" as="h3">Select Collections (Maximum 5)</Text>
                          <Text variant="bodySm" as="p" tone={selectedCollections.length >= 5 ? "critical" : "subdued"}>
                            {selectedCollections.length}/5 selected
                          </Text>
                        </InlineStack>
                        
                        {/* Display selected collections as tags */}
                        {selectedCollections.length > 0 && (
                          <InlineStack gap="100" wrap>
                            {selectedCollections.map((collectionValue, index) => {
                              const collectionOption = collectionOptions.find(opt => opt.value === collectionValue);
                              const displayLabel = collectionOption ? collectionOption.label : collectionValue;
                              return (
                                
                                <Tag
                                  key={collectionValue}
                                  onRemove={() => handleCollectionRemove(collectionValue)}
                                >
                                  {displayLabel}
                                </Tag>
                              );
                            })}
                          </InlineStack>
                        )}
                        
                        {/* Searchable dropdown */}
                        <Combobox
                          activator={
                            <Combobox.TextField
                              onChange={handleCollectionInputChange}
                              label=""
                              labelHidden
                              value={collectionInputValue}
                              placeholder={selectedCollections.length >= 5 ? "Maximum 5 collections selected" : "Search and select collections..."}
                              autoComplete="off"
                              disabled={selectedCollections.length >= 5}
                            />
                          }
                        >
                          {filteredCollectionOptions.length > 0 ? (
                            <Listbox onSelect={handleCollectionSelect}>
                              {filteredCollectionOptions.map((option) => (
                                <Listbox.Option
                                  key={option.value}
                                  value={option.value}
                                  selected={selectedCollections.includes(option.value)}
                                  disabled={selectedCollections.includes(option.value)}
                                >
                                  {option.label}
                                </Listbox.Option>
                              ))}
                            </Listbox>
                          ) : (
                            <Listbox>
                              <Listbox.Option value="" disabled>
                                {selectedCollections.length >= 5 ? "Maximum collections selected" : "No collections found"}
                              </Listbox.Option>
                            </Listbox>
                          )}
                        </Combobox>
                        
                        <Text variant="bodySm" as="p" tone="subdued">
                          Only customers who have products from the selected collections will receive this trigger. Maximum 5 collections allowed.
                        </Text>
                      </BlockStack>
                    </FormLayout>
                  )}
                </BlockStack>
              )}
            </BlockStack>
          </BlockStack>
        </Card>

            <Card>
              <InlineStack align="end">
                <ButtonGroup>
                  <Button onClick={handleCancel}>Cancel</Button>
                  <Button 
                    variant="primary" 
                    onClick={handleSave}
                    disabled={!triggerName || !messageTemplate || (triggerType === 'keywords_auto_reply' && !messageType)}
                  >
                    Save Trigger
                  </Button>
                </ButtonGroup>
              </InlineStack>
            </Card>
          </BlockStack>
        </Grid.Cell>
        
        <Grid.Cell columnSpan={{xs: 6, sm: 6, md: 4, lg: 4, xl: 4}}>
        <div style={{
    position: "sticky",
    top: "20px", 
    alignSelf: "flex-start",
    zIndex: 10
  }}>
          <BlockStack gap="500">
            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Message Preview</Text>
                
                {messageTemplate ? (
                  <BlockStack gap="300">
                    <Text as="p" tone="">
                      <Text variant="bodySm" as="p">
                      {messageTemplateOptions?.find(option => option.value === selectedTemplate)?.label}
                      </Text>
                      
                    </Text>
                    <Card sectioned background="bg-surface-secondary">
                      <WhatsAppPreview 
                        template={messageTemplate}
                        variant="bubble"
                        maxWidth="280px"
                      />
                    </Card>
                    
                    <Text variant="bodySm" as="p" tone="subdued">
                      Variables like {`{`}{`{`}{`customer_name`}{`}`}{`}`} or {`{`}{`{`}{`1`}{`}`}{`}`} will be replaced with actual customer data when sent.
                    </Text>
                  </BlockStack>
                ) : (
                  <BlockStack gap="200">
                    <Text as="p" tone="subdued">
                      Select a template to see the preview
                    </Text>
                    <LegacyCard sectioned>
                      <Text variant="bodySm" as="p" tone="subdued">
                        Your message preview will appear here once you select a template or enter a custom message.
                      </Text>
                    </LegacyCard>
                  </BlockStack>
                )}
              </BlockStack>
            </Card>
          </BlockStack>
            </div>
        </Grid.Cell>
      </Grid>

      {/* Variables Modal */}
      <Modal
        open={showVariablesModal}
        onClose={() => {
          setShowVariablesModal(false);
          setVariableSearchQuery('');
        }}
        title="All Available Variables"
        primaryAction={{
          content: 'Close',
          onAction: () => {
            setShowVariablesModal(false);
            setVariableSearchQuery('');
          },
        }}
        size="large"
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" tone="subdued">
              Click on any variable to insert it into your template:
            </Text>
            
            {/* Search Field */}
            <TextField
              label="Search Variables"
              value={variableSearchQuery}
              onChange={setVariableSearchQuery}
              placeholder="Type to search variables..."
              clearButton
              onClearButtonClick={() => setVariableSearchQuery('')}
              autoComplete="off"
            />
            
            <InlineStack gap="200" wrap>
              {availableVariables
                .filter(variable => 
                  variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
                  variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
                )
                .map((variable) => (
                  <Button
                    key={variable.value}
                    size="medium"
                    variant="tertiary"
                    onClick={() => {
                      insertVariable(variable.value);
                      setShowVariablesModal(false);
                      setVariableSearchQuery('');
                    }}
                  >
                    {variable.label}
                  </Button>
                ))}
            </InlineStack>
            
            {availableVariables.filter(variable => 
              variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
              variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
            ).length === 0 && variableSearchQuery && (
              <Text variant="bodyMd" tone="subdued" alignment="center">
                No variables found matching "{variableSearchQuery}"
              </Text>
            )}
            
            <Text variant="bodySm" tone="subdued">
              You can also create custom variables using the format: {"{{variable_name}}"}
            </Text>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </Page>
  );
}

export default CreateAutoTriggerPage;
