import {
  Page,
  LegacyCard,
  Button,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  TextField,
  Badge,
  Text,
  InlineStack,
  BlockStack,
  Pagination,
  Icon,
  Tooltip,
  Modal,
  Link,
  TextContainer,
  DropZone,
  Tabs,
} from '@shopify/polaris';
import { PlusIcon, ViewIcon, EditIcon, DeleteIcon, ImportIcon, ExportIcon } from '@shopify/polaris-icons';
import React, { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate, useLocation } from '@remix-run/react';
import { fetchStaticContactList, addNewContactListName, deleteContactList, addNewVariable } from '../utils/contactsApi';
import { API_TOKEN } from '../utils/apiConfig';
import { getTemplateVariables } from '../utils/availableVariables';

function StaticListPage() {
  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
  const navigate = useNavigate();
  const location = useLocation();
  
  // Tab navigation
  const [selectedTab, setSelectedTab] = useState(0); // Static List is now tab index 0 (first)
  
  const navigationTabs = [
    {
      id: 'static-list',
      content: 'Static List',
      accessibilityLabel: 'Static List tab',
      panelID: 'static-list-panel',
    },
    {
      id: 'live-segment',
      content: 'Live Segment',
      accessibilityLabel: 'Live Segment tab',
      panelID: 'live-segment-panel',
    },
  ];
  
  const handleTabChange = useCallback((selectedTabIndex) => {
    setSelectedTab(selectedTabIndex);
    if (selectedTabIndex === 1) {
      navigate('/app/contact-list/live-segment');
    }
  }, [navigate]);
  
  const [selectedList, setSelectedList] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  const [deleteModalActive, setDeleteModalActive] = useState(false);
  const [importModalActive, setImportModalActive] = useState(false);
  const [createListModalActive, setCreateListModalActive] = useState(false);
  const [createVariableModalActive, setCreateVariableModalActive] = useState(false);
  const [files, setFiles] = useState([]);
  const [listName, setListName] = useState('');
  const [variableName, setVariableName] = useState('');
  const [isCreating, setIsCreating] = useState(false);
  const [isDeleting, setIsDeleting] = useState(false);
  const [successMessage, setSuccessMessage] = useState('');
  
  // CSV column mapping state
  const [csvColumns, setCsvColumns] = useState([]);
  const [columnMappings, setColumnMappings] = useState({});
  const [showColumnMapping, setShowColumnMapping] = useState(false);
  const [mappingColumnIndex, setMappingColumnIndex] = useState(null);
  const [csvFile, setCsvFile] = useState(null);
  const [processingCsv, setProcessingCsv] = useState(false);
  const [csvErrorType, setCsvErrorType] = useState(null); // Track specific error type
  
  // Variables modal state
  const [availableVariables, setAvailableVariables] = useState([]);
  const [loadingVariables, setLoadingVariables] = useState(true);
  const [showVariablesModal, setShowVariablesModal] = useState(false);
  const [variableSearchQuery, setVariableSearchQuery] = useState('');
  
  // IndexFilters state
  const [itemStrings, setItemStrings] = useState([
    'All',
    'Recently Created',
    'Large Lists',
    'By Source',
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['updated desc']);
  const [sizeFilter, setSizeFilter] = useState([0, 50000]);
  const [sourceFilter, setSourceFilter] = useState('');
  
  const {mode, setMode} = useSetIndexFiltersMode();

  // Pagination state
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 20;

  // API data state
  const [staticLists, setStaticLists] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [paginationData, setPaginationData] = useState({
    current_page: 1,
    last_page: 1,
    total: 0,
    per_page: 20
  });

  // Fetch static contact lists from API
  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await fetchStaticContactList({
          page: currentPage,
          perPageData: itemsPerPage
        });

        if (response.status && response.WhatsAppBusinessTag) {
          const { data, current_page, last_page, total, per_page } = response.WhatsAppBusinessTag;
          
          // Transform API data to match component structure
          const transformedData = data.map(item => ({
            id: item.id.toString(),
            name: item.name,
            description: `Contact list with ${item.totalcontatspertag_count} contacts`,
            status: 'Active', // Default status since API doesn't provide this
            contactCount: item.totalcontatspertag_count,
            source: 'Manual Upload', // Set as requested
            lastUpdated: new Date(item.updated_at).toLocaleDateString(),
            createdAt: new Date(item.created_at).toLocaleDateString(),
            importedFrom: 'Manual Upload'
          }));

          setStaticLists(transformedData);
          setPaginationData({
            current_page,
            last_page,
            total,
            per_page
          });
        } else {
          throw new Error('Invalid API response format');
        }
      } catch (err) {
        console.error('Error fetching static contact lists:', err);
        setError(err.message);
        setStaticLists([]);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, [currentPage, itemsPerPage]);

  // Fetch variables from API on component mount
  useEffect(() => {
    const fetchVariables = async () => {
      try {
        setLoadingVariables(true);
        const variables = await getTemplateVariables();
        setAvailableVariables(variables);
      } catch (error) {
        console.error('Error fetching variables:', error);
        setAvailableVariables([]);
      } finally {
        setLoadingVariables(false);
      }
    };

    fetchVariables();
  }, []);

  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => {},
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      {
        type: 'rename',
        onAction: () => {},
        onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => {
            if (idx === index) {
              return value;
            }
            return item.content;
          });
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      {
        type: 'duplicate',
        onPrimaryAction: async (value) => {
          await sleep(1);
          duplicateView(value);
          return true;
        },
      },
      {
        type: 'edit',
      },
      {
        type: 'delete',
        onPrimaryAction: async () => {
          await sleep(1);
          deleteView(index);
          return true;
        },
      },
    ],
  }));

  const filteredLists = useMemo(() => {
    let filtered = staticLists;
    
    // Filter by tab selection
    if (selected === 1) { // Recently Created
      const sevenDaysAgo = new Date();
      sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
      filtered = filtered.filter(list => new Date(list.createdAt) >= sevenDaysAgo);
    } else if (selected === 2) { // Large Lists
      filtered = filtered.filter(list => list.contactCount >= 10000);
    } else if (selected === 3) { // By Source
      filtered = filtered.filter(list => sourceFilter !== '' && 
                                 list.source.toLowerCase().includes(sourceFilter.toLowerCase()));
    }
    
    return filtered.filter((list) => {
      const matchesSearch = list.name.toLowerCase().includes(queryValue.toLowerCase()) ||
                           list.description.toLowerCase().includes(queryValue.toLowerCase());
      const matchesSize = list.contactCount >= sizeFilter[0] && list.contactCount <= sizeFilter[1];
      const matchesSource = selected !== 3 && (sourceFilter === '' || 
                           list.source.toLowerCase().includes(sourceFilter.toLowerCase()));
      
      return matchesSearch && matchesSize && (selected === 3 || matchesSource);
    });
  }, [staticLists, queryValue, sizeFilter, sourceFilter, selected]);

  const sortedLists = useMemo(() => {
    const sorted = [...filteredLists];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');
    
    sorted.sort((a, b) => {
      let aValue, bValue;
      
      switch (sortKey) {
        case 'name':
          aValue = a.name.toLowerCase();
          bValue = b.name.toLowerCase();
          break;
        case 'status':
          aValue = a.status.toLowerCase();
          bValue = b.status.toLowerCase();
          break;
        case 'size':
          aValue = a.contactCount;
          bValue = b.contactCount;
          break;
        case 'source':
          aValue = a.source.toLowerCase();
          bValue = b.source.toLowerCase();
          break;
        case 'updated':
          aValue = new Date(a.lastUpdated);
          bValue = new Date(b.lastUpdated);
          break;
        default:
          return 0;
      }
      
      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });
    
    return sorted;
  }, [filteredLists, sortSelected]);

  // Use API data directly since pagination is handled by the API
  const paginatedLists = useMemo(() => {
    return sortedLists; // API already returns paginated data
  }, [sortedLists]);

  const handlePreviousPage = useCallback(() => {
    setCurrentPage(prev => Math.max(1, prev - 1));
  }, []);

  const handleNextPage = useCallback(() => {
    setCurrentPage(prev => Math.min(paginationData.last_page, prev + 1));
  }, [paginationData.last_page]);

  const totalItems = paginationData.total;
  const totalPages = paginationData.last_page;
  const startItem = paginationData.current_page === 1 ? 1 : ((paginationData.current_page - 1) * paginationData.per_page) + 1;
  const endItem = Math.min(paginationData.current_page * paginationData.per_page, totalItems);
  const hasNext = paginationData.current_page < paginationData.last_page;
  const hasPrevious = paginationData.current_page > 1;

  // Sort options
  const sortOptions = [
    {label: 'List Name', value: 'name asc', directionLabel: 'A-Z'},
    {label: 'List Name', value: 'name desc', directionLabel: 'Z-A'},
    {label: 'Status', value: 'status asc', directionLabel: 'A-Z'},
    {label: 'Status', value: 'status desc', directionLabel: 'Z-A'},
    {label: 'Contact Count', value: 'size asc', directionLabel: 'Low to High'},
    {label: 'Contact Count', value: 'size desc', directionLabel: 'High to Low'},
    {label: 'Source', value: 'source asc', directionLabel: 'A-Z'},
    {label: 'Source', value: 'source desc', directionLabel: 'Z-A'},
    {label: 'Last Updated', value: 'updated asc', directionLabel: 'Oldest First'},
    {label: 'Last Updated', value: 'updated desc', directionLabel: 'Newest First'},
  ];

  const handlePreviewClick = useCallback((list) => {
    // Navigate to static list contacts page with id in query for fetching
    navigate(`/app/contact-list/contacts/static-list/${encodeURIComponent(list.name)}?id=${encodeURIComponent(list.id)}`);
  }, [navigate]);

  const handleDeleteClick = useCallback((list) => {
    setSelectedList(list);
    setDeleteModalActive(true);
  }, []);

  const handleImportClick = useCallback(() => {
    setImportModalActive(true);
  }, []);

  const handleCreateListClick = useCallback(() => {
    setCreateListModalActive(true);
  }, []);
  const handleCreateVariableListClick = useCallback(() => {
    setCreateVariableModalActive(true);
  }, []);

  const handleCreateContactList = useCallback(async () => {
    if (!listName || !files.length) return;
    
    setIsCreating(true);
    try {
      const contactListPayload = {
        name: listName,
        description: `Contact list created on ${new Date().toLocaleDateString()}`
      };
      
      const result = await addNewContactListName(contactListPayload, files[0], columnMappings);
      
      if (result.contactListResponse && result.csvUploadResponse) {
        // Success - refresh the list
        window.location.reload();
      } else {
        throw new Error('Failed to create contact list');
      }
    } catch (error) {
      console.error('Error creating contact list:', error);
      alert(`Error creating contact list: ${error.message}`);
    } finally {
      setIsCreating(false);
    }
  }, [listName, files, columnMappings]);

  const handleCreateVariables = useCallback(async () => {
    if (!variableName) return;
    
    setIsCreating(true);
    function formatToCamelCase(variableName) {
  return variableName
    .split('_')                    
    .map(word => 
      word.charAt(0).toUpperCase() + word.slice(1).toLowerCase()
    )                              
    .join('');                     
}
const FormattedvariableName = formatToCamelCase(variableName);
    try {
      const CreateVariablePayload = { 
        name: FormattedvariableName,
      };
      
      const result = await addNewVariable(CreateVariablePayload);
      
      if (result.success === true) {
        setSuccessMessage(result.message || 'Label created successfully!');
       
        setVariableName('');
        
        // Reload variables list after successful creation
        try {
          setLoadingVariables(true);
          const variables = await getTemplateVariables();
          setAvailableVariables(variables);
        } catch (error) {
          console.error('Error reloading variables:', error);
          setAvailableVariables([]);
        } finally {
          setLoadingVariables(false);
        }
        
        setTimeout(() => {
         
         setSuccessMessage('');
        }, 2000);
      } else {
        throw new Error('Failed to create variable');
      }
    } catch (error) {
      console.error('Error creating variable:', error);
      alert(`Error creating variable: ${error.message}`);
    } finally {
      setIsCreating(false);
    }
  }, [variableName]);

  // Insert variable at cursor position in variable name field
  const insertVariable = useCallback((variable) => {
    const input = document.querySelector('#variable-name-field');
    if (!input) return;
    
    const start = input.selectionStart;
    const variableText = `{{${variable}}}`;
    const newValue = input.value.substring(0, start) + variableText + input.value.substring(start);
    
    setVariableName(newValue);
    
    setTimeout(() => {
      input.focus();
      input.setSelectionRange(start + variableText.length, start + variableText.length);
    }, 0);
  }, []);

  // Handle column mapping
  const handleMapColumn = useCallback((columnIndex) => {
    setMappingColumnIndex(columnIndex);
    setShowVariablesModal(true);
  }, []);

  // Handle variable selection for column mapping
  const handleVariableSelect = useCallback((variable) => {
    if (mappingColumnIndex !== null) {
      setColumnMappings(prev => ({
        ...prev,
        [mappingColumnIndex]: variable.value
      }));
    }
    setShowVariablesModal(false);
    setMappingColumnIndex(null);
    setVariableSearchQuery('');
  }, [mappingColumnIndex]);

  // Handle column mapping input change
  const handleColumnMappingChange = useCallback((columnIndex, value) => {
    setColumnMappings(prev => ({
      ...prev,
      [columnIndex]: value
    }));
  }, []);

  // Remove column mapping
  const handleRemoveColumnMapping = useCallback((columnIndex) => {
    setColumnMappings(prev => {
      const newMappings = { ...prev };
      delete newMappings[columnIndex];
      return newMappings;
    });
  }, []);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setDeleteModalActive(false);
    setImportModalActive(false);
    setCreateListModalActive(false);
    setCreateVariableModalActive(false);
    setShowVariablesModal(false);
    setSelectedList(null);
    setFiles([]);
    setListName('');
    setVariableName('');
    setIsCreating(false);
    setSuccessMessage('');
    setVariableSearchQuery('');
    // Reset CSV mapping state
    setCsvColumns([]);
    setColumnMappings({});
    setShowColumnMapping(false);
    setMappingColumnIndex(null);
    setCsvFile(null);
    setProcessingCsv(false);
    setCsvErrorType(null); // Reset error type
  }, []);

  const handleConfirmDelete = useCallback(async () => {
    if (!selectedList || isDeleting) return;
    setIsDeleting(true);
    try {
      await deleteContactList(selectedList.id);
      setStaticLists(prev => prev.filter(list => list.id !== selectedList.id));
      setDeleteModalActive(false);
      setSelectedList(null);
    } catch (error) {
      console.error('Error deleting contact list:', error);
      alert(`Failed to delete list: ${error.message || 'Unknown error'}`);
    } finally {
      setIsDeleting(false);
    }
  }, [selectedList, isDeleting]);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);
  
  const handleSizeChange = useCallback((value) => {
    setSizeFilter(value);
  }, []);
  
  const handleSourceChange = useCallback((value) => {
    setSourceFilter(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleSizeRemove = useCallback(() => setSizeFilter([0, 50000]), []);
  const handleSourceRemove = useCallback(() => setSourceFilter(''), []);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleSizeRemove();
    handleSourceRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleSizeRemove, handleSourceRemove, handleQueryValueRemove]);

  const onHandleCancel = () => {};
  const onHandleSave = async () => {
    await sleep(1);
    return true;
  };

  const primaryAction = selected === 0
    ? {
        type: 'save-as',
        onAction: onCreateNewView,
        disabled: false,
        loading: false,
      }
    : {
        type: 'save',
        onAction: onHandleSave,
        disabled: false,
        loading: false,
      };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'Active': { tone: 'success' },
      'Archived': { tone: 'subdued' },
      'Draft': { tone: 'attention' },
      'Error': { tone: 'critical' }
    };
    
    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };

  const handleDropZoneDrop = useCallback(
    (_dropFiles, acceptedFiles, _rejectedFiles) => {
      setFiles((files) => [...files, ...acceptedFiles]);
    },
    [],
  );

  const handleCreateListDropZoneDrop = useCallback(
    async (_dropFiles, acceptedFiles, _rejectedFiles) => {
      const file = acceptedFiles[0];
      if (file && file.type === 'text/csv') {
        setFiles([file]);
        setCsvFile(file);
        await processCsvFile(file);
      }
    },
    [],
  );

  // Process CSV file to extract columns
  const processCsvFile = useCallback(async (file) => {
    setProcessingCsv(true);
    setCsvErrorType(null); // Reset error type
    try {
      const text = await file.text();
      const lines = text.split('\n').filter(line => line.trim());
      
      if (lines.length > 0) {
        const headers = lines[0].split(',').map(header => header.trim().replace(/"/g, ''));
        
        // Validate that CSV has at least 2 columns
        if (headers.length < 2) {
          setCsvErrorType('column_format');
          throw new Error('CSV must have at least 2 columns');
        }
        
        // Validate second column of second row contains only integer values (phone numbers)
        if (lines.length > 1) {
          const secondRow = lines[1].split(',').map(cell => cell.trim().replace(/"/g, ''));
          if (secondRow.length > 1) {
            const secondColumnValue = secondRow[1];
            
            // Check if it's a valid integer (no decimal points, no letters, only digits)
            if (!/^\d+$/.test(secondColumnValue)) {
              setCsvErrorType('number_format');
              throw new Error(`Second column must contain only integer numbers (phone numbers). Found: "${secondColumnValue}"`);
            }
          }
        }
        
        // Create columns starting from the third column (index 2), skipping first two columns
        const columns = headers.slice(2).map((header, index) => ({
          index: index + 2, 
          name: `Column ${index + 3}`, 
          firstRowValue: header || `Column ${index + 3}`,
          displayName: `Column ${index + 3} (${header || `Column ${index + 3}`})`
        }));
        
        setCsvColumns(columns);
        setColumnMappings({});
        setShowColumnMapping(true);
      }
    } catch (error) {
      console.error('Error processing CSV:', error);
      alert(`CSV Validation Error: ${error.message}`);
      // Reset file state on validation error
      setFiles([]);
      setCsvFile(null);
      setCsvColumns([]);
      setColumnMappings({});
      setShowColumnMapping(false);
    } finally {
      setProcessingCsv(false);
    }
  }, []);

  const validImageTypes = ['image/gif', 'image/jpeg', 'image/png'];

  const fileUpload = !files.length && <DropZone.FileUpload />;
  const uploadedFiles = files.length > 0 && (
    <div style={{ padding: '16px' }}>
      <BlockStack gap="200">
        {files.map((file, index) => (
          <InlineStack alignment="center" key={index}>
            <div>
              {validImageTypes.includes(file.type) ? (
                <img
                  alt={file.name}
                  width="50"
                  height="50"
                  style={{ objectFit: 'cover', borderRadius: '4px' }}
                  src={
                    validImageTypes.includes(file.type)
                      ? window.URL.createObjectURL(file)
                      : ''
                  }
                />
              ) : (
                <div style={{
                  width: '50px',
                  height: '50px',
                  backgroundColor: 'var(--p-color-bg-surface-secondary)',
                  borderRadius: '4px',
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center'
                }}>
                  📄
                </div>
              )}
            </div>
            <div>
              <Text variant="bodyMd" as="p">
                {file.name}
              </Text>
              <Text variant="bodySm" as="p" tone="subdued">
                {file.size} bytes
              </Text>
            </div>
          </InlineStack>
        ))}
      </BlockStack>
    </div>
  );

  const filters = [
    {
      key: 'size',
      label: 'Contact Count',
      filter: (
        <RangeSlider
          label="Contact count"
          labelHidden
          value={sizeFilter || [0, 50000]}
          suffix=" contacts"
          output
          min={0}
          max={50000}
          step={100}
          onChange={handleSizeChange}
        />
      ),
    },
    {
      key: 'source',
      label: 'Source',
      filter: (
        <TextField
          label="Source"
          value={sourceFilter}
          onChange={handleSourceChange}
          autoComplete="off"
          labelHidden
        />
      ),
      shortcut: true,
    },
  ];

  const appliedFilters = [];
  if (sizeFilter && (sizeFilter[0] !== 0 || sizeFilter[1] !== 50000)) {
    appliedFilters.push({
      key: 'size',
      label: disambiguateLabel('size', sizeFilter),
      onRemove: handleSizeRemove,
    });
  }
  if (!isEmpty(sourceFilter)) {
    appliedFilters.push({
      key: 'source',
      label: disambiguateLabel('source', sourceFilter),
      onRemove: handleSourceRemove,
    });
  }

  const resourceName = {
    singular: 'list',
    plural: 'lists',
  };

  const {selectedResources, allResourcesSelected, handleSelectionChange} =
    useIndexResourceState(paginatedLists);

  function disambiguateLabel(key, value) {
    switch (key) {
      case 'size':
        return `Contact count: ${value[0].toLocaleString()}-${value[1].toLocaleString()}`;
      case 'source':
        return `Source contains "${value}"`;
      case 'status':
        return value.map((val) => `Status: ${val.charAt(0).toUpperCase() + val.slice(1)}`).join(', ');
      default:
        return value;
    }
  }

  function isEmpty(value) {
    if (Array.isArray(value)) {
      return value.length === 0;
    } else {
      return value === '' || value == null;
    }
  }

  const rowMarkup = paginatedLists.map(
    (list, index) => (
      <IndexTable.Row
        id={list.id}
        key={list.id}
        selected={selectedResources.includes(list.id)}
        position={index}
      >
        <IndexTable.Cell>
          <BlockStack gap="100">
            <Text variant="bodyMd" fontWeight="semibold" as="span">
              {list.name}
            </Text>
            <Text variant="bodySm" as="span" tone="subdued">
              {list.description}
            </Text>
          </BlockStack>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span" numeric>
            {list.contactCount.toLocaleString()}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" as="span">
            {list.source}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" as="span">
            {list.lastUpdated}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end" gap="200">
            <Tooltip content="View list details">
              <Button
                size="slim"
                onClick={(event) => {
                  event.stopPropagation();
                  handlePreviewClick(list);
                }}
              >
                <Icon source={ViewIcon} />
              </Button>
            </Tooltip>
            <Tooltip content="Export list">
              <Button
                size="slim"
                onClick={(event) => {
                  event.stopPropagation();
                  console.log('Export list:', list.id);

                  window.open(`https://app.getgabs.com/partners/broadcast/export/exportMultipleContactLists?ids=${list.id}&token=${API_TOKEN}`, '_blank');
                }}
              >
                <Icon source={ExportIcon} />
              </Button>
            </Tooltip>
            
            <Tooltip content="Delete list">
              <Button
                size="slim"
                tone="critical"
                onClick={(event) => {
                  event.stopPropagation();
                  handleDeleteClick(list);
                }}
              >
                <Icon source={DeleteIcon} />
              </Button>
            </Tooltip>
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  return (
    <>
      <Page 
        title="Contact List" 
        subtitle="Manage your customer segments and contact lists"
        fullWidth
        primaryAction={{
          content: (
            <InlineStack alignment="center">
             
              <Text as="span">Create Contact List</Text>
            </InlineStack>
          ),
          onAction: handleCreateListClick
        }}
        secondaryActions={[
                 {
                   content: (
                     <InlineStack alignment="center">
                     
                       <Text as="span">Create Variables</Text>
                     </InlineStack>
                   ),
                   onAction: handleCreateVariableListClick,
                 }
               ]}
      >
        <BlockStack gap="400">
          <Tabs tabs={navigationTabs} selected={selectedTab} onSelect={handleTabChange}>
            <div id="static-list-panel">
        <LegacyCard>
          <IndexFilters
            sortOptions={sortOptions}
            sortSelected={sortSelected}
            queryValue={queryValue}
            queryPlaceholder="Search lists..."
            onQueryChange={handleFiltersQueryChange}
            onQueryClear={() => setQueryValue('')}
            onSort={setSortSelected}
            cancelAction={{
              onAction: onHandleCancel,
              disabled: false,
              loading: false,
            }}
            tabs={tabs}
            selected={selected}
            onSelect={setSelected}
            canCreateNewView
            onCreateNewView={onCreateNewView}
            filters={filters}
            appliedFilters={appliedFilters}
            onClearAll={handleFiltersClearAll}
            mode={mode}
            setMode={setMode}
          />
          <IndexTable
            resourceName={resourceName}
            itemCount={paginatedLists.length}
            selectedItemsCount={
              allResourcesSelected ? 'All' : selectedResources.length
            }
            onSelectionChange={handleSelectionChange}
            headings={[
              {title: 'List Name'},
              {title: 'Contact Count'},
              {title: 'Source'},
              {title: 'Last Updated'},
              {title: 'Actions', alignment: 'end'},
            ]}
            emptyState={
              <div style={{ textAlign: 'center', padding: '60px 20px' }}>
                <Text variant="headingMd" as="h3">
                  {loading ? 'Loading contact lists...' : 
                   error ? 'Error loading contact lists' :
                   filteredLists.length === 0 && staticLists.length > 0 
                    ? 'No static lists found'
                    : 'No static lists available'}
                </Text>
                <Text variant="bodyMd" as="p" tone="subdued">
                  {loading ? 'Please wait while we fetch your contact lists.' :
                   error ? `Error: ${error}` :
                   filteredLists.length === 0 && staticLists.length > 0 
                    ? 'Try adjusting your filters to see more results.'
                    : 'Create your first static list or import contacts to get started.'}
                </Text>
              </div>
            }
          >
            {rowMarkup}
          </IndexTable>
          
          {totalItems > itemsPerPage && (
            <div
              style={{
                maxWidth: '700px',
                margin: 'auto',
                border: '1px solid var(--p-color-border)',
                marginTop: '16px'
              }}
            >
              <Pagination
                onPrevious={hasPrevious ? handlePreviousPage : undefined}
                onNext={hasNext ? handleNextPage : undefined}
                type="table"
                hasNext={hasNext}
                hasPrevious={hasPrevious}
                label={`${startItem}-${endItem} of ${totalItems} lists`}
              />
            </div>
          )}
        </LegacyCard>
            </div>
          </Tabs>
        </BlockStack>
      </Page>

      {/* Preview Modal */}
      {selectedList && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Static List: ${selectedList.name}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
          secondaryActions={[
            {
              content: 'Edit List',
              onAction: () => {
                console.log('Edit list:', selectedList.id);
                handleModalClose();
              }
            },
            {
              content: 'Export List',
              onAction: () => {
                console.log('Export list:', selectedList.id);
                handleModalClose();
              }
            }
          ]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <BlockStack gap="200">
                <Text variant="headingMd" as="h3">
                  {selectedList.name}
                </Text>
                <Text variant="bodyLg" as="p">
                  {selectedList.description}
                </Text>
                <InlineStack gap="200">
                  {getStatusBadge(selectedList.status)}
                </InlineStack>
              </BlockStack>
              
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Contact Count:</strong> {selectedList.contactCount.toLocaleString()} contacts
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Source:</strong> {selectedList.source}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Imported From:</strong> {selectedList.importedFrom}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Last Updated:</strong> {selectedList.lastUpdated}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Created:</strong> {selectedList.createdAt}
                </Text>
              </TextContainer>

            </BlockStack>
          </Modal.Section>
        </Modal>
      )}

      {/* Delete Confirmation Modal */}
      {selectedList && (
        <Modal
          open={deleteModalActive}
          onClose={handleModalClose}
          title="Delete Static List"
          primaryAction={{
            content: isDeleting ? 'Deleting...' : 'Delete',
            destructive: true,
            onAction: handleConfirmDelete,
            disabled: isDeleting,
            loading: isDeleting
          }}
          secondaryActions={[
            {
              content: 'Cancel',
              onAction: handleModalClose
            }
          ]}
        >
          <Modal.Section>
            <TextContainer>
              <Text variant="bodyMd" as="p">
                Are you sure you want to delete the list "{selectedList.name}"? 
                This action cannot be undone and will remove all {selectedList.contactCount.toLocaleString()} contacts from this list.
              </Text>
              <Text variant="bodyMd" as="p" tone="subdued">
                Any active campaigns using this list will be affected.
              </Text>
            </TextContainer>
          </Modal.Section>
        </Modal>
      )}

      {/* Import Modal */}
      <Modal
        open={importModalActive}
        onClose={handleModalClose}
        title="Import Contacts"
        primaryAction={{
          content: 'Import',
          onAction: () => {
            console.log('Import contacts:', files);
            handleModalClose();
          },
          disabled: files.length === 0
        }}
        secondaryActions={[
          {
            content: 'Cancel',
            onAction: handleModalClose
          }
        ]}
      >
        <Modal.Section>
          <BlockStack gap="400">
            <TextContainer>
              <Text variant="bodyMd" as="p">
                Upload a CSV file with your contacts. The file should include columns for name, phone number, and email.
              </Text>
              <Text variant="bodySm" as="p" tone="subdued">
                Supported formats: CSV, Excel (.xlsx), JSON
              </Text>
            </TextContainer>
            
            <DropZone onDrop={handleDropZoneDrop}>
              {uploadedFiles}
              {fileUpload}
            </DropZone>

            {files.length > 0 && (
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Selected files:</strong> {files.length} file(s)
                </Text>
                <Text variant="bodySm" as="p" tone="subdued">
                  Click "Import" to process these files and add contacts to a new static list.
                </Text>
              </TextContainer>
            )}
          </BlockStack>
        </Modal.Section>
      </Modal>

      {/* Create Contact List Modal */}
      <Modal
        open={createListModalActive}
        onClose={handleModalClose}
        title="Create Contact List"
        primaryAction={{
          content: isCreating ? 'Creating...' : 'Create List',
          onAction: handleCreateContactList,
          disabled: !listName || !files.length || isCreating,
          loading: isCreating
        }}
        secondaryActions={[
          {
            content: 'Cancel',
            onAction: handleModalClose
          }
        ]}
      >
        <Modal.Section>
          <BlockStack gap="400">
            {/* Contact List Name */}
            <TextField
              label="Contact List Name"
              value={listName}
              onChange={setListName}
              placeholder="Enter a name for your contact list"
              autoComplete="off"
              helpText="Choose a descriptive name for your contact list"
            />

            {/* CSV Upload */}
            <BlockStack gap="200">
              <Text variant="headingMd" as="h4">
                Upload CSV File
              </Text>
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  Upload a CSV file with your contact data. The system will automatically process the file and create your contact list.
                </Text>
                <Text variant="bodySm" as="p" tone="subdued">
                  Supported format: CSV files only
                </Text>
                <Text variant="bodySm" as="p" tone="info">
                  Required format: First two columns must be "Name" and "Number", mapping starts from third column
                </Text>
                {csvErrorType === 'number_format' && (
                  <Text variant="bodySm" as="p" tone="critical">
                    Number column must contain only integer values (no decimals, and country code without +)
                  </Text>
                )}
                <Text variant="bodySm" as="p" tone="subdued">
                  Example format:  <Link href="https://app.getgabs.com/betaversion/templatedesign/template_design/csv/Contact Upload example.csv" target="_blank">Download Example CSV</Link>
                </Text>
              </TextContainer>
              
              <DropZone 
                onDrop={handleCreateListDropZoneDrop}
                accept=".csv,text/csv"
                type="file"
                disabled={processingCsv}
              >
                {files.length > 0 ? (
                  <div style={{ padding: '16px' }}>
                    <InlineStack alignment="center" gap="200">
                      <div style={{
                        width: '40px',
                        height: '40px',
                        backgroundColor: 'var(--p-color-bg-success)',
                        borderRadius: '4px',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'center'
                      }}>
                        📄
                      </div>
                      <BlockStack gap="100">
                        <Text variant="bodyMd" as="p" fontWeight="semibold">
                          {files[0].name}
                        </Text>
                        <Text variant="bodySm" as="p" tone="subdued">
                          {(files[0].size / 1024).toFixed(1)} KB
                        </Text>
                      </BlockStack>
                    </InlineStack>
                  </div>
                ) : (
                  <DropZone.FileUpload actionTitle={processingCsv ? "Processing CSV..." : "Upload CSV"} actionHint="or drag and drop" />
                )}
              </DropZone>
            </BlockStack>

            {/* CSV Column Mapping */}
            {showColumnMapping && csvColumns.length > 0 && (
              <BlockStack gap="300">
                <Text variant="headingMd" as="h4">
                  Map CSV Columns
                </Text>
                <TextContainer>
                  <Text variant="bodyMd" as="p">
                    Map each CSV column to a variable for your contact list.
                  </Text>
                  <Text variant="bodySm" as="p" tone="subdued">
                    Click "Map" to select a variable, or type a variable name directly.
                  </Text>
                </TextContainer>
                
                <BlockStack gap="200">
                   <div style={{
                backgroundColor: '#008060',
                color: 'white',
                padding: '12px 16px',
                borderRadius: '4px',
                fontSize: '14px',
                
              }}>
                <InlineStack align="space-between" blockAlign="center">
  <Text variant="bodyMd" as="p">
  You can’t map Column1 and Column2 variable because it is always {"{{name}} and {{number}}"}
</Text>

  
</InlineStack>

              </div>
                  
                  {csvColumns.map((column) => (
                    <div key={column.index} style={{
                      border: '1px solid var(--p-color-border)',
                      borderRadius: '8px',
                      padding: '16px',
                      backgroundColor: 'var(--p-color-bg-surface-secondary)'
                    }}>
                      <InlineStack alignment="center" gap="200" blockAlign="start">
                        <div style={{ flex: 1 }}>
                          <Text variant="bodyMd" as="p" fontWeight="semibold">
                            {column.displayName}
                          </Text>
                          <Text variant="bodySm" as="p" tone="subdued">
                            First row value: "{column.firstRowValue}"
                          </Text>
                        </div>
                        
                        <div style={{ flex: 2 }}>
                          <InlineStack alignment="center" gap="200">
                            <TextField
                              label=""
                              placeholder="Select variable using Map button"
                              value={columnMappings[column.index] || ''}
                              onChange={() => {}} 
                              autoComplete="off"
                              readOnly
                              helpText="Click 'Map' button to select a variable"
                            />
                            
                            <Button
                              size="medium"
                              variant="secondary"
                              onClick={() => handleMapColumn(column.index)}
                            >
                              Map
                            </Button>
                            
                            {columnMappings[column.index] && (
                              <Button
                                size="medium"
                                variant="plain"
                                tone="critical"
                                onClick={() => handleRemoveColumnMapping(column.index)}
                              >
                                Clear
                              </Button>
                            )}
                          </InlineStack>
                        </div>
                      </InlineStack>
                    </div>
                  ))}
                </BlockStack>
                
                <TextContainer>
                  <Text variant="bodySm" as="p" tone="subdued">
                    💡 Tip: First two columns (Name, Number) are automatically handled. Map additional columns starting from Column 3 (e.g., "Email" to email, "City" to city)
                  </Text>
                </TextContainer>
              </BlockStack>
            )}

            {/* Ready to Create */}
            {files.length > 0 && listName && (
              <div style={{
                backgroundColor: 'var(--p-color-bg-success-subdued)',
                padding: '16px',
                borderRadius: '8px'
              }}>
                <BlockStack gap="200">
                  <Text variant="bodyMd" as="p">
                    Ready to create contact list "{listName}" with the uploaded CSV file.
                  </Text>
                  {showColumnMapping && Object.keys(columnMappings).length > 0 && (
                    <Text variant="bodySm" as="p" tone="subdued">
                      {Object.keys(columnMappings).length} column(s) mapped: {Object.entries(columnMappings).map(([col, varName]) => `Column ${parseInt(col) + 1} → ${varName}`).join(', ')}
                    </Text>
                  )}
                </BlockStack>
              </div>
            )}
          </BlockStack>
        </Modal.Section>
      </Modal>

      <Modal
        open={createVariableModalActive}
        onClose={handleModalClose}
        title="Create Contact List Variable"
        primaryAction={{
          content: isCreating ? 'Creating...' : 'Create Variable',
          onAction: handleCreateVariables,
          disabled: !variableName || isCreating,
          loading: isCreating
        }}
        secondaryActions={[
          {
            content: 'Cancel',
            onAction: handleModalClose
          }
        ]}
      >
        <Modal.Section>
          <BlockStack gap="400">
            <div style={{
                backgroundColor: '#008060',
                color: 'white',
                padding: '12px 16px',
                borderRadius: '4px',
                fontSize: '14px',
                
              }}>
                <InlineStack align="space-between" blockAlign="center">
  <Text variant="bodyMd" as="p">
    See All Created Variables 👉
  </Text>
  <Button
    size="micro"
    variant="secondary"
    onClick={() => setShowVariablesModal(true)}
    disabled={loadingVariables}
  >
    {loadingVariables ? 'Loading...' : `Variables (${availableVariables.length})`}
  </Button>
</InlineStack>

              </div>
            {/* Contact List Variable Name */}
            <TextField
              id="variable-name-field"
              label="Contact List Variable Name"
              value={variableName}
              onChange={(value) => setVariableName(value.replace(/\s+/g, '_'))}
              placeholder="Enter a variable name for your contact list"
              autoComplete="off"
              helpText="Choose a descriptive name for your contact list (max 20 characters)"
              maxLength={20}
             
            />
            
            
            {/* Success Message */}
            {successMessage && (
              <div style={{
                backgroundColor: '#008060',
                color: 'white',
                padding: '12px 16px',
                borderRadius: '4px',
                fontSize: '14px',
                display: 'flex',
                alignItems: 'center',
                gap: '8px'
              }}>
                <span>✓</span>
                <span>{successMessage}</span>
              </div>
            )}
          </BlockStack>
        </Modal.Section>
      </Modal>

      {/* Variables Modal */}
      <Modal
        open={showVariablesModal}
        onClose={() => {
          setShowVariablesModal(false);
          setVariableSearchQuery('');
        }}
        title="All Available Variables"
        primaryAction={{
          content: 'Close',
          onAction: () => {
            setShowVariablesModal(false);
            setVariableSearchQuery('');
          },
        }}
        size="large"
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" tone="subdued">
              Click on any variable to insert it into your variable name:
            </Text>
            
            {/* Search Field */}
            <TextField
              label="Search Variables"
              value={variableSearchQuery}
              onChange={setVariableSearchQuery}
              placeholder="Type to search variables..."
              clearButton
              onClearButtonClick={() => setVariableSearchQuery('')}
              autoComplete="off"
            />
            
            <InlineStack gap="200" wrap>
              {availableVariables
                .filter(variable => 
                  variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
                  variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
                )
                .map((variable) => (
                  <Button
                    key={variable.value}
                    size="medium"
                    variant="tertiary"
                    onClick={() => {
                      if (mappingColumnIndex !== null) {
                        handleVariableSelect(variable);
                      } else {
                        insertVariable(variable.value);
                        setShowVariablesModal(false);
                        setVariableSearchQuery('');
                      }
                    }}
                  >
                    {variable.label}
                  </Button>
                ))}
            </InlineStack>
            
            {availableVariables.filter(variable => 
              variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
              variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
            ).length === 0 && variableSearchQuery && (
              <Text variant="bodyMd" tone="subdued" alignment="center">
                No variables found matching "{variableSearchQuery}"
              </Text>
            )}
            
            <Text variant="bodySm" tone="subdued">
              You can also create custom variables using the format: {"{{variable_name}}"}
            </Text>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </>
  );
}

export default StaticListPage;
