import {
  Page,
  LegacyCard,
  Button,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  TextField,
  Badge,
  Text,
  InlineStack,
  BlockStack,
  Pagination,
  Icon,
  Tooltip,
  Modal,
  TextContainer,
  Tabs,
  Banner,
} from '@shopify/polaris';
import { PlusIcon, ViewIcon, EditIcon, DeleteIcon } from '@shopify/polaris-icons';
import React, { useState, useCallback, useMemo } from 'react';
import { useNavigate, useLocation } from '@remix-run/react';

function LiveSegmentPage() {
  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
  const navigate = useNavigate();
  const location = useLocation();
  
  // Tab navigation
  const [selectedTab, setSelectedTab] = useState(1); // Live Segment is now tab index 1
  
  const navigationTabs = [
    {
      id: 'static-list',
      content: 'Static List',
      accessibilityLabel: 'Static List tab',
      panelID: 'static-list-panel',
    },
    {
      id: 'live-segment',
      content: 'Live Segment',
      accessibilityLabel: 'Live Segment tab',
      panelID: 'live-segment-panel',
    },
  ];
  
  const handleTabChange = useCallback((selectedTabIndex) => {
    setSelectedTab(selectedTabIndex);
    if (selectedTabIndex === 0) {
      navigate('/app/contact-list/static-list');
    }
  }, [navigate]);
  
  const [selectedSegment, setSelectedSegment] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  const [deleteModalActive, setDeleteModalActive] = useState(false);
  
  // IndexFilters state
  const [itemStrings, setItemStrings] = useState([
    'All',
    'Recently Created',
    'High Engagement',
    'Large Segments',
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['name asc']);
  const [audienceSizeFilter, setAudienceSizeFilter] = useState([0, 100000]);
  const [engagementFilter, setEngagementFilter] = useState('');
  
  const {mode, setMode} = useSetIndexFiltersMode();

  // Pagination state
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 20;

  // Sample live segment data
  const liveSegments = [
    {
      id: '1',
      name: 'High Value Customers',
      description: 'Customers with orders above $500 in last 30 days',
      status: 'Active',
      audienceSize: 2456,
      engagementRate: 85.2,
      lastUpdated: '2024-01-15',
      criteria: {
        orderValue: { min: 500, period: 30 },
        location: ['US', 'CA'],
        tags: ['VIP', 'Premium']
      },
      createdAt: '2024-01-10'
    },
    {
      id: '2',
      name: 'Abandoned Cart Users',
      description: 'Users who abandoned cart in last 7 days',
      status: 'Active',
      audienceSize: 8934,
      engagementRate: 62.8,
      lastUpdated: '2024-01-15',
      criteria: {
        cartAbandonment: { period: 7 },
        excludePurchased: true
      },
      createdAt: '2024-01-12'
    },
    {
      id: '3',
      name: 'New Subscribers',
      description: 'Recently subscribed users in last 14 days',
      status: 'Active',
      audienceSize: 1247,
      engagementRate: 91.5,
      lastUpdated: '2024-01-15',
      criteria: {
        subscriptionDate: { period: 14 },
        source: ['website', 'social']
      },
      createdAt: '2024-01-13'
    },
    {
      id: '4',
      name: 'Inactive Customers',
      description: 'No purchase in last 90 days',
      status: 'Inactive',
      audienceSize: 5678,
      engagementRate: 23.4,
      lastUpdated: '2024-01-10',
      criteria: {
        lastPurchase: { period: 90, operator: 'greater_than' },
        previousOrders: { min: 1 }
      },
      createdAt: '2024-01-05'
    },
    {
      id: '5',
      name: 'Birthday Campaign',
      description: 'Customers with birthdays this month',
      status: 'Active',
      audienceSize: 892,
      engagementRate: 78.9,
      lastUpdated: '2024-01-15',
      criteria: {
        birthday: { month: 'current' },
        optedIn: true
      },
      createdAt: '2024-01-01'
    }
  ];

  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => {},
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      {
        type: 'rename',
        onAction: () => {},
        onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => {
            if (idx === index) {
              return value;
            }
            return item.content;
          });
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      {
        type: 'duplicate',
        onPrimaryAction: async (value) => {
          await sleep(1);
          duplicateView(value);
          return true;
        },
      },
      {
        type: 'edit',
      },
      {
        type: 'delete',
        onPrimaryAction: async () => {
          await sleep(1);
          deleteView(index);
          return true;
        },
      },
    ],
  }));

  const filteredSegments = useMemo(() => {
    let filtered = liveSegments;
    
    // Filter by tab selection
    if (selected === 1) { // Recently Created
      const sevenDaysAgo = new Date();
      sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
      filtered = filtered.filter(segment => new Date(segment.createdAt) >= sevenDaysAgo);
    } else if (selected === 2) { // High Engagement
      filtered = filtered.filter(segment => segment.engagementRate >= 80);
    } else if (selected === 3) { // Large Segments
      filtered = filtered.filter(segment => segment.audienceSize >= 10000);
    }
    
    return filtered.filter((segment) => {
      const matchesSearch = segment.name.toLowerCase().includes(queryValue.toLowerCase()) ||
                           segment.description.toLowerCase().includes(queryValue.toLowerCase());
      const matchesAudienceSize = segment.audienceSize >= audienceSizeFilter[0] && 
                                 segment.audienceSize <= audienceSizeFilter[1];
      const matchesEngagement = engagementFilter === '' || 
                               segment.engagementRate >= parseFloat(engagementFilter);
      
      return matchesSearch && matchesAudienceSize && matchesEngagement;
    });
  }, [liveSegments, queryValue, audienceSizeFilter, engagementFilter, selected]);

  const sortedSegments = useMemo(() => {
    const sorted = [...filteredSegments];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');
    
    sorted.sort((a, b) => {
      let aValue, bValue;
      
      switch (sortKey) {
        case 'name':
          aValue = a.name.toLowerCase();
          bValue = b.name.toLowerCase();
          break;
        case 'status':
          aValue = a.status.toLowerCase();
          bValue = b.status.toLowerCase();
          break;
        case 'audience':
          aValue = a.audienceSize;
          bValue = b.audienceSize;
          break;
        case 'engagement':
          aValue = a.engagementRate;
          bValue = b.engagementRate;
          break;
        case 'updated':
          aValue = new Date(a.lastUpdated);
          bValue = new Date(b.lastUpdated);
          break;
        default:
          return 0;
      }
      
      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });
    
    return sorted;
  }, [filteredSegments, sortSelected]);

  const paginatedSegments = useMemo(() => {
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    return sortedSegments.slice(startIndex, endIndex);
  }, [sortedSegments, currentPage, itemsPerPage]);

  const handlePreviousPage = useCallback(() => {
    setCurrentPage(prev => Math.max(1, prev - 1));
  }, []);

  const handleNextPage = useCallback(() => {
    const totalPages = Math.ceil(sortedSegments.length / itemsPerPage);
    setCurrentPage(prev => Math.min(totalPages, prev + 1));
  }, [sortedSegments.length, itemsPerPage]);

  const totalItems = sortedSegments.length;
  const totalPages = Math.ceil(totalItems / itemsPerPage);
  const startItem = (currentPage - 1) * itemsPerPage + 1;
  const endItem = Math.min(currentPage * itemsPerPage, totalItems);
  const hasNext = currentPage < totalPages;
  const hasPrevious = currentPage > 1;

  // Sort options
  const sortOptions = [
    {label: 'Segment Name', value: 'name asc', directionLabel: 'A-Z'},
    {label: 'Segment Name', value: 'name desc', directionLabel: 'Z-A'},
    {label: 'Status', value: 'status asc', directionLabel: 'A-Z'},
    {label: 'Status', value: 'status desc', directionLabel: 'Z-A'},
    {label: 'Audience Size', value: 'audience asc', directionLabel: 'Low to High'},
    {label: 'Audience Size', value: 'audience desc', directionLabel: 'High to Low'},
    {label: 'Engagement Rate', value: 'engagement asc', directionLabel: 'Low to High'},
    {label: 'Engagement Rate', value: 'engagement desc', directionLabel: 'High to Low'},
    {label: 'Last Updated', value: 'updated asc', directionLabel: 'Oldest First'},
    {label: 'Last Updated', value: 'updated desc', directionLabel: 'Newest First'},
  ];

  const handlePreviewClick = useCallback((segment) => {
    // Navigate to live segment contacts page (name-based for now)
    navigate(`/app/contact-list/contacts/live-segment/${encodeURIComponent(segment.name)}`);
  }, [navigate]);

  const handleDeleteClick = useCallback((segment) => {
    setSelectedSegment(segment);
    setDeleteModalActive(true);
  }, []);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setDeleteModalActive(false);
    setSelectedSegment(null);
  }, []);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);
  
  const handleAudienceSizeChange = useCallback((value) => {
    setAudienceSizeFilter(value);
  }, []);
  
  const handleEngagementChange = useCallback((value) => {
    setEngagementFilter(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleAudienceSizeRemove = useCallback(() => setAudienceSizeFilter([0, 100000]), []);
  const handleEngagementRemove = useCallback(() => setEngagementFilter(''), []);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleAudienceSizeRemove();
    handleEngagementRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleAudienceSizeRemove, handleEngagementRemove, handleQueryValueRemove]);

  const onHandleCancel = () => {};
  const onHandleSave = async () => {
    await sleep(1);
    return true;
  };

  const primaryAction = selected === 0
    ? {
        type: 'save-as',
        onAction: onCreateNewView,
        disabled: false,
        loading: false,
      }
    : {
        type: 'save',
        onAction: onHandleSave,
        disabled: false,
        loading: false,
      };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'Active': { tone: 'success' },
      'Inactive': { tone: 'subdued' },
      'Draft': { tone: 'attention' },
      'Error': { tone: 'critical' }
    };
    
    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };

  const filters = [
    {
      key: 'audienceSize',
      label: 'Audience Size',
      filter: (
        <RangeSlider
          label="Audience size"
          labelHidden
          value={audienceSizeFilter || [0, 100000]}
          suffix=" contacts"
          output
          min={0}
          max={100000}
          step={100}
          onChange={handleAudienceSizeChange}
        />
      ),
    },
    {
      key: 'engagement',
      label: 'Min Engagement Rate',
      filter: (
        <TextField
          label="Minimum engagement rate"
          value={engagementFilter}
          onChange={handleEngagementChange}
          autoComplete="off"
          labelHidden
          suffix="%"
          type="number"
        />
      ),
      shortcut: true,
    },
  ];

  const appliedFilters = [];
  if (audienceSizeFilter && (audienceSizeFilter[0] !== 0 || audienceSizeFilter[1] !== 100000)) {
    appliedFilters.push({
      key: 'audienceSize',
      label: disambiguateLabel('audienceSize', audienceSizeFilter),
      onRemove: handleAudienceSizeRemove,
    });
  }
  if (!isEmpty(engagementFilter)) {
    appliedFilters.push({
      key: 'engagement',
      label: disambiguateLabel('engagement', engagementFilter),
      onRemove: handleEngagementRemove,
    });
  }

  const resourceName = {
    singular: 'segment',
    plural: 'segments',
  };

  const {selectedResources, allResourcesSelected, handleSelectionChange} =
    useIndexResourceState(paginatedSegments);

  function disambiguateLabel(key, value) {
    switch (key) {
      case 'audienceSize':
        return `Audience size: ${value[0].toLocaleString()}-${value[1].toLocaleString()}`;
      case 'engagement':
        return `Min engagement: ${value}%`;
      case 'status':
        return value.map((val) => `Status: ${val.charAt(0).toUpperCase() + val.slice(1)}`).join(', ');
      default:
        return value;
    }
  }

  function isEmpty(value) {
    if (Array.isArray(value)) {
      return value.length === 0;
    } else {
      return value === '' || value == null;
    }
  }

  const rowMarkup = paginatedSegments.map(
    (segment, index) => (
      <IndexTable.Row
        id={segment.id}
        key={segment.id}
        selected={selectedResources.includes(segment.id)}
        position={index}
      >
        <IndexTable.Cell>
          <BlockStack gap="100">
            <Text variant="bodyMd" fontWeight="semibold" as="span">
              {segment.name}
            </Text>
            <Text variant="bodySm" as="span" tone="subdued">
              {segment.description}
            </Text>
          </BlockStack>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span" numeric>
            {segment.audienceSize.toLocaleString()}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" as="span">
            {segment.lastUpdated}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end" gap="200">
            <Tooltip content="View segment details">
              <Button
                size="slim"
                onClick={(event) => {
                  event.stopPropagation();
                  handlePreviewClick(segment);
                }}
              >
                <Icon source={ViewIcon} />
              </Button>
            </Tooltip>
            <Tooltip content="Delete segment">
              <Button
                size="slim"
                tone="critical"
                onClick={(event) => {
                  event.stopPropagation();
                  handleDeleteClick(segment);
                }}
              >
                <Icon source={DeleteIcon} />
              </Button>
            </Tooltip>
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  return (
    <>
      <Page 
        title="Contact List" 
        subtitle="Manage your customer segments and contact lists"
        fullWidth
        // primaryAction={{
        //   content: (
        //     <InlineStack alignment="center">
        //       <Icon source={PlusIcon} />
        //       <Text as="span">Create Live Segment</Text>
        //     </InlineStack>
        //   ),
        //   onAction: () => console.log('Create live segment clicked')
        // }}
      >
        <BlockStack gap="400">
          <Tabs tabs={navigationTabs} selected={selectedTab} onSelect={handleTabChange}>
            <div id="live-segment-panel">
              <Banner title="Coming Soon" onDismiss={() => {}}>
      <p>This feature is coming soon. now showing demo data</p>
    </Banner>
        <LegacyCard>
          <IndexFilters
            sortOptions={sortOptions}
            sortSelected={sortSelected}
            queryValue={queryValue}
            queryPlaceholder="Search segments..."
            onQueryChange={handleFiltersQueryChange}
            onQueryClear={() => setQueryValue('')}
            onSort={setSortSelected}
            cancelAction={{
              onAction: onHandleCancel,
              disabled: false,
              loading: false,
            }}
            tabs={tabs}
            selected={selected}
            onSelect={setSelected}
            canCreateNewView
            onCreateNewView={onCreateNewView}
            filters={filters}
            appliedFilters={appliedFilters}
            onClearAll={handleFiltersClearAll}
            mode={mode}
            setMode={setMode}
          />

           
          <IndexTable
            resourceName={resourceName}
            itemCount={paginatedSegments.length}
            selectedItemsCount={
              allResourcesSelected ? 'All' : selectedResources.length
            }
            onSelectionChange={handleSelectionChange}
            headings={[
              {title: 'Segment Name'},
              {title: 'Audience Size'},
              {title: 'Last Updated'},
              {title: 'Actions', alignment: 'end'},
            ]}
            emptyState={
              <div style={{ textAlign: 'center', padding: '60px 20px' }}>
                <Text variant="headingMd" as="h3">
                  No live segments found
                </Text>
                <Text variant="bodyMd" as="p" tone="subdued">
                  {filteredSegments.length === 0 && liveSegments.length > 0 
                    ? 'Try adjusting your filters to see more results.'
                    : 'Create your first live segment to start targeting customers dynamically.'
                  }
                </Text>
              </div>
            }
          >
            {rowMarkup}
          </IndexTable>
          
          {totalItems > itemsPerPage && (
            <div
              style={{
                maxWidth: '700px',
                margin: 'auto',
                border: '1px solid var(--p-color-border)',
                marginTop: '16px'
              }}
            >
              <Pagination
                onPrevious={hasPrevious ? handlePreviousPage : undefined}
                onNext={hasNext ? handleNextPage : undefined}
                type="table"
                hasNext={hasNext}
                hasPrevious={hasPrevious}
                label={`${startItem}-${endItem} of ${totalItems} segments`}
              />
            </div>
          )}
        </LegacyCard>
            </div>
          </Tabs>
        </BlockStack>
      </Page>

      {/* Preview Modal */}
      {selectedSegment && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Live Segment: ${selectedSegment.name}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
          secondaryActions={[
            {
              content: 'Edit Segment',
              onAction: () => {
                console.log('Edit segment:', selectedSegment.id);
                handleModalClose();
              }
            }
          ]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <BlockStack gap="200">
                <Text variant="headingMd" as="h3">
                  {selectedSegment.name}
                </Text>
                <Text variant="bodyLg" as="p">
                  {selectedSegment.description}
                </Text>
                <InlineStack gap="200">
                  {getStatusBadge(selectedSegment.status)}
                </InlineStack>
              </BlockStack>
              
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Audience Size:</strong> {selectedSegment.audienceSize.toLocaleString()} contacts
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Engagement Rate:</strong> {selectedSegment.engagementRate}%
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Last Updated:</strong> {selectedSegment.lastUpdated}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Created:</strong> {selectedSegment.createdAt}
                </Text>
              </TextContainer>

              <BlockStack gap="200">
                <Text variant="headingMd" as="h4">
                  Segment Criteria
                </Text>
                <div style={{ 
                  backgroundColor: 'var(--p-color-bg-surface-secondary)', 
                  padding: '16px', 
                  borderRadius: '8px' 
                }}>
                  <pre style={{ 
                    fontFamily: 'monospace', 
                    fontSize: '12px', 
                    margin: 0,
                    whiteSpace: 'pre-wrap'
                  }}>
                    {JSON.stringify(selectedSegment.criteria, null, 2)}
                  </pre>
                </div>
              </BlockStack>
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}

      {/* Delete Confirmation Modal */}
      {selectedSegment && (
        <Modal
          open={deleteModalActive}
          onClose={handleModalClose}
          title="Delete Live Segment"
          primaryAction={{
            content: 'Delete',
            destructive: true,
            onAction: () => {
              console.log('Delete segment:', selectedSegment.id);
              handleModalClose();
            }
          }}
          secondaryActions={[
            {
              content: 'Cancel',
              onAction: handleModalClose
            }
          ]}
        >
          <Modal.Section>
            <TextContainer>
              <Text variant="bodyMd" as="p">
                Are you sure you want to delete the segment "{selectedSegment.name}"? 
                This action cannot be undone and will affect any active campaigns using this segment.
              </Text>
              <Text variant="bodyMd" as="p" tone="subdued">
                Current audience size: {selectedSegment.audienceSize.toLocaleString()} contacts
              </Text>
            </TextContainer>
          </Modal.Section>
        </Modal>
      )}
    </>
  );
}

export default LiveSegmentPage;
