import {
  Page,
  LegacyCard,
  Button,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  TextField,
  Badge,
  Text,
  InlineStack,
  BlockStack,
  Pagination,
  Icon,
  Tooltip,
  Modal,
  TextContainer,
} from '@shopify/polaris';
import { ArrowLeftIcon, EditIcon } from '@shopify/polaris-icons';
import React, { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate, useParams, useSearchParams } from '@remix-run/react';
import { fetchStaticContactListContacts, updateSingleContact, addSingleContact } from '../utils/contactsApi';

function StaticListContactsPage() {
  const navigate = useNavigate();
  const { name } = useParams();
  const [searchParams] = useSearchParams();
  const decodedName = decodeURIComponent(name);
  const listId = searchParams.get('id');

  const [selectedContact, setSelectedContact] = useState(null);
  const [editModalActive, setEditModalActive] = useState(false);
  const [savingEdit, setSavingEdit] = useState(false);
  const [editName, setEditName] = useState('');
  const [editCountryCode, setEditCountryCode] = useState('');
  const [editPhone, setEditPhone] = useState('');
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [addModalActive, setAddModalActive] = useState(false);
  const [addingContact, setAddingContact] = useState(false);
  const [addName, setAddName] = useState('');
  const [addCountryCode, setAddCountryCode] = useState('');
  const [addPhone, setAddPhone] = useState('');

  const [itemStrings, setItemStrings] = useState([
    'All',
    'Recently Added',
    'High Value',
    'Active',
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['updated desc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [locationFilter, setLocationFilter] = useState('');
  const [orderValueFilter, setOrderValueFilter] = useState([0, 10000]);

  const {mode, setMode} = useSetIndexFiltersMode();

  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 20;

  const [contacts, setContacts] = useState([]);
  const [totalItems, setTotalItems] = useState(0);
  const [serverPagination, setServerPagination] = useState({ current_page: 1, last_page: 1, per_page: itemsPerPage });

  useEffect(() => {
    const loadContacts = async () => {
      try {
        setLoading(true);
        setError(null);
        const response = await fetchStaticContactListContacts({ page: currentPage, perPageData: itemsPerPage, contactListId: Number(listId || 0) });
        if (response.status && response.WhatsAppBusinessContact) {
          const { data, total, per_page, current_page, last_page } = response.WhatsAppBusinessContact;
          const transformed = data.map(item => ({
            id: item.id.toString(),
            name: item.name || 'Unknown',
            number: item.number || '',
            createdOn: item.created_at ? new Date(item.created_at).toISOString().slice(0,10) : '',
          }));
          setContacts(transformed);
          setTotalItems(total || transformed.length);
          setServerPagination({ current_page: current_page || 1, last_page: last_page || 1, per_page: per_page || itemsPerPage });
        } else {
          // fallback to empty
          setContacts([]);
          setTotalItems(0);
        }
      } catch (err) {
        setError(err.message);
        setContacts([]);
        setTotalItems(0);
      } finally {
        setLoading(false);
      }
    };
    loadContacts();
  }, [currentPage, itemsPerPage, listId]);

  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => {},
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      { type: 'rename', onAction: () => {}, onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => (idx === index ? value : item.content));
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      { type: 'duplicate', onPrimaryAction: async (value) => { await sleep(1); duplicateView(value); return true; } },
      { type: 'edit' },
      { type: 'delete', onPrimaryAction: async () => { await sleep(1); deleteView(index); return true; } },
    ],
  }));

  const filteredContacts = useMemo(() => {
    let filtered = contacts;
    return filtered.filter((contact) => {
      const matchesSearch = (contact.name || '').toLowerCase().includes(queryValue.toLowerCase()) ||
        (contact.number || '').includes(queryValue);
      return matchesSearch;
    });
  }, [contacts, queryValue]);

  const sortedContacts = useMemo(() => {
    const sorted = [...filteredContacts];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');
    sorted.sort((a, b) => {
      let aValue, bValue;
      switch (sortKey) {
        case 'name': aValue = (a.name || '').toLowerCase(); bValue = (b.name || '').toLowerCase(); break;
        default: return 0;
      }
      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });
    return sorted;
  }, [filteredContacts, sortSelected]);

  const paginatedContacts = useMemo(() => {
   
    return sortedContacts;
  }, [sortedContacts]);

  const totalPages = serverPagination.last_page || Math.ceil((totalItems || 0) / (serverPagination.per_page || itemsPerPage));
  const startItem = (serverPagination.current_page - 1) * (serverPagination.per_page || itemsPerPage) + 1;
  const endItem = Math.min(serverPagination.current_page * (serverPagination.per_page || itemsPerPage), totalItems);
  const hasNext = serverPagination.current_page < (serverPagination.last_page || totalPages);
  const hasPrevious = serverPagination.current_page > 1;

  const handlePreviousPage = useCallback(() => { setCurrentPage(prev => Math.max(1, prev - 1)); }, []);
  const handleNextPage = useCallback(() => {
    setCurrentPage(prev => Math.min((serverPagination.last_page || 1), prev + 1));
  }, [serverPagination.last_page]);

  const sortOptions = [
    {label: 'Name', value: 'name asc', directionLabel: 'A-Z'},
    {label: 'Name', value: 'name desc', directionLabel: 'Z-A'},
    {lable: 'Updated On', value: 'updated asc', directionLabel: 'Oldest First'},
    {lable: 'Updated On', value: 'updated desc', directionLabel: 'Newest First'},
  ];

  const handleEditClick = useCallback((contact) => { setSelectedContact(contact); setEditModalActive(true); }, []);
  const handleModalClose = useCallback(() => { setEditModalActive(false); setSelectedContact(null); }, []);

  useEffect(() => {
    if (selectedContact) {
      const originalNumber = selectedContact.number || '';
      // naive split: if number begins with country code like 91xxxxxxxxxx or +91xxxxxxxxxx
      let derivedCountry = '';
      let derivedPhone = '';
      const digitsOnly = (originalNumber || '').replace(/\D/g, '');
      if (digitsOnly.length > 10) {
        derivedCountry = digitsOnly.slice(0, digitsOnly.length - 10);
        derivedPhone = digitsOnly.slice(-10);
      } else {
        derivedPhone = digitsOnly;
      }
      setEditName(selectedContact.name || '');
      setEditCountryCode(derivedCountry);
      setEditPhone(derivedPhone);
    } else {
      setEditName('');
      setEditCountryCode('');
      setEditPhone('');
    }
  }, [selectedContact]);

  const handleEditSave = useCallback(async () => {
    if (!selectedContact) return;
    try {
      setSavingEdit(true);
      const payload = {
        tag: Number(listId || 0),
        contactid: Number(selectedContact.id),
        allNewTagsAdd: [],
        name: editName,
        countrycode: editCountryCode,
        phone: editPhone,
      };
      const res = await updateSingleContact(payload);
      // optimistic UI update
      const updatedNumber = `${editCountryCode ? `+${editCountryCode} ` : ''}${editPhone}`.trim();
      setContacts(prev => prev.map(c => c.id === selectedContact.id ? { ...c, name: editName, number: updatedNumber } : c));
      setEditModalActive(false);
      setSelectedContact(null);
    } catch (e) {
      console.error('Failed to update contact', e);
      setError(e?.message || 'Failed to update contact');
    } finally {
      setSavingEdit(false);
    }
  }, [selectedContact, listId, editName, editCountryCode, editPhone]);

  const getStatusBadge = (status) => {
    const statusConfig = { 'Active': { tone: 'success' }, 'Inactive': { tone: 'subdued' }, 'Pending': { tone: 'attention' }, 'Blocked': { tone: 'critical' } };
    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };

  const filters = [];

  const appliedFilters = [];

  const resourceName = { singular: 'contact', plural: 'contacts' };
  const {selectedResources, allResourcesSelected, handleSelectionChange} = useIndexResourceState(paginatedContacts);

  const tableHeadings = [
    {title: 'Name'},
    {title: 'Number'},
    {title: 'Created On'},
    {title: 'Actions', alignment: 'end'},
  ];

  const rowMarkup = paginatedContacts.map((contact, index) => (
    <IndexTable.Row id={contact.id} key={contact.id} selected={selectedResources.includes(contact.id)} position={index}>
      <IndexTable.Cell>
        <Text variant="bodyMd" fontWeight="semibold" as="span">{contact.name}</Text>
      </IndexTable.Cell>
      <IndexTable.Cell>
        <Text variant="bodyMd" as="span">{contact.number}</Text>
      </IndexTable.Cell>
      <IndexTable.Cell>
        <Text variant="bodyMd" as="span">{contact.createdOn}</Text>
      </IndexTable.Cell>
      <IndexTable.Cell>
        <InlineStack align="end" gap="200">
          <Tooltip content="Edit contact">
            <Button size="slim" onClick={(event) => { event.stopPropagation(); handleEditClick(contact); }}>
              <Icon source={EditIcon} />
            </Button>
          </Tooltip>
        </InlineStack>
      </IndexTable.Cell>
    </IndexTable.Row>
  ));

  return (
    <>
      <Page 
        title={`Static List: ${decodedName}`}
        subtitle={`${totalItems} contacts in this list`}
        fullWidth
        breadcrumbs={[{ content: 'Contact Lists', onAction: () => navigate('/app/contact-list/static-list') }]}
        primaryAction={{ content: (<InlineStack alignment="center"><Text as="span">Add single contact</Text></InlineStack>), onAction: () => setAddModalActive(true) }}
        secondaryActions={[{ content: 'Back to Lists', icon: ArrowLeftIcon, onAction: () => navigate('/app/contact-list/static-list') }]}
      >
        <BlockStack gap="400">
          <LegacyCard>
            <IndexFilters
              sortOptions={sortOptions}
              sortSelected={sortSelected}
              queryValue={queryValue}
              queryPlaceholder="Search contacts..."
              onQueryChange={setQueryValue}
              onQueryClear={() => setQueryValue('')}
              onSort={setSortSelected}
              cancelAction={{ onAction: () => {}, disabled: false, loading: false }}
              tabs={tabs}
              selected={selected}
              onSelect={setSelected}
              canCreateNewView
              onCreateNewView={onCreateNewView}
              filters={filters}
              appliedFilters={appliedFilters}
              onClearAll={() => { setQueryValue(''); }}
              mode={mode}
              setMode={setMode}
            />
            <IndexTable
              resourceName={resourceName}
              itemCount={paginatedContacts.length}
              selectedItemsCount={allResourcesSelected ? 'All' : selectedResources.length}
              onSelectionChange={handleSelectionChange}
              headings={tableHeadings}
              emptyState={<div style={{ textAlign: 'center', padding: '60px 20px' }}><Text variant="headingMd" as="h3">{loading ? 'Loading contacts...' : error ? 'Error loading contacts' : 'No contacts found'}</Text><Text variant="bodyMd" as="p" tone="subdued">{loading ? 'Please wait while we fetch the contacts.' : error ? `Error: ${error}` : 'Try adjusting your filters to see more results.'}</Text></div>}
            >
              {rowMarkup}
            </IndexTable>
            {totalItems > itemsPerPage && (
              <div style={{ maxWidth: '700px', margin: 'auto', border: '1px solid var(--p-color-border)', marginTop: '16px' }}>
                <Pagination onPrevious={hasPrevious ? handlePreviousPage : undefined} onNext={hasNext ? handleNextPage : undefined} type="table" hasNext={hasNext} hasPrevious={hasPrevious} label={`${startItem}-${endItem} of ${totalItems} contacts`} />
              </div>
            )}
          </LegacyCard>
        </BlockStack>
      </Page>

      <Modal
        open={addModalActive}
        onClose={() => { setAddModalActive(false); }}
        title="Add contact"
        primaryAction={{
          content: 'Add contact',
          loading: addingContact,
          onAction: async () => {
            try {
              setAddingContact(true);
              setError(null);
              const digitsInLabel = (addCountryCode || '').match(/\d+/g);
              const countryDigits = digitsInLabel ? digitsInLabel.join('') : (addCountryCode || '').replace(/\D/g, '');
              const phoneDigits = (addPhone || '').replace(/\D/g, '');
              const fullNumber = `${countryDigits}${phoneDigits}`;
              const payload = {
                tag: Number(listId || 0),
                number: fullNumber,
                messageType: 'on',
                countrycode: addCountryCode,
                phone: phoneDigits,
                name: addName,
              };
              await addSingleContact(payload);
              // Optimistic UI update
              const displayNumber = `${countryDigits ? `+${countryDigits} ` : ''}${phoneDigits}`.trim();
              setContacts(prev => [{ id: `new-${Date.now()}`, name: addName, number: displayNumber, createdOn: new Date().toISOString().slice(0,10) }, ...prev]);
              setTotalItems(prev => (prev || 0) + 1);
              setAddModalActive(false);
              setAddName('');
              setAddCountryCode('');
              setAddPhone('');
            } catch (e) {
              console.error('Failed to add contact', e);
              setError(e?.message || 'Failed to add contact');
            } finally {
              setAddingContact(false);
            }
          },
        }}
        secondaryActions={[{ content: 'Cancel', onAction: () => setAddModalActive(false) }]}
      >
        <Modal.Section>
          <BlockStack gap="400">
            <TextField
              label="Name"
              value={addName}
              onChange={setAddName}
              autoComplete="off"
            />
            <InlineStack gap="200">
              <TextField
                label="Country Code"
                value={addCountryCode}
                onChange={setAddCountryCode}
                autoComplete="off"
                placeholder="India (+91)"
              />
              <TextField
                label="Phone"
                value={addPhone}
                onChange={setAddPhone}
                autoComplete="off"
                placeholder="xxxxxxx9999"
              />
            </InlineStack>
          </BlockStack>
        </Modal.Section>
      </Modal>

      {selectedContact && (
        <Modal
          open={editModalActive}
          onClose={handleModalClose}
          title={`Edit Contact: ${selectedContact.name}`}
          primaryAction={{ content: 'Update', onAction: handleEditSave, loading: savingEdit }}
          secondaryActions={[{ content: 'Cancel', onAction: handleModalClose }]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <TextField
                label="Name"
                value={editName}
                onChange={setEditName}
                autoComplete="off"
              />
              <InlineStack gap="200">
                <TextField
                  label="Country Code"
                  value={editCountryCode}
                  onChange={setEditCountryCode}
                  autoComplete="off"
                  placeholder="91"
                />
                <TextField
                  label="Phone"
                  value={editPhone}
                  onChange={setEditPhone}
                  autoComplete="off"
                  placeholder="7678696517"
                />
              </InlineStack>
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}
    </>
  );
}

export default StaticListContactsPage;


