import {
  Page,
  LegacyCard,
  Button,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  TextField,
  Badge,
  Text,
  InlineStack,
  BlockStack,
  Pagination,
  Icon,
  Tooltip,
  Modal,
  TextContainer,
} from '@shopify/polaris';
import { ArrowLeftIcon, EditIcon, RefreshIcon } from '@shopify/polaris-icons';
import React, { useState, useCallback, useMemo } from 'react';
import { useNavigate, useParams } from '@remix-run/react';

function LiveSegmentContactsPage() {
  const navigate = useNavigate();
  const { name } = useParams();
  const decodedName = decodeURIComponent(name);

  const [selectedContact, setSelectedContact] = useState(null);
  const [editModalActive, setEditModalActive] = useState(false);
  const [refreshing, setRefreshing] = useState(false);

  const [itemStrings, setItemStrings] = useState([
    'All',
    'Recently Added',
    'High Engagement',
    'Active Orders',
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['name asc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [locationFilter, setLocationFilter] = useState('');
  const [engagementFilter, setEngagementFilter] = useState([0, 100]);

  const {mode, setMode} = useSetIndexFiltersMode();

  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 20;

  // Sample data (same shape as unified page)
  const contacts = [
    { id: '1', customerName: 'John Smith', phone: '+1 (555) 123-4567', email: 'john.smith@email.com', status: 'Active', location: 'New York, NY', orderValue: 1250.00, lastOrder: '2024-01-15', joinDate: '2023-08-10', tags: ['VIP', 'Newsletter'], totalOrders: 8, company: 'Tech Corp', engagementRate: 92.5 },
    { id: '2', customerName: 'Sarah Johnson', phone: '+1 (555) 234-5678', email: 'sarah.j@email.com', status: 'Active', location: 'Los Angeles, CA', orderValue: 890.50, lastOrder: '2024-01-14', joinDate: '2023-09-15', tags: ['Regular'], totalOrders: 5, company: 'Design Studio', engagementRate: 87.3 },
    { id: '3', customerName: 'Mike Wilson', phone: '+1 (555) 345-6789', email: 'mike.wilson@email.com', status: 'Inactive', location: 'Chicago, IL', orderValue: 2340.75, lastOrder: '2023-12-20', joinDate: '2023-06-22', tags: ['VIP', 'Bulk Buyer'], totalOrders: 12, company: 'Wilson & Co', engagementRate: 78.9 },
  ];

  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => {},
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      { type: 'rename', onAction: () => {}, onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => (idx === index ? value : item.content));
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      { type: 'duplicate', onPrimaryAction: async (value) => { await sleep(1); duplicateView(value); return true; } },
      { type: 'edit' },
      { type: 'delete', onPrimaryAction: async () => { await sleep(1); deleteView(index); return true; } },
    ],
  }));

  const filteredContacts = useMemo(() => {
    let filtered = contacts;
    if (selected === 1) {
      const sevenDaysAgo = new Date();
      sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
      filtered = filtered.filter(contact => new Date(contact.joinDate) >= sevenDaysAgo);
    } else if (selected === 2) {
      filtered = filtered.filter(contact => contact.engagementRate >= 90);
    } else if (selected === 3) {
      const thirtyDaysAgo = new Date();
      thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
      filtered = filtered.filter(contact => new Date(contact.lastOrder) >= thirtyDaysAgo);
    }

    return filtered.filter((contact) => {
      const matchesSearch = contact.customerName.toLowerCase().includes(queryValue.toLowerCase()) ||
        contact.email.toLowerCase().includes(queryValue.toLowerCase()) ||
        contact.phone.includes(queryValue) ||
        contact.company.toLowerCase().includes(queryValue.toLowerCase());
      const matchesStatus = statusFilter.length === 0 || statusFilter.includes(contact.status.toLowerCase());
      const matchesLocation = locationFilter === '' || contact.location.toLowerCase().includes(locationFilter.toLowerCase());
      const matchesMetric = contact.engagementRate >= engagementFilter[0] && contact.engagementRate <= engagementFilter[1];
      return matchesSearch && matchesStatus && matchesLocation && matchesMetric;
    });
  }, [contacts, queryValue, statusFilter, locationFilter, engagementFilter, selected]);

  const sortedContacts = useMemo(() => {
    const sorted = [...filteredContacts];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');
    sorted.sort((a, b) => {
      let aValue, bValue;
      switch (sortKey) {
        case 'name': aValue = a.customerName.toLowerCase(); bValue = b.customerName.toLowerCase(); break;
        case 'email': aValue = a.email.toLowerCase(); bValue = b.email.toLowerCase(); break;
        case 'engagement': aValue = a.engagementRate; bValue = b.engagementRate; break;
        default: return 0;
      }
      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });
    return sorted;
  }, [filteredContacts, sortSelected]);

  const paginatedContacts = useMemo(() => {
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    return sortedContacts.slice(startIndex, endIndex);
  }, [sortedContacts, currentPage, itemsPerPage]);

  const totalItems = sortedContacts.length;
  const totalPages = Math.ceil(totalItems / itemsPerPage);
  const startItem = (currentPage - 1) * itemsPerPage + 1;
  const endItem = Math.min(currentPage * itemsPerPage, totalItems);
  const hasNext = currentPage < totalPages;
  const hasPrevious = currentPage > 1;

  const handlePreviousPage = useCallback(() => { setCurrentPage(prev => Math.max(1, prev - 1)); }, []);
  const handleNextPage = useCallback(() => { const totalPages = Math.ceil(sortedContacts.length / itemsPerPage); setCurrentPage(prev => Math.min(totalPages, prev + 1)); }, [sortedContacts.length, itemsPerPage]);

  const sortOptions = [
    {label: 'Customer Name', value: 'name asc', directionLabel: 'A-Z'},
    {label: 'Customer Name', value: 'name desc', directionLabel: 'Z-A'},
    {label: 'Email', value: 'email asc', directionLabel: 'A-Z'},
    {label: 'Email', value: 'email desc', directionLabel: 'Z-A'},
    {label: 'Engagement Rate', value: 'engagement asc', directionLabel: 'Low to High'},
    {label: 'Engagement Rate', value: 'engagement desc', directionLabel: 'High to Low'},
  ];

  const handleEditClick = useCallback((contact) => { setSelectedContact(contact); setEditModalActive(true); }, []);
  const handleRefreshSegment = useCallback(async () => { setRefreshing(true); setTimeout(() => setRefreshing(false), 2000); }, []);
  const handleModalClose = useCallback(() => { setEditModalActive(false); setSelectedContact(null); }, []);

  const getStatusBadge = (status) => {
    const statusConfig = { 'Active': { tone: 'success' }, 'Inactive': { tone: 'subdued' }, 'Pending': { tone: 'attention' }, 'Blocked': { tone: 'critical' } };
    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };

  const getEngagementBadge = (rate) => {
    if (rate >= 90) return <Badge tone="success">{rate}%</Badge>;
    if (rate >= 70) return <Badge tone="info">{rate}%</Badge>;
    if (rate >= 50) return <Badge tone="attention">{rate}%</Badge>;
    return <Badge tone="critical">{rate}%</Badge>;
  };

  const filters = [
    { key: 'status', label: 'Status', filter: (
        <ChoiceList title="Status" titleHidden choices={[{label: 'Active', value: 'active'},{label: 'Inactive', value: 'inactive'},{label: 'Pending', value: 'pending'},{label: 'Blocked', value: 'blocked'},]} selected={statusFilter || []} onChange={setStatusFilter} allowMultiple />
      ), shortcut: true, },
    { key: 'location', label: 'Location', filter: (
        <TextField label="Location" value={locationFilter} onChange={setLocationFilter} autoComplete="off" labelHidden />
      ), shortcut: true },
    { key: 'engagement', label: 'Engagement Rate', filter: (
        <RangeSlider label="Engagement rate" labelHidden value={engagementFilter || [0, 100]} suffix="%" output min={0} max={100} step={5} onChange={setEngagementFilter} />
      ) },
  ];

  const appliedFilters = [];
  if (statusFilter.length > 0) appliedFilters.push({ key: 'status', label: `Status: ${statusFilter.map(s => s.charAt(0).toUpperCase() + s.slice(1)).join(', ')}`, onRemove: () => setStatusFilter([]) });
  if (locationFilter) appliedFilters.push({ key: 'location', label: `Location contains "${locationFilter}"`, onRemove: () => setLocationFilter('') });

  const resourceName = { singular: 'contact', plural: 'contacts' };
  const {selectedResources, allResourcesSelected, handleSelectionChange} = useIndexResourceState(paginatedContacts);

  const tableHeadings = [
    {title: 'Customer Name'},
    {title: 'Phone'},
    {title: 'Email'},
    {title: 'Location'},
    {title: 'Status'},
    {title: 'Engagement'},
    {title: 'Order Value'},
    {title: 'Last Order'},
    {title: 'Actions', alignment: 'end'},
  ];

  const rowMarkup = paginatedContacts.map((contact, index) => (
    <IndexTable.Row id={contact.id} key={contact.id} selected={selectedResources.includes(contact.id)} position={index}>
      <IndexTable.Cell>
        <BlockStack gap="100">
          <Text variant="bodyMd" fontWeight="semibold" as="span">{contact.customerName}</Text>
          <Text variant="bodySm" as="span" tone="subdued">{contact.company}</Text>
        </BlockStack>
      </IndexTable.Cell>
      <IndexTable.Cell><InlineStack gap="200" alignment="center"><Text variant="bodyMd" as="span">{contact.phone}</Text></InlineStack></IndexTable.Cell>
      <IndexTable.Cell><InlineStack gap="200" alignment="center"><Text variant="bodyMd" as="span">{contact.email}</Text></InlineStack></IndexTable.Cell>
      <IndexTable.Cell><Text variant="bodyMd" as="span">{contact.location}</Text></IndexTable.Cell>
      <IndexTable.Cell>{getStatusBadge(contact.status)}</IndexTable.Cell>
      <IndexTable.Cell>{getEngagementBadge(contact.engagementRate)}</IndexTable.Cell>
      <IndexTable.Cell><Text as="span" numeric>${contact.orderValue.toLocaleString()}</Text></IndexTable.Cell>
      <IndexTable.Cell><Text variant="bodyMd" as="span">{contact.lastOrder}</Text></IndexTable.Cell>
      <IndexTable.Cell>
        <InlineStack align="end" gap="200">
          <Tooltip content="View contact details">
            <Button size="slim" onClick={(event) => { event.stopPropagation(); handleEditClick(contact); }}>
              <Icon source={EditIcon} />
            </Button>
          </Tooltip>
        </InlineStack>
      </IndexTable.Cell>
    </IndexTable.Row>
  ));

  return (
    <>
      <Page 
        title={`Live Segment: ${decodedName}`}
        subtitle={`${totalItems} contacts currently match this segment criteria`}
        fullWidth
        breadcrumbs={[{ content: 'Contact Lists', onAction: () => navigate('/app/contact-list/live-segment') }]}
        primaryAction={{ content: (<InlineStack alignment="center"><Text as="span">Export Contacts</Text></InlineStack>), onAction: () => console.log('Export contacts clicked') }}
        secondaryActions={[{ content: 'Refresh Segment', icon: RefreshIcon, loading: refreshing, onAction: handleRefreshSegment }, { content: 'Back to Segments', icon: ArrowLeftIcon, onAction: () => navigate('/app/contact-list/live-segment') }]}
      >
        <BlockStack gap="400">
          <LegacyCard>
            <IndexFilters
              sortOptions={sortOptions}
              sortSelected={sortSelected}
              queryValue={queryValue}
              queryPlaceholder="Search contacts..."
              onQueryChange={setQueryValue}
              onQueryClear={() => setQueryValue('')}
              onSort={setSortSelected}
              cancelAction={{ onAction: () => {}, disabled: false, loading: false }}
              tabs={tabs}
              selected={selected}
              onSelect={setSelected}
              canCreateNewView
              onCreateNewView={onCreateNewView}
              filters={filters}
              appliedFilters={appliedFilters}
              onClearAll={() => { setStatusFilter([]); setLocationFilter(''); setQueryValue(''); setEngagementFilter([0, 100]); }}
              mode={mode}
              setMode={setMode}
            />
            <IndexTable
              resourceName={resourceName}
              itemCount={paginatedContacts.length}
              selectedItemsCount={allResourcesSelected ? 'All' : selectedResources.length}
              onSelectionChange={handleSelectionChange}
              headings={tableHeadings}
              emptyState={<div style={{ textAlign: 'center', padding: '60px 20px' }}><Text variant="headingMd" as="h3">No contacts found</Text><Text variant="bodyMd" as="p" tone="subdued">{`No contacts currently match this segment criteria.`}</Text></div>}
            >
              {rowMarkup}
            </IndexTable>
            {totalItems > itemsPerPage && (
              <div style={{ maxWidth: '700px', margin: 'auto', border: '1px solid var(--p-color-border)', marginTop: '16px' }}>
                <Pagination onPrevious={hasPrevious ? handlePreviousPage : undefined} onNext={hasNext ? handleNextPage : undefined} type="table" hasNext={hasNext} hasPrevious={hasPrevious} label={`${startItem}-${endItem} of ${totalItems} contacts`} />
              </div>
            )}
          </LegacyCard>
        </BlockStack>
      </Page>

      {selectedContact && (
        <Modal open={editModalActive} onClose={handleModalClose} title={`Contact Details: ${selectedContact.customerName}`} primaryAction={{ content: 'Close', onAction: handleModalClose }}>
          <Modal.Section>
            <BlockStack gap="400">
              <TextContainer>
                <Text variant="bodyMd" as="p"><strong>Name:</strong> {selectedContact.customerName}</Text>
                <Text variant="bodyMd" as="p"><strong>Email:</strong> {selectedContact.email}</Text>
                <Text variant="bodyMd" as="p"><strong>Phone:</strong> {selectedContact.phone}</Text>
                <Text variant="bodyMd" as="p"><strong>Company:</strong> {selectedContact.company}</Text>
                <Text variant="bodyMd" as="p"><strong>Location:</strong> {selectedContact.location}</Text>
                <Text variant="bodyMd" as="p"><strong>Status:</strong> {selectedContact.status}</Text>
                <Text variant="bodyMd" as="p"><strong>Engagement Rate:</strong> {selectedContact.engagementRate}%</Text>
                <Text variant="bodyMd" as="p"><strong>Total Orders:</strong> {selectedContact.totalOrders}</Text>
                <Text variant="bodyMd" as="p"><strong>Order Value:</strong> ${selectedContact.orderValue.toLocaleString()}</Text>
                <Text variant="bodyMd" as="p"><strong>Last Order:</strong> {selectedContact.lastOrder}</Text>
                <Text variant="bodyMd" as="p"><strong>Join Date:</strong> {selectedContact.joinDate}</Text>
              </TextContainer>
              <BlockStack gap="200">
                <Text variant="headingMd" as="h4">Tags</Text>
                <InlineStack gap="200">{selectedContact.tags.map((tag, index) => (<Badge key={index} tone="info">{tag}</Badge>))}</InlineStack>
              </BlockStack>
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}
    </>
  );
}

export default LiveSegmentContactsPage;


