import React, { useState, useEffect, useRef, useCallback, useMemo } from 'react';
import {
  Text,
  TextField,
  Button,
  Avatar,
  Badge,
  Icon,
  BlockStack,
  InlineStack,
  Scrollable,
  Popover,
  ActionList,
  Spinner,
  Tabs,
  Box,
  Banner,
} from '@shopify/polaris';
import {
  SearchIcon,
  SendIcon,
  AttachmentIcon,
  ChatIcon,
  PhoneIcon,
  PlayCircleIcon,
  MenuVerticalIcon,
  ImageIcon,
  FileIcon,
  MicrophoneIcon,
  ThemeTemplateIcon,
  LocationIcon,
  AlertTriangleIcon
} from '@shopify/polaris-icons';
import {
  fetchCustomerList,
  fetchCustomerListByUrl,
  fetchCustomerChat,
  transformCustomerData,
  transformMessageData,
  formatMessageTime,
  handleApiError,
  sendMessage,
  sendImageMessage,
  sendVideoMessage,
  sendDocumentMessage,
  sendMediaMessage,
  sendReplyMessage,
  sendTemplateMessage
} from '../utils/chatApi';

import { parseMarkdown, convertWhatsAppMessageToPreview } from '../utils/whatsappMessagePreview';
import { getAllTemplateList, getSelectedTemplatebyName, saveTemporaryFile } from '../utils/templateApi';
import EmojiPicker from '../components/EmojiPicker';
import WhatsAppPreview from '../components/WhatsAppPreview';
import { useSocket } from '../hooks/useSocket.jsx';
import { userInfo } from '../utils/apiConfig';
import { accountDetails } from '../utils/apiConfig';
const connected = accountDetails.isAccountConnected;

export default function ChatsPage() {
  // State management
  const [selectedCustomer, setSelectedCustomer] = useState(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [messageText, setMessageText] = useState('');
  const [showAttachmentMenu, setShowAttachmentMenu] = useState(false);
  const [showCustomerActions, setShowCustomerActions] = useState(false);
  const [showTemplateSelector, setShowTemplateSelector] = useState(false);
  const [templateSearchTerm, setTemplateSearchTerm] = useState('');
  const [apiTemplates, setApiTemplates] = useState([]);
  const [loadingTemplates, setLoadingTemplates] = useState(false);
  const [uploadingFile, setUploadingFile] = useState(false);
  const [selectedTab, setSelectedTab] = useState(0);

  // Template popup state
  const [showTemplatePopup, setShowTemplatePopup] = useState(false);
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [templateVariableMapping, setTemplateVariableMapping] = useState({});
  const [templateMediaMapping, setTemplateMediaMapping] = useState({});
  const [uploadingMedia, setUploadingMedia] = useState({});

  // Reply functionality state
  const [replyingToMessage, setReplyingToMessage] = useState(null);
  const [sendingReply, setSendingReply] = useState(false);
  const [highlightedMessageId, setHighlightedMessageId] = useState(null);

  // API-related state
  const [activeCustomers, setActiveCustomers] = useState([]);
  const [rollingOverCustomers, setRollingOverCustomers] = useState([]);
  const [messages, setMessages] = useState({});
  const [loadingCustomers, setLoadingCustomers] = useState(true);
  const [loadingMessages, setLoadingMessages] = useState(false);
  const [apiError, setApiError] = useState(null);

  // Separate pagination state for active and rolling over customers
  const [activeCustomersPage, setActiveCustomersPage] = useState(1);
  const [rollingOverCustomersPage, setRollingOverCustomersPage] = useState(1);
  const [hasMoreActiveCustomers, setHasMoreActiveCustomers] = useState(true);
  const [hasMoreRollingOverCustomers, setHasMoreRollingOverCustomers] = useState(true);
  const [hasMoreMessages, setHasMoreMessages] = useState({});
  const [loadingMoreCustomers, setLoadingMoreCustomers] = useState(false);
  const [loadingMoreMessages, setLoadingMoreMessages] = useState(false);
  const [activeCustomersNextUrl, setActiveCustomersNextUrl] = useState(null);
  const [rollingOverCustomersNextUrl, setRollingOverCustomersNextUrl] = useState(null);
  const [lastChatIds, setLastChatIds] = useState({});

  // Track if rolling over customers have been loaded
  const [rollingOverLoaded, setRollingOverLoaded] = useState(false);
  
  // Track new customers that need to be added
  const [newCustomersToAdd, setNewCustomersToAdd] = useState([]);

  // Socket-related state
  const [socketConnected, setSocketConnected] = useState(false);
  const [sendingMessage, setSendingMessage] = useState(false);





  const { socket, isConnected, connectionError, addEventListener, removeEventListener } = useSocket(userInfo);

  // Function to handle new customers that don't exist in either list
  const handleNewCustomer = useCallback((socketData) => {
    const messageData = socketData.data || socketData;
    const customerKey = messageData.profile_wa_key;
    const messageTimestamp = new Date(messageData.updated_at || messageData.created_at);
    
    // Debug logging to see the socket data structure
    console.log('🔍 Socket data for new customer:', {
      customerprofilename: socketData.customerprofilename,
      customerprofile_wa_id: socketData.customerprofile_wa_id,
      messageData: messageData
    });
    
    // Extract customer info from socket data (customerprofilename, customerprofile_wa_id)
    const customerName = socketData.customerprofilename || messageData.profile_name || `User ${customerKey}`;
    const customerPhone = socketData.customerprofile_wa_id || messageData.profile_wa_id || customerKey;
    
    // Create a new customer object from the socket data
    const newCustomer = {
      id: Date.now(), // Use timestamp as unique ID
      profileWaId: customerPhone,
      profileWaKey: customerKey,
      name: customerName,
      phone: `+${customerPhone}`,
      avatar: `https://ui-avatars.com/api/?name=${encodeURIComponent(customerName)}&background=random`,
      lastMessageTime: messageData.updated_at || messageData.created_at,
      lastMessageTimestamp: messageTimestamp,
      isOnline: true, // New customer is considered online
      unreadCount: 1, // They just sent a message
    };
    
    console.log('🆕 Adding new customer to active list:', newCustomer.name);
    
    // Add the new customer to the active list at the top
    setActiveCustomers(prev => {
      // Make sure customer doesn't already exist
      const exists = prev.some(c => c.profileWaKey === customerKey);
      if (exists) {
        console.log('⚠️ Customer already exists in active list');
        return prev;
      }
      
      const newActiveList = [newCustomer, ...prev];
      
      // Sort by latest message timestamp (most recent first)
      return newActiveList.sort((a, b) => {
        const aTime = a.lastMessageTimestamp || new Date(0);
        const bTime = b.lastMessageTimestamp || new Date(0);
        return bTime - aTime;
      });
    });
    
    // Add the message to the new customer's chat
    const transformedMessage = transformMessageData(messageData);
    setMessages(prev => ({
      ...prev,
      [newCustomer.id]: [transformedMessage]
    }));
  }, []);

  const messagesEndRef = useRef(null);
  const messagesContainerRef = useRef(null);
  const messageInputRef = useRef(null);
  const fileInputRef = useRef(null);
  const imageInputRef = useRef(null);
  const videoInputRef = useRef(null);
  const audioInputRef = useRef(null);
  const documentInputRef = useRef(null);
  const customerListRef = useRef(null);



  const fetchApiTemplates = useCallback(async () => {
    try {
      setLoadingTemplates(true);
      const response = await getAllTemplateList();

      if (response.status && response.data && response.data.customtemplates) {
        const transformedTemplates = response.data.customtemplates.map(template => ({
          id: template.template_id ? template.template_id.toString() : 'unknown',
          name: template.template_name || 'Unknown Template',
          templateId: template.template_id,
          userId: template.user_id
        }));

        setApiTemplates(transformedTemplates);
      } else {
        console.error('Invalid API response format:', response);
        setApiTemplates([]);
      }
    } catch (error) {
      console.error('Error fetching templates:', error);
      setApiError(`Failed to load templates: ${error.message}`);
      setApiTemplates([]);
    } finally {
      setLoadingTemplates(false);
    }
  }, []);

  // Filter templates based on search term
  const filteredTemplates = apiTemplates.filter(template => {
    if (!templateSearchTerm.trim()) return true;

    const searchLower = templateSearchTerm.toLowerCase();
    return template.name.toLowerCase().includes(searchLower);
  });

  // Load customers from API
  const loadCustomers = useCallback(async (isRollingOver = false, page = 1, append = false) => {
    try {
      if (!append) {
        setLoadingCustomers(true);
        setApiError(null);
      }

      const response = await fetchCustomerList(page, isRollingOver);

      if (response.data && response.data.data) {
        const transformedCustomers = response.data.data.map((customer, index) => {
          const baseIndex = append ?
            (isRollingOver ? rollingOverCustomers.length : activeCustomers.length) + index :
            index;

          return transformCustomerData(customer, baseIndex);
        });

        if (isRollingOver) {
          setRollingOverCustomers(prev => {
            const newList = append ? [...prev, ...transformedCustomers] : transformedCustomers;
            // Sort by latest message timestamp (most recent first)
            return newList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
          setRollingOverCustomersNextUrl(response.data.next_page_url);
        } else {
          setActiveCustomers(prev => {
            const newList = append ? [...prev, ...transformedCustomers] : transformedCustomers;
            // Sort by latest message timestamp (most recent first)
            return newList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
          setActiveCustomersNextUrl(response.data.next_page_url);
        }

        // Update hasMore state based on customer type
        if (isRollingOver) {
          setHasMoreRollingOverCustomers(response.data.next_page_url !== null);
        } else {
          setHasMoreActiveCustomers(response.data.next_page_url !== null);
        }

      }
    } catch (error) {
      const errorResponse = handleApiError(error);
      setApiError(errorResponse.error);
    } finally {
      setLoadingCustomers(false);
    }
  }, []); // Remove dependencies to prevent infinite loop


  // Load more customers using next page URL
  const loadMoreCustomers = useCallback(async () => {
    const isRollingOver = selectedTab === 1;
    const nextUrl = isRollingOver ? rollingOverCustomersNextUrl : activeCustomersNextUrl;
    const hasMore = isRollingOver ? hasMoreRollingOverCustomers : hasMoreActiveCustomers;

    if (!nextUrl || loadingMoreCustomers || !hasMore) {
      console.log('⚠️ Cannot load more customers:', { nextUrl: !!nextUrl, loadingMoreCustomers, hasMore });
      return;
    }

    try {
      setLoadingMoreCustomers(true);
      const response = await fetchCustomerListByUrl(nextUrl);

      if (response.data && response.data.data) {
        const currentLength = isRollingOver ? rollingOverCustomers.length : activeCustomers.length;
        const transformedCustomers = response.data.data.map((customer, index) => {
          return transformCustomerData(customer, currentLength + index);
        });

        if (isRollingOver) {
          setRollingOverCustomers(prev => {
            const newList = [...prev, ...transformedCustomers];
            // Sort by latest message timestamp (most recent first)
            return newList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
          setRollingOverCustomersNextUrl(response.data.next_page_url);
        } else {
          setActiveCustomers(prev => {
            const newList = [...prev, ...transformedCustomers];
            // Sort by latest message timestamp (most recent first)
            return newList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
          setActiveCustomersNextUrl(response.data.next_page_url);
        }

        // Update hasMore state based on customer type
        if (isRollingOver) {
          setHasMoreRollingOverCustomers(response.data.next_page_url !== null);
          setRollingOverCustomersPage(prev => prev + 1);
        } else {
          setHasMoreActiveCustomers(response.data.next_page_url !== null);
          setActiveCustomersPage(prev => prev + 1);
        }

      }
    } catch (error) {
      const errorResponse = handleApiError(error);
      setApiError(errorResponse.error);
    } finally {
      setLoadingMoreCustomers(false);
    }
  }, [selectedTab, activeCustomersNextUrl, rollingOverCustomersNextUrl, loadingMoreCustomers, rollingOverCustomers.length, activeCustomers.length]);

  // Load messages for selected customer
  const loadMessages = useCallback(async (customer, append = false) => {
    if (!customer) return;

    try {
      if (!append) {
        setLoadingMessages(true);
      } else {
        setLoadingMoreMessages(true);
      }

      // Get the last chat ID for this customer (for pagination)
      const lastChatId = append ? (lastChatIds[customer.id] || 0) : 0;

      const response = await fetchCustomerChat(customer.profileWaKey, 1, lastChatId);

      if (response.chats && response.chats.data) {
        const transformedMessages = response.chats.data.map(transformMessageData);

   
        const orderedMessages = append ? transformedMessages : transformedMessages.reverse();

        setMessages(prev => {
          const existingMessages = prev[customer.id] || [];
          const newMessages = append
            ? [...orderedMessages, ...existingMessages]
            : orderedMessages; 

          return {
            ...prev,
            [customer.id]: newMessages
          };
        });

       
        if (transformedMessages.length > 0) {
          const oldestMessage = transformedMessages[transformedMessages.length - 1];
          setLastChatIds(prev => ({
            ...prev,
            [customer.id]: oldestMessage.id
          }));
        }

        setHasMoreMessages(prev => ({
          ...prev,
          [customer.id]: transformedMessages.length >= 20 
        }));
      }
    } catch (error) {
      if(connected){
         console.error('Error loading messages:', error);
      setApiError(handleApiError(error).error);
      } else {
        const accountConnectionMessage="Please connect your account with meta to load messages"
      setApiError(accountConnectionMessage);
      }
     
    } finally {
      setLoadingMessages(false);
      setLoadingMoreMessages(false);
    }
  }, [lastChatIds]);

  // Load initial data - only load active customers on mount
  useEffect(() => {
    let isMounted = true;

    const loadInitialData = async () => {
      if (isMounted) {
        console.log('🚀 Loading initial active customers - Page 1');
        await loadCustomers(false, 1); // Only load active customers initially
      }
    };

    loadInitialData();

    return () => {
      isMounted = false;
    };
  }, []);


  useEffect(() => {
    if (selectedCustomer) {
      console.log(`💬 Loading chat messages for customer: ${selectedCustomer.name} (${selectedCustomer.profileWaKey})`);


      setLastChatIds(prev => ({
        ...prev,
        [selectedCustomer.id]: 0
      }));

      setHasMoreMessages(prev => ({
        ...prev,
        [selectedCustomer.id]: true
      }));


      const loadInitialMessages = async () => {
        try {
          setLoadingMessages(true);
          const response = await fetchCustomerChat(selectedCustomer.profileWaKey, 1, 0);

          if (response.chats && response.chats.data) {
            const transformedMessages = response.chats.data.map(transformMessageData).reverse();

            setMessages(prev => ({
              ...prev,
              [selectedCustomer.id]: transformedMessages
            }));


            if (transformedMessages.length > 0) {
              const oldestMessage = response.chats.data[response.chats.data.length - 1];
              setLastChatIds(prev => ({
                ...prev,
                [selectedCustomer.id]: oldestMessage.id
              }));
            }

            // Check if there are more messages
            setHasMoreMessages(prev => ({
              ...prev,
              [selectedCustomer.id]: response.chats.data.length >= 20
            }));
          }
        } catch (error) {
         if(connected){
         console.error('Error loading messages:', error);
      setApiError(handleApiError(error).error);
      } else {
        const accountConnectionMessage="Please connect your account with meta to load messages"
      setApiError(accountConnectionMessage);
      }
        } finally {
          setLoadingMessages(false);
        }
      };

      // Add a small delay to prevent rapid API calls when switching customers quickly
      const timeoutId = setTimeout(() => {
        loadInitialMessages();
      }, 100);

      return () => clearTimeout(timeoutId);
    }
  }, [selectedCustomer]);


  useEffect(() => {
    if (!socket) return;


    const handleIncomingMessage = (data) => {
      console.log('📨 Incoming message received:', data);
      console.log('📨 Message data:', data.data);
      console.log('📨 Sender:', data.data?.sender);

      const messageData = data.data;
      const customerKey = messageData.profile_wa_key;

      if (messageData.sender === 2) {
        console.log('⏭️ Skipping own outgoing message from chatdata');
        return;
      }

      // Find customer in both lists
      const activeCustomer = activeCustomers.find(c => c.profileWaKey === customerKey);
      const rollingOverCustomer = rollingOverCustomers.find(c => c.profileWaKey === customerKey);
      const customer = activeCustomer || rollingOverCustomer;

      if (customer) {
        const transformedMessage = transformMessageData(messageData);
        const messageTimestamp = new Date(messageData.updated_at || messageData.created_at);

        // Add message to chat
        setMessages(prev => {
          const existingMessages = prev[customer.id] || [];
          const messageExists = existingMessages.some(msg =>
            msg.messageId === transformedMessage.messageId ||
            msg.id === transformedMessage.id ||
            (msg.content === transformedMessage.content &&
              Math.abs(new Date(msg.timestamp) - new Date(transformedMessage.timestamp)) < 5000)
          );

          if (!messageExists) {
            console.log('✅ Adding new message to chat');
            return {
              ...prev,
              [customer.id]: [...existingMessages, transformedMessage]
            };
          } else {
            console.log('⏭️ Message already exists, skipping');
            return prev;
          }
        });

        // Update customer with latest message info
        const updatedCustomer = {
          ...customer,
          lastMessageTime: messageData.updated_at || messageData.created_at,
          lastMessageTimestamp: messageTimestamp,
          unreadCount: selectedCustomer?.id === customer.id ? 0 : (customer.unreadCount || 0) + 1
        };

        // If customer was in rolling over list, move them to active list
        if (rollingOverCustomer && !activeCustomer) {
          console.log('🔄 Moving customer from rolling over to active list:', customer.name);
          
          // Remove from rolling over list
          setRollingOverCustomers(prev => prev.filter(c => c.profileWaKey !== customerKey));
          
          // Add to active list at the top (sorted by timestamp)
          setActiveCustomers(prev => {
            const filteredActive = prev.filter(c => c.profileWaKey !== customerKey);
            const newActiveList = [updatedCustomer, ...filteredActive];
            
            // Sort by latest message timestamp (most recent first)
            return newActiveList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
        } else {
          // Customer is already in active list, just update and resort
          setActiveCustomers(prev => {
            const updatedList = prev.map(c =>
              c.profileWaKey === customerKey ? updatedCustomer : c
            );
            
            // Sort by latest message timestamp (most recent first)
            return updatedList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
          
          // Also update rolling over list if customer exists there
          setRollingOverCustomers(prev =>
            prev.map(c =>
              c.profileWaKey === customerKey ? updatedCustomer : c
            )
          );
        }
      } else {
        // Handle new customer that doesn't exist in either list
        console.log('🆕 New customer detected, adding to active list');
        handleNewCustomer(data); // Pass the complete socket data
      }
    };


    const handleMessageStatus = (data) => {
      console.log(' Message status update:', data);

      const messageData = data.data || data;
      const customerKey = messageData.profile_wa_key;
      const messageId = messageData.message_id;
      const newStatus = messageData.delivery_status;

      console.log(' Status update details:', {
        customerKey,
        messageId,
        newStatus,
        selectedCustomer: selectedCustomer?.profileWaKey
      });


      const customer = [...activeCustomers, ...rollingOverCustomers].find(
        c => c.profileWaKey === customerKey
      );

      if (customer) {

        setMessages(prev => {
          const customerMessages = prev[customer.id] || [];
          const updatedMessages = customerMessages.map(msg => {

            const isTargetMessage = msg.messageId === messageId ||
              (msg.type === 'sent' && !msg.messageId &&
                Math.abs(new Date(msg.timestamp) - new Date()) < 30000);

            if (isTargetMessage) {
              console.log('✅ Updating message status:', {
                messageId: msg.messageId,
                oldStatus: msg.status,
                newStatus: newStatus
              });
              return { ...msg, status: newStatus, messageId: messageId };
            }
            return msg;
          });

          return {
            ...prev,
            [customer.id]: updatedMessages
          };
        });
      }
    };


    const handleSyncMessages = (data) => {
      console.log('🔄 Sync message received:', data);

      const messageData = data.data;
      const customerKey = messageData.profile_wa_key;

      // Find customer in both lists
      const activeCustomer = activeCustomers.find(c => c.profileWaKey === customerKey);
      const rollingOverCustomer = rollingOverCustomers.find(c => c.profileWaKey === customerKey);
      const customer = activeCustomer || rollingOverCustomer;

      if (customer) {
        const transformedMessage = transformMessageData(messageData);
        const messageTimestamp = new Date(messageData.updated_at || messageData.created_at);

        // Add message regardless of sender - this handles both incoming and our own sent messages
        setMessages(prev => {
          const existingMessages = prev[customer.id] || [];
          const messageExists = existingMessages.some(msg =>
            msg.messageId === transformedMessage.messageId ||
            msg.id === transformedMessage.id ||
            (msg.content === transformedMessage.content &&
              Math.abs(new Date(msg.timestamp) - new Date(transformedMessage.timestamp)) < 5000)
          );

          if (!messageExists) {
            console.log('✅ Adding sync message to chat');
            return {
              ...prev,
              [customer.id]: [...existingMessages, transformedMessage]
            };
          } else {
            console.log('⏭️ Sync message already exists, skipping');
            return prev;
          }
        });

        // Update customer with latest message info
        const updatedCustomer = {
          ...customer,
          lastMessageTime: messageData.updated_at || messageData.created_at,
          lastMessageTimestamp: messageTimestamp,
          unreadCount: selectedCustomer?.id === customer.id ? 0 : (customer.unreadCount || 0) + (transformedMessage.type === 'received' ? 1 : 0)
        };

        // If this is an incoming message and customer was in rolling over list, move to active
        if (transformedMessage.type === 'received' && rollingOverCustomer && !activeCustomer) {
          console.log('🔄 Moving customer from rolling over to active list (sync):', customer.name);
          
          // Remove from rolling over list
          setRollingOverCustomers(prev => prev.filter(c => c.profileWaKey !== customerKey));
          
          // Add to active list at the top (sorted by timestamp)
          setActiveCustomers(prev => {
            const filteredActive = prev.filter(c => c.profileWaKey !== customerKey);
            const newActiveList = [updatedCustomer, ...filteredActive];
            
            // Sort by latest message timestamp (most recent first)
            return newActiveList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
        } else {
          // Update existing lists and resort by timestamp
          setActiveCustomers(prev => {
            const updatedList = prev.map(c =>
              c.profileWaKey === customerKey ? updatedCustomer : c
            );
            
            // Sort by latest message timestamp (most recent first)
            return updatedList.sort((a, b) => {
              const aTime = a.lastMessageTimestamp || new Date(0);
              const bTime = b.lastMessageTimestamp || new Date(0);
              return bTime - aTime;
            });
          });
          
          setRollingOverCustomers(prev =>
            prev.map(c =>
              c.profileWaKey === customerKey ? updatedCustomer : c
            )
          );
        }
      } else {
        // Handle new customer for sync messages too
        if (transformedMessage.type === 'received') {
          console.log('🆕 New customer detected in sync, adding to active list');
          // For sync messages, we might not have the full socket structure, so pass messageData
          handleNewCustomer({ data: messageData });
        }
      }
    };

    // Add event listeners
    addEventListener('chatdata', handleIncomingMessage);
    addEventListener('messagestatus', handleMessageStatus);
    addEventListener('synchmessages', handleSyncMessages);

    // Additional event listeners for message status updates
    addEventListener('messagedelivered', handleMessageStatus);
    addEventListener('messageread', handleMessageStatus);
    addEventListener('messageseen', handleMessageStatus);

    // Cleanup
    return () => {
      removeEventListener('chatdata', handleIncomingMessage);
      removeEventListener('messagestatus', handleMessageStatus);
      removeEventListener('synchmessages', handleSyncMessages);
      removeEventListener('messagedelivered', handleMessageStatus);
      removeEventListener('messageread', handleMessageStatus);
      removeEventListener('messageseen', handleMessageStatus);
    };
  }, [socket, activeCustomers, rollingOverCustomers, selectedCustomer, userInfo.id, addEventListener, removeEventListener, handleNewCustomer]);

 
  useEffect(() => {
    setSocketConnected(isConnected);
  }, [isConnected]);

 
  useEffect(() => {
    if (!selectedCustomer || !isConnected) return;

    const refreshMessageStatus = async () => {
      try {
        const response = await fetchCustomerChat(selectedCustomer.profileWaKey, 1, 0);

        if (response.chats && response.chats.data) {
          const latestMessages = response.chats.data.map(transformMessageData);

          
          setMessages(prev => {
            const existingMessages = prev[selectedCustomer.id] || [];
            const updatedMessages = existingMessages.map(existingMsg => {
              const latestMsg = latestMessages.find(latest =>
                latest.messageId === existingMsg.messageId ||
                (latest.id === existingMsg.id && existingMsg.messageId)
              );

              if (latestMsg && latestMsg.status !== existingMsg.status) {
                console.log('🔄 Refreshing message status:', {
                  messageId: existingMsg.messageId,
                  oldStatus: existingMsg.status,
                  newStatus: latestMsg.status
                });
                return { ...existingMsg, status: latestMsg.status };
              }

              return existingMsg;
            });

            return {
              ...prev,
              [selectedCustomer.id]: updatedMessages
            };
          });
        }
      } catch (error) {
        console.error('Error refreshing message status:', error);
      }
    };

    // Refresh every 30 seconds for sent messages
    const interval = setInterval(refreshMessageStatus, 30000);

    return () => clearInterval(interval);
  }, [selectedCustomer, isConnected]);

  const handleTabChange = async (newTab) => {
    console.log(`📋 Tab changed to: ${newTab === 0 ? 'Active' : 'Rolling Over'}`);
    setSelectedTab(newTab);
    setSelectedCustomer(null);

   
    if (newTab === 1 && !rollingOverLoaded) {
      console.log('🔄 Loading rolling over customers - Page 1');
      setRollingOverLoaded(true);
      await loadCustomers(true, 1);
    }
  };

  
  const handleCustomerListScroll = useCallback((e) => {
    const { scrollTop, scrollHeight, clientHeight } = e.target;
    const isNearBottom = scrollTop + clientHeight >= scrollHeight - 100; 

    const isRollingOver = selectedTab === 1;
    const hasMore = isRollingOver ? hasMoreRollingOverCustomers : hasMoreActiveCustomers;
    const currentPage = isRollingOver ? rollingOverCustomersPage : activeCustomersPage;

    if (isNearBottom && hasMore && !loadingMoreCustomers && !loadingCustomers) {
      console.log(`📜 Scrolled to bottom - Loading page ${currentPage + 1} for ${isRollingOver ? 'Rolling Over' : 'Active'} customers`);
      loadMoreCustomers();
    }
  }, [selectedTab, hasMoreActiveCustomers, hasMoreRollingOverCustomers, activeCustomersPage, rollingOverCustomersPage, loadingMoreCustomers, loadingCustomers, loadMoreCustomers]);

  const currentCustomers = selectedTab === 0 ? activeCustomers : rollingOverCustomers;

  const filteredCustomers = currentCustomers.filter(customer =>
    customer.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    customer.phone.includes(searchTerm)
  );

  const currentMessages = selectedCustomer ? messages[selectedCustomer.id] || [] : [];

 
  const loadMoreMessages = useCallback(async () => {
    const customerHasMore = selectedCustomer ? hasMoreMessages[selectedCustomer.id] : false;
    if (!selectedCustomer || !customerHasMore || loadingMoreMessages) return;

    console.log(`📜 Loading more messages for customer: ${selectedCustomer.name}`);

    
    const container = messagesContainerRef.current;
    const scrollHeightBefore = container ? container.scrollHeight : 0;

    await loadMessages(selectedCustomer, true);

    
    setTimeout(() => {
      if (container) {
        const scrollHeightAfter = container.scrollHeight;
        const heightDifference = scrollHeightAfter - scrollHeightBefore;
        container.scrollTop = container.scrollTop + heightDifference;
      }
    }, 100);
  }, [selectedCustomer, hasMoreMessages, loadingMoreMessages, loadMessages]);

  // Handle scroll to load more messages (scroll to top)
  const handleMessagesScroll = useCallback((e) => {
    const { scrollTop } = e.target;
    const isNearTop = scrollTop <= 100;

    const customerHasMore = selectedCustomer ? hasMoreMessages[selectedCustomer.id] : false;

    if (isNearTop && customerHasMore && !loadingMoreMessages && !loadingMessages && selectedCustomer) {
      console.log(`📜 Scrolled to top - Loading more messages for ${selectedCustomer.name}`);
      loadMoreMessages();
    }
  }, [hasMoreMessages, loadingMoreMessages, loadingMessages, selectedCustomer, loadMoreMessages]);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  const scrollToMessage = (messageId) => {
    const messageElement = document.getElementById(`message-${messageId}`);
    if (messageElement) {

      messageElement.scrollIntoView({
        behavior: 'smooth',
        block: 'center'
      });


      setHighlightedMessageId(messageId);


      setTimeout(() => {
        setHighlightedMessageId(null);
      }, 3000);
    }
  };

  const handleSendMessage = useCallback(async () => {
    if (!messageText.trim() || !selectedCustomer || sendingMessage) return;

    const tempId = `temp_${Date.now()}_${Math.random()}`;
    const messageContent = messageText.trim();


    if (replyingToMessage) {
      return handleSendReplyMessage(messageContent, tempId);
    }


    const newMessage = {
      id: tempId,
      type: 'sent',
      content: messageContent,
      timestamp: new Date(),
      status: 'sending',
      messageType: 'text',
      tempId: tempId
    };

    setMessages(prev => ({
      ...prev,
      [selectedCustomer.id]: [...(prev[selectedCustomer.id] || []), newMessage]
    }));

    setMessageText('');
    setSendingMessage(true);

    try {

      const response = await sendMessage(selectedCustomer.profileWaKey, messageContent, 'text');

      console.log('Message sent successfully:', response);


      setMessages(prev => ({
        ...prev,
        [selectedCustomer.id]: prev[selectedCustomer.id].map(msg =>
          msg.tempId === tempId
            ? {
              ...msg,
              status: 'sent',
              id: response.message_id || tempId,
              tempId: undefined
            }
            : msg
        )
      }));



    } catch (error) {
      console.error(' Error sending message:', error);


      setMessages(prev => ({
        ...prev,
        [selectedCustomer.id]: prev[selectedCustomer.id].map(msg =>
          msg.tempId === tempId
            ? {
              ...msg,
              status: 'failed',
              error: error.message || 'Failed to send message'
            }
            : msg
        )
      }));


      setApiError(`Failed to send message: ${error.message || 'Unknown error'}`);
    } finally {
      setSendingMessage(false);
    }
  }, [messageText, selectedCustomer, sendingMessage, replyingToMessage]);

  const handleSendReplyMessage = useCallback(async (replyContent, tempId) => {
    if (!replyingToMessage || !selectedCustomer || sendingReply) return;

    setSendingReply(true);


    const newReplyMessage = {
      id: tempId,
      type: 'sent',
      content: replyContent,
      timestamp: new Date(),
      status: 'sending',
      messageType: 'reply_msg',
      tempId: tempId,
      previewData: {
        originalMessageId: replyingToMessage.messageId,
        replyText: replyContent,
        originalData: {
          msgcontent: replyContent,
          message_id: replyingToMessage.messageId,
          content_send_type: "6"
        }
      }
    };

    console.log(' Creating optimistic reply message:', {
      replyingToMessage: replyingToMessage,
      newReplyMessage: newReplyMessage
    });

    setMessages(prev => ({
      ...prev,
      [selectedCustomer.id]: [...(prev[selectedCustomer.id] || []), newReplyMessage]
    }));

    setMessageText('');
    setReplyingToMessage(null);

    try {

      const response = await sendReplyMessage(
        selectedCustomer.profileWaKey,
        replyContent,
        replyingToMessage.messageId
      );

      console.log(' Reply message sent successfully:', response);


      setMessages(prev => ({
        ...prev,
        [selectedCustomer.id]: prev[selectedCustomer.id].map(msg =>
          msg.tempId === tempId
            ? {
              ...msg,
              status: 'sent',
              id: response.message_id || tempId,
              tempId: undefined
            }
            : msg
        )
      }));

    } catch (error) {
      console.error(' Error sending reply message:', error);


      setMessages(prev => ({
        ...prev,
        [selectedCustomer.id]: prev[selectedCustomer.id].map(msg =>
          msg.tempId === tempId
            ? {
              ...msg,
              status: 'failed',
              error: error.message || 'Failed to send reply'
            }
            : msg
        )
      }));


      setApiError(`Failed to send reply: ${error.message || 'Unknown error'}`);
    } finally {
      setSendingReply(false);
    }
  }, [replyingToMessage, selectedCustomer, sendingReply]);


  useEffect(() => {
    
      scrollToBottom();
    console.log('currentMessages', currentMessages)
  }, [currentMessages ]);

  useEffect(() => {
    const handleKeyDown = (e) => {
      if (e.key === 'Enter' && !e.shiftKey) {
        e.preventDefault();
        handleSendMessage();
      }
    };

    const timer = setTimeout(() => {
      const textareaElement = messageInputRef.current?.querySelector('textarea') ||
        messageInputRef.current?.querySelector('input') ||
        document.querySelector('[placeholder*="Type a message"]');

      if (textareaElement) {
        textareaElement.addEventListener('keydown', handleKeyDown);
      }
    }, 100);

    return () => {
      clearTimeout(timer);
      const textareaElement = messageInputRef.current?.querySelector('textarea') ||
        messageInputRef.current?.querySelector('input') ||
        document.querySelector('[placeholder*="Type a message"]');

      if (textareaElement) {
        textareaElement.removeEventListener('keydown', handleKeyDown);
      }
    };
  }, [handleSendMessage]);

  const handleEmojiSelect = (emoji) => {
    setMessageText(prev => prev + " " + emoji);
    messageInputRef.current?.focus();
  };
  const handleFileUpload = useCallback(async (file, messageType) => {
    if (!selectedCustomer || !file) return;


    const MAX_VIDEO_SIZE = 16 * 1024 * 1024;
    const MAX_DOCUMENT_SIZE = 100 * 1024 * 1024;

    if (messageType === 'video' && file.size > MAX_VIDEO_SIZE) {
      setApiError('Video file size exceeds the maximum limit of 16MB');
      return;
    }

    if (messageType === 'document' && file.size > MAX_DOCUMENT_SIZE) {
      setApiError('Document file size exceeds the maximum limit of 100MB');
      return;
    }

    setUploadingFile(true);
    setShowAttachmentMenu(false);

    // Don't create optimistic message for media files
    // Let socket handle message display when actually sent/received

    try {
      let response;

      if (messageType === 'image') {
        console.log('Sending image:', file.name, file.type);
        response = await sendImageMessage(selectedCustomer.profileWaKey, file);
      } else if (messageType === 'video') {
        console.log(' Sending video:', file.name, file.type);
        response = await sendVideoMessage(selectedCustomer.profileWaKey, file);
      } else if (messageType === 'document') {
        console.log(' Sending document:', file.name, file.type);
        response = await sendDocumentMessage(selectedCustomer.profileWaKey, file);
      } else {
        console.log(' Unsupported file type:', file.name, file.type);
        throw new Error(`File type "${messageType}" is not supported. Supported types: image, video, document.`);
      }

      console.log(' File sent successfully:', response);
      // Message will appear via socket when actually sent/received

    } catch (error) {
      console.error(' File upload failed:', error);
      setApiError(`Failed to send file: ${error.message || 'Unknown error'}`);
    } finally {
      setUploadingFile(false);
    }
  }, [selectedCustomer]);


  const handleTemplateSelect = useCallback(async (template) => {
    if (!selectedCustomer) return;

    setShowTemplateSelector(false);
    setLoadingTemplateDetails(true);
    setShowTemplatePopup(true);

    try {
      const response = await getSelectedTemplatebyName(template.name);

      if (response.data && response.data.length > 0) {
        const templateData = response.data[0];
        setSelectedTemplateData(templateData);

        const variableMapping = {};
        const mediaMapping = {};

        if (templateData.components) {
          templateData.components.forEach((component, index) => {
            if (component.type === 'HEADER' && component.format === 'IMAGE') {
              mediaMapping[`header_${index}`] = {
                type: 'image',
                url: component.example?.header_handle?.[0] || '',
                required: true
              };
            } else if (component.type === 'HEADER' && component.format === 'VIDEO') {
              mediaMapping[`header_${index}`] = {
                type: 'video',
                url: component.example?.header_handle?.[0] || '',
                required: true
              };
            } else if (component.type === 'HEADER' && component.format === 'DOCUMENT') {
              mediaMapping[`header_${index}`] = {
                type: 'document',
                url: component.example?.header_handle?.[0] || '',
                required: true
              };
            } else if (component.type === 'BODY' && component.text) {
              // Extract variables from body text (format: {{1}}, {{2}}, {{customer_name}}, etc.)
              const variables = component.text.match(/\{\{([^}]+)\}\}/g);
              if (variables) {
                variables.forEach((variable, varIndex) => {
                  const varName = variable.match(/\{([^}]+)\}/)[1];
                  variableMapping[`body_var_${varName}`] = {
                    placeholder: `{{${varName}}}`,
                    value: '',
                    required: true
                  };
                });
              }
            } else if (component.type === 'FOOTER' && component.text) {
              // Extract variables from footer text (format: {{1}}, {{2}}, {{customer_name}}, etc.)
              const variables = component.text.match(/\{\{([^}]+)\}\}/g);
              if (variables) {
                variables.forEach((variable, varIndex) => {
                  const varName = variable.match(/\{([^}]+)\}/)[1];
                  variableMapping[`footer_var_${varName}`] = {
                    placeholder: `{{${varName}}}`,
                    value: '',
                    required: true
                  };
                });
              }
            } else if (component.type === 'HEADER' && component.text) {
              // Extract variables from header text (format: {{1}}, {{2}}, {{customer_name}}, etc.)
              const variables = component.text.match(/\{\{([^}]+)\}\}/g);
              if (variables) {
                variables.forEach((variable, varIndex) => {
                  const varName = variable.match(/\{([^}]+)\}/)[1];
                  variableMapping[`header_var_${varName}`] = {
                    placeholder: `{{${varName}}}`,
                    value: '',
                    required: true
                  };
                });
              }
            }
          });
        }

        setTemplateVariableMapping(variableMapping);
        setTemplateMediaMapping(mediaMapping);
      } else {
        console.error('No template data found');
        setApiError('Template not found or invalid');
        setShowTemplatePopup(false);
      }
    } catch (error) {
      console.error('Error fetching template details:', error);
      setApiError(`Failed to load template: ${error.message}`);
      setShowTemplatePopup(false);
    } finally {
      setLoadingTemplateDetails(false);
    }
  }, [selectedCustomer]);

  // Handle template variable changes
  const handleTemplateVariableChange = useCallback((variableKey, value) => {
    setTemplateVariableMapping(prev => ({
      ...prev,
      [variableKey]: {
        ...prev[variableKey],
        value: value
      }
    }));
  }, []);

  // Handle template media changes
  const handleTemplateMediaChange = useCallback(async (mediaKey, file) => {
    if (file) {
      setUploadingMedia(prev => ({ ...prev, [mediaKey]: true }));

      try {
        const reader = new FileReader();
        reader.onload = async () => {
          try {
            const base64String = reader.result.split(',')[1];

            let fileType = 'image';
            if (file.type.startsWith('video/')) {
              fileType = 'video';
            } else if (file.type.startsWith('audio/')) {
              fileType = 'audio';
            } else if (file.type.includes('pdf') || file.type.includes('document') || file.type.includes('text')) {
              fileType = 'document';
            }

            const filePayload = {
              fileName: file.name,
              fileSrc: `data:${file.type};base64,${base64String}`,
              fileType: fileType
            };

            const uploadResponse = await saveTemporaryFile(filePayload);

            if (uploadResponse.status && uploadResponse.filename) {
              const getGabsUrl = `https://app.getgabs.com/customers/mediafile/newfiles/uploads/${uploadResponse.filename}`;

              setTemplateMediaMapping(prev => ({
                ...prev,
                [mediaKey]: {
                  ...prev[mediaKey],
                  file: file,
                  fileUrl: getGabsUrl,
                  fileName: uploadResponse.filename,
                  uploaded: true,
                  uploadError: null
                }
              }));
            } else {
              throw new Error(uploadResponse.message || 'Failed to upload file');
            }
          } catch (error) {
            console.error('Error uploading file:', error);
            setTemplateMediaMapping(prev => ({
              ...prev,
              [mediaKey]: {
                ...prev[mediaKey],
                uploadError: error.message
              }
            }));
            setApiError(`Failed to upload file: ${error.message}`);
          } finally {
            // Clear loading state
            setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
          }
        };

        reader.onerror = () => {
          setApiError('Failed to read file');
          setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
        };

        reader.readAsDataURL(file);
      } catch (error) {
        console.error('Error uploading file:', error);
        setApiError(`Failed to upload file: ${error.message}`);
        setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
      }
    }
  }, []);

  // Send template message with variable mapping
  const handleSendTemplate = useCallback(async () => {
    if (!selectedCustomer || !selectedTemplateData) return;

    try {
      // Generate the WhatsApp API format template JSON
      const generateTemplateJson = () => {
        const templateJson = {
          to: selectedCustomer.phone || "receiver_number",
          type: "template",
          template: {
            name: selectedTemplateData.name,
            language: {
              code: selectedTemplateData.language || "en_US"
            },
            components: []
          },
          recipient_type: "individual",
          messaging_product: "whatsapp"
        };

        // Process each component
        selectedTemplateData.components.forEach((component, index) => {
          if (component.type === 'HEADER') {
            if (component.format === 'IMAGE' || component.format === 'VIDEO' || component.format === 'DOCUMENT') {
              const mediaKey = `header_${index}`;
              const mediaMapping = templateMediaMapping[mediaKey];
              if (mediaMapping && mediaMapping.fileUrl) {
                templateJson.template.components.push({
                  type: "HEADER",
                  parameters: [{
                    type: component.format,
                    [component.format.toLowerCase()]: {
                      link: mediaMapping.fileUrl
                    }
                  }]
                });
              }
            } else if (component.text) {
              // Handle text header with variables
              const variables = component.text.match(/\{\{([^}]+)\}\}/g);
              if (variables) {
                const parameters = variables.map(variable => {
                  const varName = variable.match(/\{([^}]+)\}/)[1];
                  const mapping = templateVariableMapping[`header_var_${varName}`];
                  return {
                    type: "text",
                    text: mapping?.value || `{{${varName}}}`
                  };
                });
                templateJson.template.components.push({
                  type: "HEADER",
                  parameters: parameters
                });
              }
            }
          } else if (component.type === 'BODY') {
            const variables = component.text.match(/\{\{([^}]+)\}\}/g);
            if (variables) {
              const parameters = variables.map(variable => {
                const varName = variable.match(/\{([^}]+)\}/)[1];
                const mapping = templateVariableMapping[`body_var_${varName}`];
                return {
                  type: "text",
                  text: mapping?.value || `{{${varName}}}`
                };
              });
              templateJson.template.components.push({
                type: "BODY",
                parameters: parameters
              });
            }
          } else if (component.type === 'FOOTER') {
            const variables = component.text.match(/\{\{([^}]+)\}\}/g);
            if (variables) {
              const parameters = variables.map(variable => {
                const varName = variable.match(/\{([^}]+)\}/)[1];
                const mapping = templateVariableMapping[`footer_var_${varName}`];
                return {
                  type: "text",
                  text: mapping?.value || `{{${varName}}}`
                };
              });
              templateJson.template.components.push({
                type: "FOOTER",
                parameters: parameters
              });
            }
          }
        });

        return templateJson;
      };

      const templateJson = generateTemplateJson();

      // Close popup first
      setShowTemplatePopup(false);
      setSelectedTemplateData(null);
      setTemplateVariableMapping({});
      setTemplateMediaMapping({});
      setUploadingMedia({});

      const templatePayload = {
        customerKey: selectedCustomer.profileWaKey || "",
        campaign_name: selectedTemplateData.name,
        templateJson: templateJson,
        currentChoosedTemplateJson: {
          data: [selectedTemplateData]
        }
      };

      try {
        const apiResponse = await sendTemplateMessage(templatePayload);
        console.log('Template sent successfully:', apiResponse);
        // Template message will appear via socket when actually sent/received
      } catch (apiError) {
        console.error('Failed to send template via API:', apiError);
        setApiError(`Failed to send template: ${apiError.message}`);
      }

    } catch (error) {
      console.error('Error sending template:', error);
      setApiError(`Failed to send template: ${error.message}`);
    }
  }, [selectedCustomer, selectedTemplateData, templateVariableMapping, templateMediaMapping]);

  const handleImageUpload = (e) => {
    const file = e.target.files[0];
    if (file) handleFileUpload(file, 'image');
    e.target.value = '';
  };

  const handleVideoUpload = (e) => {
    const file = e.target.files[0];
    if (file) handleFileUpload(file, 'video');
    e.target.value = '';
  };

  const handleAudioUpload = (e) => {
    const file = e.target.files[0];
    if (file) handleFileUpload(file, 'audio');
    e.target.value = '';
  };

  const handleDocumentUpload = (e) => {
    const file = e.target.files[0];
    if (file) handleFileUpload(file, 'document');
    e.target.value = '';
  };

  const formatFileSize = (bytes) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const formatTimestamp = (timestamp) => {
    return formatMessageTime(timestamp);
  };

  // Function to check if two dates are the same day
  const isSameDay = (date1, date2) => {
    return date1.getDate() === date2.getDate() &&
           date1.getMonth() === date2.getMonth() &&
           date1.getFullYear() === date2.getFullYear();
  };

  // Function to format date for display
  const formatDateSeparator = (date) => {
    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);

    if (isSameDay(date, today)) {
      return 'Today';
    } else if (isSameDay(date, yesterday)) {
      return 'Yesterday';
    } else {
      return date.toLocaleDateString('en-US', {
        weekday: 'long',
        year: 'numeric',
        month: 'long',
        day: 'numeric'
      });
    }
  };

  // Function to render date separator
  const renderDateSeparator = (date) => {
    return (
      <div
        key={`date-${date.getTime()}`}
        style={{
          display: 'flex',
          justifyContent: 'center',
          margin: '20px 0 16px 0'
        }}
      >
        <div
          style={{
            backgroundColor: '#f0f2f5',
            color: '#667781',
            padding: '6px 12px',
            borderRadius: '8px',
            fontSize: '12px',
            fontWeight: '500',
            textAlign: 'center'
          }}
        >
          {formatDateSeparator(date)}
        </div>
      </div>
    );
  };

  // Function to group messages by date and render with date separators
  const renderMessagesWithDates = (messages) => {
    if (!messages || messages.length === 0) return [];

    const result = [];
    let lastDate = null;

    messages.forEach((message, index) => {
      const messageDate = new Date(message.timestamp);
      
      // Check if we need to show a date separator
      if (!lastDate || !isSameDay(messageDate, lastDate)) {
        result.push(renderDateSeparator(messageDate));
        lastDate = messageDate;
      }

      // Add the message
      result.push(renderMessage(message));
    });

    return result;
  };

  const renderMessage = (message) => {

    const isReceived = message.type === 'received';
    const parsedContent = parseMarkdown(message.content);
    const renderMessageContent = () => {
      switch (message.messageType) {
        case 'image':
          return (
            <div style={{ width: 'fit-content' }}>
              {message.mediaUrl && (
                <img
                  src={message.mediaUrl}
                  alt={message.fileName || 'Image'}
                  style={{
                    maxWidth: '100%',
                    maxHeight: '300px',
                    width: 'auto',
                    height: 'auto',
                    borderRadius: '8px',
                    marginBottom: '8px',
                    display: 'block',
                  }}
                  onError={(e) => {
                    console.error('Image failed to load:', {
                      src: e.target.src,
                      fileName: message.fileName,
                      mediaUrl: message.mediaUrl,
                      error: e,
                    });
                    e.target.style.display = 'none';
                    if (e.target.nextSibling) {
                      e.target.nextSibling.style.display = 'block';
                    }
                  }}
                />
              )}

            </div>
          );

        case 'video':
          return (
            <div style={{ width: 'fit-content' }}>
              {message.mediaUrl ? (
                <video
                  src={message.mediaUrl}
                  controls
                  style={{
                    maxWidth: '100%',
                    maxHeight: '300px',
                    width: 'auto',
                    height: 'auto',
                    borderRadius: '8px',
                    marginBottom: '8px',
                    display: 'block'
                  }}
                  onError={(e) => {
                    e.target.style.display = 'none';
                    e.target.nextSibling.style.display = 'block';
                  }}
                />
              ) : null}

            </div>
          );


        case 'document':
          return (
            <div
              style={{
                display: 'flex',
                alignItems: 'center',
                gap: '12px',
                padding: '8px',
                backgroundColor: 'rgba(0,0,0,0.05)',
                borderRadius: '8px',
                cursor: message.mediaUrl ? 'pointer' : 'default'
              }}
              onClick={() => message.mediaUrl && window.open(message.mediaUrl, '_blank')}
            >
              <Icon source={FileIcon} tone="subdued" />
              <div style={{ flex: 1 }}>
                <div style={{ fontSize: '14px', fontWeight: '500' }}>
                  📄 {message.fileName || 'Document'}
                </div>
                <div style={{ fontSize: '12px', color: '#667781' }}>
                  {message.fileSize ? formatFileSize(message.fileSize) : 'Document'} {message.mediaUrl ? '• Click to open' : ''}
                </div>
                {message.content && message.content !== message.fileName && message.content !== 'Document' && (
                  <div style={{
                    fontSize: '13px',
                    marginTop: '4px',
                    wordBreak: 'break-word'
                  }}>
                    {message.content}
                  </div>
                )}
              </div>
            </div>
          );

        case 'sticker':
          return (
            <div style={{ width: 'fit-content' }}>
              {message.mediaUrl && (
                <img
                  src={message.mediaUrl}
                  alt="Sticker"
                  style={{
                    maxWidth: '150px',
                    maxHeight: '150px',
                    width: 'auto',
                    height: 'auto',
                    borderRadius: '8px',
                    display: 'block',
                  }}
                  onError={(e) => {
                    console.error('Sticker failed to load:', {
                      src: e.target.src,
                      fileName: message.fileName,
                      mediaUrl: message.mediaUrl,
                      error: e,
                    });
                    e.target.style.display = 'none';
                    if (e.target.nextSibling) {
                      e.target.nextSibling.style.display = 'block';
                    }
                  }}
                />
              )}


            </div>
          );

        case 'template':
        case 'interactive':
        case 'template_with_image':


          return (
            <div>
              <div style={{
                fontSize: '12px',
                color: '#667781',
                marginBottom: '8px',
                display: 'flex',
                alignItems: 'center',
                gap: '4px',
                padding: '4px 8px',
                backgroundColor: 'rgba(0,0,0,0.05)',
                borderRadius: '4px'
              }}>
                {message.templateName || 'Template Message'}
              </div>
              <div style={{
                marginTop: '8px',
                position: 'relative'
              }}>
                {message.content ? (
                  <WhatsAppPreview
                    template={message.content}
                    variant="bubble"
                    maxWidth="100%"
                    variableMapping={{
                      customer_name: selectedCustomer?.name || 'Customer',
                      first_name: selectedCustomer?.name?.split(' ')[0] || 'Customer',
                      last_name: selectedCustomer?.name?.split(' ')[1] || '',

                    }}
                    mediaMapping={message.mediaUrl ? { 'header_0': { link: message.mediaUrl } } : {}}
                  />
                ) : (
                  <div style={{
                    fontSize: '14px',
                    fontStyle: 'italic',
                    color: '#667781',
                    padding: '8px',
                    backgroundColor: 'rgba(0,0,0,0.05)',
                    borderRadius: '8px'
                  }}>
                    Template preview not available
                  </div>
                )}
                <style dangerouslySetInnerHTML={{
                  __html: `
                    /* Remove white background from WhatsAppPreview in chat */
                    .template-content + div > div[style*="backgroundColor: #fff"],
                    .template-content + div > div[style*="background-color: #fff"],
                    .template-content + div > div[style*="backgroundColor:#fff"],
                    .template-content + div > div[style*="background-color:#fff"] {
                      background-color: transparent !important;
                      background: transparent !important;
                      box-shadow: none !important;
                      border: none !important;
                      padding: 0 !important;
                    }
                    /* Ensure proper text formatting */
                    .template-content + div strong { font-weight: 600 !important; }
                    .template-content + div em { font-style: italic !important; }
                    .template-content + div del { text-decoration: line-through !important; }
                    .template-content + div code.monospace { 
                      font-family: 'Courier New', monospace !important; 
                      background: rgba(0,0,0,0.1) !important; 
                      padding: 2px 4px !important; 
                      border-radius: 3px !important; 
                    }
                    .template-content + div code.inline { 
                      font-family: 'Courier New', monospace !important; 
                      background: rgba(0,0,0,0.1) !important; 
                      padding: 1px 3px !important; 
                      border-radius: 2px !important; 
                      font-size: 13px !important; 
                    }
                    .template-content + div blockquote { 
                      border-left: 3px solid #ccc !important; 
                      margin: 4px 0 !important; 
                      padding-left: 8px !important; 
                      font-style: italic !important; 
                      color: #666 !important; 
                    }
                    .template-content + div li.bullet { 
                      list-style: none !important; 
                      position: relative !important; 
                      padding-left: 16px !important; 
                      margin: 2px 0 !important; 
                    }
                    .template-content + div li.bullet::before { 
                      content: '•' !important; 
                      position: absolute !important; 
                      left: 0 !important; 
                      color: #666 !important; 
                    }
                    .template-content + div li.numbered { 
                      list-style: none !important; 
                      position: relative !important; 
                      padding-left: 20px !important; 
                      margin: 2px 0 !important; 
                    }
                    .template-content + div li.numbered::before { 
                      content: attr(data-number) '. ' !important; 
                      position: absolute !important; 
                      left: 0 !important; 
                      color: #666 !important; 
                      font-weight: 500 !important; 
                    }
                  `
                }} />
              </div>
            </div>
          );

        case 'interactive_button':
          // Handle outgoing interactive button messages
          try {
            const buttonData = JSON.parse(message.content);
            return (
              <div>
                <div style={{
                  fontSize: '12px',
                  color: '#667781',
                  marginBottom: '8px',
                  display: 'flex',
                  alignItems: 'center',
                  gap: '4px',
                  padding: '4px 8px',
                  backgroundColor: 'rgba(0,0,0,0.05)',
                  borderRadius: '4px'
                }}>
                  🔘 Interactive Button Message
                </div>
                <div style={{
                  fontSize: '14px',
                  lineHeight: '1.4',
                  marginBottom: '12px',
                  wordBreak: 'break-word'
                }}>
                  {buttonData.body}
                </div>
                <div style={{
                  display: 'flex',
                  flexDirection: 'column',
                  gap: '6px'
                }}>
                  {buttonData.buttons?.map((buttonTitle, index) => (
                    <div
                      key={index}
                      style={{
                        padding: '8px 12px',
                        border: '1px solid #e1e3e5',
                        borderRadius: '20px',
                        fontSize: '13px',
                        textAlign: 'center',
                        backgroundColor: '#f8f9fa',
                        color: '#0084ff',
                        fontWeight: '500'
                      }}
                    >
                      {buttonTitle}
                    </div>
                  ))}
                </div>
              </div>
            );
          } catch (e) {
            return (
              <div style={{
                fontSize: '14px',
                fontStyle: 'italic',
                color: '#667781'
              }}>
                Interactive button message
              </div>
            );
          }

        case 'button_reply':
        case 'button':
         
          
          return (
            <div>
              <div style={{
                fontSize: '12px',
                color: '#667781',
                marginBottom: '6px',
                display: 'flex',
                alignItems: 'center',
                gap: '4px'
              }}>
                🔘 Button Reply
              </div>
              <div style={{
                backgroundColor: 'rgb(240, 240, 240)', padding: '8px 12px', borderRadius: '5px', border: '1px solid rgb(221, 221, 221)', textAlign: 'center', fontSize: '12px', color: 'inherit', display: 'flex', alignItems: 'center', justifyContent: 'center', gap: '6px'
              }}>
                {message.content}
              </div>
              {message.previewData?.buttonId && (
                <div style={{
                  fontSize: '11px',
                  color: '#999',
                  marginTop: '4px',
                  fontStyle: 'italic'
                }}>

                </div>
              )}
            </div>
          );

        case 'reply_msg':
          
          const originalMessage = message.previewData?.originalMessageId ?
            currentMessages.find(msg => msg.messageId === message.previewData.originalMessageId) : null;

          return (
            <div>
              <div style={{
                fontSize: '12px',
                color: '#667781',
                marginBottom: '6px',
                display: 'flex',
                alignItems: 'center',
                gap: '4px'
              }}>
                Reply to Message
              </div>

             
              {message.previewData?.originalMessageId && (
                <div
                  onClick={() => {
                    if (originalMessage) {
                      scrollToMessage(originalMessage.id);
                    }
                  }}
                  style={{
                    backgroundColor: 'rgba(0,0,0,0.05)',
                    borderLeft: '3px solid #00a884',
                    padding: '6px 10px',
                    marginBottom: '8px',
                    borderRadius: '0 6px 6px 0',

                    fontSize: '12px',
                    color: '#667781',
                    cursor: originalMessage ? 'pointer' : 'default',
                    transition: 'background-color 0.2s ease'
                  }}
                  onMouseEnter={(e) => {
                    if (originalMessage) {
                      e.target.style.backgroundColor = 'rgba(0,0,0,0.08)';
                    }
                  }}
                  onMouseLeave={(e) => {
                    e.target.style.backgroundColor = 'rgba(0,0,0,0.05)';
                  }}
                  title={originalMessage ? 'Click to jump to original message' : 'Original message not found in current chat'}
                >
                  <div style={{ fontWeight: '500', marginBottom: '2px', fontSize: '11px' }}>
                    {originalMessage ? 'Replying to:' : 'Original Message'}

                  </div>
                  {originalMessage ? (
                    <div style={{
                      fontSize: '12px',
                      color: '#555',
                      maxHeight: '40px',
                      overflow: 'hidden',
                      textOverflow: 'ellipsis',
                      display: '-webkit-box',
                      WebkitLineClamp: 2,
                      WebkitBoxOrient: 'vertical'
                    }}>
                      {originalMessage.content.length > 100
                        ? `${originalMessage.content.substring(0, 100)}...`
                        : originalMessage.content}
                    </div>
                  ) : (
                    <div style={{ fontSize: '11px', fontStyle: 'italic' }}>
                      Message ID: {message.previewData.originalMessageId.slice(-8)}...
                    </div>
                  )}
                </div>
              )}

         
              <div style={{
                fontSize: '14px',
                lineHeight: '1.4',
                wordBreak: 'break-word'
              }}>
                {message.content}
              </div>

            </div>
          );

        default:
          return (
            <div
              style={{
                fontSize: '14px',
                lineHeight: '1.4',
                wordBreak: 'break-word'
              }}
              dangerouslySetInnerHTML={{ __html: parsedContent }}
            />
          );
      }
    };


    const isMediaMessage = ['image', 'video', 'audio', 'document'].includes(message.messageType);
    const isTemplateMessage = ['template', 'template_with_image', 'interactive', 'interactive_button', 'button_reply', 'reply_msg'].includes(message.messageType);

    return (
      <div
        key={message.id}
        id={`message-${message.id}`}
        style={{
          display: 'flex',
          justifyContent: isReceived ? 'flex-start' : 'flex-end',
          marginBottom: '12px',
          padding: highlightedMessageId === message.id ? '8px' : '0',
          borderRadius: highlightedMessageId === message.id ? '8px' : '0',
          backgroundColor: highlightedMessageId === message.id ? 'rgba(0, 168, 132, 0.1)' : 'transparent',
          transition: 'all 0.3s ease',
          border: highlightedMessageId === message.id ? '2px solid rgba(0, 168, 132, 0.3)' : '2px solid transparent'
        }}
      >
        <div
          style={{
            position: 'relative',
            maxWidth: (isMediaMessage || isTemplateMessage) ? '500px' : 'min(500px, 80%)',
            minWidth: (message.messageType === 'text' || message.messageType === 'sticker' || message.messageType === 'image') ? 'auto' : '200px',
            width: (isMediaMessage || isTemplateMessage) ? 'fit-content' : 'auto',
          }}
          onMouseEnter={() => {
         
            const replyBtn = document.getElementById(`reply-btn-${message.id}`);
            if (replyBtn) replyBtn.style.opacity = '1';
          }}
          onMouseLeave={() => {
        
            const replyBtn = document.getElementById(`reply-btn-${message.id}`);
            if (replyBtn) replyBtn.style.opacity = '0';
          }}
        >
          {/* Reply Button */}
          {message.messageId && (
            <button
              id={`reply-btn-${message.id}`}
              onClick={() => setReplyingToMessage(message)}
              style={{
                position: 'absolute',
                top: '50%',
                right: isReceived ? '-35px' : 'auto',
                left: isReceived ? 'auto' : '-35px',
                transform: 'translateY(-50%)',
                backgroundColor: '#f0f0f0',
                border: '1px solid #ddd',
                borderRadius: '50%',
                width: '28px',
                height: '28px',
                display: 'flex',
                alignItems: 'center',
                justifyContent: 'center',
                cursor: 'pointer',
                opacity: '0',
                transition: 'opacity 0.2s ease',
                fontSize: '12px',
                zIndex: 10
              }}
              title="Reply to this message"
            >
              ↩
            </button>
          )}

          <div
            style={{
              backgroundColor: isReceived ? '#ffffff' : '#dcf8c6',
              padding: isTemplateMessage ? '6px 10px' : '8px 12px',
              borderRadius: isReceived ? '18px 18px 18px 4px' : '18px 18px 4px 18px',
              boxShadow: '0 1px 2px rgba(0,0,0,0.1)',
              border: isReceived ? '1px solid #e1e3e5' : 'none'
            }}
          >
            {renderMessageContent()}
            <div
              style={{
                display: 'flex',
                justifyContent: 'flex-end',
                alignItems: 'center',
                gap: '4px',
                marginTop: '4px',
                fontSize: '11px',
                color: '#667781'
              }}
            >
              <span>{message.timestamp.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}</span>
             
                <span>
                  {message.status === 'sending' && <Spinner size="small" />}
                  {message.status === 'sent' && '✓'}
                  {message.status === 'delivered' && '✓✓'}
                  {message.status === 'read' && <span style={{ color: '#4fc3f7' }}>✓✓</span>}
                  {message.status === 'failed' && <span style={{ color: '#d72c0d' }} title={message.error || 'Failed to send'}>⚠️</span>}
                </span>
             
            </div>
          </div>
        </div>
      </div>
    );
  };

  const handleCustomerSelect = (customer) => {
    setSelectedCustomer(customer);

    // Reset unread count for the selected customer
    if (customer.unreadCount > 0) {
      const updateCustomerList = (customers) =>
        customers.map(c =>
          c.id === customer.id
            ? { ...c, unreadCount: 0 }
            : c
        );

      setActiveCustomers(updateCustomerList);
      setRollingOverCustomers(updateCustomerList);
    }
  };

  const renderCustomerItem = (customer) => (
    <Box
      key={customer.id}
      onClick={() => handleCustomerSelect(customer)}
      background={selectedCustomer?.id === customer.id ? "bg-surface-hover" : "bg-surface"}
      padding="300"
      borderBlockEndWidth="025"
      borderColor="border-subdued"
      borderBlockEndStyle="solid"
      cursor="pointer"
    >
      <InlineStack align="space-between" blockAlign="center">
        {/* Left Side: Avatar + Info */}
        <InlineStack gap="300" blockAlign="center">
          <Avatar
            customer
            size="medium"
            name={customer.name}
            source={customer.avatar}
          />
          <BlockStack gap="050">
            <Text variant="bodyMd" fontWeight="semibold">
              {customer?.name?.length > 20
                ? `${customer.name.substring(0, 20)}...`
                : customer?.name}

            </Text>
            <Text variant="bodySm" tone="subdued" truncate>
              {customer.phone}
            </Text>
          </BlockStack>
        </InlineStack>


        <BlockStack gap="100" align="end">
          <Text variant="caption" tone="subdued">
            {formatTimestamp(customer.lastMessageTimestamp)}
          </Text>
          {customer.unreadCount > 0 && (
            <span style={{
              backgroundColor: 'rgb(29, 170, 97)',
              color: 'rgb(255, 255, 255)',
              padding: '1px 6px',
              borderRadius: '12px',
              width: '20px',
              height: '20px',
              margin: '0 auto'
            }}>{customer.unreadCount}</span>
          )}
        </BlockStack>
      </InlineStack>
    </Box>
  );
  return (
    <div style={{
      height: '100vh',
      display: 'flex',
      flexDirection: 'column',
      backgroundColor: '#f8f9fa'
    }}>
      {/* Page Header */}
      <div style={{
        backgroundColor: '#ffffff',
        borderBottom: '1px solid #e1e3e5',
        padding: '16px 24px',
        boxShadow: '0 1px 3px rgba(0,0,0,0.1)'
      }}>
        <Text variant="headingLg" fontWeight="bold">
          WhatsApp Chats
        </Text>
        <Text variant="bodyMd" tone="subdued">
          Manage customer conversations
        </Text>

        {/* API Error Banner */}
        {apiError && (
          <div style={{ marginTop: '12px' }}>
            <Banner tone="critical" icon={AlertTriangleIcon}>
              
                {connected ? (
                <Text variant="bodyMd">
         Error loading data: {apiError}
         </Text>
     
      ) : (
        <Text variant="bodyMd">
         Account not conneted with meta
         </Text>
      )}
              
            </Banner>
          </div>
        )}

        {/* Socket Connection Status */}
        {connectionError && (
          <div style={{ marginTop: '12px' }}>
            <Banner tone="critical">
              <Text variant="bodyMd">
                Connection error: {connectionError}
              </Text>
            </Banner>
          </div>
        )}

        {!isConnected && !connectionError && (
          <div style={{ marginTop: '12px' }}>
            <Banner tone="warning">
              <Text variant="bodyMd">
                Connecting to chat server...
              </Text>
            </Banner>
          </div>
        )}

       
      </div>

      {/* Chat Container */}
      <div style={{
        display: 'flex',
        height: 'calc(100vh - 120px)',
        overflow: 'hidden'
      }}>
        {/* Customer List Sidebar */}
        <div style={{ width: '350px', borderRight: '1px solid #e1e3e5', backgroundColor: '#ffffff' }}>
          {/* Search Header */}
          <div style={{ padding: '16px', borderBottom: '1px solid #f1f3f4' }}>
            <TextField
              placeholder="Search customers..."
              value={searchTerm}
              onChange={setSearchTerm}
              prefix={<Icon source={SearchIcon} />}
              clearButton
              onClearButtonClick={() => setSearchTerm('')}
            />
          </div>

          {/* Chat Category Tabs */}
          <div style={{ borderBottom: '1px solid #f1f3f4' }}>
            <Tabs
              tabs={[
                {
                  id: 'active',
                  content: `Active`,
                  accessibilityLabel: 'Active chats',
                },
                {
                  id: 'rolling-over',
                  content: `Rolling Over`,
                  accessibilityLabel: 'Rolling over chats',
                }
              ]}
              selected={selectedTab}
              onSelect={handleTabChange}
            />
          </div>

          {/* Customer List */}
          <Scrollable
            style={{ height: 'calc(100% - 130px)' }}
            onScrolledToBottom={handleCustomerListScroll}
            onScroll={handleCustomerListScroll}
          >
            {loadingCustomers ? (
              <div style={{ padding: '32px', textAlign: 'center' }}>
                <Spinner size="large" />
                <div style={{ marginTop: '16px' }}>
                  <Text variant="bodyMd" tone="subdued">
                    Loading customers...
                  </Text>
                </div>
              </div>
            ) : filteredCustomers.length > 0 ? (
              <>
                {filteredCustomers.map(renderCustomerItem)}

                {/* Load More Indicator */}
                {loadingMoreCustomers && (
                  <div style={{ padding: '16px', textAlign: 'center' }}>
                    <Spinner size="small" />
                    <div style={{ marginTop: '8px' }}>
                      <Text variant="bodySm" tone="subdued">
                        Loading more customers...
                      </Text>
                    </div>
                  </div>
                )}

                {/* No More Customers Indicator */}
                {!(selectedTab === 0 ? hasMoreActiveCustomers : hasMoreRollingOverCustomers) && filteredCustomers.length > 0 && !loadingMoreCustomers && (
                  <div style={{ padding: '16px', textAlign: 'center' }}>
                    <Text variant="bodySm" tone="subdued">
                      No more customers to load
                    </Text>
                  </div>
                )}
              </>
            ) : (
              <div style={{ padding: '32px', textAlign: 'center' }}>
                <Text variant="bodyMd" tone="subdued">
                  {apiError ? 'Unable to load customers' : 'No customers found'}
                </Text>
              </div>
            )}
          </Scrollable>
        </div>

        {/* Chat Area */}
        <div style={{ flex: 1, display: 'flex', flexDirection: 'column', backgroundColor: '#f8f9fa' }}>
          {selectedCustomer ? (
            <>
              {/* Chat Header */}
              <div
                style={{
                  padding: '16px 20px',
                  backgroundColor: '#ffffff',
                  borderBottom: '1px solid #e1e3e5',
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'space-between'
                }}
              >
                <InlineStack gap="300" align="center">
                  <div style={{ position: 'relative', height: '40px', width: '40px', }}>
                    <Avatar
                      customer
                      size="extralarge"
                      name={selectedCustomer.name}
                      source={selectedCustomer.avatar}
                    />

                  </div>
                  <BlockStack gap="050">
                    <Text variant="bodyLg" fontWeight="semibold">
                      {selectedCustomer.name}

                    </Text>
                    <Text variant="caption" tone="subdued">
                      {selectedCustomer.phone}
                    </Text>
                  </BlockStack>
                </InlineStack>

                <Popover
                  active={showCustomerActions}
                  activator={
                    <Button
                      icon={MenuVerticalIcon}
                      variant="tertiary"
                      onClick={() => setShowCustomerActions(!showCustomerActions)}
                    />
                  }
                  onClose={() => setShowCustomerActions(false)}
                >
                  <ActionList
                    items={[
                      { content: 'View Profile', icon: ChatIcon },
                      { content: 'Call Customer', icon: PhoneIcon },
                      { content: 'Video Call', icon: PlayCircleIcon },
                      { content: 'Block Customer', destructive: true }
                    ]}
                  />
                </Popover>
              </div>

              {/* Messages Area */}
              <div
                ref={messagesContainerRef}
                style={{
                  flex: 1,
                  padding: '16px 20px',
                  overflowY: 'auto',
                  maxHeight: '100%'
                }}
                onScroll={handleMessagesScroll}
              >
                <div style={{ minHeight: '100%', display: 'flex', flexDirection: 'column', justifyContent: 'flex-end' }}>
                  {/* Load More Messages Indicator */}
                  {loadingMoreMessages && (
                    <div style={{ textAlign: 'center', padding: '16px', marginBottom: '12px' }}>
                      <Spinner size="small" />
                      <div style={{ marginTop: '8px' }}>
                        <Text variant="caption" tone="subdued">
                          Loading older messages...
                        </Text>
                      </div>
                    </div>
                  )}

                  {loadingMessages ? (
                    <div style={{ textAlign: 'center', padding: '32px' }}>
                      <Spinner size="large" />
                      <div style={{ marginTop: '16px' }}>
                        <Text variant="bodyMd" tone="subdued">
                          Loading messages...
                        </Text>
                      </div>
                    </div>
                  ) : currentMessages.length > 0 ? (
                    renderMessagesWithDates(currentMessages)
                  ) : (
                    <div style={{ textAlign: 'center', padding: '32px' }}>
                      <Text variant="bodyMd" tone="subdued">
                        No messages yet. Start a conversation!
                      </Text>
                    </div>
                  )}
                  {/* {isTyping && (
                    <div style={{ display: 'flex', justifyContent: 'flex-start', marginBottom: '12px' }}>
                      <div
                        style={{
                          backgroundColor: '#ffffff',
                          padding: '8px 12px',
                          borderRadius: '18px 18px 18px 4px',
                          boxShadow: '0 1px 2px rgba(0,0,0,0.1)',
                          border: '1px solid #e1e3e5'
                        }}
                      >
                        <Spinner size="small" />
                        <Text variant="caption" tone="subdued" style={{ marginLeft: '8px' }}>
                          typing...
                        </Text>
                      </div>
                    </div>
                  )} */}
                  <div ref={messagesEndRef} />
                </div>
              </div>


              {/* Message Composition Area */}
              <div
                style={{
                  padding: '16px 20px',
                  backgroundColor: '#ffffff',
                  borderTop: '1px solid #e1e3e5',
                }}
              >
                {selectedTab === 1 ? (
                  // Rolling Over customers - Template messages only
                  <div style={{ textAlign: 'center' }}>
                    <Button
                      primary
                      size="large"
                      icon={ThemeTemplateIcon}
                      onClick={() => {
                        setShowTemplateSelector(true);
                        fetchApiTemplates();
                      }}
                      disabled={!isConnected}
                    >
                      Send Template Message
                    </Button>
                    <div style={{ marginTop: '8px' }}>
                      <Text variant="bodySm" tone="subdued">
                        Rolling over customers can only receive template messages
                      </Text>
                    </div>
                  </div>
                ) : (
                  // Active customers - Full message composition
                  <>
                    {/* Reply Preview */}
                    {replyingToMessage && (
                      <div style={{
                        backgroundColor: '#f8f9fa',
                        border: '1px solid #e1e3e5',
                        borderRadius: '8px',
                        padding: '8px 12px',
                        marginBottom: '12px',
                        display: 'flex',
                        alignItems: 'center',
                        justifyContent: 'space-between'
                      }}>
                        <div style={{ flex: 1 }}>
                          <div style={{
                            fontSize: '12px',
                            color: '#667781',
                            marginBottom: '4px',
                            fontWeight: '500'
                          }}>
                            Replying to:
                          </div>
                          <div style={{
                            fontSize: '13px',
                            color: '#333',
                            maxHeight: '40px',
                            overflow: 'hidden',
                            textOverflow: 'ellipsis',
                            display: '-webkit-box',
                            WebkitLineClamp: 2,
                            WebkitBoxOrient: 'vertical'
                          }}>
                            {replyingToMessage.content.length > 80
                              ? `${replyingToMessage.content.substring(0, 80)}...`
                              : replyingToMessage.content}
                          </div>
                        </div>
                        <button
                          onClick={() => setReplyingToMessage(null)}
                          style={{
                            backgroundColor: 'transparent',
                            border: 'none',
                            cursor: 'pointer',
                            padding: '4px',
                            marginLeft: '8px',
                            borderRadius: '4px',
                            fontSize: '16px',
                            color: '#667781'
                          }}
                          title="Cancel reply"
                        >
                          ✕
                        </button>
                      </div>
                    )}

                    <InlineStack gap="200" align="center" distribution="justify" blockAlign="center">
                      <Popover
                        active={showAttachmentMenu}
                        activator={
                          <Button
                            icon={AttachmentIcon}
                            variant="tertiary"
                            onClick={() => setShowAttachmentMenu(!showAttachmentMenu)}
                          />
                        }
                        onClose={() => setShowAttachmentMenu(false)}
                      >
                        <ActionList
                          items={[
                            {
                              content: 'Upload Image',
                              icon: ImageIcon,
                              onAction: () => imageInputRef.current?.click()
                            },
                            {
                              content: 'Upload Video',
                              icon: PlayCircleIcon,
                              onAction: () => videoInputRef.current?.click()
                            },
                            // {
                            //   content: 'Upload Audio',
                            //   icon: MicrophoneIcon,
                            //   onAction: () => audioInputRef.current?.click()
                            // },
                            {
                              content: 'Upload Document',
                              icon: FileIcon,
                              onAction: () => documentInputRef.current?.click()
                            },
                            {
                              content: 'Send Template',
                              icon: ThemeTemplateIcon,
                              onAction: () => {
                                setShowAttachmentMenu(false);
                                setShowTemplateSelector(true);
                                fetchApiTemplates();
                              }
                            }
                            // { 
                            //   content: 'Send Location', 
                            //   icon: LocationIcon,
                            //   onAction: () => {
                            //     setShowAttachmentMenu(false);
                            //     // Location functionality can be added later
                            //   }
                            // }
                          ]}
                        />
                      </Popover>

                      <div style={{ flex: 1 }}>
                        <TextField
                          ref={messageInputRef}
                          placeholder="Type a message... (Press Enter to send, Shift+Enter for new line)"
                          value={messageText}
                          onChange={setMessageText}
                          multiline={2}
                        />
                      </div>

                      <EmojiPicker onEmojiSelect={handleEmojiSelect} buttonSize="medium" />

                      <Button
                        primary
                        onClick={handleSendMessage}
                        disabled={!messageText.trim() || sendingMessage || sendingReply || !isConnected}
                        loading={sendingMessage || uploadingFile || sendingReply}
                      >
                        {uploadingFile ? 'Uploading...' :
                          sendingReply ? 'Sending Reply...' :
                            sendingMessage ? 'Sending...' :
                              replyingToMessage ? 'Reply' : 'Send'}
                      </Button>
                    </InlineStack>
                  </>
                )}
              </div>


                <Popover
                  active={showTemplateSelector}
                  activator={<div />}
                  onClose={() => {
                    setShowTemplateSelector(false);
                    setTemplateSearchTerm('');
                  }}
                  preferredPosition="above"
                >
                  <div style={{ padding: '16px', minWidth: '350px' }}>
                    <Text variant="headingMd" as="h3" style={{ marginBottom: '12px' }}>
                      Select Template
                    </Text>

                    {/* Search Field */}
                    <div style={{ marginBottom: '12px' }}>
                      <TextField
                        placeholder="Search templates..."
                        value={templateSearchTerm}
                        onChange={setTemplateSearchTerm}
                        prefix={<Icon source={SearchIcon} />}
                        clearButton
                        onClearButtonClick={() => setTemplateSearchTerm('')}
                      />
                    </div>

                    <div style={{ maxHeight: '400px', overflowY: 'auto' }}>
                      {loadingTemplates ? (
                        <div style={{
                          padding: '20px',
                          textAlign: 'center',
                          color: '#667781',
                          fontSize: '14px'
                        }}>
                          <Spinner size="small" />
                          <div style={{ marginTop: '8px' }}>Loading templates...</div>
                        </div>
                      ) : filteredTemplates.length === 0 ? (
                        <div style={{
                          padding: '20px',
                          textAlign: 'center',
                          color: '#667781',
                          fontSize: '14px'
                        }}>
                          {templateSearchTerm.trim() ? 'No templates found matching your search.' : 'No templates available.'}
                        </div>
                      ) : (
                        filteredTemplates.map((template) => {
                          return (
                            <div
                              key={template.id}
                              onClick={() => handleTemplateSelect(template)}
                              style={{
                                padding: '12px',
                                borderRadius: '8px',
                                cursor: 'pointer',
                                marginBottom: '8px',
                                border: '1px solid #e1e3e5',
                                backgroundColor: '#ffffff',
                                transition: 'all 0.2s ease'
                              }}
                              onMouseEnter={(e) => {
                                e.currentTarget.style.backgroundColor = '#f6f6f7';
                                e.currentTarget.style.borderColor = '#c9cccf';
                              }}
                              onMouseLeave={(e) => {
                                e.currentTarget.style.backgroundColor = '#ffffff';
                                e.currentTarget.style.borderColor = '#e1e3e5';
                              }}
                            >
                              <div style={{ display: 'flex', alignItems: 'center', gap: '8px', marginBottom: '8px' }}>
                                <Text variant="bodyMd" fontWeight="semibold">
                                  {template.name}
                                </Text>

                              </div>

                            </div>
                          );
                        })
                      )}
                    </div>
                  </div>
                </Popover>

                {/* Template Preview Popup */}
                <Popover
                  active={showTemplatePopup}
                  activator={<div />}
                  onClose={() => {
                    setShowTemplatePopup(false);
                    setSelectedTemplateData(null);
                    setTemplateVariableMapping({});
                    setTemplateMediaMapping({});
                  }}
                  preferredPosition="above"
                >
                  <div style={{ padding: '20px', }}>
                    {loadingTemplateDetails ? (
                      <div style={{ textAlign: 'center', padding: '40px' }}>
                        <Spinner size="large" />
                        <div style={{ marginTop: '16px' }}>
                          <Text variant="bodyMd" tone="subdued">
                            Loading template details...
                          </Text>
                        </div>
                      </div>
                    ) : selectedTemplateData ? (
                      <div>
                        {/* Template Header */}
                        <div style={{ marginBottom: '20px' }}>
                          <Text variant="headingMd" as="h3" style={{ marginBottom: '8px' }}>
                            {selectedTemplateData.name}
                          </Text>
                          <div style={{ display: 'flex', gap: '12px', alignItems: 'center' }}>
                            <Badge tone={selectedTemplateData.status === 'APPROVED' ? 'success' : 'warning'}>
                              {selectedTemplateData.status}
                            </Badge>
                            <Text variant="bodySm" tone="subdued">
                              {selectedTemplateData.category}
                            </Text>
                            <Text variant="bodySm" tone="subdued">
                              {selectedTemplateData.language}
                            </Text>
                          </div>
                        </div>

                        {/* Template Preview */}
                        <div style={{ marginBottom: '20px' }}>
                          <Text variant="bodyMd" fontWeight="semibold" style={{ marginBottom: '8px' }}>
                            Preview
                          </Text>
                          <div style={{
                            border: '1px solid #e1e3e5',
                            borderRadius: '8px',
                            padding: '16px',
                            backgroundColor: '#f8f9fa'
                          }}>
                            <WhatsAppPreview
                              template={JSON.stringify({
                                sender: "",
                                campaign_id: "put_campaign_id",
                                messaging_product: "whatsapp",
                                recipient_type: "individual",
                                to: "receiver_number",
                                type: "template",
                                template: {
                                  name: selectedTemplateData.name,
                                  language: {
                                    code: selectedTemplateData.language || 'en_US'
                                  },
                                  components: selectedTemplateData.components || []
                                }
                              })}
                              variant="bubble"
                              maxWidth="100%"
                              variableMapping={{
                                customer_name: selectedCustomer?.name || 'Customer',
                                first_name: selectedCustomer?.name?.split(' ')[0] || 'Customer',
                                last_name: selectedCustomer?.name?.split(' ')[1] || '',
                                ...Object.fromEntries(
                                  Object.entries(templateVariableMapping).map(([key, mapping]) => [
                                    key.replace('body_var_', '').replace('footer_var_', '').replace('header_var_', ''),
                                    mapping.value || mapping.placeholder
                                  ])
                                )
                              }}
                            />
                          </div>
                        </div>

                        {/* Variable Mapping Fields */}
                        {Object.keys(templateVariableMapping).length > 0 && (
                          <div style={{ marginBottom: '20px' }}>
                            <Text variant="bodyMd" fontWeight="semibold" style={{ marginBottom: '12px' }}>
                              Template Variables
                            </Text>
                            <BlockStack gap="300">
                              {Object.entries(templateVariableMapping).map(([key, mapping]) => (
                                <TextField
                                  key={key}
                                  label={mapping.placeholder}
                                  value={mapping.value}
                                  onChange={(value) => handleTemplateVariableChange(key, value)}
                                  placeholder={`Enter ${mapping.placeholder.toLowerCase()}`}
                                  required={mapping.required}
                                />
                              ))}
                            </BlockStack>
                          </div>
                        )}

                        {/* Media Mapping Fields */}
                        {Object.keys(templateMediaMapping).length > 0 && (
                          <div style={{ marginBottom: '20px' }}>
                            <Text variant="bodyMd" fontWeight="semibold" style={{ marginBottom: '12px' }}>
                              Media Files
                            </Text>
                            <BlockStack gap="300">
                              {Object.entries(templateMediaMapping).map(([key, mapping]) => (
                                <div key={key}>
                                  <Text variant="bodySm" style={{ marginBottom: '8px' }}>
                                    {mapping.type === 'image' ? 'Header Image' :
                                      mapping.type === 'video' ? 'Header Video' :
                                        'Header Document'} {mapping.required && '*'}
                                  </Text>
                                  {uploadingMedia[key] ? (
                                    <div style={{
                                      display: 'flex',
                                      alignItems: 'center',
                                      gap: '12px',
                                      padding: '8px',
                                      border: '1px solid #e1e3e5',
                                      borderRadius: '4px',
                                      backgroundColor: '#f8f9fa'
                                    }}>
                                      <Spinner size="small" />
                                      <Text variant="bodySm" tone="subdued">
                                        Uploading file...
                                      </Text>
                                    </div>
                                  ) : mapping.uploadError ? (
                                    <div style={{
                                      padding: '8px',
                                      border: '1px solid #d72c0d',
                                      borderRadius: '4px',
                                      backgroundColor: '#fef7f7'
                                    }}>
                                      <Text variant="bodySm" tone="critical">
                                        Upload failed: {mapping.uploadError}
                                      </Text>
                                      <Button
                                        size="slim"
                                        variant="tertiary"
                                        onClick={() => {
                                          const input = document.createElement('input');
                                          input.type = 'file';
                                          input.accept = mapping.type === 'image' ? 'image/*' :
                                            mapping.type === 'video' ? 'video/*' :
                                              '.pdf,.doc,.docx,.txt';
                                          input.onchange = (e) => {
                                            const file = e.target.files[0];
                                            if (file) handleTemplateMediaChange(key, file);
                                          };
                                          input.click();
                                        }}
                                        style={{ marginTop: '8px' }}
                                      >
                                        Try Again
                                      </Button>
                                    </div>
                                  ) : mapping.fileUrl ? (
                                    <div style={{
                                      display: 'flex',
                                      alignItems: 'center',
                                      gap: '12px',
                                      padding: '8px',
                                      border: '1px solid #e1e3e5',
                                      borderRadius: '4px',
                                      backgroundColor: '#f8f9fa'
                                    }}>
                                      {mapping.type === 'image' && (
                                        <img
                                          src={mapping.fileUrl}
                                          alt="Preview"
                                          style={{ width: '40px', height: '40px', objectFit: 'cover', borderRadius: '4px' }}
                                        />
                                      )}
                                      <div style={{ flex: 1 }}>
                                        <Text variant="bodySm" fontWeight="medium">
                                          {mapping.fileName}
                                        </Text>
                                        <Text variant="caption" tone="subdued">
                                          {mapping.type.toUpperCase()} {mapping.uploaded && '• Uploaded'}
                                        </Text>
                                      </div>
                                      <Button
                                        size="slim"
                                        variant="tertiary"
                                        onClick={() => {
                                          const input = document.createElement('input');
                                          input.type = 'file';
                                          input.accept = mapping.type === 'image' ? 'image/*' :
                                            mapping.type === 'video' ? 'video/*' :
                                              '.pdf,.doc,.docx,.txt';
                                          input.onchange = (e) => {
                                            const file = e.target.files[0];
                                            if (file) handleTemplateMediaChange(key, file);
                                          };
                                          input.click();
                                        }}
                                      >
                                        Change
                                      </Button>
                                    </div>
                                  ) : (
                                    <Button
                                      variant="secondary"
                                      onClick={() => {
                                        const input = document.createElement('input');
                                        input.type = 'file';
                                        input.accept = mapping.type === 'image' ? 'image/*' :
                                          mapping.type === 'video' ? 'video/*' :
                                            '.pdf,.doc,.docx,.txt';
                                        input.onchange = (e) => {
                                          const file = e.target.files[0];
                                          if (file) handleTemplateMediaChange(key, file);
                                        };
                                        input.click();
                                      }}
                                    >
                                      Upload {mapping.type === 'image' ? 'Image' :
                                        mapping.type === 'video' ? 'Video' :
                                          'Document'}
                                    </Button>
                                  )}
                                </div>
                              ))}
                            </BlockStack>
                          </div>
                        )}

                        {/* Action Buttons */}
                        <div style={{ display: 'flex', gap: '12px', justifyContent: 'flex-end' }}>
                          <Button
                            onClick={() => {
                              setShowTemplatePopup(false);
                              setSelectedTemplateData(null);
                              setTemplateVariableMapping({});
                              setTemplateMediaMapping({});
                              setUploadingMedia({});
                            }}
                          >
                            Cancel
                          </Button>
                          <Button
                            primary
                            onClick={handleSendTemplate}
                            disabled={Object.values(templateVariableMapping).some(mapping =>
                              mapping.required && !mapping.value.trim()
                            ) || Object.values(templateMediaMapping).some(mapping =>
                              mapping.required && !mapping.fileUrl
                            ) || Object.values(uploadingMedia).some(uploading => uploading)}
                          >
                            Send Template
                          </Button>
                        </div>
                      </div>
                    ) : (
                      <div style={{ textAlign: 'center', padding: '40px' }}>
                        <Text variant="bodyMd" tone="subdued">
                          No template data available
                        </Text>
                      </div>
                    )}
                  </div>
                </Popover>

                {/* Hidden File Inputs */}
                <input
                  ref={imageInputRef}
                  type="file"
                  accept="image/*"
                  style={{ display: 'none' }}
                  onChange={handleImageUpload}
                />
                <input
                  ref={videoInputRef}
                  type="file"
                  accept="video/*"
                  style={{ display: 'none' }}
                  onChange={handleVideoUpload}
                />
                <input
                  ref={audioInputRef}
                  type="file"
                  accept="audio/*"
                  style={{ display: 'none' }}
                  onChange={handleAudioUpload}
                />
                <input
                  ref={documentInputRef}
                  type="file"
                  accept=".pdf,.doc,.docx,.txt,.xls,.xlsx,.ppt,.pptx"
                  style={{ display: 'none' }}
                  onChange={handleDocumentUpload}
                />
              
            </>
          ) : (
            <div style={{ flex: 1, display: 'flex', flexDirection: 'column', backgroundColor: '#f8f9fa', alignItems: 'center', justifyContent: 'center' }}>
              <div
                style={{

                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center',
                  flexDirection: 'column',
                  gap: '16px'
                }}
              >
                <Icon source={ChatIcon} tone="subdued" />
                <BlockStack gap="100" align="center">
                  <Text variant="headingMd" tone="subdued">
                    Select a customer to start chatting
                  </Text>

                </BlockStack>
              </div>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}