import {
  Page, 
  Badge, 
  LegacyCard, 
  Button, 
  Modal, 
  TextContainer,
  Text,
  InlineStack,
  BlockStack,
  TextField,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  Pagination,
  Icon,
  Tooltip,
  Banner,
  Spinner,
} from '@shopify/polaris';
import { ViewIcon, PlusIcon, RefreshIcon } from '@shopify/polaris-icons';
import React, { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { messageTemplateOptions, getTemplateByLabel, getTemplateById } from '../utils/messageTemplates';
import { fetchBroadcastList } from '../utils/broadcastApi';
import { getSelectedTemplatebyName } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';

function BroadcastPage() {
  const navigate = useNavigate();
  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
  
  const [selectedBroadcast, setSelectedBroadcast] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  const [broadcasts, setBroadcasts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [syncing, setSyncing] = useState(false);
  const [syncMessage, setSyncMessage] = useState(null);
  
  // Template preview states
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [templateError, setTemplateError] = useState(null);
  

  const [itemStrings, setItemStrings] = useState([
    'All',
    'In Progress',
    'Waiting',
    'Paused',
    'Finished',
    'Payment Issues'
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [campaignFilter, setCampaignFilter] = useState('');
  const [dateRange, setDateRange] = useState([0, 30]);
  
  const {mode, setMode} = useSetIndexFiltersMode();

 
  const [currentPage, setCurrentPage] = useState(1);
  const [paginationData, setPaginationData] = useState({
    current_page: 1,
    last_page: 1,
    per_page: 20,
    total: 0,
    from: 0,
    to: 0
  });
  const itemsPerPage = 20;

  // Transform API data to match UI structure
  const transformApiData = (apiData) => {
    if (!apiData || !apiData.WhatsAppBusinessCampaign || !apiData.WhatsAppBusinessCampaign.data) {
      return [];
    }

    return apiData.WhatsAppBusinessCampaign.data.map(campaign => ({
      id: campaign.id.toString(),
      broadcastName: campaign.name,
      campaignId:  campaign.id.toString(),
      status: campaign.status.charAt(0).toUpperCase() + campaign.status.slice(1),
      scheduledOn: new Date(campaign.schedule).toLocaleDateString() + ' ' + new Date(campaign.schedule).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'}),
      targetAudience: campaign.failed + campaign.pending +campaign.totalsent+campaign.totaldelivered+campaign.totalread,
      executed: campaign.executed,
      failed: campaign.failed,
      templateId: campaign.template_name || 'unknown',
      templateName: campaign.template_name || 'Unknown Template',
      templateType: 'campaign',
      createdAt: campaign.created_at.split('T')[0],
      message: campaign.message,
      totalread: campaign.totalread,
      totaldelivered: campaign.totaldelivered,
      totalsent: campaign.totalsent,
      pending: campaign.pending
    }));
  };

  
  useEffect(() => {
    const loadBroadcasts = async () => {
      try {
        setLoading(true);
        setError(null);
        
   
        const statusMap = {
          0: 'All',
          1: 'processing', 
          2: 'waiting',
          3: 'paused',
          4: 'finished',
          5: 'payment-issue'
        };
        
        const statusType = statusMap[selected] || 'All';
        
        const response = await fetchBroadcastList({
          statusType,
          page: currentPage,
          perPageData: itemsPerPage
        });
        
        if (response.status && response.WhatsAppBusinessCampaign) {
          const transformedData = transformApiData(response);
          setBroadcasts(transformedData);
          
          
          const apiPagination = response.WhatsAppBusinessCampaign;
          setPaginationData({
            current_page: apiPagination.current_page || 1,
            last_page: apiPagination.last_page || 1,
            per_page:  parseInt(apiPagination.per_page) || 20,
            total: apiPagination.total || 0,
            from: apiPagination.from || 0,
            to: apiPagination.to || 0
          });
        } else {
          throw new Error('Invalid API response format');
        }
      } catch (err) {
        console.error('Error loading broadcasts:', err);
        setError(err.message);
        setBroadcasts([]);
      } finally {
        setLoading(false);
      }
    };

    loadBroadcasts();
  }, [selected, currentPage, itemsPerPage]); 

  const handleSync = useCallback(async () => {
    try {
      setSyncing(true);
      setSyncMessage(null);
      setError(null);
      
      // Force refresh by clearing current data first
      setBroadcasts([]);
      
      // Load fresh data
      const statusMap = {
        0: 'All',
        1: 'processing', 
        2: 'waiting',
        3: 'paused',
        4: 'finished',
        5: 'payment-issue'
      };
      
      const statusType = statusMap[selected] || 'All';
      
      const response = await fetchBroadcastList({
        statusType,
        page: currentPage,
        perPageData: itemsPerPage
      });
      
      if (response.status && response.WhatsAppBusinessCampaign) {
        const transformedData = transformApiData(response);
        setBroadcasts(transformedData);
        
        const apiPagination = response.WhatsAppBusinessCampaign;
        setPaginationData({
          current_page: apiPagination.current_page || 1,
          last_page: apiPagination.last_page || 1,
          per_page: parseInt(apiPagination.per_page) || 20,
          total: apiPagination.total || 0,
          from: apiPagination.from || 0,
          to: apiPagination.to || 0
        });
        
        setSyncMessage('Broadcast data synchronized successfully!');
        
        // Clear success message after 3 seconds
        setTimeout(() => {
          setSyncMessage(null);
        }, 3000);
      } else {
        throw new Error('Invalid API response format');
      }
    } catch (err) {
      console.error('Error syncing broadcasts:', err);
      setError(err.message);
      setSyncMessage('Failed to sync broadcast data. Please try again.');
      
      // Clear error message after 5 seconds
      setTimeout(() => {
        setSyncMessage(null);
      }, 5000);
    } finally {
      setSyncing(false);
    }
  }, [selected, currentPage, itemsPerPage]); 


  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => {},
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      {
        type: 'rename',
        onAction: () => {},
        onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => {
            if (idx === index) {
              return value;
            }
            return item.content;
          });
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      {
        type: 'duplicate',
        onPrimaryAction: async (value) => {
          await sleep(1);
          duplicateView(value);
          return true;
        },
      },
      {
        type: 'edit',
      },
      {
        type: 'delete',
        onPrimaryAction: async () => {
          await sleep(1);
          deleteView(index);
          return true;
        },
      },
    ],
  }));

  const filteredBroadcasts = useMemo(() => {
 
    return broadcasts.filter((broadcast) => {
      const matchesSearch = broadcast.broadcastName.toLowerCase().includes(queryValue.toLowerCase()) ||
                           broadcast.campaignId.toLowerCase().includes(queryValue.toLowerCase());
      const matchesStatus = statusFilter.length === 0 || statusFilter.includes(broadcast.status.toLowerCase());
      const matchesCampaign = campaignFilter === '' || broadcast.campaignId.toLowerCase().includes(campaignFilter.toLowerCase());
      
      let matchesDate = true;
      if (dateRange && (dateRange[0] !== 0 || dateRange[1] !== 30)) {
        const createdDate = new Date(broadcast.createdAt);
        const today = new Date();
        const daysSinceCreated = Math.floor((today - createdDate) / (1000 * 60 * 60 * 24));
        matchesDate = daysSinceCreated >= dateRange[0] && daysSinceCreated <= dateRange[1];
      }
      
      return matchesSearch && matchesStatus && matchesCampaign && matchesDate;
    });
  }, [broadcasts, queryValue, statusFilter, campaignFilter, dateRange]);

  const sortedBroadcasts = useMemo(() => {
    const sorted = [...filteredBroadcasts];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');
    
    sorted.sort((a, b) => {
      let aValue, bValue;
      
      switch (sortKey) {
        case 'name':
          aValue = a.broadcastName.toLowerCase();
          bValue = b.broadcastName.toLowerCase();
          break;
        case 'status':
          aValue = a.status.toLowerCase();
          bValue = b.status.toLowerCase();
          break;
        case 'campaign':
          aValue = a.campaignId.toLowerCase();
          bValue = b.campaignId.toLowerCase();
          break;
        case 'audience':
          aValue = a.targetAudience;
          bValue = b.targetAudience;
          break;
        case 'scheduled':
          aValue = new Date(a.scheduledOn);
          bValue = new Date(b.scheduledOn);
          break;
        case 'executed':
          aValue = a.executed;
          bValue = b.executed;
          break;
        default:
          return 0;
      }
      
      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });
    
    return sorted;
  }, [filteredBroadcasts, sortSelected]);


  const paginatedBroadcasts = sortedBroadcasts;

  const handlePreviousPage = useCallback(() => {
    setCurrentPage(prev => Math.max(1, prev - 1));
  }, []);

  const handleNextPage = useCallback(() => {
    setCurrentPage(prev => Math.min(paginationData.last_page, prev + 1));
  }, [paginationData.last_page]);

 
  const totalItems = paginationData.total;
  const totalPages = paginationData.last_page;
  const startItem = paginationData.from;
  const endItem = paginationData.to;
  const hasNext = currentPage < paginationData.last_page;
  const hasPrevious = currentPage > 1;


  const sortOptions = [
    {label: 'Broadcast Name', value: 'name asc', directionLabel: 'A-Z'},
    {label: 'Broadcast Name', value: 'name desc', directionLabel: 'Z-A'},
    {label: 'Status', value: 'status asc', directionLabel: 'A-Z'},
    {label: 'Status', value: 'status desc', directionLabel: 'Z-A'},
    {label: 'Campaign ID', value: 'campaign asc', directionLabel: 'A-Z'},
    {label: 'Campaign ID', value: 'campaign desc', directionLabel: 'Z-A'},
    {label: 'Target Audience', value: 'audience asc', directionLabel: 'Low to High'},
    {label: 'Target Audience', value: 'audience desc', directionLabel: 'High to Low'},
    {label: 'Scheduled Date', value: 'scheduled asc', directionLabel: 'Earliest First'},
    {label: 'Scheduled Date', value: 'scheduled desc', directionLabel: 'Latest First'},
    {label: 'Executed Count', value: 'executed asc', directionLabel: 'Low to High'},
    {label: 'Executed Count', value: 'executed desc', directionLabel: 'High to Low'},
  ];

  // Function to fetch template details from API
  const fetchTemplateDetails = useCallback(async (templateName) => {
    if (!templateName || templateName === 'unknown' || templateName === 'Unknown Template') {
      return null;
    }

    try {
      setLoadingTemplateDetails(true);
      setTemplateError(null);
      
      const templateData = await getSelectedTemplatebyName(templateName);
      setSelectedTemplateData(templateData);
      
      return templateData;
    } catch (error) {
      console.error('Error fetching template details:', error);
      setTemplateError(error.message || 'Failed to fetch template details');
      return null;
    } finally {
      setLoadingTemplateDetails(false);
    }
  }, []);

  const handlePreviewClick = useCallback(async (broadcast) => {
    setSelectedBroadcast(broadcast);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setPreviewModalActive(true);
    
    // Fetch template details if template name is available
    if (broadcast.templateName && broadcast.templateName !== 'Unknown Template') {
      await fetchTemplateDetails(broadcast.templateName);
    }
  }, [fetchTemplateDetails]);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setSelectedBroadcast(null);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setLoadingTemplateDetails(false);
  }, []);

  const handleCreateBroadcast = useCallback(() => {
    console.log('Create broadcast button clicked');
    try {
      navigate('/app/create-broadcast');
    } catch (error) {
      console.error('Navigation error:', error);
  
      window.location.href = '/app/create-broadcast';
    }
  }, [navigate]);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);
  
  const handleCampaignChange = useCallback((value) => {
    setCampaignFilter(value);
  }, []);
  
  const handleDateRangeChange = useCallback((value) => {
    setDateRange(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleCampaignRemove = useCallback(() => setCampaignFilter(''), []);
  const handleDateRangeRemove = useCallback(() => setDateRange([0, 30]), []);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleCampaignRemove();
    handleDateRangeRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleCampaignRemove, handleDateRangeRemove, handleQueryValueRemove]);

  const onHandleCancel = () => {};
  const onHandleSave = async () => {
    await sleep(1);
    return true;
  };

  const primaryAction = selected === 0
    ? {
        type: 'save-as',
        onAction: onCreateNewView,
        disabled: false,
        loading: false,
      }
    : {
        type: 'save',
        onAction: onHandleSave,
        disabled: false,
        loading: false,
      };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'Inprogress': { tone: 'info' },
      'In Progress': { tone: 'info' },
      'Waiting': { tone: 'attention' },
      'Paused': { tone: 'warning' },
      'Finished': { tone: 'success' },
      'Payment Issue': { tone: 'critical' },
      'Payment Issues': { tone: 'critical' },
      'Active': { tone: 'info' },
      'Scheduled': { tone: 'attention' },
      'Completed': { tone: 'success' },
      'Draft': { tone: 'subdued' },
      'Failed': { tone: 'critical' }
    };
    
    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };

  const filters = [
    {
      key: 'status',
      label: 'Campaign Status',
      filter: (
        <ChoiceList
          title="Campaign Status"
          titleHidden
          choices={[
            {label: 'In Progress', value: 'inprogress'},
            {label: 'Waiting', value: 'waiting'},
            {label: 'Paused', value: 'paused'},
            {label: 'Finished', value: 'finished'},
            {label: 'Payment Issue', value: 'payment issue'},
            {label: 'Active', value: 'active'},
            {label: 'Scheduled', value: 'scheduled'},
            {label: 'Completed', value: 'completed'},
            {label: 'Draft', value: 'draft'},
            {label: 'Failed', value: 'failed'},
          ]}
          selected={statusFilter || []}
          onChange={handleStatusChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
    {
      key: 'campaign',
      label: 'Campaign ID',
      filter: (
        <TextField
          label="Campaign ID"
          value={campaignFilter}
          onChange={handleCampaignChange}
          autoComplete="off"
          labelHidden
        />
      ),
      shortcut: true,
    },
    {
      key: 'dateRange',
      label: 'Days Since Created',
      filter: (
        <RangeSlider
          label="Days since created"
          labelHidden
          value={dateRange || [0, 30]}
          suffix=" days"
          output
          min={0}
          max={30}
          step={1}
          onChange={handleDateRangeChange}
        />
      ),
    },
  ];

  const appliedFilters = [];
  if (statusFilter && !isEmpty(statusFilter)) {
    appliedFilters.push({
      key: 'status',
      label: disambiguateLabel('status', statusFilter),
      onRemove: handleStatusRemove,
    });
  }
  if (dateRange && (dateRange[0] !== 0 || dateRange[1] !== 30)) {
    appliedFilters.push({
      key: 'dateRange',
      label: disambiguateLabel('dateRange', dateRange),
      onRemove: handleDateRangeRemove,
    });
  }
  if (!isEmpty(campaignFilter)) {
    appliedFilters.push({
      key: 'campaign',
      label: disambiguateLabel('campaign', campaignFilter),
      onRemove: handleCampaignRemove,
    });
  }

  const resourceName = {
    singular: 'broadcast',
    plural: 'broadcasts',
  };

  const {selectedResources, allResourcesSelected, handleSelectionChange} =
    useIndexResourceState(paginatedBroadcasts);

  function disambiguateLabel(key, value) {
    switch (key) {
      case 'dateRange':
        return `Created ${value[0]}-${value[1]} days ago`;
      case 'campaign':
        return `Campaign ID contains "${value}"`;
      case 'status':
        return value.map((val) => `Status: ${val.charAt(0).toUpperCase() + val.slice(1)}`).join(', ');
      default:
        return value;
    }
  }

  function isEmpty(value) {
    if (Array.isArray(value)) {
      return value.length === 0;
    } else {
      return value === '' || value == null;
    }
  }

  const rowMarkup = paginatedBroadcasts.map(
    (broadcast, index) => (
      <IndexTable.Row
        id={broadcast.id}
        key={broadcast.id}
        selected={selectedResources.includes(broadcast.id)}
        position={index}
      >
        <IndexTable.Cell>
          <Text variant="bodyMd" fontWeight="semibold" as="span">
            {broadcast.broadcastName}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" as="span">
            {broadcast.campaignId}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span" numeric>
            {broadcast.targetAudience.toLocaleString()}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>{getStatusBadge(broadcast.status)}</IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" as="span">
            {broadcast.scheduledOn}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span" numeric>
            {broadcast.executed.toLocaleString()}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span" numeric>
            {(broadcast.totalsent+broadcast.totalread+broadcast.totaldelivered).toLocaleString()}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text as="span" numeric tone={broadcast.failed > 0 ? 'critical' : 'subdued'}>
            {broadcast.failed.toLocaleString()}
          </Text> 
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end">
            <Tooltip content="Preview template">
              <Button
                size="slim"
                onClick={(event) => {
                  event.stopPropagation();
                  handlePreviewClick(broadcast);
                }}
              >
                <Icon source={ViewIcon} />
              </Button>
            </Tooltip>
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  console.log('Debug Info:', {
    totalBroadcasts: broadcasts.length,
    filteredBroadcasts: filteredBroadcasts.length,
    sortedBroadcasts: sortedBroadcasts.length,
    paginatedBroadcasts: paginatedBroadcasts.length,
    selectedTab: selected,
    queryValue,
    statusFilter,
    dateRange,
    currentPage,
    paginationData
  });

  // Show loading state
  if (loading) {
    return (
      <Page 
        title="Broadcast Campaigns" 
        subtitle="Manage your WhatsApp broadcast campaigns"
        fullWidth
      >
        <LegacyCard>
          <div style={{ textAlign: 'center', padding: '60px 20px' }}>
            <Text variant="headingMd" as="h3">
              Loading broadcast campaigns...
            </Text>
          </div>
        </LegacyCard>
      </Page>
    );
  }

  // Show error state
  if (error) {
    return (
      <Page 
        title="Broadcast Campaigns" 
        subtitle="Manage your WhatsApp broadcast campaigns"
        fullWidth
      >
        <LegacyCard>
          <div style={{ textAlign: 'center', padding: '60px 20px' }}>
            <Text variant="headingMd" as="h3" tone="critical">
              Error loading broadcasts
            </Text>
            <Text variant="bodyMd" as="p" tone="subdued">
              {error}
            </Text>
          </div>
        </LegacyCard>
      </Page>
    );
  }

  return (
    <>
      <Page 
        title="Broadcast Campaigns" 
        subtitle="Manage your WhatsApp broadcast campaigns"
        fullWidth
        primaryAction={{
          content: (
            <InlineStack alignment="center">
              
              <Text as="span">Create Broadcast</Text>
            </InlineStack>
          ),
          onAction: handleCreateBroadcast
        }}
        secondaryActions={[
          {
            content: (
                         <InlineStack alignment="center">
                           <Icon source={RefreshIcon} />
                           <Text as="span">Sync</Text>
                         </InlineStack>
                       ),
            onAction: handleSync,
            disabled: syncing,
            loading: syncing
          }
        ]}
      >
        {/* Sync Message Banner */}
        {syncMessage && (
          <div style={{ marginBottom: '16px' }}>
            <Banner
              title={syncMessage.includes('successfully') ? 'Success' : 'Sync Error'}
              tone={syncMessage.includes('successfully') ? 'success' : 'critical'}
              onDismiss={() => setSyncMessage(null)}
            >
              <Text variant="bodyMd" as="p">
                {syncMessage}
              </Text>
            </Banner>
          </div>
        )}
        <LegacyCard>
          <IndexFilters
            sortOptions={sortOptions}
            sortSelected={sortSelected}
            queryValue={queryValue}
            queryPlaceholder="Search broadcasts..."
            onQueryChange={handleFiltersQueryChange}
            onQueryClear={() => setQueryValue('')}
            onSort={setSortSelected}
            cancelAction={{
              onAction: onHandleCancel,
              disabled: false,
              loading: false,
            }}
            tabs={tabs}
            selected={selected}
            onSelect={setSelected}
            canCreateNewView
            onCreateNewView={onCreateNewView}
            filters={filters}
            appliedFilters={appliedFilters}
            onClearAll={handleFiltersClearAll}
            mode={mode}
            setMode={setMode}
          />
          <div className="broadcast-table-wrapper">
            <style>{`
              .broadcast-table-wrapper .Polaris-IndexTable__Table th:nth-child(2),
              .broadcast-table-wrapper .Polaris-IndexTable__Table td:nth-child(2) {
                padding-left: 8px !important;
              }
              .broadcast-table-wrapper .Polaris-IndexTable__CheckboxWrapper {
                margin-right: 4px !important;
              }
            `}</style>
            <IndexTable
              resourceName={resourceName}
              itemCount={paginatedBroadcasts.length}
              selectedItemsCount={
                allResourcesSelected ? 'All' : selectedResources.length
              }
              onSelectionChange={handleSelectionChange}
              headings={[
                {title: 'Broadcast name'},
                {title: 'Campaign id'},
                {title: 'Target Audience'},
                {title: 'Status'},
                {title: 'Scheduled On'},
                {title: 'Executed'},
                 {title: 'Sent'},
                {title: 'Failed'},
               
                {title: 'Preview Template & report', alignment: 'end'},
              ]}
            emptyState={
              <div style={{ textAlign: 'center', padding: '60px 20px' }}>
                <Text variant="headingMd" as="h3">
                  No broadcasts found
                </Text>
                <Text variant="bodyMd" as="p" tone="subdued">
                  {filteredBroadcasts.length === 0 && broadcasts.length > 0 
                    ? 'Try adjusting your filters to see more results.'
                    : 'Create your first broadcast campaign to get started.'
                  }
                </Text>
              </div>
            }
          >
            {rowMarkup}
          </IndexTable>
          </div>
          
          {paginationData.total > paginationData.per_page && (
            <div
              style={{
                maxWidth: '700px',
                margin: 'auto',
                border: '1px solid var(--p-color-border)',
                marginTop: '16px'
              }}
            >
              <Pagination
                onPrevious={hasPrevious ? handlePreviousPage : undefined}
                onNext={hasNext ? handleNextPage : undefined}
                type="table"
                hasNext={hasNext}
                hasPrevious={hasPrevious}
                label={`${startItem || 0}-${endItem || 0} of ${totalItems || 0} broadcasts`}
              />
            </div>
          )}
        </LegacyCard>
      </Page>

      {selectedBroadcast && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Preview: ${selectedBroadcast.broadcastName}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
          // secondaryActions={[
          //   {
          //     content: 'Edit Campaign',
          //     onAction: () => {
          //       console.log('Edit broadcast:', selectedBroadcast.id);
          //       handleModalClose();
          //     }
          //   }
          // ]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <BlockStack gap="200">
                <InlineStack gap="200">
                  <BlockStack gap="200">
                    <Text variant="headingMd" as="h3">
                      {selectedBroadcast.broadcastName}
                    </Text>
                    <Text variant="bodyLg" as="p">
                      Campaign ID: {selectedBroadcast.campaignId}
                    </Text>
                  </BlockStack>
                </InlineStack>
                <InlineStack gap="200">
                  {getStatusBadge(selectedBroadcast.status)}
                </InlineStack> 
              </BlockStack>
              
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Template:</strong> {selectedBroadcast.templateName}
                </Text>
                <InlineStack gap="200">
              
                <Text variant="bodyMd" as="p">
                  <strong>Total Executed:</strong> {selectedBroadcast.executed}
                </Text>
                 <Text variant="bodyMd" as="p">
                  <strong>Total Sent:</strong> {selectedBroadcast.totalsent+selectedBroadcast.totaldelivered+selectedBroadcast.totalread}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Total Delivered:</strong> {selectedBroadcast.totaldelivered + selectedBroadcast.totalread}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Total Read:</strong> {selectedBroadcast.totalread}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Total Failed:</strong> {selectedBroadcast.failed}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Total Pending:</strong> {selectedBroadcast.pending}
                </Text>
               
               
                </InlineStack>
                <InlineStack gap="200">
                  <Text variant="bodyMd" as="p">
                  <strong>Execution Rate:</strong> {selectedBroadcast.executed > 0 ? 
                    (((selectedBroadcast.failed + selectedBroadcast.pending +selectedBroadcast.totalsent+selectedBroadcast.totaldelivered+selectedBroadcast.totalread) / selectedBroadcast.executed ) * 100).toFixed(1) + '%' : 
                    'N/A'} 
                   
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Delivery Rate:</strong> {selectedBroadcast.targetAudience > 0 ? 
                    (((selectedBroadcast.totaldelivered + selectedBroadcast.totalread) / (selectedBroadcast.failed + selectedBroadcast.pending +selectedBroadcast.totalsent+selectedBroadcast.totaldelivered+selectedBroadcast.totalread)) * 100).toFixed(1) + '%' : 
                    'N/A'}
                </Text>
                </InlineStack>
              </TextContainer>

              {/* Template Error Banner */}
              {templateError && (
                <Banner
                  title="Template Loading Error"
                  tone="critical"
                  onDismiss={() => setTemplateError(null)}
                >
                  <Text variant="bodyMd" as="p">
                    {templateError}
                  </Text>
                </Banner>
              )}

              {/* Template Preview Section */}
              <BlockStack gap="200">
                <Text variant="headingMd" as="h4">
                  WhatsApp Template Preview
                </Text>
                
                {loadingTemplateDetails ? (
                  <div style={{ textAlign: 'center', padding: '40px 20px' }}>
                    <Spinner accessibilityLabel="Loading template details" size="large" />
                    <Text variant="bodyMd" as="p" tone="subdued" alignment="center">
                      Loading template details...
                    </Text>
                  </div>
                ) : selectedTemplateData ? (
                  <WhatsAppPreview 
                    template={selectedTemplateData} 
                    variant="full" 
                    showWhatsAppHeader={true}
                  />
                ) : (
                  (() => {
                    try {
                      // Fallback: Try to parse the message from API
                      if (selectedBroadcast.message) {
                        const messageData = JSON.parse(selectedBroadcast.message);
                        if (messageData.template) {
                          return (
                            <WhatsAppPreview 
                              template={messageData.template} 
                              variant="full" 
                              showWhatsAppHeader={true}
                            />
                          );
                        }
                      }
                      
                      // Fallback to template library
                      const templateData = getTemplateById(selectedBroadcast.templateId);
                      if (templateData) {
                        const templateObject = JSON.parse(templateData.value);
                        return (
                          <WhatsAppPreview 
                            template={templateObject.template} 
                            variant="full" 
                            showWhatsAppHeader={true}
                          />
                        );
                      } else {
                        return (
                          <Text variant="bodyMd" as="p" tone="subdued">
                            Template preview not available for "{selectedBroadcast.templateName}"
                          </Text>
                        );
                      }
                    } catch (error) {
                      console.error('Error parsing template data:', error);
                      return (
                        <Text variant="bodyMd" as="p" tone="subdued">
                          Unable to preview template "{selectedBroadcast.templateName}"
                        </Text>
                      );
                    }
                  })()
                )}
              </BlockStack>
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}
    </>
  );
}

export default BroadcastPage;