import {
  Page,
  Text,
  InlineStack,
  Icon,
  LegacyCard,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  Badge,
  Button,
  useIndexResourceState,
  Tooltip,
  Spinner,
  Banner,
  Pagination,
  ChoiceList,
  Modal,
  BlockStack,
  TextContainer,
} from '@shopify/polaris';
import { PlusIcon, EditIcon, ViewIcon } from '@shopify/polaris-icons';
import React from 'react';
import { useState, useCallback, useEffect, useMemo } from 'react';
import { useNavigate } from '@remix-run/react';
import { fetchOrderConfirmationTriggerList } from '../utils/orderConfirmationApi';
import { getSelectedTemplatebyName } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';

/**
 * Transform API trigger data to component format
 * @param {Object} apiTrigger - Raw trigger data from API
 * @returns {Object} Transformed trigger data
 */
const transformTriggerData = (apiTrigger) => {
  return {
    id: apiTrigger.id.toString(),
    triggerName: apiTrigger.triggerName || 'Unnamed Trigger',
    templateName: apiTrigger.template_name || 'Unnamed Template',
    collections: ['All products'], 
    lastUpdated: new Date(apiTrigger.updated_at).toLocaleDateString('en-US', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit'
    }),
    isActive: apiTrigger.status === 1,
    originalData: apiTrigger // Store original API data for reference
  };
};

function TriggerIndexTable({ triggers, onToggleStatus, onViewTemplate, loading }) {
  const navigate = useNavigate();

  const resourceName = {
    singular: 'trigger',
    plural: 'triggers',
  };

  const { selectedResources, allResourcesSelected, handleSelectionChange } =
    useIndexResourceState(triggers);

  const handleToggleActive = useCallback((triggerId, event) => {
    event.stopPropagation();
    onToggleStatus(triggerId);
  }, [onToggleStatus]);

  // const handleEditTrigger = useCallback((triggerId, event) => {
  //   event.stopPropagation();
  //   navigate(`/app/edit-auto-trigger/${triggerId}`);
  // }, [navigate]);
  const handleViewTemplateTrigger = useCallback((triggerId, event) => {
    event.stopPropagation();
    onViewTemplate(triggerId);
  }, [onViewTemplate]);


  if (loading) {
    return (
      <div style={{ padding: '40px', textAlign: 'center' }}>
        <Spinner size="large" />
        <Text variant="bodyMd" color="subdued" as="p" alignment="center">
          Loading order confirmation triggers...
        </Text>
      </div>
    );
  }

  if (triggers.length === 0) {
    return (
      <div style={{ padding: '40px', textAlign: 'center' }}>
        <Text variant="bodyMd" color="subdued" as="p" alignment="center">
          No order confirmation triggers found matching your search criteria.
        </Text>
      </div>
    );
  }

  const rowMarkup = triggers.map(
    ({ id, triggerName, templateName, collections, lastUpdated, isActive }, index) => (
      <IndexTable.Row
        id={id}
        key={id}
        selected={selectedResources.includes(id)}
        position={index}
      >
        <IndexTable.Cell>
          <Text variant="bodyMd" fontWeight="bold">
            {triggerName}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">
            {templateName}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">
            {lastUpdated}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Badge tone={isActive ? 'success' : 'critical'}>
            {isActive ? 'Active' : 'Inactive'}
          </Badge>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">
          {collections.join(', ').length > 15
  ? collections.join(', ').slice(0, 15) + '...'
  : collections.join(', ')}

          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack>
            <Button
              size="slim"
              onClick={(event) => handleToggleActive(id, event)}
            >
              {isActive ? 'Deactivate' : 'Activate'}
            </Button>
          </InlineStack>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end">
            <Tooltip content="view template">
              <Button
                size="slim"
                icon={ViewIcon}
                onClick={(event) => handleViewTemplateTrigger(id, event)}
              />
            </Tooltip>
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  return (
    <IndexTable
      resourceName={resourceName}
      itemCount={triggers.length}
      selectedItemsCount={
        allResourcesSelected ? 'All' : selectedResources.length
      }
      onSelectionChange={handleSelectionChange}
      headings={[
        { title: 'Trigger Name' },
        { title: 'Template Name' },
        { title: 'Last Updated' },
        { title: 'Status' },
        { title: 'Collections' },
        { title: 'Actions' },
        { title: '' },
      ]}
    >
      {rowMarkup}
    </IndexTable>
  );
}

function OrderConfirmationTriggersPage() {
  const navigate = useNavigate();
  const [orderConfirmationData, setOrderConfirmationData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  
  // Template preview states
  const [selectedTrigger, setSelectedTrigger] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [templateError, setTemplateError] = useState(null);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['triggerName asc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [collectionFilter, setCollectionFilter] = useState([]);
  const [pagination, setPagination] = useState({
    currentPage: 1,
    totalPages: 1,
    total: 0,
    perPage: 20
  });
  
  const {mode, setMode} = useSetIndexFiltersMode();

  // Fetch triggers from API
  const fetchTriggers = useCallback(async (page = 1) => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await fetchOrderConfirmationTriggerList({
        page,
        perPageData: 20,
        TriggerCategory: 2 
      });

      if (response.status && response.data) {
        const transformedTriggers = response.data.data.map(transformTriggerData);
        setOrderConfirmationData(transformedTriggers);
        
        setPagination({
          currentPage: response.data.current_page,
          totalPages: response.data.last_page,
          total: response.data.total,
          perPage: response.data.per_page
        });
      } else {
        throw new Error('Invalid API response format');
      }
    } catch (err) {
      console.error('Error fetching order confirmation triggers:', err);
      setError('Failed to load order confirmation triggers. Please try again.');
      setOrderConfirmationData([]);
    } finally {
      setLoading(false);
    }
  }, []);

  // Load triggers on component mount
  useEffect(() => {
    fetchTriggers();
  }, [fetchTriggers]);

  // Function to fetch template details from API
  const fetchTemplateDetails = useCallback(async (templateName) => {
    if (!templateName || templateName === 'No Template') {
      return null;
    }

    try {
      setLoadingTemplateDetails(true);
      setTemplateError(null);
      
      const templateData = await getSelectedTemplatebyName(templateName);
      setSelectedTemplateData(templateData);
      
      return templateData;
    } catch (error) {
      console.error('Error fetching template details:', error);
      setTemplateError(error.message || 'Failed to fetch template details');
      return null;
    } finally {
      setLoadingTemplateDetails(false);
    }
  }, []);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);
  
  const handleCollectionChange = useCallback((value) => {
    setCollectionFilter(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleCollectionRemove = useCallback(() => setCollectionFilter([]), []);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleCollectionRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleCollectionRemove, handleQueryValueRemove]);

  // Filter triggers based on search query and filters
  const filteredTriggers = useMemo(() => {
    if (!Array.isArray(orderConfirmationData)) return [];
    
    return orderConfirmationData.filter(trigger => {
      const matchesSearch = !queryValue.trim() || (
        trigger.triggerName?.toLowerCase().includes(queryValue.toLowerCase()) ||
        trigger.templateName?.toLowerCase().includes(queryValue.toLowerCase()) ||
        trigger.collections?.some(collection => 
          collection.toLowerCase().includes(queryValue.toLowerCase())
        )
      );
      
      const matchesStatus = !Array.isArray(statusFilter) || statusFilter.length === 0 || statusFilter.includes(trigger.isActive ? 'active' : 'inactive');
      const matchesCollection = !Array.isArray(collectionFilter) || collectionFilter.length === 0 || 
        trigger.collections?.some(collection => collectionFilter.includes(collection.toLowerCase()));
      
      return matchesSearch && matchesStatus && matchesCollection;
    });
  }, [orderConfirmationData, queryValue, statusFilter, collectionFilter]);

  // Sort options
  const sortOptions = [
    {label: 'Trigger Name', value: 'triggerName asc', directionLabel: 'A-Z'},
    {label: 'Trigger Name', value: 'triggerName desc', directionLabel: 'Z-A'},
    {label: 'Template Name', value: 'templateName asc', directionLabel: 'A-Z'},
    {label: 'Template Name', value: 'templateName desc', directionLabel: 'Z-A'},
    {label: 'Last Updated', value: 'lastUpdated asc', directionLabel: 'Oldest first'},
    {label: 'Last Updated', value: 'lastUpdated desc', directionLabel: 'Newest first'},
  ];

  // Filters configuration
  const filters = [
    {
      key: 'status',
      label: 'Status',
      filter: (
        <ChoiceList
          title="Status"
          titleHidden
          choices={[
            {label: 'Active', value: 'active'},
            {label: 'Inactive', value: 'inactive'},
          ]}
          selected={statusFilter || []}
          onChange={handleStatusChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
    {
      key: 'collection',
      label: 'Collection',
      filter: (
        <ChoiceList
          title="Collection"
          titleHidden
          choices={[
            {label: 'All Products', value: 'all products'},
            {label: 'Electronics', value: 'electronics'},
            {label: 'Clothing', value: 'clothing'},
            {label: 'Home & Garden', value: 'home & garden'},
          ]}
          selected={collectionFilter || []}
          onChange={handleCollectionChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
  ];

  // Applied filters
  const appliedFilters = [];
  if (Array.isArray(statusFilter) && statusFilter.length > 0) {
    appliedFilters.push({
      key: 'status',
      label: `Status: ${statusFilter.map(s => s.charAt(0).toUpperCase() + s.slice(1)).join(', ')}`,
      onRemove: handleStatusRemove,
    });
  }
  if (Array.isArray(collectionFilter) && collectionFilter.length > 0) {
    appliedFilters.push({
      key: 'collection',
      label: `Collection: ${collectionFilter.map(c => c.charAt(0).toUpperCase() + c.slice(1)).join(', ')}`,
      onRemove: handleCollectionRemove,
    });
  }

  const handleCreateTrigger = useCallback(() => {
    navigate('/app/auto-triggers/create-order-confirmation');
  }, [navigate]);

  const handlePageChange = useCallback((page) => {
    fetchTriggers(page);
  }, [fetchTriggers]);

  const handleRefresh = useCallback(() => {
    fetchTriggers(pagination.currentPage);
  }, [fetchTriggers, pagination.currentPage]);

  const dismissError = useCallback(() => {
    setError(null);
  }, []);

  const handleViewTemplateTrigger = useCallback(async (triggerId) => {
    // Find the trigger data
    const trigger = orderConfirmationData.find(t => t.id === triggerId);
    if (!trigger) return;
    
    setSelectedTrigger(trigger);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setPreviewModalActive(true);
    
    // Fetch template details if template name is available
    if (trigger.templateName && trigger.templateName !== 'No Template') {
      await fetchTemplateDetails(trigger.templateName);
    }
  }, [orderConfirmationData, fetchTemplateDetails]);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setSelectedTrigger(null);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setLoadingTemplateDetails(false);
  }, []);

  const handleToggleStatus = useCallback((triggerId) => {
    setOrderConfirmationData(prev =>
      prev.map(trigger => {
        if (trigger.id === triggerId) {
        
          return { ...trigger, isActive: !trigger.isActive };
        } else {
       
          const clickedTrigger = prev.find(t => t.id === triggerId);
          if (clickedTrigger && !clickedTrigger.isActive) {
            return { ...trigger, isActive: false };
          }
          return trigger;
        }
      })
    );
  }, []);

  return (
    <Page
      title="Order Confirmation Triggers"
      subtitle="Send automated order updates and confirmations via WhatsApp"
      fullWidth
      primaryAction={{
        content: (
          <InlineStack alignment="center">
            
            <Text as="span">Create Order Trigger</Text>
          </InlineStack>
        ),
        onAction: handleCreateTrigger
      }}
      secondaryActions={[
        {
          content: 'Refresh',
          onAction: handleRefresh
        },
        {
          content: 'Back to Auto Triggers',
          onAction: () => navigate('/app/auto-triggers')
        }
      ]}
    >
      {error && (
        <Banner
          title="Error loading triggers"
          tone="critical"
          onDismiss={dismissError}
        >
          <Text variant="bodyMd">{error}</Text>
        </Banner>
      )}
      
      <LegacyCard>
        <IndexFilters
          sortOptions={sortOptions}
          sortSelected={sortSelected}
          queryValue={queryValue}
          queryPlaceholder="Search triggers by name, template, collection..."
          onQueryChange={handleFiltersQueryChange}
          onQueryClear={handleQueryValueRemove}
          onSort={setSortSelected}
          filters={filters}
          appliedFilters={appliedFilters}
          onClearAll={handleFiltersClearAll}
          mode={mode}
          setMode={setMode}
          tabs={[]}
          selected={0}
          onSelect={() => {}}
        />
        
        <TriggerIndexTable
          triggers={filteredTriggers}
          onToggleStatus={handleToggleStatus}
          onViewTemplate={handleViewTemplateTrigger}
          loading={loading}
        />
        
        {!loading && pagination.totalPages > 1 && (
          <LegacyCard.Section>
            <InlineStack align="center">
              <Pagination
                hasPrevious={pagination.currentPage > 1}
                onPrevious={() => handlePageChange(pagination.currentPage - 1)}
                hasNext={pagination.currentPage < pagination.totalPages}
                onNext={() => handlePageChange(pagination.currentPage + 1)}
                label={`Page ${pagination.currentPage} of ${pagination.totalPages}`}
              />
            </InlineStack>
          </LegacyCard.Section>
        )}
      </LegacyCard>

      {/* Template Preview Modal */}
      {selectedTrigger && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Preview: ${selectedTrigger.triggerName}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
          // secondaryActions={[
          //   {
          //     content: 'Edit Trigger',
          //     onAction: () => {
          //       console.log('Edit trigger:', selectedTrigger.id);
          //       handleModalClose();
          //     }
          //   }
          // ]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <BlockStack gap="200">
                <InlineStack gap="200">
                  <BlockStack gap="200">
                    <Text variant="headingMd" as="h3">
                      {selectedTrigger.triggerName}
                    </Text>
                    <Text variant="bodyLg" as="p">
                      Template: {selectedTrigger.templateName}
                    </Text>
                  </BlockStack>
                </InlineStack>
                <InlineStack gap="200">
                  <Badge tone={selectedTrigger.isActive ? 'success' : 'critical'}>
                    {selectedTrigger.isActive ? 'Active' : 'Inactive'}
                  </Badge>
                </InlineStack> 
              </BlockStack>
              
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Collections:</strong> {selectedTrigger.collections.join(', ')}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Last Updated:</strong> {selectedTrigger.lastUpdated}
                </Text>
              </TextContainer>

              {/* Template Error Banner */}
              {templateError && (
                <Banner
                  title="Template Loading Error"
                  tone="critical"
                  onDismiss={() => setTemplateError(null)}
                >
                  <Text variant="bodyMd" as="p">
                    {templateError}
                  </Text>
                </Banner>
              )}

              {/* Template Preview Section */}
              <BlockStack gap="200">
                <Text variant="headingMd" as="h4">
                  WhatsApp Template Preview
                </Text>
                
                {loadingTemplateDetails ? (
                  <div style={{ textAlign: 'center', padding: '40px 20px' }}>
                    <Spinner accessibilityLabel="Loading template details" size="large" />
                    <Text variant="bodyMd" as="p" tone="subdued" alignment="center">
                      Loading template details...
                    </Text>
                  </div>
                ) : selectedTemplateData ? (
                  <WhatsAppPreview 
                    template={selectedTemplateData} 
                    variant="full" 
                    showWhatsAppHeader={true}
                  />
                ) : (
                  <Text variant="bodyMd" as="p" tone="subdued">
                    {selectedTrigger.templateName === 'No Template' 
                      ? 'No template assigned to this trigger'
                      : `Template preview not available for "${selectedTrigger.templateName}"`
                    }
                  </Text>
                )}
              </BlockStack>
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}
    </Page>
  );
}

export default OrderConfirmationTriggersPage;
