import {
  Page,
  Card,
  FormLayout,
  TextField,
  Select,
  Button,
  ButtonGroup,
  Text,
  InlineStack,
  BlockStack,
  Banner,
  Checkbox,
  Grid,
  Tooltip,
  LegacyCard,
  Combobox,
  Listbox,
  Tag,
  Autocomplete,
  Spinner,
  Thumbnail,
  Badge
} from '@shopify/polaris';
import { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { extractVariablesFromWhatsAppMessage } from '../utils/whatsappMessagePreview';
import { messageTemplateOptions } from '../utils/messageTemplates';
import { getAllTemplateList, getSelectedTemplatebyName, saveTemporaryFile, getAllTemplateListForOrderConfirmation } from '../utils/templateApi';
import { createAbandonedCartTrigger } from '../utils/abandonedCartApi';
import { fetchallCategoryList } from '../utils/productApi';
import { autoReplyToken } from '../utils/apiConfig';
import EmojiPicker from '../components/EmojiPicker';
import WhatsAppPreview from '../components/WhatsAppPreview';

function CreateOrderConfirmationTriggerPage() {
  const navigate = useNavigate();
  
  // Form state - fixed to order confirmation type
  const [triggerName, setTriggerName] = useState('');
  const triggerType = 'order_confirmation'; 
  const [selectedTemplate, setSelectedTemplate] = useState('');
  const [selectedTemplateId, setSelectedTemplateId] = useState('');
  const [templateInputValue, setTemplateInputValue] = useState('');
  const [messageTemplate, setMessageTemplate] = useState('');
  const [delayTime, setDelayTime] = useState('5');
  const [delayUnit, setDelayUnit] = useState('minutes');
  const [isActive, setIsActive] = useState(true);
  const targetAudience = ['order_confirmation_customers']; 
  const [variableSelections, setVariableSelections] = useState({});
  const [filterByCollection, setFilterByCollection] = useState(false);
  const [selectedCollections, setSelectedCollections] = useState([]);
  const [collectionInputValue, setCollectionInputValue] = useState('');
  
  // API Collections state
  const [apiCollections, setApiCollections] = useState([]);
  const [loadingCollections, setLoadingCollections] = useState(false);
  const [collectionsError, setCollectionsError] = useState(null);
  
  // API Template state
  const [apiTemplates, setApiTemplates] = useState([]);
  const [loadingTemplates, setLoadingTemplates] = useState(false);
  const [templateError, setTemplateError] = useState(null);
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [mediaMapping, setMediaMapping] = useState({});
  const [uploadingMedia, setUploadingMedia] = useState({});
  
  // URL button variables state
  const [urlButtonVariables, setUrlButtonVariables] = useState({});
  
  // API state for creating trigger
  const [creatingTrigger, setCreatingTrigger] = useState(false);
  const [triggerError, setTriggerError] = useState(null);
  
  // Static variables array
  const availableVariables = [

  { label: 'order_confirmation_id', value: 'order_confirmation_id' },
  { label: 'order_confirmation_date', value: 'order_confirmation_date' },
  { label: 'order_confirmation_total_price', value: 'order_confirmation_total_price' },
  { label: 'order_confirmation_currency', value: 'order_confirmation_currency' },
  { label: 'order_confirmation_payment_status', value: 'order_confirmation_payment_status' },
  { label: 'order_confirmation_fulfillment_status', value: 'order_confirmation_fulfillment_status' },
  { label: 'order_confirmation_status', value: 'order_confirmation_status' },
  { label: 'order_confirmation_number', value: 'order_confirmation_number' },


  { label: 'order_confirmation_customer_id', value: 'order_confirmation_customer_id' },
  { label: 'order_confirmation_first_name', value: 'order_confirmation_first_name' },
  { label: 'order_confirmation_last_name', value: 'order_confirmation_last_name' },
  { label: 'order_confirmation_email', value: 'order_confirmation_email' },
  { label: 'order_confirmation_phone', value: 'order_confirmation_phone' },
  { label: 'order_confirmation_default_address', value: 'order_confirmation_default_address' },
  { label: 'order_confirmation_billing_address', value: 'order_confirmation_billing_address' },


  { label: 'order_confirmation_shipping_method', value: 'order_confirmation_shipping_method' },
  { label: 'order_confirmation_shipping_price', value: 'order_confirmation_shipping_price' },
  { label: 'order_confirmation_tracking_number', value: 'order_confirmation_tracking_number' },
  { label: 'order_confirmation_tracking_url', value: 'order_confirmation_tracking_url' },

  { label: 'order_confirmation_item_id', value: 'order_confirmation_item_id' },
  { label: 'order_confirmation_item_title', value: 'order_confirmation_item_title' },
  { label: 'order_confirmation_item_quantity', value: 'order_confirmation_item_quantity' },
  { label: 'order_confirmation_item_price', value: 'order_confirmation_item_price' },
  { label: 'order_confirmation_item_variant_title', value: 'order_confirmation_item_variant_title' },
  { label: 'order_confirmation_item_variant_image', value: 'order_confirmation_item_variant_image' },

  { label: 'order_confirmation_subtotal', value: 'order_confirmation_subtotal' },
  { label: 'order_confirmation_total_discounts', value: 'order_confirmation_total_discounts' },
  { label: 'order_confirmation_total_tax', value: 'order_confirmation_total_tax' }
];


  // Fetch API templates
  const fetchApiTemplates = useCallback(async () => {
    try {
      setLoadingTemplates(true);
      setTemplateError(null);
      const response = await getAllTemplateListForOrderConfirmation();

      if (response.status && response.data && response.data.customtemplates) {
        const transformedTemplates = response.data.customtemplates.map(template => ({
          id: template.template_id ? template.template_id.toString() : 'unknown',
          name: template.template_name || 'Unknown Template',
          templateId: template.template_id,
          userId: template.user_id,
          label: template.template_name || 'Unknown Template',
          value: template.template_id ? template.template_id.toString() : 'unknown'
        }));

        setApiTemplates(transformedTemplates);
      } else {
        console.error('Invalid API response format:', response);
        setApiTemplates([]);
      }
    } catch (error) {
      console.error('Error fetching templates:', error);
      setTemplateError(`Failed to load templates: ${error.message}`);
      setApiTemplates([]);
    } finally {
      setLoadingTemplates(false);
    }
  }, []);

  useEffect(() => {
    fetchApiTemplates();
  }, [fetchApiTemplates]);

  // Fetch API collections
  const fetchApiCollections = useCallback(async () => {
    try {
      setLoadingCollections(true);
      setCollectionsError(null);
      const response = await fetchallCategoryList({ perPageData: 200 });

      if (response.status && response.data && response.data.data) {
        const transformedCollections = response.data.data
          .filter(category => category.status === 1) // Only active categories
          .map(category => ({
            id: category.id,
            label: category.category_name,
            value: category.id.toString() // Use ID as value for selection
          }));

        setApiCollections(transformedCollections);
      } else {
        console.error('Invalid collections API response format:', response);
        setCollectionsError('Invalid API response format');
        setApiCollections([]);
      }
    } catch (error) {
      console.error('Error fetching collections:', error);
      setCollectionsError(`Failed to load collections: ${error.message}`);
      setApiCollections([]);
    } finally {
      setLoadingCollections(false);
    }
  }, []);

  // Fetch collections when component mounts
  useEffect(() => {
    fetchApiCollections();
  }, [fetchApiCollections]);
  
  // Also fetch collections when filterByCollection is enabled (if not already loaded)
  useEffect(() => {
    if (filterByCollection && apiCollections.length === 0 && !loadingCollections) {
      fetchApiCollections();
    }
  }, [filterByCollection, apiCollections.length, loadingCollections, fetchApiCollections]);

  // API template options only
  const allTemplateOptions = useMemo(() => {
    console.log('Template options:', {
      api: apiTemplates.length,
      total: apiTemplates.length,
      apiTemplates: apiTemplates
    });
    return apiTemplates;
  }, [apiTemplates]);

  // Filter templates based on search term
  const filteredTemplateOptions = useMemo(() => {
    if (!templateInputValue.trim()) return allTemplateOptions;
    
    const searchLower = templateInputValue.toLowerCase();
    return allTemplateOptions.filter(template => 
      template.label?.toLowerCase().includes(searchLower) ||
      template.name?.toLowerCase().includes(searchLower)
    );
  }, [allTemplateOptions, templateInputValue]);

  // Handlers
  const handleTriggerNameChange = useCallback((value) => setTriggerName(value), []);
  const handleTemplateChange = useCallback(async (value) => {
    setSelectedTemplateId(value);
    
    const selectedTemplate = allTemplateOptions.find(t => t.id === value || t.value === value);
    if (selectedTemplate) {
      setTemplateInputValue(selectedTemplate.label);
      setSelectedTemplate(selectedTemplate.value);
      
      try {
        setLoadingTemplateDetails(true);
        setTemplateError(null);
        
        const response = await getSelectedTemplatebyName(selectedTemplate.name);
        
        if (response.data && response.data.length > 0) {
          const templateData = response.data[0];
          setSelectedTemplateData(templateData);
          
          let templateContent = '';
          if (templateData.components) {
            templateData.components.forEach(component => {
              if (component.type === 'HEADER' && component.text) {
                templateContent += component.text + '\n\n';
              } else if (component.type === 'BODY' && component.text) {
                templateContent += component.text + '\n\n';
              } else if (component.type === 'FOOTER' && component.text) {
                templateContent += component.text;
              }
            });
          }
          setMessageTemplate(templateContent.trim());
        } else {
          console.error('No template data found');
          setTemplateError('Template not found or invalid');
          setSelectedTemplateData(null);
        }
      } catch (error) {
        console.error('Error fetching template details:', error);
        setTemplateError(`Failed to load template: ${error.message}`);
        setSelectedTemplateData(null);
      } finally {
        setLoadingTemplateDetails(false);
      }
    } else {
      setTemplateInputValue('');
      setSelectedTemplate('');
      setMessageTemplate('');
      setSelectedTemplateData(null);
    }
    
    // Reset variable selections and media mapping
    setVariableSelections({});
    setMediaMapping({});
    setUrlButtonVariables({});
  }, [allTemplateOptions]);
  const handleMessageTemplateChange = useCallback((value) => setMessageTemplate(value), []);
  const handleDelayTimeChange = useCallback((value) => setDelayTime(value), []);
  const handleDelayUnitChange = useCallback((value) => setDelayUnit(value), []);
  const handleIsActiveChange = useCallback((value) => setIsActive(value), []);
  const handleVariableSelectionChange = useCallback((variable, value) => {
    setVariableSelections(prev => ({
      ...prev,
      [variable]: value
    }));
  }, []);

  const handleUrlButtonVariableChange = useCallback((buttonIndex, value) => {
    setUrlButtonVariables(prev => ({
      ...prev,
      [buttonIndex]: value
    }));
  }, []);

  const handleFilterByCollectionChange = useCallback((value) => {
    setFilterByCollection(value);
    if (!value) {
      setSelectedCollections([]);
      setCollectionInputValue('');
    }
  }, []);

  const handleSelectedCollectionsChange = useCallback((value) => {
    setSelectedCollections(value);
  }, []);

  const handleCollectionInputChange = useCallback((value) => {
    setCollectionInputValue(value);
  }, []);

  const handleCollectionSelect = useCallback((selectedValue) => {
    if (selectedValue && !selectedCollections.includes(selectedValue) && selectedCollections.length < 5) {
      setSelectedCollections([...selectedCollections, selectedValue]);
    }
    setCollectionInputValue('');
  }, [selectedCollections]);

  const handleCollectionRemove = useCallback((collectionToRemove) => {
    setSelectedCollections(selectedCollections.filter(collection => collection !== collectionToRemove));
  }, [selectedCollections]);

  const handleMediaMappingChange = useCallback(async (mediaKey, file) => {
    if (file) {
      setUploadingMedia(prev => ({ ...prev, [mediaKey]: true }));

      try {
        const reader = new FileReader();
        reader.onload = async () => {
          try {
            const base64String = reader.result.split(',')[1];

            let fileType = 'image';
            if (file.type.startsWith('video/')) {
              fileType = 'video';
            } else if (file.type.startsWith('audio/')) {
              fileType = 'audio';
            } else if (file.type.includes('pdf') || file.type.includes('document') || file.type.includes('text')) {
              fileType = 'document';
            }

            const filePayload = {
              fileName: file.name,
              fileSrc: `data:${file.type};base64,${base64String}`,
              fileType: fileType
            };

            const uploadResponse = await saveTemporaryFile(filePayload);

            if (uploadResponse.status && uploadResponse.filename) {
              const getGabsUrl = `https://app.getgabs.com/customers/mediafile/newfiles/uploads/${uploadResponse.filename}`;

              setMediaMapping(prev => ({
                ...prev,
                [mediaKey]: {
                  ...prev[mediaKey],
                  file: file,
                  fileUrl: getGabsUrl,
                  fileName: uploadResponse.filename,
                  uploaded: true,
                  uploadError: null
                }
              }));
            } else {
              throw new Error(uploadResponse.message || 'Failed to upload file');
            }
          } catch (error) {
            console.error('Error uploading file:', error);
            setMediaMapping(prev => ({
              ...prev,
              [mediaKey]: {
                ...prev[mediaKey],
                uploadError: error.message
              }
            }));
            setTemplateError(`Failed to upload file: ${error.message}`);
          } finally {
            setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
          }
        };

        reader.onerror = () => {
          setTemplateError('Failed to read file');
          setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
        };

        reader.readAsDataURL(file);
      } catch (error) {
        console.error('Error uploading file:', error);
        setTemplateError(`Failed to upload file: ${error.message}`);
        setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
      }
    } else {
      setMediaMapping(prev => ({
        ...prev,
        [mediaKey]: {
          ...prev[mediaKey],
          file: null,
          fileUrl: null,
          fileName: null,
          uploaded: false,
          uploadError: null
        }
      }));
    }
  }, []);

  const handleSave = useCallback(async () => {
    try {
      setCreatingTrigger(true);
      setTriggerError(null);
      
      if (!triggerName.trim()) {
        throw new Error('Please enter a trigger name');
      }
      
      // Template payload structure
      if (!selectedTemplateData) {
        throw new Error('Please select a template');
      }
      
      // Build template components with media and variables
      const components = [];
      
      // Add components from selected template
      if (selectedTemplateData.components) {
        selectedTemplateData.components.forEach((component, index) => {
          if (component.type === 'HEADER') {
            if (component.format === 'IMAGE' || component.format === 'VIDEO' || component.format === 'DOCUMENT') {
              const mediaKey = `header_${index}`;
              const mediaFile = mediaMapping[mediaKey];
              
              components.push({
                type: 'HEADER',
                parameters: [{
                  type: component.format,
                  [component.format.toLowerCase()]: {
                    link: mediaFile?.fileUrl || component.example?.header_handle?.[0] || ''
                  }
                }]
              });
            } else if (component.text) {
              // Extract variables from header text
              const headerVariables = extractVariablesFromWhatsAppMessage(component.text);
              if (headerVariables.length > 0) {
                components.push({
                  type: 'HEADER',
                  parameters: headerVariables.map(variable => ({
                    type: 'text',
                    parameter_name:variableSelections[variable],
                    text: `{${variableSelections[variable]}}` 
                  }))
                });
              }
            }
          } else if (component.type === 'BODY' && component.text) {
            // Extract variables from body text
            const bodyVariables = extractVariablesFromWhatsAppMessage(component.text);
            if (bodyVariables.length > 0) {
              components.push({
                type: 'BODY',
                parameters: bodyVariables.map(variable => ({
                  type: 'text',
                  parameter_name:variableSelections[variable],
                  text: `{${variableSelections[variable]}}` 
                }))
              });
            }
          } else if (component.type === 'BUTTONS' && component.buttons) {
            // Handle URL buttons with variables
            const buttonParameters = [];
            component.buttons.forEach((button, index) => {
              if (button.type === 'URL' && button.url && /\{\{\d+\}\}/.test(button.url)) {
                buttonParameters.push({
                  type: 'button',
                  sub_type: 'URL',
                  index: index,
                  parameters: [{
                    type: 'text',
                    text: urlButtonVariables[index] || button.example?.[0] || '' 
                  }]
                });
              }
            });
            
            if (buttonParameters.length > 0) {
              components.push({
                type: 'BUTTONS',
                parameters: buttonParameters
              });
            }
          }
        });
      }
      
      const payload = {
        _token: autoReplyToken,
        template_type: 'template',
        formId: 'orderConfirmationTemplateForm',
        templateJson: {
          to: 'receiver_number',
          type: 'template',
          template: {
            name: selectedTemplateData.name || 'template_name',
            language: {
              code: selectedTemplateData.language || 'en_US'
            },
            components: components
          },
          recipient_type: 'individual',
          messaging_product: 'whatsapp'
        },
        currentChoosedTemplateJson: {
          choosedTemplate: {
            name: selectedTemplateData.name || 'template_name',
            parameter_format: selectedTemplateData.parameter_format || 'POSITIONAL',
            components: selectedTemplateData.components || [],
            language: selectedTemplateData.language || 'en_US',
            status: selectedTemplateData.status || 'APPROVED',
            category: selectedTemplateData.category || 'MARKETING',
            id: selectedTemplateData.id || selectedTemplateData.template_id || ''
          }
        },
        triggerName: triggerName,
        triggerType: 'order_confirmation',
        delayTime: delayTime,
        delayUnit: delayUnit,
        selectedCollections: filterByCollection ? JSON.stringify(selectedCollections) : '[All]'
      };
      
      console.log('Creating order confirmation trigger with payload:', payload);
      
      // Use the same API function as abandoned cart but with different payload
      const response = await createAbandonedCartTrigger(payload);
      
      console.log('Order confirmation trigger created successfully:', response);
      
      navigate('/app/auto-triggers/order-confirmation');
      
    } catch (error) {
      console.error('Error creating order confirmation trigger:', error);
      setTriggerError(error.message || 'Failed to create order confirmation trigger');
    } finally {
      setCreatingTrigger(false);
    }
  }, [triggerName, messageTemplate, selectedTemplateData, mediaMapping, variableSelections, urlButtonVariables, delayTime, delayUnit, filterByCollection, selectedCollections, navigate]);

  const handleCancel = useCallback(() => {
    navigate('/app/auto-triggers/order-confirmation');
  }, [navigate]);

  const delayUnitOptions = [
    { label: 'Minutes', value: 'minutes' },
    { label: 'Hours', value: 'hours' },
    { label: 'Days', value: 'days' }
  ];

  // Use API collections data
  const collectionOptions = useMemo(() => {
    return apiCollections;
  }, [apiCollections]);

  // Filter collections based on search input
  const filteredCollectionOptions = useMemo(() => {
    if (selectedCollections.length >= 5) return [];
    if (!collectionInputValue) return collectionOptions.filter(option => !selectedCollections.includes(option.value));
    return collectionOptions.filter(option =>
      option.label.toLowerCase().includes(collectionInputValue.toLowerCase()) &&
      !selectedCollections.includes(option.value)
    );
  }, [collectionOptions, collectionInputValue, selectedCollections]);

  // Extract variables 
  const extractVariables = useMemo(() => {
    return extractVariablesFromWhatsAppMessage(messageTemplate);
  }, [messageTemplate]);


  // Template media extraction
  const templateMedia = useMemo(() => {
    if (!selectedTemplateData || !selectedTemplateData.components) return {};
    
    const media = {};
    selectedTemplateData.components.forEach((component, index) => {
      if (component.type === 'HEADER' && (component.format === 'IMAGE' || component.format === 'VIDEO' || component.format === 'DOCUMENT')) {
        const mediaKey = `header_${index}`;
        media[mediaKey] = {
          type: component.format.toLowerCase(),
          url: component.example?.header_handle?.[0] || null,
          required: true
        };
      }
    });
    
    return media;
  }, [selectedTemplateData]);

  // Extract URL buttons from template
  const templateUrlButtons = useMemo(() => {
    if (!selectedTemplateData || !selectedTemplateData.components) return [];
    
    const urlButtons = [];
    selectedTemplateData.components.forEach((component) => {
      if (component.type === 'BUTTONS' && component.buttons) {
        component.buttons.forEach((button, index) => {
          if (button.type === 'URL' && button.url) {
            // Check if URL contains variables like {{1}}, {{2}}, etc.
            const hasVariable = /\{\{\d+\}\}/.test(button.url);
            if (hasVariable) {
              urlButtons.push({
                index: index,
                text: button.text,
                url: button.url,
                example: button.example?.[0] || ''
              });
            }
          }
        });
      }
    });
    
    return urlButtons;
  }, [selectedTemplateData]);

  // Initialize URL button variables when template changes
  useEffect(() => {
    if (templateUrlButtons.length > 0) {
      const initialUrlVariables = {};
      templateUrlButtons.forEach((button) => {
        if (!urlButtonVariables[button.index]) {
          initialUrlVariables[button.index] = button.example;
        }
      });
      if (Object.keys(initialUrlVariables).length > 0) {
        setUrlButtonVariables(prev => ({ ...prev, ...initialUrlVariables }));
      }
    }
  }, [templateUrlButtons]);

  // Check if form is valid
  const isFormValid = useMemo(() => {
    if (!triggerName.trim()) return false;
    if (!messageTemplate) return false;
    if (creatingTrigger) return false;
    
    // Check if all URL button variables are filled
    if (templateUrlButtons.length > 0) {
      for (const button of templateUrlButtons) {
        if (!urlButtonVariables[button.index] || urlButtonVariables[button.index].trim() === '') {
          return false;
        }
      }
    }
    
    return true;
  }, [triggerName, messageTemplate, creatingTrigger, templateUrlButtons, urlButtonVariables]);

  return (
    <Page
      title="Create Order Confirmation Trigger"
      subtitle="Set up automated messages for order confirmations and updates"
      backAction={{
        content: 'Order Confirmation Triggers',
        onAction: () => navigate('/app/auto-triggers/order-confirmation')
      }}
      primaryAction={{
        content: 'Save Trigger',
        onAction: handleSave,
        disabled: !isFormValid,
        loading: creatingTrigger
      }}
      secondaryActions={[
        {
          content: 'Cancel',
          onAction: handleCancel
        }
      ]}
      fullWidth
    >
      <Grid>
        <Grid.Cell columnSpan={{xs: 6, sm: 6, md: 8, lg: 8, xl: 8}}>
          <BlockStack gap="500">
            <Banner status="info">
              <Text as="p">
                Create automated messages to keep customers informed about their orders. 
                Set up confirmations, shipping updates, and delivery notifications.
              </Text>
            </Banner>
            
            {templateError && (
              <Banner status="critical" onDismiss={() => setTemplateError(null)}>
                <Text as="p">{templateError}</Text>
              </Banner>
            )}
            
            {triggerError && (
              <Banner status="critical" onDismiss={() => setTriggerError(null)}>
                <Text as="p">{triggerError}</Text>
              </Banner>
            )}

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Basic Information</Text>
                
                <FormLayout>
                  <TextField
                    label="Trigger Name"
                    value={triggerName}
                    onChange={handleTriggerNameChange}
                    placeholder="e.g., Order Confirmation - Immediate"
                    helpText="Give your trigger a descriptive name"
                    requiredIndicator
                  />

                  <TextField
                    label="Trigger Type"
                    value="Order Confirmation"
                    disabled
                    helpText="This trigger is specifically for order confirmations and updates"
                  />
                </FormLayout>
              </BlockStack>
            </Card>

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Template Selection</Text>
                
                <FormLayout>
                  {loadingTemplates ? (
                    <div style={{ display: 'flex', alignItems: 'center', gap: '8px', padding: '12px' }}>
                      <Spinner size="small" />
                      <Text variant="bodySm" tone="subdued">Loading templates...</Text>
                    </div>
                  ) : allTemplateOptions.length > 0 ? (
                    <BlockStack gap="200">
                      <Autocomplete
                        label="Search and select template"
                        options={filteredTemplateOptions.map(template => ({
                          value: template.id || template.value,
                          label: template.label
                        }))}
                        selected={selectedTemplateId ? [selectedTemplateId] : []}
                        onSelect={(selected) => {
                          const selectedValue = selected[0];
                          if (selectedValue) {
                            handleTemplateChange(selectedValue);
                          }
                        }}
                        textField={
                          <Autocomplete.TextField
                            onChange={(value) => setTemplateInputValue(value)}
                            label="Search and select template"
                            value={templateInputValue}
                            placeholder="Type to search templates..."
                            autoComplete="off"
                          />
                        }
                        loading={loadingTemplates || loadingTemplateDetails}
                        emptyState={loadingTemplates ? "Loading templates..." : "No templates found"}
                        preferredPosition="below"
                        willLoadMoreResults={false}
                      />
                      <Text variant="bodySm" tone="subdued">
                        Found {allTemplateOptions.length} templates from API
                      </Text>
                      
                      {selectedTemplateData && (
                        <Banner status="info">
                          <Text variant="bodyMd" as="p">
                            Template selected: <strong>{allTemplateOptions.find(t => t.id === selectedTemplateId || t.value === selectedTemplateId)?.label}</strong>
                            {loadingTemplateDetails && <span> (Loading details...)</span>}
                          </Text>
                        </Banner>
                      )}
                    </BlockStack>
                  ) : (
                    <div style={{ padding: '12px', textAlign: 'center' }}>
                      <Text variant="bodySm" tone="subdued">
                        No templates available. Please check your API connection.
                      </Text>
                    </div>
                  )}
                  <Text variant="bodySm" tone="subdued">
                    Search and select from available templates
                  </Text>
                </FormLayout>
              </BlockStack>
            </Card>

            {extractVariables.length > 0 && (
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Variable Mapping</Text>
                  <Text variant="bodySm" as="p" tone="subdued">
                    Map the template variables to actual data fields:
                  </Text>
                  <FormLayout>
                    {extractVariables.map((variable, index) => (
                      <FormLayout.Group key={variable}>
                        <TextField
                          label={`Variable: {{${variable}}}`}
                          value={`{{${variable}}}`}
                          disabled
                          helpText="Template variable"
                        />
                        <Select
                          label="Map to"
                          options={[
                            { label: 'Select mapping...', value: '' },
                            ...availableVariables
                          ]}
                          value={variableSelections[variable] || ''}
                          onChange={(value) => handleVariableSelectionChange(variable, value)}
                          placeholder={`${variable}`}
                        />
                      </FormLayout.Group>
                    ))}
                  </FormLayout>
                </BlockStack>
              </Card>
            )}

            {/* Media Mapping Section */}
            {Object.keys(templateMedia).length > 0 && (
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Media Files</Text>
                  <Text variant="bodySm" as="p" tone="subdued">
                    Upload media files required by this template:
                  </Text>
                  
                  <BlockStack gap="400">
                    {Object.entries(templateMedia).map(([mediaKey, mediaData]) => {
                      const uploadedFile = mediaMapping[mediaKey]?.file;
                      const fileUrl = mediaMapping[mediaKey]?.fileUrl || mediaData.url;
                      const isUploading = uploadingMedia[mediaKey];
                      const uploadError = mediaMapping[mediaKey]?.uploadError;
                      const isUploaded = mediaMapping[mediaKey]?.uploaded;
                      
                      return (
                        <BlockStack key={mediaKey} gap="200">
                          {/* File row */}
                          <InlineStack gap="200" align="space-between" blockAlign="center">
                            <InlineStack gap="200" blockAlign="center">
                              <Thumbnail
                                source={
                                  uploadedFile
                                    ? fileUrl
                                    : mediaData.url ||
                                      (mediaData.type === "document" ? "📄" : "📁")
                                }
                                alt={uploadedFile?.name || `Header ${mediaData.type}`}
                                size="small"
                              />
                              <BlockStack gap="050">
                                <InlineStack gap="200" blockAlign="center">
                                  <Text variant="bodyMd" as="p">
                                    {uploadedFile ? uploadedFile.name : `Header ${mediaData.type}`}
                                  </Text>
                                  {isUploading && (
                                    <Badge tone="info">Uploading...</Badge>
                                  )}
                                  {isUploaded && !isUploading && (
                                    <Badge tone="success">Uploaded</Badge>
                                  )}
                                </InlineStack>
                                <Text variant="bodySm" as="p" tone="subdued">
                                  {uploadedFile
                                    ? `${(uploadedFile.size / 1024 / 1024).toFixed(2)} MB`
                                    : `Original ${mediaData.type}`}
                                </Text>
                                {uploadError && (
                                  <Text variant="bodySm" as="p" tone="critical">
                                    Error: {uploadError}
                                  </Text>
                                )}
                              </BlockStack>
                            </InlineStack>

                            {/* Remove / Replace actions */}
                            {uploadedFile ? (
                              <InlineStack gap="100">
                                <Button
                                  variant="plain"
                                  tone="critical"
                                  size="slim"
                                  onClick={() => handleMediaMappingChange(mediaKey, null)}
                                  disabled={isUploading}
                                >
                                  Remove
                                </Button>
                                <Button
                                  variant="plain"
                                  size="slim"
                                  onClick={() => document.getElementById(`upload-${mediaKey}`).click()}
                                  disabled={isUploading}
                                >
                                  Replace
                                </Button>
                              </InlineStack>
                            ) : (
                              <Button
                                variant="plain"
                                size="slim"
                                onClick={() => document.getElementById(`upload-${mediaKey}`).click()}
                                disabled={isUploading}
                                loading={isUploading}
                              >
                                Upload {mediaData.type}
                              </Button>
                            )}
                          </InlineStack>

                          {/* Hidden file input */}
                          <input
                            id={`upload-${mediaKey}`}
                            type="file"
                            accept={
                              mediaData.type === "image"
                                ? "image/*"
                                : mediaData.type === "video"
                                ? "video/*"
                                : "application/pdf,.doc,.docx,.txt"
                            }
                            style={{ display: "none" }}
                            onChange={(e) => {
                              if (e.target.files?.length > 0) {
                                handleMediaMappingChange(mediaKey, e.target.files[0]);
                              }
                            }}
                          />
                        </BlockStack>
                      );
                    })}
                  </BlockStack>
                </BlockStack>
              </Card>
            )}

            {/* URL Button Variables Section */}
            {templateUrlButtons.length > 0 && (
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">URL Button Variables</Text>
                  <Text variant="bodySm" as="p" tone="subdued">
                    Configure the default values for URL button variables:
                  </Text>
                  
                  <FormLayout>
                    {templateUrlButtons.map((button, index) => (
                      <FormLayout.Group key={`url_button_${button.index}`}>
                        <TextField
                          label={`Button: ${button.text}`}
                          value={button.url}
                          disabled
                          helpText="URL template with variable placeholder"
                        />
                        <TextField
                          label="Variable value"
                          value={urlButtonVariables[button.index] || button.example}
                          onChange={(value) => handleUrlButtonVariableChange(button.index, value)}
                          placeholder="Enter default URL value"
                          helpText="This value will replace the variable in the URL"
                        />
                      </FormLayout.Group>
                    ))}
                  </FormLayout>
                </BlockStack>
              </Card>
            )}

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Timing Settings</Text>
                
                <FormLayout>
                  <FormLayout.Group>
                    <TextField
                      label="Delay Time"
                      type="number"
                      value={delayTime}
                      onChange={handleDelayTimeChange}
                      min="0"
                      helpText="How long to wait after order is placed before sending (0 for immediate)"
                    />
                    
                    <Select
                      label="Delay Unit"
                      options={delayUnitOptions}
                      value={delayUnit}
                      onChange={handleDelayUnitChange}
                    />
                  </FormLayout.Group>
                </FormLayout>
              </BlockStack>
            </Card>

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Target Audience</Text>
                
                <BlockStack gap="300">
                  <Text as="p" tone="subdued">
                    This trigger will target customers who place orders.
                  </Text>
                  <Checkbox
                    label="Order Confirmation Customers"
                    checked={true}
                    disabled={true}
                    helpText="This audience is automatically selected for order confirmation triggers"
                  />
                  
                  {/* Collection filtering */}
                  <BlockStack gap="300">
                    <Checkbox
                      label="Do you want to filter target customers based on your product category/collection?"
                      checked={filterByCollection}
                      onChange={handleFilterByCollectionChange}
                      helpText="Filter customers based on the products they have ordered"
                    />
                    
                    {filterByCollection && (
                      <FormLayout>
                        {/* Collections error banner */}
                        {collectionsError && (
                          <Banner
                            title="Collections Error"
                            tone="critical"
                            onDismiss={() => setCollectionsError(null)}
                          >
                            <p>{collectionsError}</p>
                          </Banner>
                        )}
                        
                        {/* Collections loading state */}
                        {loadingCollections && (
                          <InlineStack gap="200" align="center">
                            <Spinner accessibilityLabel="Loading collections" size="small" />
                            <Text variant="bodySm" as="p" tone="subdued">Loading collections...</Text>
                          </InlineStack>
                        )}
                        <BlockStack gap="200">
                          <InlineStack align="space-between">
                            <Text variant="bodyMd" as="h3">Select Collections (Maximum 5)</Text>
                            <Text variant="bodySm" as="p" tone={selectedCollections.length >= 5 ? "critical" : "subdued"}>
                              {selectedCollections.length}/5 selected
                            </Text>
                          </InlineStack>
                          
                          {/* Display selected collections as tags */}
                          {selectedCollections.length > 0 && (
                            <InlineStack gap="100" wrap>
                              {selectedCollections.map((collectionValue, index) => {
                                const collectionOption = collectionOptions.find(opt => opt.value === collectionValue);
                                const displayLabel = collectionOption ? collectionOption.label : collectionValue;
                                return (
                                  <Tag
                                    key={collectionValue}
                                    onRemove={() => handleCollectionRemove(collectionValue)}
                                  >
                                    {displayLabel}
                                  </Tag>
                                );
                              })}
                            </InlineStack>
                          )}
                          
                          {/* Searchable dropdown */}
                          <Combobox
                            activator={
                              <Combobox.TextField
                                onChange={handleCollectionInputChange}
                                label=""
                                labelHidden
                                value={collectionInputValue}
                                placeholder={loadingCollections ? "Loading collections..." : selectedCollections.length >= 5 ? "Maximum 5 collections selected" : "Search and select collections..."}
                                autoComplete="off"
                                disabled={selectedCollections.length >= 5 || loadingCollections}
                              />
                            }
                          >
                            {filteredCollectionOptions.length > 0 ? (
                              <Listbox onSelect={handleCollectionSelect}>
                                {filteredCollectionOptions.map((option) => (
                                  <Listbox.Option
                                    key={option.value}
                                    value={option.value}
                                    selected={selectedCollections.includes(option.value)}
                                    disabled={selectedCollections.includes(option.value)}
                                  >
                                    {option.label}
                                  </Listbox.Option>
                                ))}
                              </Listbox>
                            ) : (
                              <Listbox>
                                <Listbox.Option value="" disabled>
                                  {loadingCollections ? "Loading collections..." : selectedCollections.length >= 5 ? "Maximum collections selected" : "No collections found"}
                                </Listbox.Option>
                              </Listbox>
                            )}
                          </Combobox>
                          
                          <Text variant="bodySm" as="p" tone="subdued">
                            Only customers who order products from the selected collections will receive this trigger. Maximum 5 collections allowed.
                          </Text>
                        </BlockStack>
                      </FormLayout>
                    )}
                  </BlockStack>
                </BlockStack>
              </BlockStack>
            </Card>

            <Card>
              <InlineStack align="end">
                <ButtonGroup>
                  <Button onClick={handleCancel}>Cancel</Button>
                  <Button 
                    variant="primary" 
                    onClick={handleSave}
                    disabled={!isFormValid}
                    loading={creatingTrigger}
                  >
                    Save Trigger
                  </Button>
                </ButtonGroup>
              </InlineStack>
            </Card>
          </BlockStack>
        </Grid.Cell>
        
        <Grid.Cell columnSpan={{xs: 6, sm: 6, md: 4, lg: 4, xl: 4}}>
          <div style={{
            position: "sticky",
            top: "20px", 
            alignSelf: "flex-start",
            zIndex: 10
          }}>
            <BlockStack gap="500">
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Message Preview</Text>
                  
                  {messageTemplate ? (
                    <BlockStack gap="300">
                      <Text as="p" tone="">
                        <Text variant="bodySm" as="p">
                          {allTemplateOptions.find(t => t.id === selectedTemplateId || t.value === selectedTemplateId)?.label}
                        </Text>
                      </Text>
                      <Card sectioned background="bg-surface-secondary">
                        <WhatsAppPreview 
                          template={selectedTemplateData || messageTemplate}
                          variant="bubble"
                          maxWidth="280px"
                          mediaMapping={mediaMapping}
                        />
                      </Card>
                      
                      <Text variant="bodySm" as="p" tone="subdued">
                        Variables like {`{`}{`{`}{`customer_name`}{`}`}{`}`} or {`{`}{`{`}{`1`}{`}`}{`}`} will be replaced with actual customer data when sent.
                      </Text>
                    </BlockStack>
                  ) : (
                    <BlockStack gap="200">
                      <Text as="p" tone="subdued">
                        Select a template to see the preview
                      </Text>
                      <LegacyCard sectioned>
                        <Text variant="bodySm" as="p" tone="subdued">
                          Your message preview will appear here once you select a template.
                        </Text>
                      </LegacyCard>
                    </BlockStack>
                  )}
                </BlockStack>
              </Card>
            </BlockStack>
          </div>
        </Grid.Cell>
      </Grid>
    </Page>
  );
}

export default CreateOrderConfirmationTriggerPage;
