import {
  Page,
  Card,
  FormLayout,
  TextField,
  Select,
  Button,
  ButtonGroup,
  Text,
  InlineStack,
  BlockStack,
  Banner,
  Checkbox,
  Grid,
  Tooltip,
  LegacyCard,
  Autocomplete,
  Spinner,
  Thumbnail,
  Badge,
  Modal
} from '@shopify/polaris';
import { useState, useCallback, useMemo, useEffect } from 'react';
import { useNavigate } from '@remix-run/react';
import { extractVariablesFromWhatsAppMessage } from '../utils/whatsappMessagePreview';
import { getTemplateVariables } from '../utils/availableVariables';
import { getAllTemplateList, getSelectedTemplatebyName, saveTemporaryFile } from '../utils/templateApi';
import { createAutoReply } from '../utils/automationApi';
import { autoReplyToken } from '../utils/apiConfig';
import EmojiPicker from '../components/EmojiPicker';
import WhatsAppPreview from '../components/WhatsAppPreview';

function CreateAutoReplyTriggerPage() {
  const navigate = useNavigate();
 
  const triggerType = 'keywords_auto_reply'; 
  const [selectedTemplate, setSelectedTemplate] = useState('');
  const [selectedTemplateId, setSelectedTemplateId] = useState('');
  const [templateInputValue, setTemplateInputValue] = useState('');
  const [messageTemplate, setMessageTemplate] = useState('');
  const [isActive, setIsActive] = useState(true);
  const targetAudience = ['all_customers']; 
  const [keywords, setKeywords] = useState('');
  const [variableSelections, setVariableSelections] = useState({});
  const [messageType, setMessageType] = useState(''); 
  const [customServiceMessage, setCustomServiceMessage] = useState('');
  const [selectedText, setSelectedText] = useState('');
  const [cursorPosition, setCursorPosition] = useState(0);
  
  // API Template state
  const [apiTemplates, setApiTemplates] = useState([]);
  const [loadingTemplates, setLoadingTemplates] = useState(false);
  const [templateError, setTemplateError] = useState(null);
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [mediaMapping, setMediaMapping] = useState({});
  const [uploadingMedia, setUploadingMedia] = useState({});
  
  // URL button variables state
  const [urlButtonVariables, setUrlButtonVariables] = useState({});
  
  // API state for creating auto reply
  const [creatingAutoReply, setCreatingAutoReply] = useState(false);
  const [autoReplyError, setAutoReplyError] = useState(null);
  
  // Condition type state
  const [conditionType, setConditionType] = useState('Equal');
  
  // API variables state
  const [availableVariables, setAvailableVariables] = useState([]);
  const [loadingVariables, setLoadingVariables] = useState(true);
  const [showVariablesModal, setShowVariablesModal] = useState(false);
  const [variableSearchQuery, setVariableSearchQuery] = useState('');
  const [variableInputValues, setVariableInputValues] = useState({});

  // Fetch API templates
  const fetchApiTemplates = useCallback(async () => {
    try {
      setLoadingTemplates(true);
      setTemplateError(null);
      const response = await getAllTemplateList();

      if (response.status && response.data && response.data.customtemplates) {
        const transformedTemplates = response.data.customtemplates.map(template => ({
          id: template.template_id ? template.template_id.toString() : 'unknown',
          name: template.template_name || 'Unknown Template',
          templateId: template.template_id,
          userId: template.user_id,
          label: template.template_name || 'Unknown Template',
          value: template.template_id ? template.template_id.toString() : 'unknown'
        }));

        setApiTemplates(transformedTemplates);
      } else {
        console.error('Invalid API response format:', response);
        setApiTemplates([]);
      }
    } catch (error) {
      console.error('Error fetching templates:', error);
      setTemplateError(`Failed to load templates: ${error.message}`);
      setApiTemplates([]);
    } finally {
      setLoadingTemplates(false);
    }
  }, []);

  useEffect(() => {
    fetchApiTemplates();
  }, [fetchApiTemplates]);

  // API template options only
  const allTemplateOptions = useMemo(() => {
    console.log('Template options:', {
      api: apiTemplates.length,
      total: apiTemplates.length,
      apiTemplates: apiTemplates
    });
    return apiTemplates;
  }, [apiTemplates]);

  // Filter templates based on search term
  const filteredTemplateOptions = useMemo(() => {
    if (!templateInputValue.trim()) return allTemplateOptions;
    
    const searchLower = templateInputValue.toLowerCase();
    return allTemplateOptions.filter(template => 
      template.label?.toLowerCase().includes(searchLower) ||
      template.name?.toLowerCase().includes(searchLower)
    );
  }, [allTemplateOptions, templateInputValue]);

  const handleTemplateChange = useCallback(async (value) => {
    setSelectedTemplateId(value);
    
    const selectedTemplate = allTemplateOptions.find(t => t.id === value || t.value === value);
    if (selectedTemplate) {
      setTemplateInputValue(selectedTemplate.label);
      setSelectedTemplate(selectedTemplate.value);
      
     
      try {
        setLoadingTemplateDetails(true);
        setTemplateError(null);
        
        const response = await getSelectedTemplatebyName(selectedTemplate.name);
        
        if (response.data && response.data.length > 0) {
          const templateData = response.data[0];
          setSelectedTemplateData(templateData);
          
        
          let templateContent = '';
          if (templateData.components) {
            templateData.components.forEach(component => {
              if (component.type === 'HEADER' && component.text) {
                templateContent += component.text + '\n\n';
              } else if (component.type === 'BODY' && component.text) {
                templateContent += component.text + '\n\n';
              } else if (component.type === 'FOOTER' && component.text) {
                templateContent += component.text;
              }
            });
          }
          setMessageTemplate(templateContent.trim());
        } else {
          console.error('No template data found');
          setTemplateError('Template not found or invalid');
          setSelectedTemplateData(null);
        }
      } catch (error) {
        console.error('Error fetching template details:', error);
        setTemplateError(`Failed to load template: ${error.message}`);
        setSelectedTemplateData(null);
      } finally {
        setLoadingTemplateDetails(false);
      }
    } else {
      setTemplateInputValue('');
      setSelectedTemplate('');
      setMessageTemplate('');
      setSelectedTemplateData(null);
    }
    
    
    setVariableSelections({});
    setVariableInputValues({});
    setMediaMapping({});
    setUrlButtonVariables({});
  }, [allTemplateOptions]);

  const handleMediaMappingChange = useCallback(async (mediaKey, file) => {
    if (file) {
      setUploadingMedia(prev => ({ ...prev, [mediaKey]: true }));

      try {
        const reader = new FileReader();
        reader.onload = async () => {
          try {
            const base64String = reader.result.split(',')[1];

            let fileType = 'image';
            if (file.type.startsWith('video/')) {
              fileType = 'video';
            } else if (file.type.startsWith('audio/')) {
              fileType = 'audio';
            } else if (file.type.includes('pdf') || file.type.includes('document') || file.type.includes('text')) {
              fileType = 'document';
            }

            const filePayload = {
              fileName: file.name,
              fileSrc: `data:${file.type};base64,${base64String}`,
              fileType: fileType
            };

            const uploadResponse = await saveTemporaryFile(filePayload);

            if (uploadResponse.status && uploadResponse.filename) {
              const getGabsUrl = `https://app.getgabs.com/customers/mediafile/newfiles/uploads/${uploadResponse.filename}`;

              setMediaMapping(prev => ({
                ...prev,
                [mediaKey]: {
                  ...prev[mediaKey],
                  file: file,
                  fileUrl: getGabsUrl,
                  fileName: uploadResponse.filename,
                  uploaded: true,
                  uploadError: null
                }
              }));
            } else {
              throw new Error(uploadResponse.message || 'Failed to upload file');
            }
          } catch (error) {
            console.error('Error uploading file:', error);
            setMediaMapping(prev => ({
              ...prev,
              [mediaKey]: {
                ...prev[mediaKey],
                uploadError: error.message
              }
            }));
            setTemplateError(`Failed to upload file: ${error.message}`);
          } finally {
            setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
          }
        };

        reader.onerror = () => {
          setTemplateError('Failed to read file');
          setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
        };

        reader.readAsDataURL(file);
      } catch (error) {
        console.error('Error uploading file:', error);
        setTemplateError(`Failed to upload file: ${error.message}`);
        setUploadingMedia(prev => ({ ...prev, [mediaKey]: false }));
      }
    } else {
      setMediaMapping(prev => ({
        ...prev,
        [mediaKey]: {
          ...prev[mediaKey],
          file: null,
          fileUrl: null,
          fileName: null,
          uploaded: false,
          uploadError: null
        }
      }));
    }
  }, []);
  const handleMessageTemplateChange = useCallback((value) => setMessageTemplate(value), []);
  const handleIsActiveChange = useCallback((value) => setIsActive(value), []);
  const handleKeywordsChange = useCallback((value) => setKeywords(value), []);
  const handleVariableSelectionChange = useCallback((variable, value) => {
    setVariableSelections(prev => ({
      ...prev,
      [variable]: value
    }));
  }, []);

  const handleUrlButtonVariableChange = useCallback((buttonIndex, value) => {
    setUrlButtonVariables(prev => ({
      ...prev,
      [buttonIndex]: value
    }));
  }, []);
  const handleMessageTypeChange = useCallback((value) => {
    setMessageType(value);
    
    setSelectedTemplate('');
    setCustomServiceMessage('');
    setMessageTemplate('');
  }, []);
  const handleCustomServiceMessageChange = useCallback((value) => {
    setCustomServiceMessage(value);
    setMessageTemplate(value);
  }, []);
  
  const handleConditionTypeChange = useCallback((value) => {
    setConditionType(value);
  }, []);

  const handleTextSelection = useCallback((event) => {
    const textarea = event.target;
    setSelectedText(textarea.value.substring(textarea.selectionStart, textarea.selectionEnd));
    setCursorPosition(textarea.selectionStart);
  }, []);

  const applyFormatting = useCallback((format) => {
    const textarea = document.querySelector('#custom-service-message');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const end = textarea.selectionEnd;
    const selectedText = textarea.value.substring(start, end);
    
    let formattedText = '';
    let cursorOffset = 0;
    
    if (selectedText) {
      // If text is selected, wrap it with formatting
      switch (format) {
        case 'bold':
          formattedText = `*${selectedText}*`;
          cursorOffset = formattedText.length;
          break;
        case 'italic':
          formattedText = `_${selectedText}_`;
          cursorOffset = formattedText.length;
          break;
        case 'strikethrough':
          formattedText = `~${selectedText}~`;
          cursorOffset = formattedText.length;
          break;
        case 'monospace':
          formattedText = `\`\`\`${selectedText}\`\`\``;
          cursorOffset = formattedText.length;
          break;
        case 'inline-code':
          formattedText = `\`${selectedText}\``;
          cursorOffset = formattedText.length;
          break;
        case 'quote':
          // Ensure quote starts on new line
          const needsNewLineBeforeQuote = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterQuote = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeQuote ? '\n' : '') + `> ${selectedText}` + (needsNewLineAfterQuote ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          const bulletLines = selectedText.split('\n').map(line => line.trim() ? `* ${line}` : line).join('\n');
          // Ensure list starts on new line
          const needsNewLineBefore = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfter = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBefore ? '\n' : '') + bulletLines + (needsNewLineAfter ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          const numberedLines = selectedText.split('\n').filter(line => line.trim()).map((line, index) => `${index + 1}. ${line}`).join('\n');
          // Ensure list starts on new line
          const needsNewLineBeforeNum = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          const needsNewLineAfterNum = end < textarea.value.length && textarea.value.charAt(end) !== '\n';
          formattedText = (needsNewLineBeforeNum ? '\n' : '') + numberedLines + (needsNewLineAfterNum ? '\n' : '');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = selectedText;
          cursorOffset = formattedText.length;
      }
    } else {
      // If no text is selected, insert formatting characters with cursor between them
      switch (format) {
        case 'bold':
          formattedText = '**';
          cursorOffset = 1;
          break;
        case 'italic':
          formattedText = '__';
          cursorOffset = 1;
          break;
        case 'strikethrough':
          formattedText = '~~';
          cursorOffset = 1;
          break;
        case 'monospace':
          formattedText = '\`\`\`\n\n\`\`\`';
          cursorOffset = 4;
          break;
        case 'inline-code':
          formattedText = '\`\`';
          cursorOffset = 1;
          break;
        case 'quote':
          // Ensure quote starts on new line
          const needsNewLineBeforeQuoteEmpty = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeQuoteEmpty ? '\n> ' : '> ');
          cursorOffset = formattedText.length;
          break;
        case 'bullet-list':
          // Ensure list starts on new line
          const needsNewLineBeforeBullet = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeBullet ? '\n* ' : '* ');
          cursorOffset = formattedText.length;
          break;
        case 'numbered-list':
          // Ensure list starts on new line
          const needsNewLineBeforeNumber = start > 0 && textarea.value.charAt(start - 1) !== '\n';
          formattedText = (needsNewLineBeforeNumber ? '\n1. ' : '1. ');
          cursorOffset = formattedText.length;
          break;
        default:
          formattedText = '';
          cursorOffset = 0;
      }
    }
    
    const newValue = textarea.value.substring(0, start) + formattedText + textarea.value.substring(end);
    setCustomServiceMessage(newValue);
    setMessageTemplate(newValue);
    
    // Set cursor position
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + cursorOffset, start + cursorOffset);
    }, 0);
  }, []);

  // Keyboard shortcuts
  useEffect(() => {
    const handleKeyDown = (event) => {
      if ((event.ctrlKey || event.metaKey) && event.target.id === 'custom-service-message') {
        switch (event.key.toLowerCase()) {
          case 'b':
            event.preventDefault();
            applyFormatting('bold');
            break;
          case 'i':
            event.preventDefault();
            applyFormatting('italic');
            break;
          case 's':
            event.preventDefault();
            applyFormatting('strikethrough');
            break;
          case 'm':
            event.preventDefault();
            applyFormatting('monospace');
            break;
          case 'k':
            event.preventDefault();
            applyFormatting('inline-code');
            break;
          case 'q':
            event.preventDefault();
            applyFormatting('quote');
            break;
          case 'l':
            event.preventDefault();
            applyFormatting('bullet-list');
            break;
          case 'n':
            event.preventDefault();
            applyFormatting('numbered-list');
            break;
        }
      }
    };

    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [applyFormatting]);

  const insertVariable = useCallback((variable) => {
    const textarea = document.querySelector('#custom-service-message');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const variableText = `{{${variable}}}`;
    const newValue = textarea.value.substring(0, start) + variableText + textarea.value.substring(start);
    
    setCustomServiceMessage(newValue);
    setMessageTemplate(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + variableText.length, start + variableText.length);
    }, 0);
  }, []);

  const insertEmoji = useCallback((emoji) => {
    const textarea = document.querySelector('#custom-service-message');
    if (!textarea) return;
    
    const start = textarea.selectionStart;
    const newValue = textarea.value.substring(0, start) + emoji + textarea.value.substring(start);
    
    setCustomServiceMessage(newValue);
    setMessageTemplate(newValue);
    
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + emoji.length, start + emoji.length);
    }, 0);
  }, []);

  const handleSave = useCallback(async () => {
    try {
      setCreatingAutoReply(true);
      setAutoReplyError(null);
      
      // Parse keywords into array
      const keywordArray = keywords.split(',').map(k => k.trim()).filter(k => k.length > 0);
      
      if (keywordArray.length === 0) {
        throw new Error('Please enter at least one keyword');
      }
      
      let payload;
      
      if (messageType === 'template') {
        // Template payload structure
        if (!selectedTemplateData) {
          throw new Error('Please select a template');
        }
        
        // Build template components with media and variables
        const components = [];
        
        // Add components from selected template
        if (selectedTemplateData.components) {
          selectedTemplateData.components.forEach((component, index) => {
            if (component.type === 'HEADER') {
              if (component.format === 'IMAGE' || component.format === 'VIDEO' || component.format === 'DOCUMENT') {
                const mediaKey = `header_${index}`;
                const mediaFile = mediaMapping[mediaKey];
                
                components.push({
                  type: 'HEADER',
                  parameters: [{
                    type: component.format,
                    [component.format.toLowerCase()]: {
                      link: mediaFile?.fileUrl || component.example?.header_handle?.[0] || ''
                    }
                  }]
                });
              } else if (component.text) {
                // Extract variables from header text
                const headerVariables = extractVariablesFromWhatsAppMessage(component.text);
                if (headerVariables.length > 0) {
                  components.push({
                    type: 'HEADER',
                    parameters: headerVariables.map(variable => ({
                      type: 'text',
                      text: variableSelections[variable] || `{{${variable}}}`
                    }))
                  });
                }
              }
            } else if (component.type === 'BODY' && component.text) {
              // Extract variables from body text
              const bodyVariables = extractVariablesFromWhatsAppMessage(component.text);
              if (bodyVariables.length > 0) {
                components.push({
                  type: 'BODY',
                  parameters: bodyVariables.map(variable => ({
                    type: 'text',
                    text: variableSelections[variable] || `{{${variable}}}`
                  }))
                });
              }
            } else if (component.type === 'FOOTER' && component.text) {
              // Extract variables from footer text
              const footerVariables = extractVariablesFromWhatsAppMessage(component.text);
              if (footerVariables.length > 0) {
                components.push({
                  type: 'FOOTER',
                  parameters: footerVariables.map(variable => ({
                    type: 'text',
                    text: variableSelections[variable] || `{{${variable}}}`
                  }))
                });
              }
            } else if (component.type === 'BUTTONS' && component.buttons) {
              // Handle URL buttons with variables
              const buttonParameters = [];
              component.buttons.forEach((button, index) => {
                if (button.type === 'URL' && button.url && /\{\{\d+\}\}/.test(button.url)) {
                  buttonParameters.push({
                    type: 'button',
                    sub_type: 'URL',
                    index: index,
                    parameters: [{
                      type: 'text',
                      text: urlButtonVariables[index] || button.example?.[0] || ''
                    }]
                  });
                }
              });
              
              if (buttonParameters.length > 0) {
                components.push({
                  type: 'BUTTONS',
                  parameters: buttonParameters
                });
              }
            }
          });
        }
        
        payload = {
          _token: autoReplyToken,
          template_type: 'template',
          formId: 'autoReplytemplateTypeSelectform',
          templateJsonData: JSON.stringify({
            to: 'receiver_number',
            type: 'template',
            template: {
              name: selectedTemplateData.name || 'template_name',
              language: {
                code: selectedTemplateData.language || 'en_US'
              },
              components: components
            },
            recipient_type: 'individual',
            messaging_product: 'whatsapp'
          }),
          keyWordType: conditionType,
          keyWordValues: JSON.stringify(keywordArray)
        };
      } else if (messageType === 'service') {
        // Service message payload structure
        if (!customServiceMessage.trim()) {
          throw new Error('Please enter a service message');
        }
        
        payload = {
          _token: autoReplyToken,
          template_type: 'text',
          body: customServiceMessage,
          formId: 'autoReplytextTypeSelectform',
          keyWordType: conditionType,
          keyWordValues: JSON.stringify(keywordArray)
        };
      } else {
        throw new Error('Please select a message type');
      }
      
      console.log('Creating auto reply with payload:', payload);
      
   
      const response = await createAutoReply(payload);
      
      console.log('Auto reply created successfully:', response);
      
      
      navigate('/app/auto-triggers/auto-reply');
      
    } catch (error) {
      console.error('Error creating auto reply:', error);
      setAutoReplyError(error.message || 'Failed to create auto reply trigger');
    } finally {
      setCreatingAutoReply(false);
    }
  }, [messageTemplate, messageType, keywords, selectedTemplateData, customServiceMessage, mediaMapping, variableSelections, urlButtonVariables, navigate]);

  const handleCancel = useCallback(() => {
    navigate('/app/auto-triggers/auto-reply');
  }, [navigate]);


  const extractVariables = useMemo(() => {
    return extractVariablesFromWhatsAppMessage(messageTemplate);
  }, [messageTemplate]);

  // Initialize variable input values when variables are extracted
  useEffect(() => {
    if (extractVariables.length > 0) {
      const initialInputValues = {};
      extractVariables.forEach(variable => {
        if (!variableInputValues[variable]) {
          const selectedVar = availableVariables.find(av => av.value === variableSelections[variable]);
          initialInputValues[variable] = selectedVar?.label || '';
        }
      });
      if (Object.keys(initialInputValues).length > 0) {
        setVariableInputValues(prev => ({ ...prev, ...initialInputValues }));
      }
    }
  }, [extractVariables, variableSelections, availableVariables]);

  // Fetch variables from API on component mount
  useEffect(() => {
    const fetchVariables = async () => {
      try {
        setLoadingVariables(true);
        const variables = await getTemplateVariables();
        setAvailableVariables(variables);
      } catch (error) {
        console.error('Error fetching variables:', error);
        setAvailableVariables([]);
      } finally {
        setLoadingVariables(false);
      }
    };

    fetchVariables();
  }, []);
 
  const templateMedia = useMemo(() => {
    if (!selectedTemplateData || !selectedTemplateData.components) return {};
    
    const media = {};
    selectedTemplateData.components.forEach((component, index) => {
      if (component.type === 'HEADER' && (component.format === 'IMAGE' || component.format === 'VIDEO' || component.format === 'DOCUMENT')) {
        const mediaKey = `header_${index}`;
        media[mediaKey] = {
          type: component.format.toLowerCase(),
          url: component.example?.header_handle?.[0] || null,
          required: true
        };
      }
    });
    
    return media;
  }, [selectedTemplateData]);

  // Extract URL buttons from template
  const templateUrlButtons = useMemo(() => {
    if (!selectedTemplateData || !selectedTemplateData.components) return [];
    
    const urlButtons = [];
    selectedTemplateData.components.forEach((component) => {
      if (component.type === 'BUTTONS' && component.buttons) {
        component.buttons.forEach((button, index) => {
          if (button.type === 'URL' && button.url) {
            // Check if URL contains variables like {{1}}, {{2}}, etc.
            const hasVariable = /\{\{\d+\}\}/.test(button.url);
            if (hasVariable) {
              urlButtons.push({
                index: index,
                text: button.text,
                url: button.url,
                example: button.example?.[0] || ''
              });
            }
          }
        });
      }
    });
    
    return urlButtons;
  }, [selectedTemplateData]);

  // Initialize URL button variables when template changes
  useEffect(() => {
    if (templateUrlButtons.length > 0) {
      const initialUrlVariables = {};
      templateUrlButtons.forEach((button) => {
        if (!urlButtonVariables[button.index]) {
          initialUrlVariables[button.index] = button.example;
        }
      });
      if (Object.keys(initialUrlVariables).length > 0) {
        setUrlButtonVariables(prev => ({ ...prev, ...initialUrlVariables }));
      }
    }
  }, [templateUrlButtons]);

  // Check if form is valid
  const isFormValid = useMemo(() => {
    if (!keywords.trim()) return false;
    if (messageType === 'template' && !messageTemplate) return false;
    if (messageType === 'service' && !customServiceMessage.trim()) return false;
    if (creatingAutoReply) return false;
    
    // Check if all URL button variables are filled
    if (templateUrlButtons.length > 0) {
      for (const button of templateUrlButtons) {
        if (!urlButtonVariables[button.index] || urlButtonVariables[button.index].trim() === '') {
          return false;
        }
      }
    }
    
    return true;
  }, [keywords, messageType, messageTemplate, customServiceMessage, creatingAutoReply, templateUrlButtons, urlButtonVariables]);

  const messageTypeOptions = [
    { label: 'Select Template', value: 'template' },
    { label: 'Create Service Message', value: 'service' }
  ];
  const conditionTypeOptions = [
    { label: 'Equal', value: 'Equal' },
    { label: 'Contains', value: 'Contains' }
  ];

  return (
    <Page
      title="Create Auto Reply Trigger"
      subtitle="Set up automated keyword-based responses for customer inquiries"
      backAction={{
        content: 'Auto Reply Triggers',
        onAction: () => navigate('/app/auto-triggers/auto-reply')
      }}
      primaryAction={{
        content: 'Save Trigger',
        onAction: handleSave,
        disabled: !isFormValid,
        loading: creatingAutoReply
      }}
      secondaryActions={[
        {
          content: 'Cancel',
          onAction: handleCancel
        }
      ]}
      fullWidth
    >
      <Grid>
        <Grid.Cell columnSpan={{xs: 6, sm: 6, md: 8, lg: 8, xl: 8}}>
          <BlockStack gap="500">
            <Banner status="info">
              <Text as="p">
                Create automated responses to customer inquiries based on keywords. 
                Provide instant support and information to your customers.
              </Text>
            </Banner>
            
            {templateError && (
              <Banner status="critical" onDismiss={() => setTemplateError(null)}>
                <Text as="p">{templateError}</Text>
              </Banner>
            )}
            
            {autoReplyError && (
              <Banner status="critical" onDismiss={() => setAutoReplyError(null)}>
                <Text as="p">{autoReplyError}</Text>
              </Banner>
            )}

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Basic Information</Text>
                
                <FormLayout>
                  <TextField
                    label="Trigger Type"
                    value="Keywords Auto Reply"
                    disabled
                    helpText="This trigger responds to specific keywords in customer messages"
                  />
                </FormLayout>
              </BlockStack>
            </Card>

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Keywords Configuration</Text>
                
                <FormLayout>
                  <TextField
                    label="Keywords"
                    value={keywords}
                    onChange={handleKeywordsChange}
                    placeholder="e.g., help, support, refund"
                    helpText="Enter keywords separated by commas"
                    multiline={2}
                    requiredIndicator
                  />
                  <Select
                    label="Condition"
                    options={conditionTypeOptions}
                    value={conditionType}
                    onChange={handleConditionTypeChange}
                    placeholder="Choose condition"
                    helpText="Select whether to use a equal or contains condition"
                    requiredIndicator
                  />
                  
                  <Select
                    label="Message Type"
                    options={messageTypeOptions}
                    value={messageType}
                    onChange={handleMessageTypeChange}
                    placeholder="Choose message type"
                    helpText="Select whether to use a template or service message"
                    requiredIndicator
                  />
                </FormLayout>
              </BlockStack>
            </Card>

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Message Settings</Text>
                
                <FormLayout>
                  {/* Show template selection when template type is selected */}
                  {messageType === 'template' && (
                    <BlockStack gap="200">
                     
                      {loadingTemplates ? (
                        <div style={{ display: 'flex', alignItems: 'center', gap: '8px', padding: '12px' }}>
                          <Spinner size="small" />
                          <Text variant="bodySm" tone="subdued">Loading templates...</Text>
                        </div>
                      ) : allTemplateOptions.length > 0 ? (
                        <BlockStack gap="200">
                          <Autocomplete
                            label="Search and select template"
                            options={filteredTemplateOptions.map(template => ({
                              value: template.id || template.value,
                              label: template.label
                            }))}
                            selected={selectedTemplateId ? [selectedTemplateId] : []}
                            onSelect={(selected) => {
                              const selectedValue = selected[0];
                              if (selectedValue) {
                                handleTemplateChange(selectedValue);
                              }
                            }}
                            textField={
                              <Autocomplete.TextField
                                onChange={(value) => setTemplateInputValue(value)}
                                label="Search and select template"
                                value={templateInputValue}
                                placeholder="Type to search templates..."
                                autoComplete="off"
                              />
                            }
                            loading={loadingTemplates || loadingTemplateDetails}
                            emptyState={loadingTemplates ? "Loading templates..." : "No templates found"}
                            preferredPosition="below"
                            willLoadMoreResults={false}
                          />
                          <Text variant="bodySm" tone="subdued">
                            Found {allTemplateOptions.length} templates from API
                          </Text>
                          
                          {selectedTemplateData && (
                            <Banner status="info">
                              <Text variant="bodyMd" as="p">
                                Template selected: <strong>{allTemplateOptions.find(t => t.id === selectedTemplateId || t.value === selectedTemplateId)?.label}</strong>
                                {loadingTemplateDetails && <span> (Loading details...)</span>}
                              </Text>
                            </Banner>
                          )}
                        </BlockStack>
                      ) : (
                        <div style={{ padding: '12px', textAlign: 'center' }}>
                          <Text variant="bodySm" tone="subdued">
                            No templates available. Please check your API connection.
                          </Text>
                        </div>
                      )}
                      <Text variant="bodySm" tone="subdued">
                        Search and select from available templates
                      </Text>
                    </BlockStack>
                  )}
                  
                  {/* Show custom service message editor when service type is selected */}
                  {messageType === 'service' && (
                    <BlockStack gap="300">
                      <Text variant="bodyMd" as="h3">Custom Service Message</Text>
                      
                      {/* Variable insertion buttons */}
                      <BlockStack gap="200">
                        <Text variant="bodySm" as="p" tone="subdued">
                          Click to insert variables:
                        </Text>
                        <InlineStack gap="100" wrap>
                          {availableVariables.slice(0, 15).map((variable) => (
                            <Button
                              key={variable.value}
                              size="micro"
                              variant="tertiary"
                              onClick={() => insertVariable(variable.value)}
                            >
                              {variable.label}
                            </Button>
                          ))}
                          {availableVariables.length > 15 && (
                            <Button
                              size="micro"
                              variant="primary"
                              onClick={() => setShowVariablesModal(true)}
                            >
                              More Variables ({availableVariables.length - 15})
                            </Button>
                          )}
                        </InlineStack>
                      </BlockStack>
                      
                      {/* Custom message textarea with attached formatting controls */}
                      <div style={{ position: 'relative' }}>
                        <div style={{ position: 'absolute', bottom: 12, right: 6, zIndex: 99, display: 'flex', gap: '4px', flexWrap: 'wrap' }}>
                          <ButtonGroup variant="segmented">
                            <EmojiPicker onEmojiSelect={insertEmoji} buttonSize="micro" />
                          </ButtonGroup>
                          
                          <ButtonGroup variant="segmented">
                            <Tooltip content={<InlineStack gap="200">Bold<Text as="span" variant="bodyMd" tone="subdued">⌘B</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('bold')} accessibilityLabel="Bold">
                                <Text fontWeight="bold">B</Text>
                              </Button>
                            </Tooltip>
                            <Tooltip content={<InlineStack gap="200">Italic<Text as="span" variant="bodyMd" tone="subdued">⌘I</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('italic')} accessibilityLabel="Italic">
                                <Text as="em">I</Text>
                              </Button>
                            </Tooltip>
                            <Tooltip content={<InlineStack gap="200">Strikethrough<Text as="span" variant="bodyMd" tone="subdued">⌘S</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('strikethrough')} accessibilityLabel="Strikethrough">
                                <Text as="s">S</Text>
                              </Button>
                            </Tooltip>
                          </ButtonGroup>
                          <ButtonGroup variant="segmented">
                            <Tooltip content={<InlineStack gap="200">Monospace<Text as="span" variant="bodyMd" tone="subdued">⌘M</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('monospace')} accessibilityLabel="Monospace">
                                <Text as="code"> {'</>'}</Text>
                              </Button>
                            </Tooltip>
                            <Tooltip content={<InlineStack gap="200">Inline Code<Text as="span" variant="bodyMd" tone="subdued">⌘K</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('inline-code')} accessibilityLabel="Inline Code">
                                  <Text as="code">{"<>"}</Text>
                              </Button>
                            </Tooltip> 
                            <Tooltip content={<InlineStack gap="200">Quote<Text as="span" variant="bodyMd" tone="subdued">⌘Q</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('quote')} accessibilityLabel="Quote">
                                {'"'}
                              </Button>
                            </Tooltip>
                          </ButtonGroup>
                          <ButtonGroup variant="segmented">
                            <Tooltip content={<InlineStack gap="200">Bullet List<Text as="span" variant="bodyMd" tone="subdued">⌘L</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('bullet-list')} accessibilityLabel="Bullet List">
                                •
                              </Button>
                            </Tooltip>
                            <Tooltip content={<InlineStack gap="200">Numbered List<Text as="span" variant="bodyMd" tone="subdued">⌘N</Text></InlineStack>}>
                              <Button size="micro" onClick={() => applyFormatting('numbered-list')} accessibilityLabel="Numbered List">
                                1.
                              </Button>
                            </Tooltip>
                          </ButtonGroup>
                        </div>
                       
                        <TextField
                          id="custom-service-message"
                          label="Message Content"
                          value={customServiceMessage}
                          onChange={handleCustomServiceMessageChange}
                          onSelect={handleTextSelection}
                          multiline={8}
                          maxLength={4096}
                          placeholder="Type your custom service message here... Formatting options: *bold*, _italic_, ~strikethrough~, ```monospace```, `inline code`, > quote, * bullet list, 1. numbered list. Use keyboard shortcuts or click buttons above."
                          helpText={`${customServiceMessage.length}/4096 characters`}
                        />
                      </div>
                    </BlockStack>
                  )}

                  {extractVariables.length > 0 && (
                    <BlockStack gap="300">
                      <Text variant="bodyMd" as="h3">Variable Mapping</Text>
                      <Text variant="bodySm" as="p" tone="subdued">
                        Map the template variables to actual data fields:
                      </Text>
                      {extractVariables.map((variable, index) => (
                        <FormLayout.Group key={variable}>
                          <TextField
                            label={`Variable: {{${variable}}}`}
                            value={`{{${variable}}}`}
                            disabled
                            helpText="Template variable"
                          />
                          <Autocomplete
                            options={availableVariables
                              .filter(av => {
                                const searchValue = variableInputValues[variable] || '';
                                return searchValue === '' || av.label.toLowerCase().includes(searchValue.toLowerCase());
                              })
                              .map(av => ({
                                value: av.value,
                                label: av.label
                              }))}
                            selected={variableSelections[variable] ? [variableSelections[variable]] : []}
                            onSelect={(selected) => {
                              const value = selected.length > 0 ? selected[0] : '';
                              handleVariableSelectionChange(variable, value);
                              if (value) {
                                const selectedVar = availableVariables.find(av => av.value === value);
                                setVariableInputValues(prev => ({
                                  ...prev,
                                  [variable]: selectedVar?.label || ''
                                }));
                              } else {
                                // Clear input when no selection
                                setVariableInputValues(prev => ({
                                  ...prev,
                                  [variable]: ''
                                }));
                              }
                            }}
                            textField={
                              <Autocomplete.TextField
                                label="Map to"
                                value={variableInputValues[variable] || ''}
                                onChange={(value) => {
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: value
                                  }));
                                }}
                                placeholder="Search and choose data field"
                                autoComplete="off"
                                clearButton
                                onClearButtonClick={() => {
                                  setVariableInputValues(prev => ({
                                    ...prev,
                                    [variable]: ''
                                  }));
                                  handleVariableSelectionChange(variable, '');
                                }}
                              />
                            }
                          />
                        </FormLayout.Group>
                      ))}
                    </BlockStack>
                  )}

                  {/* Media Mapping Section */}
                  {messageType === 'template' && Object.keys(templateMedia).length > 0 && (
                    <BlockStack gap="300">
                      <Text variant="bodyMd" as="h3">Media Files</Text>
                      <Text variant="bodySm" as="p" tone="subdued">
                        Upload media files required by this template:
                      </Text>
                      
                      <BlockStack gap="400">
                        {Object.entries(templateMedia).map(([mediaKey, mediaData]) => {
                          const uploadedFile = mediaMapping[mediaKey]?.file;
                          const fileUrl = mediaMapping[mediaKey]?.fileUrl || mediaData.url;
                          const isUploading = uploadingMedia[mediaKey];
                          const uploadError = mediaMapping[mediaKey]?.uploadError;
                          const isUploaded = mediaMapping[mediaKey]?.uploaded;
                          
                          return (
                            <BlockStack key={mediaKey} gap="200">
                              {/* File row */}
                              <InlineStack gap="200" align="space-between" blockAlign="center">
                                <InlineStack gap="200" blockAlign="center">
                                  <Thumbnail
                                    source={
                                      uploadedFile
                                        ? fileUrl
                                        : mediaData.url ||
                                          (mediaData.type === "document" ? "📄" : "📁")
                                    }
                                    alt={uploadedFile?.name || `Header ${mediaData.type}`}
                                    size="small"
                                  />
                                  <BlockStack gap="050">
                                    <InlineStack gap="200" blockAlign="center">
                                      <Text variant="bodyMd" as="p">
                                        {uploadedFile ? uploadedFile.name : `Header ${mediaData.type}`}
                                      </Text>
                                      {isUploading && (
                                        <Badge tone="info">Uploading...</Badge>
                                      )}
                                      {isUploaded && !isUploading && (
                                        <Badge tone="success">Uploaded</Badge>
                                      )}
                                    </InlineStack>
                                    <Text variant="bodySm" as="p" tone="subdued">
                                      {uploadedFile
                                        ? `${(uploadedFile.size / 1024 / 1024).toFixed(2)} MB`
                                        : `Original ${mediaData.type}`}
                                    </Text>
                                    {uploadError && (
                                      <Text variant="bodySm" as="p" tone="critical">
                                        Error: {uploadError}
                                      </Text>
                                    )}
                                  </BlockStack>
                                </InlineStack>

                                {/* Remove / Replace actions */}
                                {uploadedFile ? (
                                  <InlineStack gap="100">
                                    <Button
                                      variant="plain"
                                      tone="critical"
                                      size="slim"
                                      onClick={() => handleMediaMappingChange(mediaKey, null)}
                                      disabled={isUploading}
                                    >
                                      Remove
                                    </Button>
                                    <Button
                                      variant="plain"
                                      size="slim"
                                      onClick={() => document.getElementById(`upload-${mediaKey}`).click()}
                                      disabled={isUploading}
                                    >
                                      Replace
                                    </Button>
                                  </InlineStack>
                                ) : (
                                  <Button
                                    variant="plain"
                                    size="slim"
                                    onClick={() => document.getElementById(`upload-${mediaKey}`).click()}
                                    disabled={isUploading}
                                    loading={isUploading}
                                  >
                                    Upload {mediaData.type}
                                  </Button>
                                )}
                              </InlineStack>

                              {/* Hidden file input */}
                              <input
                                id={`upload-${mediaKey}`}
                                type="file"
                                accept={
                                  mediaData.type === "image"
                                    ? "image/*"
                                    : mediaData.type === "video"
                                    ? "video/*"
                                    : "application/pdf,.doc,.docx,.txt"
                                }
                                style={{ display: "none" }}
                                onChange={(e) => {
                                  if (e.target.files?.length > 0) {
                                    handleMediaMappingChange(mediaKey, e.target.files[0]);
                                  }
                                }}
                              />
                            </BlockStack>
                          );
                        })}
                      </BlockStack>
                    </BlockStack>
                  )}
                </FormLayout>
              </BlockStack>
            </Card>

            {/* URL Button Variables Section */}
            {templateUrlButtons.length > 0 && (
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">URL Button Variables</Text>
                  <Text variant="bodySm" as="p" tone="subdued">
                    Configure the default values for URL button variables:
                  </Text>
                  
                  <FormLayout>
                    {templateUrlButtons.map((button, index) => (
                      <FormLayout.Group key={`url_button_${button.index}`}>
                        <TextField
                          label={`Button: ${button.text}`}
                          value={button.url}
                          disabled
                          helpText="URL template with variable placeholder"
                        />
                        <TextField
                          label="Default Value"
                          value={urlButtonVariables[button.index] || button.example}
                          onChange={(value) => handleUrlButtonVariableChange(button.index, value)}
                          placeholder="Enter default URL value"
                          helpText="This value will replace the variable in the URL"
                        />
                      </FormLayout.Group>
                    ))}
                  </FormLayout>
                </BlockStack>
              </Card>
            )}

            <Card>
              <BlockStack gap="400">
                <Text variant="headingMd" as="h2">Target Audience</Text>
                
                <BlockStack gap="300">
                  <Text as="p" tone="subdued">
                    This trigger will respond to all customers who send messages containing the specified keywords.
                  </Text>
                  <Checkbox
                    label="All Customers"
                    checked={true}
                    disabled={true}
                    helpText="This audience is automatically selected for auto reply triggers"
                  />
                </BlockStack>
              </BlockStack>
            </Card>

            <Card>
              <InlineStack align="end">
                <ButtonGroup>
                  <Button onClick={handleCancel} disabled={creatingAutoReply}>Cancel</Button>
                  <Button 
                    variant="primary" 
                    onClick={handleSave}
                    disabled={!isFormValid}
                    loading={creatingAutoReply}
                  >
                    Save Trigger
                  </Button>
                </ButtonGroup>
              </InlineStack>
            </Card>
          </BlockStack>
        </Grid.Cell>
        
        <Grid.Cell columnSpan={{xs: 6, sm: 6, md: 4, lg: 4, xl: 4}}>
          <div style={{
            position: "sticky",
            top: "20px", 
            alignSelf: "flex-start",
            zIndex: 10
          }}>
            <BlockStack gap="500">
              <Card>
                <BlockStack gap="400">
                  <Text variant="headingMd" as="h2">Message Preview</Text>
                  
                  {messageTemplate ? (
                    <BlockStack gap="300">
                      <Text as="p" tone="">
                        <Text variant="bodySm" as="p">
                          {allTemplateOptions?.find(option => option.value === selectedTemplate || option.id === selectedTemplateId)?.label || 'Custom Service Message'}
                        </Text>
                      </Text>
                      <Card sectioned background="bg-surface-secondary">
                        <WhatsAppPreview 
                          template={selectedTemplateData || messageTemplate}
                          variant="bubble"
                          maxWidth="280px"
                          mediaMapping={mediaMapping}
                        />
                      </Card>
                      
                      <Text variant="bodySm" as="p" tone="subdued">
                        Variables like {`{`}{`{`}{`customer_name`}{`}`}{`}`} or {`{`}{`{`}{`1`}{`}`}{`}`} will be replaced with actual customer data when sent.
                      </Text>
                    </BlockStack>
                  ) : (
                    <BlockStack gap="200">
                      <Text as="p" tone="subdued">
                        Select a message type and template to see the preview
                      </Text>
                      <LegacyCard sectioned>
                        <Text variant="bodySm" as="p" tone="subdued">
                          Your message preview will appear here once you select a message type and template.
                        </Text>
                      </LegacyCard>
                    </BlockStack>
                  )}
                </BlockStack>
              </Card>
            </BlockStack>
          </div>
        </Grid.Cell>
      </Grid>

      {/* Variables Modal */}
      <Modal
        open={showVariablesModal}
        onClose={() => {
          setShowVariablesModal(false);
          setVariableSearchQuery('');
        }}
        title="All Available Variables"
        primaryAction={{
          content: 'Close',
          onAction: () => {
            setShowVariablesModal(false);
            setVariableSearchQuery('');
          },
        }}
        size="large"
      >
        <Modal.Section>
          <BlockStack gap="400">
            <Text variant="bodyMd" tone="subdued">
              Click on any variable to insert it into your template:
            </Text>
            
            {/* Search Field */}
            <TextField
              label="Search Variables"
              value={variableSearchQuery}
              onChange={setVariableSearchQuery}
              placeholder="Type to search variables..."
              clearButton
              onClearButtonClick={() => setVariableSearchQuery('')}
              autoComplete="off"
            />
            
            <InlineStack gap="200" wrap>
              {availableVariables
                .filter(variable => 
                  variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
                  variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
                )
                .map((variable) => (
                  <Button
                    key={variable.value}
                    size="medium"
                    variant="tertiary"
                    onClick={() => {
                      insertVariable(variable.value);
                      setShowVariablesModal(false);
                      setVariableSearchQuery('');
                    }}
                  >
                    {variable.label}
                  </Button>
                ))}
            </InlineStack>
            
            {availableVariables.filter(variable => 
              variable.label.toLowerCase().includes(variableSearchQuery.toLowerCase()) ||
              variable.value.toLowerCase().includes(variableSearchQuery.toLowerCase())
            ).length === 0 && variableSearchQuery && (
              <Text variant="bodyMd" tone="subdued" alignment="center">
                No variables found matching "{variableSearchQuery}"
              </Text>
            )}
            
            <Text variant="bodySm" tone="subdued">
              You can also create custom variables using the format: {"{{variable_name}}"}
            </Text>
          </BlockStack>
        </Modal.Section>
      </Modal>
    </Page>
  );
}

export default CreateAutoReplyTriggerPage;
