import {
  Page,
  Text,
  InlineStack,
  Icon,
  LegacyCard,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  Badge,
  Button,
  useIndexResourceState,
  Tooltip,
  Spinner,
  Banner,
  Pagination,
  TextField,
  BlockStack,
  ChoiceList,
  Popover,
} from '@shopify/polaris';
import { PlusIcon, EditIcon,ViewIcon,DeleteIcon } from '@shopify/polaris-icons';
import React from 'react';
import { useState, useCallback, useEffect, useMemo } from 'react';
import { useNavigate } from '@remix-run/react';
import { fetchAutoReplyList, deleteAutoReply } from '../utils/automationApi';

/**
 * Transform API auto reply data to component format
 * @param {Object} apiData - Raw API response data
 * @returns {Array} Transformed auto reply triggers
 */
const transformAutoReplyData = (apiData) => {
  if (!apiData || !apiData.data) return [];
  
  return apiData.data.map(item => {
    let replyContent = '';
    let templateName = '';
    let replyType = item.type || 'text';
    
    try {
      const replyData = JSON.parse(item.reply || '{}');
      
    
    } catch (error) {
      console.error('Error parsing reply data:', error);
      replyContent = 'Invalid reply format';
    }
    
    return {
      id: item.id.toString(),
      keyword: item.keyword || '',
      condition: item.type_keyword === 'Equal' ? 'equals' : 'contains',
      templateName: templateName,
      replyContent: replyContent,
      type: item.template_sub_type || replyType,
      isActive: true, 
      createdAt: item.created_at,
      updatedAt: item.updated_at,
      rawData: item
    };
  });
};

function TriggerIndexTable({ triggers, onToggleStatus, onDeleteTrigger, loading, deletingTriggers = [] }) {
  const navigate = useNavigate();
  const [deletePopoverActive, setDeletePopoverActive] = useState(null);

  const resourceName = {
    singular: 'trigger',
    plural: 'triggers',
  };

  const { selectedResources, allResourcesSelected, handleSelectionChange } =
    useIndexResourceState(triggers);

  const handleToggleActive = useCallback((triggerId, event) => {
    event.stopPropagation();
    onToggleStatus(triggerId);
  }, [onToggleStatus]);

  const handleEditTrigger = useCallback((triggerId, event) => {
    event.stopPropagation();
    navigate(`/app/edit-auto-trigger/${triggerId}`);
  }, [navigate]);

  const handleViewTrigger = useCallback((triggerId, event) => {
    event.stopPropagation();
    console.log('View trigger:', triggerId);
    
  }, []);

  const handleDeleteTrigger = useCallback((triggerId, event) => {
    event.stopPropagation();
    setDeletePopoverActive(triggerId);
  }, []);

  const handleConfirmDelete = useCallback((triggerId) => {
    onDeleteTrigger(triggerId);
    setDeletePopoverActive(null);
  }, [onDeleteTrigger]);

  const handleCancelDelete = useCallback(() => {
    setDeletePopoverActive(null);
  }, []);

  if (loading) {
    return (
      <div style={{ padding: '2rem', textAlign: 'center' }}>
        <Spinner size="large" />
        <Text variant="bodyMd" color="subdued" as="p" alignment="center">
          Loading auto reply triggers...
        </Text>
      </div>
    );
  }

  if (triggers.length === 0) {
    return (
      <div style={{ padding: '2rem', textAlign: 'center' }}>
        <Text variant="bodyMd" color="subdued" as="p" alignment="center">
          No auto reply triggers found. Create your first trigger to get started.
        </Text>
      </div>
    );
  }

  const rowMarkup = triggers.map(
    ({ id, keyword, condition,   type, isActive, createdAt }, index) => (
      <IndexTable.Row
        id={id}
        key={id}
        selected={selectedResources.includes(id)}
        position={index}
      >
        <IndexTable.Cell>
          <Text variant="bodyMd" fontWeight="medium">
            If keyword
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Badge tone={condition === 'equals' ? 'info' : 'attention'}>
            {condition === 'equals' ? 'Exact Match' : 'Contains'}
          </Badge>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" fontWeight="medium">
            {keyword}
          </Text>
        </IndexTable.Cell>
       
        
        <IndexTable.Cell>
         
            {type}
        
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Badge tone={isActive ? 'success' : 'critical'}>
            {isActive ? 'Active' : 'Inactive'}
          </Badge>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodySm" color="subdued">
            {new Date(createdAt).toLocaleDateString()}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end" gap="200">
           
            <Tooltip content="View Message Preview">
              <Button
                size="slim"
                icon={ViewIcon}
                onClick={(event) => handleViewTrigger(id, event)}
              />
            </Tooltip>
            <Popover
              active={deletePopoverActive === id}
              activator={
                <Tooltip content="Delete Trigger">
                  <Button
                    size="slim"
                    icon={DeleteIcon}
                    onClick={(event) => handleDeleteTrigger(id, event)}
                    loading={deletingTriggers.includes(id)}
                    disabled={deletingTriggers.includes(id)}
                  />
                </Tooltip>
              }
              onClose={handleCancelDelete}
              ariaHaspopup={false}
              sectioned
            >
              <BlockStack gap="400">
                <Text variant="headingMd" as="h3">
                  Delete Auto Reply Trigger
                </Text>
                <Text variant="bodyMd" as="p">
                  Are you sure you want to delete the auto reply trigger for keyword "{keyword}"? This action cannot be undone.
                </Text>
                <InlineStack gap="200" align="end">
                  <Button onClick={handleCancelDelete}>Cancel</Button>
                  <Button 
                    variant="primary" 
                    tone="critical" 
                    onClick={() => handleConfirmDelete(id)}
                    loading={deletingTriggers.includes(id)}
                  >
                    Delete Trigger
                  </Button>
                </InlineStack>
              </BlockStack>
            </Popover>
            {/* <Tooltip content="Edit Trigger">
              <Button
                size="slim"
                icon={EditIcon}
                onClick={(event) => handleEditTrigger(id, event)}
              />
            </Tooltip> */}
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  return (
    <IndexTable
      resourceName={resourceName}
      itemCount={triggers.length}
      selectedItemsCount={
        allResourcesSelected ? 'All' : selectedResources.length
      }
      onSelectionChange={handleSelectionChange}
      headings={[
        { title: 'If keyword' },
        { title: 'Condition' },
        { title: 'Keyword' },
        { title: 'Type' },
        { title: 'Status' },
        { title: 'Created' },
        { title: 'Actions', alignment: 'end' },
      ]}
    >
      {rowMarkup}
    </IndexTable>
  );
}

function AutoReplyTriggersPage() {
  const navigate = useNavigate();
  const [autoReplyData, setAutoReplyData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [deletingTriggers, setDeletingTriggers] = useState([]);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['keyword asc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [typeFilter, setTypeFilter] = useState([]);
  const [pagination, setPagination] = useState({
    currentPage: 1,
    totalPages: 1,
    totalItems: 0,
    perPage: 20
  });
  
  const {mode, setMode} = useSetIndexFiltersMode();


  const fetchAutoReplies = useCallback(async (page = 1) => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await fetchAutoReplyList({
        statusType: 'All',
        page: page,
        perPageData: pagination.perPage
      });
      
      if (response.status && response.WhatsappBusinessAutoReply) {
        const transformedData = transformAutoReplyData(response.WhatsappBusinessAutoReply);
        setAutoReplyData(transformedData);
        
        
        setPagination({
          currentPage: response.WhatsappBusinessAutoReply.current_page || 1,
          totalPages: response.WhatsappBusinessAutoReply.last_page || 1,
          totalItems: response.WhatsappBusinessAutoReply.total || 0,
          perPage: parseInt(response.WhatsappBusinessAutoReply.per_page) || 10
        });
      } else {
        throw new Error('Invalid API response format');
      }
    } catch (err) {
      console.error('Error fetching auto replies:', err);
      setError(err.message || 'Failed to fetch auto reply triggers');
      setAutoReplyData([]);
    } finally {
      setLoading(false);
    }
  }, [pagination.perPage]);


  useEffect(() => {
    fetchAutoReplies(1);
  }, []);

  const handleCreateTrigger = useCallback(() => {
    navigate('/app/auto-triggers/create-auto-reply');
  }, [navigate]);

  const handleToggleStatus = useCallback((triggerId) => {
  
    setAutoReplyData(prev =>
      prev.map(trigger =>
        trigger.id === triggerId
          ? { ...trigger, isActive: !trigger.isActive }
          : trigger
      )
    );
  }, []);

  const handleDeleteTrigger = useCallback(async (triggerId) => {
    try {
      // Add trigger to deleting state
      setDeletingTriggers(prev => [...prev, triggerId]);
      
      // Call the delete API
      await deleteAutoReply(triggerId);
      
      // Remove trigger from local state on success
      setAutoReplyData(prev => prev.filter(trigger => trigger.id !== triggerId));
      
      // Show success message (optional)
      console.log('Auto reply trigger deleted successfully');
      
    } catch (error) {
      console.error('Error deleting auto reply trigger:', error);
      setError(error.message || 'Failed to delete auto reply trigger');
    } finally {
      // Remove trigger from deleting state
      setDeletingTriggers(prev => prev.filter(id => id !== triggerId));
    }
  }, []);

  const handlePageChange = useCallback((page) => {
    fetchAutoReplies(page);
  }, [fetchAutoReplies]);

  const handleRefresh = useCallback(() => {
    fetchAutoReplies(pagination.currentPage);
  }, [fetchAutoReplies, pagination.currentPage]);

  const handleDismissError = useCallback(() => {
    setError(null);
  }, []);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);
  
  const handleTypeChange = useCallback((value) => {
    setTypeFilter(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleTypeRemove = useCallback(() => setTypeFilter([]), []);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleTypeRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleTypeRemove, handleQueryValueRemove]);

  // Filter triggers based on search query and filters
  const filteredTriggers = useMemo(() => {
    if (!Array.isArray(autoReplyData)) return [];
    
    return autoReplyData.filter(trigger => {
      const matchesSearch = !queryValue.trim() || (
        trigger.keyword?.toLowerCase().includes(queryValue.toLowerCase()) ||
        trigger.templateName?.toLowerCase().includes(queryValue.toLowerCase()) ||
        trigger.type?.toLowerCase().includes(queryValue.toLowerCase()) ||
        trigger.condition?.toLowerCase().includes(queryValue.toLowerCase())
      );
      
      const matchesStatus = !Array.isArray(statusFilter) || statusFilter.length === 0 || statusFilter.includes(trigger.isActive ? 'active' : 'inactive');
      const matchesType = !Array.isArray(typeFilter) || typeFilter.length === 0 || typeFilter.includes(trigger.type?.toLowerCase());
      
      return matchesSearch && matchesStatus && matchesType;
    });
  }, [autoReplyData, queryValue, statusFilter, typeFilter]);

  // Sort options
  const sortOptions = [
    {label: 'Keyword', value: 'keyword asc', directionLabel: 'A-Z'},
    {label: 'Keyword', value: 'keyword desc', directionLabel: 'Z-A'},
    {label: 'Type', value: 'type asc', directionLabel: 'A-Z'},
    {label: 'Type', value: 'type desc', directionLabel: 'Z-A'},
    {label: 'Created Date', value: 'created asc', directionLabel: 'Oldest first'},
    {label: 'Created Date', value: 'created desc', directionLabel: 'Newest first'},
  ];

  // Filters configuration
  const filters = [
    {
      key: 'status',
      label: 'Status',
      filter: (
        <ChoiceList
          title="Status"
          titleHidden
          choices={[
            {label: 'Active', value: 'active'},
            {label: 'Inactive', value: 'inactive'},
          ]}
          selected={statusFilter || []}
          onChange={handleStatusChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
    {
      key: 'type',
      label: 'Reply Type',
      filter: (
        <ChoiceList
          title="Reply Type"
          titleHidden
          choices={[
            {label: 'Text', value: 'text'},
            {label: 'Template', value: 'template'},
          ]}
          selected={typeFilter || []}
          onChange={handleTypeChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
  ];

  // Applied filters
  const appliedFilters = [];
  if (Array.isArray(statusFilter) && statusFilter.length > 0) {
    appliedFilters.push({
      key: 'status',
      label: `Status: ${statusFilter.map(s => s.charAt(0).toUpperCase() + s.slice(1)).join(', ')}`,
      onRemove: handleStatusRemove,
    });
  }
  if (Array.isArray(typeFilter) && typeFilter.length > 0) {
    appliedFilters.push({
      key: 'type',
      label: `Type: ${typeFilter.map(t => t.charAt(0).toUpperCase() + t.slice(1)).join(', ')}`,
      onRemove: handleTypeRemove,
    });
  }

  return (
    <Page
      title="Auto Reply Triggers"
      subtitle="Set up keyword-based automatic replies for customer inquiries"
      fullWidth
      primaryAction={{
        content: (
          <InlineStack alignment="center">
            <Text as="span">Create Auto Reply Trigger</Text>
          </InlineStack>
        ),
        onAction: handleCreateTrigger
      }}
      secondaryActions={[
        {
          content: 'Refresh',
          onAction: handleRefresh,
          loading: loading
        },
        {
          content: 'Back to Auto Triggers',
          onAction: () => navigate('/app/auto-triggers')
        }
      ]}
    >
      {error && (
        <Banner
          title="Error loading auto reply triggers"
          tone="critical"
          onDismiss={handleDismissError}
        >
          <p>{error}</p>
        </Banner>
      )}
      
      <LegacyCard>
        <IndexFilters
          sortOptions={sortOptions}
          sortSelected={sortSelected}
          queryValue={queryValue}
          queryPlaceholder="Search triggers by keyword, template, type..."
          onQueryChange={handleFiltersQueryChange}
          onQueryClear={handleQueryValueRemove}
          onSort={setSortSelected}
          filters={filters}
          appliedFilters={appliedFilters}
          onClearAll={handleFiltersClearAll}
          mode={mode}
          setMode={setMode}
          tabs={[]}
          selected={0}
          onSelect={() => {}}
        />
        
        <TriggerIndexTable
          triggers={filteredTriggers}
          onToggleStatus={handleToggleStatus}
          onDeleteTrigger={handleDeleteTrigger}
          loading={loading}
          deletingTriggers={deletingTriggers}
        />
        
        {!loading && pagination.totalPages > 1 && (
          <LegacyCard.Section>
            <InlineStack align="center">
              <Pagination
                hasPrevious={pagination.currentPage > 1}
                onPrevious={() => handlePageChange(pagination.currentPage - 1)}
                hasNext={pagination.currentPage < pagination.totalPages}
                onNext={() => handlePageChange(pagination.currentPage + 1)}
                label={`Page ${pagination.currentPage} of ${pagination.totalPages}`}
              />
            </InlineStack>
          </LegacyCard.Section>
        )}
      </LegacyCard>
    </Page>
  );
}

export default AutoReplyTriggersPage;
