import {
  Page,
  Text,
  InlineStack,
  Icon,
  LegacyCard,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  Badge,
  Button,
  useIndexResourceState,
  Tooltip,
  Spinner,
  Banner,
  Pagination,
  ChoiceList,
  Modal,
  BlockStack,
  TextContainer,
} from '@shopify/polaris';
import { PlusIcon, EditIcon, ViewIcon } from '@shopify/polaris-icons';
import React from 'react';
import { useState, useCallback, useEffect, useMemo } from 'react';
import { useNavigate } from '@remix-run/react';
import { fetchAbandonedCartTriggerList } from '../utils/abandonedCartApi';
import { getSelectedTemplatebyName } from '../utils/templateApi';
import WhatsAppPreview from '../components/WhatsAppPreview';

/**
 * Transform API data to component format
 */
const transformAbandonedCartData = (apiData) => {
  return apiData.map(item => ({
    id: item.id.toString(),
    triggerName: item.triggerName || 'Unnamed Trigger',
    templateName: item.template_name || 'No Template', 
    collectionName: [item.selectedCollections || 'All Products'], 
    lastUpdated: new Date(item.updated_at).toLocaleDateString('en-US', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit'
    }),
    isActive: item.status === 1,
    originalData: item 
  }));
};

function TriggerIndexTable({ triggers, onToggleStatus, onViewTemplate }) {
  const navigate = useNavigate();

  const resourceName = {
    singular: 'trigger',
    plural: 'triggers',
  };

  const { selectedResources, allResourcesSelected, handleSelectionChange } =
    useIndexResourceState(triggers);

  const handleToggleActive = useCallback((triggerId, event) => {
    event.stopPropagation();
    onToggleStatus(triggerId);
  }, [onToggleStatus]);

  // const handleEditTrigger = useCallback((triggerId, event) => {
  //   event.stopPropagation();
  //   navigate(`/app/edit-auto-trigger/${triggerId}`);
  // }, [navigate]);

  const handleViewTemplateTrigger = useCallback((triggerId, event) => {
    event.stopPropagation();
    onViewTemplate(triggerId);
  }, [onViewTemplate]);

  const rowMarkup = triggers.map(
    ({ id, triggerName, templateName, collectionName, lastUpdated, isActive }, index) => (
      <IndexTable.Row
        id={id}
        key={id}
        selected={selectedResources.includes(id)}
        position={index}
      >
        <IndexTable.Cell>
          <Text variant="bodyMd" fontWeight="bold">
            {triggerName}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">
            {templateName}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">
            {lastUpdated}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Badge tone={isActive ? 'success' : 'critical'}>
            {isActive ? 'Active' : 'Inactive'}
          </Badge>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd">
          {collectionName.join(', ').length > 20
  ? collectionName.join(', ').slice(0, 20) + '...'
  : collectionName.join(', ')}

          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack>
            <Button
              size="slim"
              onClick={(event) => handleToggleActive(id, event)}
            >
              {isActive ? 'Deactivate' : 'Activate'}
            </Button>
          </InlineStack>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end">
            <Tooltip content="Preview Template">
              <Button
                size="slim"
                icon={ViewIcon}
                onClick={(event) => handleViewTemplateTrigger(id, event)}
              />
            </Tooltip>
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  return (
    <IndexTable
      resourceName={resourceName}
      itemCount={triggers.length}
      selectedItemsCount={
        allResourcesSelected ? 'All' : selectedResources.length
      }
      onSelectionChange={handleSelectionChange}
      headings={[
        { title: 'Trigger Name' },
        { title: 'Template Name' },
        { title: 'Last Updated' },
        { title: 'Status' },
        { title: 'Collections' },
        { title: 'Actions' },
        { title: '' },
      ]}
    >
      {rowMarkup}
    </IndexTable>
  );
}

function AbandonedCartTriggersPage() {
  const navigate = useNavigate();
  
  // API State Management
  const [abandonedCartData, setAbandonedCartData] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  
  // Template preview states
  const [selectedTrigger, setSelectedTrigger] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  const [selectedTemplateData, setSelectedTemplateData] = useState(null);
  const [loadingTemplateDetails, setLoadingTemplateDetails] = useState(false);
  const [templateError, setTemplateError] = useState(null);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['triggerName asc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [collectionFilter, setCollectionFilter] = useState([]);
  const [currentPage, setCurrentPage] = useState(1);
  const [pagination, setPagination] = useState({
    current_page: 1,
    last_page: 1,
    per_page: 20,
    total: 0,
    from: 0,
    to: 0
  });
  
  const {mode, setMode} = useSetIndexFiltersMode();

  // Fetch abandoned cart triggers from API
  const fetchTriggers = useCallback(async (page = 1) => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await fetchAbandonedCartTriggerList({
        page,
        perPageData: 20,
        TriggerCategory: 1
      });

      if (response.status && response.data) {
        const transformedData = transformAbandonedCartData(response.data.data);
        setAbandonedCartData(transformedData);
        
      
        setPagination({
          current_page: response.data.current_page,
          last_page: response.data.last_page,
          per_page: response.data.per_page,
          total: response.data.total,
          from: response.data.from,
          to: response.data.to
        });
        
        setCurrentPage(response.data.current_page);
      } else {
        throw new Error('Invalid API response format');
      }
    } catch (err) {
      console.error('Error fetching abandoned cart triggers:', err);
      setError(err.message || 'Failed to fetch abandoned cart triggers');
      setAbandonedCartData([]);
    } finally {
      setLoading(false);
    }
  }, []);


  useEffect(() => {
    fetchTriggers(1);
  }, [fetchTriggers]);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handleStatusChange = useCallback((value) => {
    setStatusFilter(value);
  }, []);
  
  const handleCollectionChange = useCallback((value) => {
    setCollectionFilter(value);
  }, []);

  const handleStatusRemove = useCallback(() => setStatusFilter([]), []);
  const handleCollectionRemove = useCallback(() => setCollectionFilter([]), []);

  const handleFiltersClearAll = useCallback(() => {
    handleStatusRemove();
    handleCollectionRemove();
    handleQueryValueRemove();
  }, [handleStatusRemove, handleCollectionRemove, handleQueryValueRemove]);

  // Filter triggers based on search query and filters
  const filteredTriggers = useMemo(() => {
    if (!Array.isArray(abandonedCartData)) return [];
    
    return abandonedCartData.filter(trigger => {
      const matchesSearch = !queryValue.trim() || (
        trigger.triggerName?.toLowerCase().includes(queryValue.toLowerCase()) ||
        trigger.templateName?.toLowerCase().includes(queryValue.toLowerCase()) ||
        trigger.collectionName?.some(collection => 
          collection.toLowerCase().includes(queryValue.toLowerCase())
        )
      );
      
      const matchesStatus = !Array.isArray(statusFilter) || statusFilter.length === 0 || statusFilter.includes(trigger.isActive ? 'active' : 'inactive');
      const matchesCollection = !Array.isArray(collectionFilter) || collectionFilter.length === 0 || 
        trigger.collectionName?.some(collection => collectionFilter.includes(collection.toLowerCase()));
      
      return matchesSearch && matchesStatus && matchesCollection;
    });
  }, [abandonedCartData, queryValue, statusFilter, collectionFilter]);

  // Sort options
  const sortOptions = [
    {label: 'Trigger Name', value: 'triggerName asc', directionLabel: 'A-Z'},
    {label: 'Trigger Name', value: 'triggerName desc', directionLabel: 'Z-A'},
    {label: 'Template Name', value: 'templateName asc', directionLabel: 'A-Z'},
    {label: 'Template Name', value: 'templateName desc', directionLabel: 'Z-A'},
    {label: 'Last Updated', value: 'lastUpdated asc', directionLabel: 'Oldest first'},
    {label: 'Last Updated', value: 'lastUpdated desc', directionLabel: 'Newest first'},
  ];

  // Filters configuration
  const filters = [
    {
      key: 'status',
      label: 'Status',
      filter: (
        <ChoiceList
          title="Status"
          titleHidden
          choices={[
            {label: 'Active', value: 'active'},
            {label: 'Inactive', value: 'inactive'},
          ]}
          selected={statusFilter || []}
          onChange={handleStatusChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
    {
      key: 'collection',
      label: 'Collection',
      filter: (
        <ChoiceList
          title="Collection"
          titleHidden
          choices={[
            {label: 'All Products', value: 'all products'},
            {label: 'Electronics', value: 'electronics'},
            {label: 'Clothing', value: 'clothing'},
            {label: 'Home & Garden', value: 'home & garden'},
          ]}
          selected={collectionFilter || []}
          onChange={handleCollectionChange}
          allowMultiple
        />
      ),
      shortcut: true,
    },
  ];

  // Applied filters
  const appliedFilters = [];
  if (Array.isArray(statusFilter) && statusFilter.length > 0) {
    appliedFilters.push({
      key: 'status',
      label: `Status: ${statusFilter.map(s => s.charAt(0).toUpperCase() + s.slice(1)).join(', ')}`,
      onRemove: handleStatusRemove,
    });
  }
  if (Array.isArray(collectionFilter) && collectionFilter.length > 0) {
    appliedFilters.push({
      key: 'collection',
      label: `Collection: ${collectionFilter.map(c => c.charAt(0).toUpperCase() + c.slice(1)).join(', ')}`,
      onRemove: handleCollectionRemove,
    });
  }

  const handleCreateTrigger = useCallback(() => {
    navigate('/app/auto-triggers/create-abandoned-cart');
  }, [navigate]);

  // Function to fetch template details from API
  const fetchTemplateDetails = useCallback(async (templateName) => {
    if (!templateName || templateName === 'No Template') {
      return null;
    }

    try {
      setLoadingTemplateDetails(true);
      setTemplateError(null);
      
      const templateData = await getSelectedTemplatebyName(templateName);
      setSelectedTemplateData(templateData);
      
      return templateData;
    } catch (error) {
      console.error('Error fetching template details:', error);
      setTemplateError(error.message || 'Failed to fetch template details');
      return null;
    } finally {
      setLoadingTemplateDetails(false);
    }
  }, []);

  const handleViewTemplateTrigger = useCallback(async (triggerId) => {
    // Find the trigger data
    const trigger = abandonedCartData.find(t => t.id === triggerId);
    if (!trigger) return;
    
    setSelectedTrigger(trigger);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setPreviewModalActive(true);
    
    // Fetch template details if template name is available
    if (trigger.templateName && trigger.templateName !== 'No Template') {
      await fetchTemplateDetails(trigger.templateName);
    }
  }, [abandonedCartData, fetchTemplateDetails]);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setSelectedTrigger(null);
    setSelectedTemplateData(null);
    setTemplateError(null);
    setLoadingTemplateDetails(false);
  }, []);

  const handleToggleStatus = useCallback((triggerId) => {

    setAbandonedCartData(prev =>
      prev.map(trigger => {
        if (trigger.id === triggerId) {
          return { ...trigger, isActive: !trigger.isActive };
        } else {
          const clickedTrigger = prev.find(t => t.id === triggerId);
          if (clickedTrigger && !clickedTrigger.isActive) {
            return { ...trigger, isActive: false };
          }
          return trigger;
        }
      })
    );
  }, []);

  // Pagination handlers
  const handlePreviousPage = useCallback(() => {
    if (currentPage > 1) {
      fetchTriggers(currentPage - 1);
    }
  }, [currentPage, fetchTriggers]);

  const handleNextPage = useCallback(() => {
    if (currentPage < pagination.last_page) {
      fetchTriggers(currentPage + 1);
    }
  }, [currentPage, pagination.last_page, fetchTriggers]);

  // Dismiss error banner
  const handleDismissError = useCallback(() => {
    setError(null);
  }, []);

  // Loading state
  if (loading) {
    return (
      <Page
        title="Abandoned Cart Triggers"
        subtitle="Recover abandoned carts with automated WhatsApp messages"
        fullWidth
      >
        <LegacyCard>
          <LegacyCard.Section>
            <div style={{ display: 'flex', justifyContent: 'center', padding: '2rem' }}>
              <Spinner accessibilityLabel="Loading abandoned cart triggers" size="large" />
            </div>
          </LegacyCard.Section>
        </LegacyCard>
      </Page>
    );
  }

  return (
    <Page
      title="Abandoned Cart Triggers"
      subtitle="Recover abandoned carts with automated WhatsApp messages"
      fullWidth
      primaryAction={{
        content: (
          <InlineStack alignment="center">
            <Text as="span">Create Abandoned Cart Trigger</Text>
          </InlineStack>
        ),
        onAction: handleCreateTrigger
      }}
      secondaryActions={[
        {
          content: 'Back to Auto Triggers',
          onAction: () => navigate('/app/auto-triggers')
        }
      ]}
    >
      {/* Error Banner */}
      {error && (
        <Banner
          title="Error loading abandoned cart triggers"
          tone="critical"
          onDismiss={handleDismissError}
        >
          <p>{error}</p>
          <Button onClick={() => fetchTriggers(currentPage)}>
            Try Again
          </Button>
        </Banner>
      )}

      <LegacyCard>
        <IndexFilters
          sortOptions={sortOptions}
          sortSelected={sortSelected}
          queryValue={queryValue}
          queryPlaceholder="Search triggers by name, template, collection..."
          onQueryChange={handleFiltersQueryChange}
          onQueryClear={handleQueryValueRemove}
          onSort={setSortSelected}
          filters={filters}
          appliedFilters={appliedFilters}
          onClearAll={handleFiltersClearAll}
          mode={mode}
          setMode={setMode}
          tabs={[]}
          selected={0}
          onSelect={() => {}}
        />
        
        {filteredTriggers.length > 0 ? (
          <>
            <TriggerIndexTable
              triggers={filteredTriggers}
              onToggleStatus={handleToggleStatus}
              onViewTemplate={handleViewTemplateTrigger}
            />
            
            {/* Pagination */}
            {pagination.last_page > 1 && (
              <LegacyCard.Section>
                <InlineStack align="center">
                  <Pagination
                    hasPrevious={currentPage > 1}
                    onPrevious={handlePreviousPage}
                    hasNext={currentPage < pagination.last_page}
                    onNext={handleNextPage}
                  />
                </InlineStack>
              </LegacyCard.Section>
            )}
          </>
        ) : (
          <LegacyCard.Section>
            <div style={{ textAlign: 'center', padding: '2rem' }}>
              <Text variant="bodyMd" color="subdued">
                {loading ? 'Loading...' : 'No abandoned cart triggers found matching your search criteria.'}
              </Text>
            </div>
          </LegacyCard.Section>
        )}
      </LegacyCard>

      {/* Template Preview Modal */}
      {selectedTrigger && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Preview: ${selectedTrigger.triggerName}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
          // secondaryActions={[
          //   {
          //     content: 'Edit Trigger',
          //     onAction: () => {
          //       console.log('Edit trigger:', selectedTrigger.id);
          //       handleModalClose();
          //     }
          //   }
          // ]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <BlockStack gap="200">
                <InlineStack gap="200">
                  <BlockStack gap="200">
                    <Text variant="headingMd" as="h3">
                      {selectedTrigger.triggerName}
                    </Text>
                    <Text variant="bodyLg" as="p">
                      Template: {selectedTrigger.templateName}
                    </Text>
                  </BlockStack>
                </InlineStack>
                <InlineStack gap="200">
                  <Badge tone={selectedTrigger.isActive ? 'success' : 'critical'}>
                    {selectedTrigger.isActive ? 'Active' : 'Inactive'}
                  </Badge>
                </InlineStack> 
              </BlockStack>
              
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Collections:</strong> {selectedTrigger.collectionName.join(', ')}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Last Updated:</strong> {selectedTrigger.lastUpdated}
                </Text>
              </TextContainer>

              {/* Template Error Banner */}
              {templateError && (
                <Banner
                  title="Template Loading Error"
                  tone="critical"
                  onDismiss={() => setTemplateError(null)}
                >
                  <Text variant="bodyMd" as="p">
                    {templateError}
                  </Text>
                </Banner>
              )}

              {/* Template Preview Section */}
              <BlockStack gap="200">
                <Text variant="headingMd" as="h4">
                  WhatsApp Template Preview
                </Text>
                
                {loadingTemplateDetails ? (
                  <div style={{ textAlign: 'center', padding: '40px 20px' }}>
                    <Spinner accessibilityLabel="Loading template details" size="large" />
                    <Text variant="bodyMd" as="p" tone="subdued" alignment="center">
                      Loading template details...
                    </Text>
                  </div>
                ) : selectedTemplateData ? (
                  <WhatsAppPreview 
                    template={selectedTemplateData} 
                    variant="full" 
                    showWhatsAppHeader={true}
                  />
                ) : (
                  <Text variant="bodyMd" as="p" tone="subdued">
                    {selectedTrigger.templateName === 'No Template' 
                      ? 'No template assigned to this trigger'
                      : `Template preview not available for "${selectedTrigger.templateName}"`
                    }
                  </Text>
                )}
              </BlockStack>
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}
    </Page>
  );
}

export default AbandonedCartTriggersPage;
