import {
  Page, 
  Badge, 
  LegacyCard, 
  Button, 
  Modal, 
  TextContainer,
  Thumbnail,
  Text,
  InlineStack,
  BlockStack,
  TextField,
  IndexTable,
  IndexFilters,
  useSetIndexFiltersMode,
  useIndexResourceState,
  ChoiceList,
  RangeSlider,
  Pagination,
  Icon,
  Tooltip,
  Spinner,
  Banner,
} from '@shopify/polaris';
import { RefreshIcon, ViewIcon } from '@shopify/polaris-icons';
import React, { useState, useCallback, useMemo, useEffect } from 'react';
import { fetchAbandonedList } from '../utils/abandonedCartApi';

function AbandonedCartList() {
  const sleep = (ms) => new Promise((resolve) => setTimeout(resolve, ms));
  
  const [selectedProduct, setSelectedProduct] = useState(null);
  const [previewModalActive, setPreviewModalActive] = useState(false);
  
  // IndexFilters state
  const [itemStrings, setItemStrings] = useState([
    'All Abandoned Carts'
  ]);
  const [selected, setSelected] = useState(0);
  const [queryValue, setQueryValue] = useState('');
  const [sortSelected, setSortSelected] = useState(['name asc']);
  const [statusFilter, setStatusFilter] = useState([]);
  const [collectionFilter, setCollectionFilter] = useState('');
  const [priceRange, setPriceRange] = useState([0, 500]);
  
  const {mode, setMode} = useSetIndexFiltersMode();

  // API state
  const [abandonedCarts, setAbandonedCarts] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [apiPagination, setApiPagination] = useState(null);

  // Pagination state
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 20;

  // Fetch abandoned cart data from API
  const fetchAbandonedCartData = useCallback(async (page = 1) => {
    try {
      setLoading(true);
      setError(null);
      
      const response = await fetchAbandonedList({
        page,
        perPageData: itemsPerPage
      });

      if (response.status && response.data) {
        const transformedData = response.data.customerData.data.map(cart => ({
          id: cart.last_id,
          customerName: `${cart.customer.first_name} ${cart.customer.last_name}`.trim(),
          firstName: cart.customer.first_name,
          lastName: cart.customer.last_name,
          email: cart.customer.email || 'N/A',
          phone: cart.customer.phone,
          image: cart.customer.image,
          address: cart.customer.default_address || 'N/A',
          cartValue: `${cart.total_price} ${cart.currency}`,
          cartQuantity: cart.line_items.reduce((total, item) => total + item.quantity, 0),
          createdAt: cart.created_at,
          completedAt: cart.completed_at,
          lineItems: cart.line_items,
          lifetimeDuration: cart.customer.lifetime_duration,
          numberOfOrders: cart.customer.number_of_orders
        }));

        setAbandonedCarts(transformedData);
        setApiPagination(response.data.customerData);
      }
    } catch (err) {
      console.error('Error fetching abandoned cart data:', err);
      setError('Failed to load abandoned cart data. Please try again.');
    } finally {
      setLoading(false);
    }
  }, [itemsPerPage]);

  // Fetch data on component mount and when page changes
  useEffect(() => {
    fetchAbandonedCartData(currentPage);
  }, [fetchAbandonedCartData, currentPage]);

  // Tab management
  const deleteView = (index) => {
    const newItemStrings = [...itemStrings];
    newItemStrings.splice(index, 1);
    setItemStrings(newItemStrings);
    setSelected(0);
  };

  const duplicateView = async (name) => {
    setItemStrings([...itemStrings, name]);
    setSelected(itemStrings.length);
    await sleep(1);
    return true;
  };

  const onCreateNewView = async (value) => {
    await sleep(500);
    setItemStrings([...itemStrings, value]);
    setSelected(itemStrings.length);
    return true;
  };

  const tabs = itemStrings.map((item, index) => ({
    content: item,
    index,
    onAction: () => {},
    id: `${item}-${index}`,
    isLocked: index === 0,
    actions: index === 0 ? [] : [
      {
        type: 'rename',
        onAction: () => {},
        onPrimaryAction: async (value) => {
          const newItemsStrings = tabs.map((item, idx) => {
            if (idx === index) {
              return value;
            }
            return item.content;
          });
          await sleep(1);
          setItemStrings(newItemsStrings);
          return true;
        },
      },
      {
        type: 'duplicate',
        onPrimaryAction: async (value) => {
          await sleep(1);
          duplicateView(value);
          return true;
        },
      },
      {
        type: 'edit',
      },
      {
        type: 'delete',
        onPrimaryAction: async () => {
          await sleep(1);
          deleteView(index);
          return true;
        },
      },
    ],
  }));

  // Filter and search logic
  const filteredCarts = useMemo(() => {
    let filtered = abandonedCarts;
    
    return filtered.filter((cart) => {
      const matchesSearch = cart.customerName.toLowerCase().includes(queryValue.toLowerCase()) ||
                           cart.email.toLowerCase().includes(queryValue.toLowerCase()) ||
                           cart.phone.toLowerCase().includes(queryValue.toLowerCase());
      
      const cartValueNum = parseFloat(cart.cartValue.split(' ')[0]);
      const matchesPrice = cartValueNum >= priceRange[0] && cartValueNum <= priceRange[1];
      
      return matchesSearch && matchesPrice;
    });
  }, [abandonedCarts, queryValue, priceRange]);

  // Sort abandoned carts
  const sortedCarts = useMemo(() => {
    const sorted = [...filteredCarts];
    const [sortKey, sortDirection] = sortSelected[0].split(' ');
    
    sorted.sort((a, b) => {
      let aValue, bValue;
      
      switch (sortKey) {
        case 'name':
          aValue = a.customerName.toLowerCase();
          bValue = b.customerName.toLowerCase();
          break;
        case 'email':
          aValue = a.email.toLowerCase();
          bValue = b.email.toLowerCase();
          break;
        case 'phone':
          aValue = a.phone.toLowerCase();
          bValue = b.phone.toLowerCase();
          break;
        case 'cartValue':
          aValue = parseFloat(a.cartValue.split(' ')[0]);
          bValue = parseFloat(b.cartValue.split(' ')[0]);
          break;
        case 'cartQuantity':
          aValue = a.cartQuantity;
          bValue = b.cartQuantity;
          break;
        case 'createdAt':
          aValue = new Date(a.createdAt);
          bValue = new Date(b.createdAt);
          break;
        default:
          return 0;
      }
      
      if (aValue < bValue) return sortDirection === 'asc' ? -1 : 1;
      if (aValue > bValue) return sortDirection === 'asc' ? 1 : -1;
      return 0;
    });
    
    return sorted;
  }, [filteredCarts, sortSelected]);

  // Paginated carts (using API pagination)
  const paginatedCarts = abandonedCarts; // API handles pagination

  // Pagination handlers
  const handlePreviousPage = useCallback(() => {
    if (apiPagination && apiPagination.prev_page_url) {
      setCurrentPage(prev => Math.max(1, prev - 1));
    }
  }, [apiPagination]);

  const handleNextPage = useCallback(() => {
    if (apiPagination && apiPagination.next_page_url) {
      setCurrentPage(prev => prev + 1);
    }
  }, [apiPagination]);

  // Pagination info from API
  const totalItems = apiPagination?.total || 0;
  const totalPages = apiPagination?.last_page || 1;
  const startItem = apiPagination?.from || 0;
  const endItem = apiPagination?.to || 0;
  const hasNext = !!apiPagination?.next_page_url;
  const hasPrevious = !!apiPagination?.prev_page_url;

  // Sort options
  const sortOptions = [
    {label: 'Customer Name', value: 'name asc', directionLabel: 'A-Z'},
    {label: 'Customer Name', value: 'name desc', directionLabel: 'Z-A'},
    {label: 'Email', value: 'email asc', directionLabel: 'A-Z'},
    {label: 'Email', value: 'email desc', directionLabel: 'Z-A'},
    {label: 'Cart Value', value: 'cartValue asc', directionLabel: 'Low to High'},
    {label: 'Cart Value', value: 'cartValue desc', directionLabel: 'High to Low'},
    {label: 'Cart Quantity', value: 'cartQuantity asc', directionLabel: 'Low to High'},
    {label: 'Cart Quantity', value: 'cartQuantity desc', directionLabel: 'High to Low'},
    {label: 'Created Date', value: 'createdAt asc', directionLabel: 'Oldest First'},
    {label: 'Created Date', value: 'createdAt desc', directionLabel: 'Newest First'},
  ];

  // Event handlers
  const handlePreviewClick = useCallback((product) => {
    setSelectedProduct(product);
    setPreviewModalActive(true);
  }, []);

  const handleModalClose = useCallback(() => {
    setPreviewModalActive(false);
    setSelectedProduct(null);
  }, []);

  const handleFiltersQueryChange = useCallback((value) => {
    setQueryValue(value);
  }, []);

  const handleQueryValueRemove = useCallback(() => setQueryValue(''), []);
  
  const handlePriceRangeChange = useCallback((value) => {
    setPriceRange(value);
  }, []);

  const handlePriceRangeRemove = useCallback(() => setPriceRange([0, 500]), []);

  const handleFiltersClearAll = useCallback(() => {
    handlePriceRangeRemove();
    handleQueryValueRemove();
  }, [handlePriceRangeRemove, handleQueryValueRemove]);

  const onHandleCancel = () => {};
  const onHandleSave = async () => {
    await sleep(1);
    return true;
  };

  const primaryAction = selected === 0
    ? {
        type: 'save-as',
        onAction: onCreateNewView,
        disabled: false,
        loading: false,
      }
    : {
        type: 'save',
        onAction: onHandleSave,
        disabled: false,
        loading: false,
      };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'Active': { tone: 'success' },
      'Draft': { tone: 'attention' },
      'Archived': { tone: 'subdued' }
    };
    
    return <Badge tone={statusConfig[status]?.tone || 'info'}>{status}</Badge>;
  };

  // Filters configuration
  const filters = [
    {
      key: 'priceRange',
      label: 'Cart Value Range',
      filter: (
        <RangeSlider
          label="Cart value range"
          labelHidden
          value={priceRange || [0, 500]}
          prefix="₹"
          output
          min={0}
          max={500}
          step={5}
          onChange={handlePriceRangeChange}
        />
      ),
    },
  ];

  // Applied filters
  const appliedFilters = [];
  if (priceRange && (priceRange[0] !== 0 || priceRange[1] !== 500)) {
    appliedFilters.push({
      key: 'priceRange',
      label: disambiguateLabel('priceRange', priceRange),
      onRemove: handlePriceRangeRemove,
    });
  }

  // Resource state for IndexTable
  const resourceName = {
    singular: 'abandoned cart',
    plural: 'abandoned carts',
  };

  const {selectedResources, allResourcesSelected, handleSelectionChange} =
    useIndexResourceState(paginatedCarts);

  // Helper functions
  function disambiguateLabel(key, value) {
    switch (key) {
      case 'priceRange':
        return `Cart value is between ₹${value[0]} and ₹${value[1]}`;
      default:
        return value;
    }
  }

  function isEmpty(value) {
    if (Array.isArray(value)) {
      return value.length === 0;
    } else {
      return value === '' || value == null;
    }
  }

  // Format date helper
  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  // IndexTable row markup
  const rowMarkup = paginatedCarts.map(
    (cart, index) => (
      <IndexTable.Row
        id={cart.id}
        key={cart.id}
        selected={selectedResources.includes(cart.id)}
        position={index}
      >
        <IndexTable.Cell>
          <InlineStack gap="300" align="start">
            <Thumbnail
              source={cart.image}
              alt={cart.customerName}
              size="small"
            />
            <BlockStack gap="100">
              <Text variant="bodyMd" fontWeight="semibold" as="span">
                {cart.customerName}
              </Text>
              <Text variant="bodySm" tone="subdued" as="span">
                {cart.phone}
              </Text>
            </BlockStack>
          </InlineStack>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" as="span">
            {cart.email}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" as="span">
            {cart.address}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodyMd" fontWeight="semibold" as="span" tone="success">
            {cart.cartValue}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Badge tone="info">{cart.cartQuantity} items</Badge>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <Text variant="bodySm" as="span">
            {formatDate(cart.createdAt)}
          </Text>
        </IndexTable.Cell>
        <IndexTable.Cell>
          <InlineStack align="end">
            <Tooltip content="View cart details">
              <Button
                size="slim"
                onClick={(event) => {
                  event.stopPropagation();
                  handlePreviewClick(cart);
                }}
              >
                <Icon source={ViewIcon} />
              </Button>
            </Tooltip>
          </InlineStack>
        </IndexTable.Cell>
      </IndexTable.Row>
    ),
  );

  return (
    <>
      <Page 
        title="All Abandoned Cart List" 
        subtitle="Manage your Abandoned Cart"
        fullWidth
        primaryAction={{
          content: (
            <InlineStack alignment="center">
              <Icon source={RefreshIcon} />
              <Text as="span">Sync</Text>
            </InlineStack>
          ),
          onAction: () => fetchAbandonedCartData(currentPage)
        }}
      >
        {error && (
          <Banner tone="critical" onDismiss={() => setError(null)}>
            <p>{error}</p>
          </Banner>
        )}
        
        <LegacyCard>
          {loading ? (
            <div style={{ padding: '40px', textAlign: 'center' }}>
              <Spinner accessibilityLabel="Loading abandoned carts" size="large" />
              <Text variant="bodyMd" as="p" tone="subdued">
                Loading abandoned carts...
              </Text>
            </div>
          ) : (
            <>
              <IndexFilters
                sortOptions={sortOptions}
                sortSelected={sortSelected}
                queryValue={queryValue}
                queryPlaceholder="Search customers..."
                onQueryChange={handleFiltersQueryChange}
                onQueryClear={() => setQueryValue('')}
                onSort={setSortSelected}
                primaryAction={primaryAction}
                cancelAction={{
                  onAction: onHandleCancel,
                  disabled: false,
                  loading: false,
                }}
                tabs={tabs}
                selected={selected}
                onSelect={setSelected}
                canCreateNewView
                onCreateNewView={onCreateNewView}
                filters={filters}
                appliedFilters={appliedFilters}
                onClearAll={handleFiltersClearAll}
                mode={mode}
                setMode={setMode}
              />
              <IndexTable
                resourceName={resourceName}
                itemCount={paginatedCarts.length}
                selectedItemsCount={
                  allResourcesSelected ? 'All' : selectedResources.length
                }
                onSelectionChange={handleSelectionChange}
                headings={[
                  {title: 'Customer'},
                  {title: 'Email'},
                  {title: 'Address'},
                  {title: 'Cart Value'},
                  {title: 'Quantity'},
                  {title: 'Created'},
                  {title: 'Actions', alignment: 'end'},
                ]}
              >
                {rowMarkup}
              </IndexTable>
              
              {totalItems > 0 && (
                <div
                  style={{
                    maxWidth: '700px',
                    margin: 'auto',
                    border: '1px solid var(--p-color-border)',
                    marginTop: '16px'
                  }}
                >
                  <Pagination
                    onPrevious={hasPrevious ? handlePreviousPage : undefined}
                    onNext={hasNext ? handleNextPage : undefined}
                    type="table"
                    hasNext={hasNext}
                    hasPrevious={hasPrevious}
                    label={`${startItem}-${endItem} of ${totalItems} abandoned carts`}
                  />
                </div>
              )}
            </>
          )}
        </LegacyCard>
      </Page>

      {selectedProduct && (
        <Modal
          open={previewModalActive}
          onClose={handleModalClose}
          title={`Cart Details: ${selectedProduct.customerName}`}
          primaryAction={{
            content: 'Close',
            onAction: handleModalClose,
          }}
          secondaryActions={[
            {
              content: 'Send Message',
              onAction: () => {
                console.log('Send message to:', selectedProduct.id);
                handleModalClose();
              }
            }
          ]}
        >
          <Modal.Section>
            <BlockStack gap="400">
              <InlineStack align="center" gap="400">
                <Thumbnail
                  source={selectedProduct.image}
                  alt={selectedProduct.customerName}
                  size="large"
                />
                <BlockStack gap="200">
                  <Text variant="headingMd" as="h3">
                    {selectedProduct.customerName}
                  </Text>
                  <Text variant="headingLg" as="p" tone="success">
                    {selectedProduct.cartValue}
                  </Text>
                  <Badge tone="info">{selectedProduct.cartQuantity} items</Badge>
                </BlockStack>
              </InlineStack>
              
              <TextContainer>
                <Text variant="bodyMd" as="p">
                  <strong>Email:</strong> {selectedProduct.email}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Phone:</strong> {selectedProduct.phone}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Address:</strong> {selectedProduct.address}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Created:</strong> {formatDate(selectedProduct.createdAt)}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Customer Lifetime:</strong> {selectedProduct.lifetimeDuration}
                </Text>
                <Text variant="bodyMd" as="p">
                  <strong>Number of Orders:</strong> {selectedProduct.numberOfOrders}
                </Text>
              </TextContainer>

              {selectedProduct.lineItems && selectedProduct.lineItems.length > 0 && (
                <BlockStack gap="200">
                  <Text variant="headingSm" as="h4">Cart Items:</Text>
                  {selectedProduct.lineItems.map((item, index) => (
                    <InlineStack key={index} align="space-between">
                      <Text variant="bodyMd" as="span">
                        {item.title} {item.variant_title ? `(${item.variant_title})` : ''}
                      </Text>
                      <Text variant="bodyMd" as="span">
                        Qty: {item.quantity}
                      </Text>
                    </InlineStack>
                  ))}
                </BlockStack>
              )}
            </BlockStack>
          </Modal.Section>
        </Modal>
      )}
    </>
  );
}

export default AbandonedCartList;