import { useNavigate } from "@remix-run/react";
import { useState, useCallback, useEffect, useRef } from 'react';
import {
  Page,
  Layout,
  Text,
  Card,
  Button,
  BlockStack,
  InlineStack,
  Icon,
  Badge,
  Grid,
  TextField,
  Link, AccountConnection, Avatar, DatePicker, Tooltip, Popover, useBreakpoints, Select, Scrollable, OptionList, Box, Modal, InlineGrid, List, MediaCard, VideoThumbnail, Banner,
  LegacyCard,
} from "@shopify/polaris";
import { isPlanPurchased, getPlanName, getFormattedPlanName } from '../utils/planProtection';
import { API_TOKEN,accountDetails } from '../utils/apiConfig';
import {
  ProductIcon,
  OrderIcon,
  ChatIcon,
  PersonSegmentIcon,
  ChartVerticalFilledIcon,
  ChatReferralIcon,
  ProfileIcon,
  DomainNewIcon,
  CalendarIcon,
  ArrowRightIcon,
  NoteIcon,
  CartIcon,
  CashDollarIcon,
  InfoIcon,

} from "@shopify/polaris-icons";
import { TitleBar } from "@shopify/app-bridge-react";
import { authenticate } from "../shopify.server";
import metaIcon from "../routes/assetgetgabs/meta-lcon.svg";


export const loader = async ({ request }) => {
  await authenticate.admin(request);

  return null;
};

const lastUpdated = "12 june 2025 12:00:00 IST";
// DateRangePicker component
function DateRangePicker() {
  const { mdDown, lgUp } = useBreakpoints();
  const shouldShowMultiMonth = false;
  const today = new Date(new Date().setHours(0, 0, 0, 0));
  const yesterday = new Date(
    new Date(new Date().setDate(today.getDate() - 1)).setHours(0, 0, 0, 0)
  );
  const ranges = [

    {
      title: "Last 7 days",
      alias: "last7days",
      period: {
        since: new Date(
          new Date(new Date().setDate(today.getDate() - 7)).setHours(0, 0, 0, 0)
        ),
        until: yesterday,
      },
    },
    {
      title: "Last 30 days",
      alias: "last30days",
      period: {
        since: new Date(
          new Date(new Date().setDate(today.getDate() - 30)).setHours(0, 0, 0, 0)
        ),
        until: yesterday,
      },
    },
    {
      title: "Last 3 Months",
      alias: "last3Months",
      period: {
        since: new Date(
          new Date(new Date().setMonth(today.getMonth() - 3)).setHours(0, 0, 0, 0)
        ),
        until: yesterday,
      },
    },
    {
      title: "Last 6 Months",
      alias: "last6Months",
      period: {
        since: new Date(
          new Date(new Date().setMonth(today.getMonth() - 6)).setHours(0, 0, 0, 0)
        ),
        until: yesterday,
      },
    },
    {
      title: "Last 1 Year",
      alias: "last1Year",
      period: {
        since: new Date(
          new Date(new Date().setFullYear(today.getFullYear() - 1)).setHours(0, 0, 0, 0)
        ),
        until: yesterday,
      },
    },

  ];
  const [popoverActive, setPopoverActive] = useState(false);
  const [activeDateRange, setActiveDateRange] = useState(ranges[0]);
  const [inputValues, setInputValues] = useState({});
  const [{ month, year }, setDate] = useState({
    month: activeDateRange.period.since.getMonth(),
    year: activeDateRange.period.since.getFullYear(),
  });
  const datePickerRef = useRef(null);
  const VALID_YYYY_MM_DD_DATE_REGEX = /^\d{4}-\d{1,2}-\d{1,2}/;

  function isDate(date) {
    return !isNaN(new Date(date).getDate());
  }

  function isValidYearMonthDayDateString(date) {
    return VALID_YYYY_MM_DD_DATE_REGEX.test(date) && isDate(date);
  }

  function isValidDate(date) {
    return date.length === 10 && isValidYearMonthDayDateString(date);
  }

  function parseYearMonthDayDateString(input) {
    const [year, month, day] = input.split("-");
    return new Date(Number(year), Number(month) - 1, Number(day));
  }

  function formatDateToYearMonthDayDateString(date) {
    const year = String(date.getFullYear());
    let month = String(date.getMonth() + 1);
    let day = String(date.getDate());
    if (month.length < 2) {
      month = String(month).padStart(2, "0");
    }
    if (day.length < 2) {
      day = String(day).padStart(2, "0");
    }
    return [year, month, day].join("-");
  }

  function formatDate(date) {
    return formatDateToYearMonthDayDateString(date);
  }

  function nodeContainsDescendant(rootNode, descendant) {
    if (rootNode === descendant) {
      return true;
    }
    let parent = descendant.parentNode;
    while (parent != null) {
      if (parent === rootNode) {
        return true;
      }
      parent = parent.parentNode;
    }
    return false;
  }

  function isNodeWithinPopover(node) {
    return datePickerRef?.current
      ? nodeContainsDescendant(datePickerRef.current, node)
      : false;
  }

  function handleStartInputValueChange(value) {
    setInputValues((prevState) => {
      return { ...prevState, since: value };
    });
    if (isValidDate(value)) {
      const newSince = parseYearMonthDayDateString(value);
      setActiveDateRange((prevState) => {
        const newPeriod =
          prevState.period && newSince <= prevState.period.until
            ? { since: newSince, until: prevState.period.until }
            : { since: newSince, until: newSince };
        return {
          ...prevState,
          period: newPeriod,
        };
      });
    }
  }

  function handleEndInputValueChange(value) {
    setInputValues((prevState) => ({ ...prevState, until: value }));
    if (isValidDate(value)) {
      const newUntil = parseYearMonthDayDateString(value);
      setActiveDateRange((prevState) => {
        const newPeriod =
          prevState.period && newUntil >= prevState.period.since
            ? { since: prevState.period.since, until: newUntil }
            : { since: newUntil, until: newUntil };
        return {
          ...prevState,
          period: newPeriod,
        };
      });
    }
  }

  function handleInputBlur({ relatedTarget }) {
    const isRelatedTargetWithinPopover =
      relatedTarget != null && isNodeWithinPopover(relatedTarget);
    if (isRelatedTargetWithinPopover) {
      return;
    }
    setPopoverActive(false);
  }

  function handleMonthChange(month, year) {
    setDate({ month, year });
  }

  function handleCalendarChange({ start, end }) {
    const newDateRange = ranges.find((range) => {
      return (
        range.period.since.valueOf() === start.valueOf() &&
        range.period.until.valueOf() === end.valueOf()
      );
    }) || {
      alias: "custom",
      title: "Custom",
      period: {
        since: start,
        until: end,
      },
    };
    setActiveDateRange(newDateRange);
  }

  function apply() {
    setPopoverActive(false);
  }

  function cancel() {
    setPopoverActive(false);
  }

  useEffect(() => {
    if (activeDateRange) {
      setInputValues({
        since: formatDate(activeDateRange.period.since),
        until: formatDate(activeDateRange.period.until),
      });
      function monthDiff(referenceDate, newDate) {
        return (
          newDate.month -
          referenceDate.month +
          12 * (referenceDate.year - newDate.year)
        );
      }
      const monthDifference = monthDiff(
        { year, month },
        {
          year: activeDateRange.period.until.getFullYear(),
          month: activeDateRange.period.until.getMonth(),
        }
      );
      if (monthDifference > 1 || monthDifference < 0) {
        setDate({
          month: activeDateRange.period.until.getMonth(),
          year: activeDateRange.period.until.getFullYear(),
        });
      }
    }
  }, [activeDateRange]);

  const buttonValue =
    activeDateRange.title === "Custom"
      ? activeDateRange.period.since.toDateString() +
      " - " +
      activeDateRange.period.until.toDateString()
      : activeDateRange.title;

  return (
    <Popover
      active={popoverActive}
      autofocusTarget="none"
      preferredAlignment="left"
      preferredPosition="below"
      fluidContent
      sectioned={false}
      fullHeight
      activator={
        <Button
          size="slim"
          icon={CalendarIcon}
          onClick={() => setPopoverActive(!popoverActive)}
        >
          {buttonValue}
        </Button>
      }
      onClose={() => setPopoverActive(false)}
    >
      <Popover.Pane fixed>
        <InlineGrid
          columns={{
            xs: "1fr",
            mdDown: "1fr",
            md: "max-content max-content",
          }}
          gap={0}
          ref={datePickerRef}
        >
          <Box
            maxWidth={mdDown ? "516px" : "212px"}
            width={mdDown ? "100%" : "212px"}
            padding={{ xs: 500, md: 0 }}
            paddingBlockEnd={{ xs: 100, md: 0 }}
          >
            {mdDown ? (
              <Select
                label="dateRangeLabel"
                labelHidden
                onChange={(value) => {
                  const result = ranges.find(
                    ({ title, alias }) => title === value || alias === value
                  );
                  setActiveDateRange(result);
                }}
                value={activeDateRange?.title || activeDateRange?.alias || ""}
                options={ranges.map(({ alias, title }) => title || alias)}
              />
            ) : (
              <Scrollable style={{ height: "334px" }}>
                <OptionList
                  options={ranges.map((range) => ({
                    value: range.alias,
                    label: range.title,
                  }))}
                  selected={activeDateRange.alias}
                  onChange={(value) => {
                    setActiveDateRange(
                      ranges.find((range) => range.alias === value[0])
                    );
                  }}
                />
              </Scrollable>
            )}
          </Box>
          <Box padding={{ xs: 500 }} maxWidth={mdDown ? "320px" : "516px"}>
            <BlockStack gap="400">
              <InlineStack gap="200">
                <div style={{ flexGrow: 1 }}>
                  <TextField
                    role="combobox"
                    label={"Since"}
                    labelHidden
                    prefix={<Icon source={CalendarIcon} />}
                    value={inputValues.since}
                    onChange={handleStartInputValueChange}
                    onBlur={handleInputBlur}
                    autoComplete="off"
                  />
                </div>
                <Icon source={ArrowRightIcon} />
                <div style={{ flexGrow: 1 }}>
                  <TextField
                    role="combobox"
                    label={"Until"}
                    labelHidden
                    prefix={<Icon source={CalendarIcon} />}
                    value={inputValues.until}
                    onChange={handleEndInputValueChange}
                    onBlur={handleInputBlur}
                    autoComplete="off"
                  />
                </div>
              </InlineStack>
              <div>
                <DatePicker
                  month={month}
                  year={year}
                  selected={{
                    start: activeDateRange.period.since,
                    end: activeDateRange.period.until,
                  }}
                  onMonthChange={handleMonthChange}
                  onChange={handleCalendarChange}
                  multiMonth={shouldShowMultiMonth}
                  allowRange
                />
              </div>
            </BlockStack>
          </Box>
        </InlineGrid>
      </Popover.Pane>
      <Popover.Pane fixed>
        <Popover.Section>
          <InlineStack align="end">
            <Button onClick={cancel}>Cancel</Button>
            <Button primary onClick={apply}>
              Apply
            </Button>
          </InlineStack>
        </Popover.Section>
      </Popover.Pane>
    </Popover>
  );
}

export default function Index() {
  const navigate = useNavigate();
  const [active, setActive] = useState(false);
  const [onboardingModalActive, setOnboardingModalActive] = useState(false);
  const [planPurchaseStatus, setPlanPurchaseStatus] = useState(() => isPlanPurchased());
  const [currentPlan, setCurrentPlan] = useState(() => getPlanName());
 
  
 
  const connected = accountDetails.isAccountConnected;
  const purchased = accountDetails.isplanpurchased;

 

  const handleAction = useCallback(() => {
    if (!connected) {
      
      setOnboardingModalActive(true);
    }
  }, [connected]);

  const handleChange = useCallback(() => setActive(!active), [active]);
  const [showBanner, setShowBanner] = useState(true);
 useEffect(() => {
  
  const timer = setTimeout(() => {
    setShowBanner(false);
  }, 2000);

  
  return () => clearTimeout(timer);
}, []);

const handlescheduleAction = () => {
  window.open('https://calendly.com/getgabs/demo', '_blank');
};
  const metrics = [
    {
      id: 1,
      icon: ChatIcon,
      value: "120",
      title: "Messages sent",
      description: "Total WhatsApp messages sent to customers",
    },
    {
      id: 2,
      icon: NoteIcon,
      value: "91",
      title: "Messages read",
      description: "WhatsApp messages opened by customers",
    },
    {
      id: 3,
      icon: CartIcon,
      value: "4",
      title: "Carts recovered",
      description: "Successfully recovered abandoned carts",
    },
    {
      id: 4,
      icon: CashDollarIcon,
      value: "2020.00 USD",
      title: "Amount recovered",
      description: "Total amount recovered from abandoned carts",
    },
  ];


  const quickActions = [
    {
      title: "Business Profile",
      description: "Manage your whatsapp business profile",
      icon: ProfileIcon,
      action: () => navigate("/app/manage-whatsapp-profile"),
     
      badge: "New",
    },
    {
      title: "Create Chat Widget",
      description: "create WhatsApp chat widget for your store",
      icon: ChatReferralIcon,
      action: () => navigate("/app/create-chat-widget"),
    },
    {
      title: "Create Template",
      description: "Create Msg for broadcast and updates",
      icon: DomainNewIcon,
      action: () => navigate("/app/templates"),
    },
    {
      title: "Create Contact List",
      description: "Create contact list for broadcast and updates",
      icon: PersonSegmentIcon,
      action: () => navigate("/app/contact-list"), 
    },
  ];

  return (
    <Page>
      <TitleBar title="Getgabs" />
      <Layout>
        <Layout.Section>
          <BlockStack gap="500">
             
          
            {!purchased && connected && (
              <Banner
                title="Plan Required"
                status="warning"
                action={{
                  content: 'View Plans',
                  onAction: () => navigate('/app/pricing-plan'),
                }}
              >
                <p>You need to purchase a plan to access all features. Currently you can only access the Home page and Pricing Plan.</p>
              </Banner>
            )}
            {connected ? (

              <Card>
                <BlockStack gap="400">
                  <InlineStack gap="300" align="start">
                    <div
                      style={{
                        width: "60px",
                        height: "60px",
                        borderRadius: "50%",
                        overflow: "hidden",
                      }}
                    >
                      <img
                        src="https://cdn.shopify.com/shopifycloud/shopify/assets/admin/customers/polaris/avatar-3-d7b953912ef42b1ef360e0a80bf1d474b9cfc90cb46126bd72ff32d10b47cefb.png"
                        alt="Getgabs LLP"
                        style={{ width: "100%", height: "100%", objectFit: "cover" }}
                      />
                    </div>

                    <BlockStack gap="100">
                      <Text as="h3" variant="headingSm">
                        {accountDetails.businesName}
                      </Text>
                      <Text as="span" variant="bodySm" tone="subdued">
                        {accountDetails.connectedWhatsappNumber}
                      </Text>
                      <Text as="span" variant="bodySm" tone="success">
                        {accountDetails.isAccountConnected ? 'Account connected' : 'Account not connected'}
                      </Text>
                    </BlockStack>
                  </InlineStack>
                  <Grid>
                    <Grid.Cell columnSpan={{ xs: 12, sm: 12, md: 6, lg: 6, xl: 6 }}>
                      <Card>
                        <BlockStack gap="400">
                          <InlineStack gap="0">
                            <Text as="h3" variant="headingSm">
                              Account Quality Rating
                            </Text>
                            <Tooltip content="account quality rating evaluates how well customers receive your messages over a seven-day period. It is determined by user feedback signals such as message blocks, user-provided reasons for blocking, and other reports">
                              <Icon source={InfoIcon} />
                            </Tooltip>
                          </InlineStack>

                          <InlineStack gap="200">
                            <Badge tone="success">{accountDetails.accouuntQualityRating}</Badge>
                          </InlineStack>
                        </BlockStack>
                      </Card>
                    </Grid.Cell>
                    <Grid.Cell columnSpan={{ xs: 12, sm: 12, md: 6, lg: 6, xl: 6 }}>
                      <Card>
                        <BlockStack gap="400">
                          <InlineStack gap="0">
                            <Text as="h3" variant="headingSm">
                              Your Current Message Limit (by Meta)
                              </Text>
                              <Tooltip content="account quality rating evaluates how well customers receive your messages over a seven-day period. It is determined by user feedback signals such as message blocks, user-provided reasons for blocking, and other reports">
                                <Icon source={InfoIcon} />
                              </Tooltip>
                            
                          </InlineStack>
                          <InlineStack gap="200">
                            <Link url="https://business.facebook.com/wa/manage/home" target="_blank">
                            <Badge tone="subdued">{accountDetails.currentMessageLimit}</Badge>
                            </Link>
                          </InlineStack>
                        </BlockStack>
                      </Card>
                    </Grid.Cell>
                  </Grid>
                </BlockStack>
              </Card>

            ) : (
             <>
               
              {showBanner && (
        <Banner
          title="Please connect account with Meta"
          onDismiss={() => setShowBanner(false)}
        >
          <p>
            Connect your account with Meta to enable WhatsApp Business API for
            automated messages and customer chats.
          </p>
        </Banner>
      )}
      
              <Card>
                <Grid>
                  <Grid.Cell columnSpan={{ xs: 12, sm: 8, md: 8, lg: 8, xl: 8 }}>
                    <BlockStack gap="400">
                      <Text as="h2" variant="headingLg">
                        Connect Your WhatsApp Number
                      </Text>

                      <Text variant="bodyMd" tone="subdued">
                        Connect your WhatsApp number to enable WhatsApp Business API for
                        automated messages and customer chats.{" "}
                        <Link url="https://getgabs.com">learn more</Link>.
                      </Text>

                      <Card>
                        <BlockStack gap="400">
                          <Text as="h3" variant="headingMd">
                            Requirements
                          </Text>
                          <List type="bullet">
                            <List.Item>Verified Meta Business account</List.Item>
                            <List.Item>A business phone number</List.Item>
                          </List>
                        </BlockStack>
                      </Card>

                      <InlineStack align="start">
                        <Button onClick={handleAction} variant="primary">
                          <InlineStack gap="200" align="center" blockAlign="center">
                            <img
                              src={metaIcon}
                              alt="Custom icon"
                              style={{ width: 20, height: 20 }}
                            />
                            <Text>Connect with meta for business</Text>
                          </InlineStack>
                        </Button>
                      </InlineStack>
                    </BlockStack>
                  </Grid.Cell>

                  {/* Smaller column for image */}
                  <Grid.Cell columnSpan={{ xs: 12, sm: 4, md: 4, lg: 4, xl: 4 }}>
                    <div style={{ display: "flex", justifyContent: "center", marginTop: "40px" }}>
                      <img
                        src="https://app.getgabs.com/betaversion/templatedesign/template_design/img/register_with_facebook.gif"
                        alt="Meta illustration"
                        style={{
                          maxWidth: "200px",
                          width: "100%",
                          borderRadius: "8px",
                        }}
                      />
                    </div>
                  </Grid.Cell>
                </Grid>
              </Card>
              </>
            )}

            {connected && (
              <>
                <BlockStack gap="400">
                  <InlineStack gap="400" align="space-between">
                    <Text s="h2" variant="headingLg">
                      Abandoned Cart Metrics (coming soon..)
                    </Text>

                    {/* <DateRangePicker /> */}
                    <Text  >
                      Last updated on:{lastUpdated}
                    </Text>
                  </InlineStack>
                  <Grid columns={{ xs: 1, sm: 2, md: 4, lg: 4 }}>
                    {metrics.map((metric) => (
                      <Grid.Cell key={metric.id} columnSpan={{ xs: 1, sm: 1, md: 1, lg: 1 }}>
                        <Card>
  <BlockStack gap="300">
    <InlineStack gap="200" align="center" blockAlign="center">
      <Badge status="new">
        <Icon source={metric.icon} tone="base" />
      </Badge>
    </InlineStack>

    <BlockStack gap="050" align="center">
      <Text variant="headingMd">{metric.value}</Text>
      <Text variant="headingSm">{metric.title}</Text>
      <Text variant="bodySm" tone="subdued">
        {metric.description}
      </Text>
    </BlockStack>
  </BlockStack>
</Card>
                      </Grid.Cell>
                    ))}
                  </Grid>
                </BlockStack>

                <Card>
                  <BlockStack gap="400">
                    <Text s="h2" variant="headingLg">
                      Quick Actions
                    </Text>
                    <Grid>
                      {quickActions.map((action, index) => (
                        <Grid.Cell
                          key={index}
                          columnSpan={{ xs: 6, sm: 3, md: 3, lg: 3, xl: 3 }}
                        >
                          <Card sectioned>
                            <BlockStack gap="200">
                              <InlineStack align="space-between">
                                <Icon source={action.icon} color="base" />
                                {action.badge && (
                                  <Badge status="new">{action.badge}</Badge>
                                )}
                              </InlineStack>
                              <Text as="h3" variant="headingSm">
                                {action.title}
                              </Text>
                              <Text variant="bodySm" color="subdued">
                                {action.description}
                              </Text>
                              <Button onClick={action.action} size="slim">
                                {action.title}
                              </Button>
                            </BlockStack>
                          </Card>
                        </Grid.Cell>
                      ))}
                    </Grid>
                  </BlockStack>
                </Card>

                <Grid>
                  <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 6, lg: 6, xl: 6 }}>
                    <Card>
                      <BlockStack gap="400">
                        <Text as="h2" variant="headingLg">
                          GetGabs API Key
                        </Text>
                        <Text variant="bodyMd" as="p">
                          Use this API key to test getGabs WhatsApp Business API.
                          
                        </Text>
                        <Text variant="bodyMd" as="p">
                         <Link

                            url="https://documenter.getpostman.com/view/40451098/2sAYJ1k2VV"
                          >
                            See Documentation
                          </Link>
                        </Text>

                        <TextField
                          value= {accountDetails.api_key}
                          readOnly
                          connectedRight={
                            <Button primary onClick={() => navigator.clipboard.writeText(accountDetails.api_key)}>
                              Copy
                            </Button>
                          }
                        />
                      </BlockStack>
                    </Card>
                  </Grid.Cell>
                  <Grid.Cell columnSpan={{ xs: 6, sm: 6, md: 6, lg: 6, xl: 6 }}>
                    <Card>
                      <BlockStack gap="400">
                        <Text as="h2" variant="headingLg">
                          Active Plan
                        </Text>
                       
                        {planPurchaseStatus && (
                          <Text variant="bodyMd" as="p">
                            Current Plan: {" "}
                            <Badge tone="info">
                              {getFormattedPlanName()}
                            </Badge>
                          </Text>
                        )}
                        <Text variant="bodyMd" as="p">
                          {planPurchaseStatus 
                            ? "You have access to all features." 
                            : "Purchase a plan to unlock all features."
                          }
                        </Text>
                        <Button primary 
                        onClick={() => navigate('/app/pricing-plan')}>
                          {planPurchaseStatus ? "Manage Plan" : "Purchase Plan"}
                        </Button>
                      </BlockStack>
                    </Card>
                  </Grid.Cell>
                </Grid>
              </>
            )}


            <>
              
<Card>
      <Grid>
        <Grid.Cell columnSpan={{ xs: 7, sm: 7, md: 7, lg: 7 }}>
        <BlockStack gap="400">
          <Text variant="headingLg" as="h2">
            Complete Guide of WhatsApp Automation with GetGabs
          </Text>
          <Text as="p">
            Learn how to automate WhatsApp using GetGabs and grow your business.
          </Text>
          <InlineStack gap="200">
          <Button onClick={handleChange} primary>
            Watch Video
          </Button>
          </InlineStack>
        </BlockStack>
        </Grid.Cell>
        <Grid.Cell columnSpan={{ xs: 5, sm: 5, md: 5, lg: 5 }}>
          <VideoThumbnail
            videoLength={80}
            thumbnailUrl="https://i.ytimg.com/vi_webp/pU8F92_vfq4/maxresdefault.webp"
            onClick={handleChange}
          />
        </Grid.Cell>
      </Grid>
    </Card>
              <Modal
                open={active}
                onClose={handleChange}
                title="Complete Guide of WhatsApp Automation"
                large
              >
                <div style={{ position: 'relative', paddingBottom: '56.25%', height: 0 }}>
                  <iframe
                    src="https://www.youtube.com/embed/pU8F92_vfq4"
                    title="YouTube video"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                    allowFullScreen
                    style={{
                      position: 'absolute',
                      top: 0,
                      left: 0,
                      width: '100%',
                      height: '100%',
                      borderRadius: '8px',
                    }}
                  ></iframe>
                </div>
              </Modal>

              {/* Onboarding Modal */}
              <Modal
                open={onboardingModalActive}
                onClose={() => setOnboardingModalActive(false)}
                title="Connect with Meta for Business"
                large
                fullScreen
              >
                <div style={{ position: 'relative', paddingBottom: '56.25%', height: 0, minHeight: '600px' }}>
                  
                  <iframe
                    src={`https://app.getgabs.com/partners/accountSetup/onboarding?token=${API_TOKEN}`}
                    title="GetGabs Onboarding"
                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                    style={{
                      position: 'absolute',
                      top: -100,
                      left: 0,
                      width: '100%',
                      height: '100%',
                      border: 'none',
                      borderRadius: '8px',
                    }}
                  ></iframe>
                </div>
              </Modal>
            </>

            <Card background="bg-surface-primary">

              <BlockStack gap="400">
                <Text as="h2" variant="headingLg" alignment="center">
                  Need help ?
                </Text>

                <Text variant="bodyMd" tone="subdued" alignment="center">Book a free call with our team to understand and unlock the power of WhatsApp using GetGabs. </Text>
                <InlineStack align="center">
                  <Button onClick={handlescheduleAction} variant="primary">
                    <InlineStack gap="200" align="center" blockAlign="center">
                      <Icon source={CalendarIcon} />
                      <Text>Schedule a call</Text>
                    </InlineStack>
                  </Button>
                </InlineStack>
              </BlockStack>
            </Card>
          </BlockStack>
        </Layout.Section>
      </Layout>
    </Page>
  );
}



