import { useEffect, useRef, useState, useCallback } from 'react';
import { io } from 'socket.io-client';

export function useSocket(userInfo) {
  const socketRef = useRef(null);
  const [isConnected, setIsConnected] = useState(false);
  const [connectionError, setConnectionError] = useState(null);

  useEffect(() => {
    if (!userInfo) return;

    // Initialize GetGabs socket connection 
    socketRef.current = io('https://app.getgabs.com:56000', {
      transports: ['websocket', 'polling', 'flashsocket'],
      autoConnect: true,
      reconnection: true,
      reconnectionDelay: 2000,
      reconnectionDelayMax: 10000,
      maxReconnectionAttempts: 5,
      timeout: 20000,
      forceNew: true
    });

    const socket = socketRef.current;

    // Connection event handlers
    socket.on('connect', () => {
      console.log('GetGabs Socket connected - ID:', socket.id);
      setIsConnected(true);
      setConnectionError(null);
    });

    socket.on('disconnect', (reason) => {
      console.log(' GetGabs Socket disconnected - Reason:', reason);
      setIsConnected(false);
      
      // Don't try to reconnect if it was a manual disconnect
      if (reason === 'io client disconnect') {
        console.log('Manual disconnect - not reconnecting');
      }
    });

    socket.on('connect_error', (error) => {
      console.error(' GetGabs Socket connection error:', error);
      setConnectionError(error.message);
      setIsConnected(false);
    });

    socket.on('reconnect', (attemptNumber) => {
      console.log('GetGabs Socket reconnected after', attemptNumber, 'attempts');
      setIsConnected(true);
      setConnectionError(null);
    });

    socket.on('reconnect_error', (error) => {
      console.error(' GetGabs Socket reconnection failed:', error);
    });

    socket.on('reconnect_failed', () => {
      console.error('GetGabs Socket reconnection failed permanently');
      setConnectionError('Failed to reconnect to chat server');
    });

    // GetGabs specific event handlers
    socket.on('welcome', (data) => {
      console.log(' GetGabs welcome received:', data);
      console.log(' Sending user info:', userInfo);
      
      // Add a small delay before sending user info to ensure socket is ready
      setTimeout(() => {
        if (socket.connected) {
          socket.emit("connectchannel", userInfo);
          console.log('User info sent to connectchannel');
        } else {
          console.warn('Socket not connected, skipping connectchannel');
        }
      }, 100);
    });

    // Add handler for successful channel connection
    socket.on('channelconnected', (data) => {
      console.log(' Channel connected successfully:', data);
    });

    // Add error handler for authentication failures
    socket.on('error', (error) => {
      console.error(' GetGabs Socket error:', error);
      setConnectionError(error.message || 'Socket error occurred');
    });

    // Add handler for any other events for debugging
    socket.onAny((eventName, ...args) => {
      if (!['connect', 'disconnect', 'welcome', 'channelconnected'].includes(eventName)) {
        console.log(' GetGabs Socket event:', eventName, args);
      }
    });

    // Cleanup on unmount
    return () => {
      if (socket) {
        socket.disconnect();
      }
    };
  }, [userInfo]);

  // Event listener helpers
  const addEventListener = useCallback((event, handler) => {
    if (socketRef.current) {
      socketRef.current.on(event, handler);
    }
  }, []);

  const removeEventListener = useCallback((event, handler) => {
    if (socketRef.current) {
      socketRef.current.off(event, handler);
    }
  }, []);

  return {
    socket: socketRef.current,
    isConnected,
    connectionError,
    addEventListener,
    removeEventListener
  };
}
