import React from 'react';
import { Text, BlockStack, Icon } from '@shopify/polaris';
import { LinkIcon, PhoneIcon, UndoIcon } from '@shopify/polaris-icons';
import { convertWhatsAppMessageToPreview, getFormattedTextData } from '../utils/whatsappMessagePreview';

/**
 * Reusable WhatsApp Message Preview Component
 * 
 * @param {Object} props
 * @param {string} props.template - The WhatsApp template JSON string
 * @param {string} props.variant - Preview style variant ('bubble' or 'full')
 * @param {string} props.maxWidth - Maximum width of the preview
 * @param {boolean} props.showWhatsAppHeader - Whether to show WhatsApp-style header
 * @param {Object} props.variableMapping - Object mapping template variables to actual values
 */
const WhatsAppPreview = ({ 
  template, 
  variant = 'bubble', 
  maxWidth = '280px',
  showWhatsAppHeader = false,
  variableMapping = {},
  mediaMapping = {}
}) => {
  if (!template) {
    return (
      <Text variant="bodyMd" tone="subdued">
        No preview available
      </Text>
    );
  }

  // Function to apply variable mapping to text
  const applyVariableMapping = (text) => {
    if (!text || typeof text !== 'string') return text;
    
    let processedText = text;
    Object.entries(variableMapping).forEach(([variable, mappedValue]) => {
      if (mappedValue) {
        // Get the label for the mapped value from availableVariables
        const variableRegex = new RegExp(`\\{\\{${variable}\\}\\}`, 'g');
        const displayValue = getVariableDisplayValue(mappedValue);
        processedText = processedText.replace(variableRegex, displayValue);
      }
    });
    
    return processedText;
  };

  // Function to get display value for a variable
  const getVariableDisplayValue = (variableValue) => {
    const sampleData = {
      customer_name: 'John Doe',
      first_name: 'John',
      last_name: 'Doe',
      product_name: 'Premium Widget',
      order_id: 'ORD-12345',
      order_number: '#12345',
      tracking_link: 'https://track.example.com/12345',
      company_name: 'Acme Corp',
      support_email: 'support@business.com',
      support_phone: '+1 800-123-4567'
    };
    
    return sampleData[variableValue] || `[${variableValue}]`;
  };

  // Apply variable mapping to the template before parsing
  let processedTemplate = template;
  if (typeof template === 'string' && Object.keys(variableMapping).length > 0) {
    try {
      const templateObj = JSON.parse(template);
      processedTemplate = JSON.stringify(templateObj, (key, value) => {
        if (typeof value === 'string') {
          return applyVariableMapping(value);
        }
        return value;
      });
    } catch (e) {
      // If not JSON, treat as plain text
      processedTemplate = applyVariableMapping(template);
    }
  }

  const previewComponents = convertWhatsAppMessageToPreview(processedTemplate);
  if (!previewComponents) {
    return (
      <Text variant="bodyMd" tone="subdued">
        Unable to generate preview
      </Text>
    );
  }

  const renderFormattedText = (text) => {
    if (!text) return null;
    const processedText = applyVariableMapping(text);
    const elements = getFormattedTextData(processedText);
    return elements.map((element, index) => {
      if (element.isBullet) {
        return (
          <div key={element.key || index} style={{ display: 'flex', alignItems: 'flex-start', marginBottom: '4px' }}>
            <span style={{ marginRight: '8px', fontSize: '14px' }}>•</span>
            <span style={element.style}>{element.text}</span>
          </div>
        );
      }
      if (element.isNumbered) {
        return (
          <div key={element.key || index} style={{ display: 'flex', alignItems: 'flex-start', marginBottom: '4px' }}>
            <span style={{ marginRight: '8px', fontSize: '14px', minWidth: '1px' }}>
              {element.text.match(/^(\d+)\./)?.[1] || '1'}.
            </span>
            <span style={element.style}>{element.text.replace(/^\d+\.\s*/, '')}</span>
          </div>
        );
      }
      if (element.isQuote) {
        return (
          <div key={element.key || index} style={{ 
            borderLeft: '3px solid #ccc', 
            paddingLeft: '10px', 
            fontStyle: 'italic',
            marginBottom: '4px',
            backgroundColor: '#f9f9f9'
          }}>
            <span style={element.style}>{element.text}</span>
          </div>
        );
      }
      return (
        <span key={element.key || index}>
          <span style={element.style}>
            {element.text}
          </span>
          {element.isLineBreak && <br />}
        </span>
      );
    });
  };

  const renderHeader = (header, mediaId = null) => {
    if (!header) return null;
    
    // Check if there's a mapped media file for this header
    const mappedFile = mediaId && mediaMapping[mediaId];
    let mediaUrl = header.content;
    
    // Handle different types of media mapping
    if (mappedFile) {
      if (mappedFile.fileUrl) {
        // GetGabs API URL mapping
        mediaUrl = mappedFile.fileUrl;
      } else if (mappedFile.link) {
        // Direct URL mapping
        mediaUrl = mappedFile.link;
      } else if (mappedFile.file && mappedFile.file instanceof File) {
        // File object mapping
        mediaUrl = URL.createObjectURL(mappedFile.file);
      } else if (mappedFile instanceof File) {
        // Direct file mapping
        mediaUrl = URL.createObjectURL(mappedFile);
      } else if (typeof mappedFile === 'string') {
        // Direct URL string
        mediaUrl = mappedFile;
      }
    }
    
    return (
      <div>
        {header.type === 'image' && (
          <img 
            src={mediaUrl} 
            alt="Header image" 
            style={{
              width: '100%',
              // maxWidth: '250px',
              height: 'auto',
              borderRadius: '8px'
            }}
          />
        )}
        {header.type === 'video' && (
          <video 
            src={mediaUrl} 
            controls
            style={{
              width: '100%',
              // maxWidth: '250px',
              height: 'auto',
              borderRadius: '8px'
            }}
          />
        )}
        {header.type === 'text' && (
          <Text variant="bodyMd" as="div" fontWeight="bold">
            {renderFormattedText(header.content)}
          </Text>
        )}
        {header.type === 'document' && (
          <div style={{
            padding: '8px 12px',
            backgroundColor: '#f5f5f5',
            borderRadius: '8px',
            border: '1px solid #ddd'
          }}>
            📄 {mappedFile ? (mappedFile.file?.name || mappedFile.fileName || mappedFile.name) : (header.filename || 'Document')}
          </div>
        )}
      </div>
    );
  };

  const renderButtons = (buttons) => {
    if (!buttons || !Array.isArray(buttons) || buttons.length === 0) return null;
    
    return (
      <BlockStack gap="100">
        {buttons.map((button, index) => (
          <div key={index} style={{
            backgroundColor: button.type === 'URL' ? '#e3f2fd' : 
                           button.type === 'CATALOG' ? '#e8f5e8' : '#f0f0f0',
            padding: '8px 12px',
            borderRadius: '5px',
            border: '1px solid #ddd',
            textAlign: 'center',
            fontSize: '12px',
            color: button.type === 'URL' ? '#1976d2' : 
                   button.type === 'CATALOG' ? '#2e7d32' : 'inherit',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            gap: '6px'
          }}>
            {button.type === 'URL' && <span><Icon source={LinkIcon} tone="base" /></span>}
            {button.type === 'PHONE_NUMBER' && <span><Icon source={PhoneIcon} tone="base" /></span>}
            {(button.type === 'QUICK_REPLY' || (!button.type || button.type === 'REPLY')) && <span><Icon source={UndoIcon} tone="base" /></span>} 
            {button.type === 'CATALOG' && ' '}
            <span>{button.text}</span>
          </div>
        ))}
      </BlockStack>
    );
  };

  const renderCarousel = (carousel) => {
    if (!carousel || !Array.isArray(carousel) || carousel.length === 0) return null;
    
    return (
      <div style={{
        display: 'flex',
        gap: '8px',
        overflowX: 'auto',
        padding: '4px',
        maxWidth: '100%'
      }}>
        {carousel.map((card, index) => (
          <div key={card.id || index} style={{
            minWidth: '200px',
            maxWidth: '200px',
            backgroundColor: '#f8f9fa',
            border: '1px solid #e1e3e5',
            borderRadius: '8px',
            overflow: 'hidden',
            flexShrink: 0
          }}>
            <BlockStack gap="100">
              {/* Card Header */}
              {card.components?.header && (
                <div style={{ padding: '0' }}>
                  {renderHeader(card.components.header, `carousel_${index}_header`)}
                </div>
              )}
              
              {/* Card Body */}
              <div style={{ padding: '8px 12px' }}>
                {card.components?.body && (
                  <Text variant="bodySm" as="div">
                    {renderFormattedText(card.components.body)}
                  </Text>
                )}
              </div>
              
              {/* Card Buttons */}
              {card.components?.buttons && (
                <div style={{ padding: '0 8px 8px 8px' }}>
                  {renderButtons(card.components.buttons)}
                </div>
              )}
            </BlockStack>
          </div>
        ))}
      </div>
    );
  };

  // Full WhatsApp-style preview with header
  if (variant === 'full' && showWhatsAppHeader) {
    return (
      <div style={{
        maxWidth: '350px',
        margin: '0 auto',
        border: '1px solid #e1e3e5',
        borderRadius: '12px',
        backgroundColor: '#ffffff',
        overflow: 'hidden',
        boxShadow: '0 2px 8px rgba(0,0,0,0.1)'
      }}>
        {/* WhatsApp Header */}
        <div style={{
          backgroundColor: '#075e54',
          color: 'white',
          padding: '12px 16px',
          display: 'flex',
          alignItems: 'center',
          gap: '12px'
        }}>
          <div style={{
            width: '32px',
            height: '32px',
            borderRadius: '50%',
            backgroundColor: '#128c7e',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            fontSize: '14px',
            fontWeight: 'bold'
          }}>
            B
          </div>
          <div>
            <Text variant="bodyMd" fontWeight="semibold"  style={{
              color: 'white'
            }}>
              Business Name
            </Text>
            <div style={{ fontSize: '12px', opacity: 0.8 }}>
              online
            </div>
          </div>
        </div>

        {/* Message Content */}
        <div style={{
          padding: '16px',
          backgroundColor: '#e5ddd5',
          minHeight: '100px'
        }}>
          <div style={{
            backgroundColor: '#ffffff',
            borderRadius: '8px',
            padding: '12px',
            boxShadow: '0 1px 2px rgba(0,0,0,0.1)',
            position: 'relative'
          }}>
            {/* Message bubble tail */}
            <div style={{
              position: 'absolute',
              left: '-8px',
              top: '10px',
              width: 0,
              height: 0,
              borderTop: '8px solid transparent',
              borderBottom: '8px solid transparent',
              borderRight: '8px solid #ffffff'
            }} />

            <BlockStack gap="200">
              {/* Handle carousel templates in full view */}
              {previewComponents.type === 'carousel' ? (
                <>
                  {previewComponents.body && (
                    <Text variant="bodySm" as="div">
                      {renderFormattedText(previewComponents.body)}
                    </Text>
                  )}
                  {renderCarousel(previewComponents.carousel)}
                  <Text variant="caption" as="div" tone="subdued" alignment="center">
                    Swipe to see more cards →
                  </Text>
                </>
              ) : (
                <>
                  {renderHeader(previewComponents.header, 'header_0')}
                  
                  {previewComponents.body && (
                    <Text variant="bodySm" as="div">
                      {renderFormattedText(previewComponents.body)}
                    </Text>
                  )}

                  {previewComponents.footer && (
                    <Text variant="caption" as="div" tone="subdued">
                      {renderFormattedText(previewComponents.footer)}
                    </Text>
                  )}

                  {renderButtons(previewComponents.buttons)}
                </>
              )}
            </BlockStack>

            {/* Timestamp */}
            <div style={{
              textAlign: 'right',
              marginTop: '8px',
              fontSize: '11px',
              color: '#667781'
            }}>
              {new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Handle carousel templates
  if (previewComponents.type === 'carousel') {
    return (
      <div style={{
        backgroundColor: '#fff',
        padding: '12px 16px',
        borderRadius: '4px 18px 18px 18px',
        boxShadow: 'rgba(0, 0, 0, 0.1) 0px 0px 13px',
        maxWidth: maxWidth
      }}>
        <BlockStack gap="200">
          {/* Main body text for carousel */}
          {previewComponents.body && (
            <Text variant="bodySm" as="div">
              {renderFormattedText(previewComponents.body)}
            </Text>
          )}
          
          {/* Carousel cards */}
          {renderCarousel(previewComponents.carousel)}
          
          <Text variant="caption" as="div" tone="subdued" alignment="center">
            Swipe to see more cards →
          </Text>
        </BlockStack>
      </div>
    );
  }

  // Simple bubble preview (default)
  return (
    <div style={{
      backgroundColor: '#fff',
      padding: '12px 16px',
      borderRadius: '4px 18px 18px 18px',
      boxShadow: 'rgba(0, 0, 0, 0.1) 0px 0px 13px',
      maxWidth: maxWidth
    }}>
      <BlockStack gap="200">
        {renderHeader(previewComponents.header, 'header_0')}
        
        {previewComponents.body && (
          <Text variant="bodySm" as="div">
            {renderFormattedText(previewComponents.body)}
          </Text>
        )}

        {previewComponents.footer && (
          <Text variant="caption" as="div" tone="subdued">
            {renderFormattedText(previewComponents.footer)}
          </Text>
        )}

        {renderButtons(previewComponents.buttons)}
      </BlockStack>
    </div>
  );
};

export default WhatsAppPreview;
