import React, { useState, useCallback, useRef, useEffect } from 'react';
import {
  Button,
  Popover,
  TextField,
  Text,
  InlineStack,
  BlockStack,
  Tabs,
  Scrollable,
  Tooltip,
  Box,
  Grid,
  Divider,
  EmptyState
} from '@shopify/polaris';
import {
  emojiCategories,
  searchEmojis,
  getFrequentlyUsedEmojis,
  getRecentlyUsedEmojis,
  addToRecentlyUsed
} from '../utils/emojiLibrary';

export function EmojiPicker({ onEmojiSelect, buttonSize = 'micro', disabled = false }) {
  const [popoverActive, setPopoverActive] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedTab, setSelectedTab] = useState(0);
  const [recentEmojis, setRecentEmojis] = useState(getRecentlyUsedEmojis());
  const buttonRef = useRef(null);

  const togglePopoverActive = useCallback(() => {
    setPopoverActive((popoverActive) => !popoverActive);
  }, []);

  const handleEmojiClick = useCallback((emoji) => {
    if (onEmojiSelect) {
      onEmojiSelect(emoji);
    }
    
    addToRecentlyUsed(emoji);
    setRecentEmojis(getRecentlyUsedEmojis());
    
    setPopoverActive(false);
  }, [onEmojiSelect]);

  const handleSearchChange = useCallback((value) => {
    setSearchQuery(value);
  }, []);

  const handleTabChange = useCallback((selectedTabIndex) => {
    setSelectedTab(selectedTabIndex);
    setSearchQuery(''); 
  }, []);

  const categoryTabs = [
    { id: 'recent', content: '🕒', accessibilityLabel: 'Recently Used' },
    { id: 'frequent', content: '⭐', accessibilityLabel: 'Frequently Used' },
    ...Object.entries(emojiCategories).map(([key, category]) => ({
      id: key,
      content: category.icon,
      accessibilityLabel: category.name
    }))
  ];

  const getEmojisToDisplay = () => {
    if (searchQuery) {
      return searchEmojis(searchQuery);
    }

    switch (selectedTab) {
      case 0: 
        return recentEmojis.length > 0 ? recentEmojis : getFrequentlyUsedEmojis();
      case 1: 
        return getFrequentlyUsedEmojis();
      default: 
        const categoryKey = Object.keys(emojiCategories)[selectedTab - 2];
        return categoryKey ? emojiCategories[categoryKey].emojis : [];
    }
  };

  const emojisToDisplay = getEmojisToDisplay();

  const activator = (
    <div ref={buttonRef}>
      <Tooltip content="Insert emoji">
        <Button
          size={buttonSize}
          onClick={togglePopoverActive}
          disabled={disabled}
          accessibilityLabel="Insert emoji"
        >
          😀
        </Button>
      </Tooltip>
    </div>
  );

  return (
    <Popover
      active={popoverActive}
      activator={activator}
      onClose={togglePopoverActive}
      ariaHaspopup={false}
      sectioned={false}
      preferredAlignment="left"
      preferredPosition="above"
    >
      <Box width="380px" maxHeight="450px">
        <BlockStack gap="0">
          {/* Search Bar */}
          <Box paddingInlineStart="300" paddingInlineEnd="300" paddingBlockStart="300" paddingBlockEnd="200">
            <TextField
              placeholder="Search emojis... like happy, sad, etc."
              value={searchQuery}
              onChange={handleSearchChange}
              clearButton
              onClearButtonClick={() => setSearchQuery('')}
              autoComplete="off"
            />
          </Box>

          {/* Category Tabs - Always visible */}
          <Box paddingInlineStart="300" paddingInlineEnd="300" paddingBlockEnd="200">
            <Scrollable style={{ maxWidth: '100%' }} horizontal>
              <InlineStack gap="100" wrap={false}>
                {categoryTabs.map((tab, index) => (
                  <Button
                    key={tab.id}
                    onClick={() => handleTabChange(index)}
                    variant={selectedTab === index ? "primary" : "tertiary"}
                    size="micro"
                    accessibilityLabel={tab.accessibilityLabel}
                  >
                    <Text variant="bodySm" as="span">
                      {tab.content}
                    </Text>
                  </Button>
                ))}
              </InlineStack>
            </Scrollable>
          </Box>

          {/* Category Name */}
          {!searchQuery && (
            <Box paddingInlineStart="300" paddingInlineEnd="300" paddingBlockEnd="200">
              <Text variant="headingSm" tone="subdued">
                {selectedTab === 0 && 'Recently Used'}
                {selectedTab === 1 && 'Frequently Used'}
                {selectedTab > 1 && Object.values(emojiCategories)[selectedTab - 2]?.name}
              </Text>
            </Box>
          )}

          <Divider />

          {/* Emoji Grid */}
          <Box paddingInlineStart="300" paddingInlineEnd="300" paddingBlockStart="200" paddingBlockEnd="300">
            <Scrollable style={{ height: '300px' }}>
              <Box paddingBlockEnd="200">
                {emojisToDisplay.length > 0 ? (
                  <div
                    style={{
                      display: 'grid',
                      gridTemplateColumns: 'repeat(9, 1fr)',
                      gap: 'var(--p-space-100)',
                    }}
                  >
                    {emojisToDisplay.map((emoji, index) => (
                      <Button
                        key={`${emoji}-${index}`}
                        onClick={() => handleEmojiClick(emoji)}
                        variant="plain"
                        size="micro"
                        textAlign="center"
                        accessibilityLabel={`Insert ${emoji} emoji`}
                        fullWidth
                      >
                        <Text variant="headingMd" as="span">
                          {emoji}
                        </Text>
                      </Button>
                    ))}
                  </div>
                ) : (
                  <Box paddingBlockStart="500" paddingBlockEnd="500">
                    <EmptyState
                      heading={searchQuery ? 'No emojis found' : 'No emojis available'}
                      image=""
                    >
                      <Text variant="bodySm" tone="subdued">
                        {searchQuery 
                          ? 'Try searching with different keywords' 
                          : 'Start using emojis to see them here'
                        }
                      </Text>
                    </EmptyState>
                  </Box>
                )}
              </Box>
            </Scrollable>
          </Box>
        </BlockStack>
      </Box>
    </Popover>
  );
}

export default EmojiPicker;
